#include "../h/conf.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/tty.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/buf.h"
#include "../h/vmcf.h"
#include "../h/ioconf.h"

/*
 * General VMCF driver
 * Supports 4 protocols:
 *    1) SEND (both ends)
 *    2) SEND/RECV (send end)
 *    3) SEND/RECV (receive/reply end)
 *    4) SENDX (both ends)
 * See The VM System Programmer's Guide (GC20-1807)
 */

#define VMCIPRI TTIPRI

struct vmc {
	char    v_user[8];      /* user with whom we're communicating */
	int     v_prot;         /* current protocol (see above) */
	int     v_state;        /* state bits */
	int     v_chan;         /* VMCF channel descriptor */
	int     v_mid;          /* message ID */
	caddr_t v_buf;          /* page size buffer */
	int     v_len;          /* length for reads */
} vmcdev[NVMC];

/* bits in v_state */
#define OPEN    01      /* this vmcf device is open */
#define SNDPND  02      /* there is a send pending */

/*
 * Open VMCF device
 */
/* ARGSUSED */
vmcopen(dev, flag)
dev_t dev;
{
	register struct vmc *v;

	if(minor(dev) >= NVMC) {
		u.u_error = ENXIO;
		return;
	}
	v = &vmcdev[minor(dev)];
	if(v->v_state & OPEN) {
		u.u_error = EBUSY;
		return;
	}
	v->v_state = OPEN;
	v->v_buf = getpage();
	v->v_chan = -1;
	v->v_len = 0;
}

/*
 * Close VMCF device
 */
vmcclose(dev)
dev_t dev;
{
	register struct vmc *v = &vmcdev[minor(dev)];

	if(v->v_chan != -1)
		vmc_drop(v->v_chan);
	freepag(v->v_buf);
	v->v_state = 0;
}

/*
 * Read from VMCF device
 */
vmcread(dev)
dev_t dev;
{
	register struct vmc *v = &vmcdev[minor(dev)];

	if(v->v_chan == -1)
		return;
	if(u.u_count > PSIZE) {
		u.u_error = EINVAL;
		return;
	}
	switch(v->v_prot) {

	case VMC_P1:    /* read = RECEIVE, write = SEND */
	case VMC_P3:    /* read = RECEIVE, write = REPLY */
		while((v->v_state & SNDPND) == 0)
			sleep((caddr_t)&v->v_state, VMCIPRI);
		if(v->v_len <= u.u_count) {
			if(vmc_recv(v->v_chan, v->v_mid, v->v_buf, v->v_len))
				u.u_error = EIO;
			else
				iomove(v->v_buf, v->v_len, B_READ);
		} else {
			vmc_rjct(v->v_chan, v->v_mid);
			u.u_error = EIO;
		}
		v->v_state &= ~SNDPND;
		break;

	case VMC_P2:    /* read = move data, write = SEND/RECV */
		if(v->v_len > 0) {
			iomove(v->v_buf, v->v_len, B_READ);
			v->v_len = 0;
		}
		break;

	case VMC_P4:    /* read = receive SENDX data, write = SENDX */
		while((v->v_state & SNDPND) == 0)
			sleep((caddr_t)&v->v_state, VMCIPRI);
		if(v->v_len > 0)
			iomove(v->v_buf, v->v_len, B_READ);
		v->v_state &= ~SNDPND;
		break;

	default:
		u.u_error = EINVAL;
		break;
	}
}

/*
 * Write to VMCF device
 */
vmcwrite(dev)
dev_t dev;
{
	register struct vmc *v = &vmcdev[minor(dev)];
	int resid;

	if(v->v_chan == -1)
		return;
	if(u.u_count > PSIZE) {
		u.u_error = EINVAL;
		return;
	}
	v->v_len = u.u_count;
	iomove(v->v_buf, v->v_len, B_WRITE);
	if(u.u_error)
		return;
	switch(v->v_prot) {

	case VMC_P1:    /* write = SEND, read = RECEIVE */
		if(vmc_send(v->v_chan, 0, v->v_buf, v->v_len))
			u.u_error = EIO;
		break;

	case VMC_P2:    /* write = SEND/RECV, read = move data */
		if(vmc_senr(v->v_chan, 0, v->v_buf, v->v_len, v->v_buf, PSIZE, &resid))
			u.u_error = EIO;
		else
			v->v_len = PSIZE - resid;
		break;

	case VMC_P3:    /* write = REPLY, read = RECEIVE */
		if(vmc_repl(v->v_chan, v->v_mid, v->v_buf, v->v_len))
			u.u_error = EIO;
		break;

	case VMC_P4:    /* write = SENDX, read = receive SENDX data */
		if(vmc_senx(v->v_chan, 0, v->v_buf, v->v_len))
			u.u_error = EIO;
		break;

	default:
		u.u_error = EINVAL;
		break;
	}
}

/*
 * ioctl for VMCF
 */
/* ARGSUSED */
vmcioctl(dev, cmd, argp, flag)
dev_t dev;
caddr_t argp;
{
	register struct vmc *v = &vmcdev[minor(dev)];
	int i, c, vmcintr();
	char *cp;

	switch(cmd) {

	case VMCUSER:   /* open a channel to a user */
		if(v->v_chan != -1) {
			vmc_drop(v->v_chan);
			v->v_chan = -1;
			v->v_state = 0;
		}
		cp = v->v_user;
		for(i=0; i<8; i++) {
			c = fubyte(argp);
			if(c == -1) {
				u.u_error = EFAULT;
				return;
			}
			*cp++ = c;
			if(c) argp++;
		}
		v->v_chan = vmc_catch(v->v_user, vmcintr, (int)v);
		break;

	case VMCPROT:   /* set protocol */
		v->v_prot = (int)argp;
		break;

	case VMCIDENT:  /* issue IDENTIFY */
		if(v->v_chan != -1)
			vmc_iden(v->v_chan);
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}

/*
 * Our VMCF interrupts come here
 */
vmcintr(v, h)
register struct vmc *v;
register struct vmcmhdr *h;
{
	switch(h->m_func) {

	case P_SEND:
		if(v->v_prot != VMC_P1 || (v->v_state&SNDPND) != 0 || h->m_lena > PSIZE)
			return(0);
		v->v_state |= SNDPND;
		v->v_mid = h->m_mid;
		v->v_len = h->m_lena;
		wakeup((caddr_t)&v->v_state);
		break;

	case P_SENR:
		if(v->v_prot != VMC_P3 || (v->v_state&SNDPND) != 0 || h->m_lena > PSIZE)
			return(0);
		v->v_state |= SNDPND;
		v->v_mid = h->m_mid;
		v->v_len = h->m_lena;
		wakeup((caddr_t)&v->v_state);
		break;

	case P_SENX:
		if(v->v_prot != VMC_P4 || (v->v_state&SNDPND) != 0 || h->m_lena > PSIZE)
			return(0);
		v->v_state |= SNDPND;
		v->v_len = h->m_lena;
		a_mvcl(h->m_buf, v->v_len, v->v_buf, v->v_len);
		wakeup((caddr_t)&v->v_state);
		break;

	default:
		return(0);
	}
	return(1);
}
