#include "../h/conf.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/io.h"
#include "../h/ioconf.h"
#include "../h/tty.h"

/*
 * IBM 3420 tape dirver.
 * This program has been modified to handle errors according to
 * the IBM manual GA32-0021.
 * The unit check handling is modelled after the tape unit check
 * routines in the MTS operating system written by Paul Whaley.
 *
 *                              Peter Madderom
 *                              October 1980
 */

#define CMASK      0xFF            /* tape command mask       */

/*
 * states of a tape
 */

#define OPENING  1
#define JUSTOPEN 2
#define OPEN     3
#define CLOSED   4

#define FREE       0
#define BUSY       1
#define DONE       2
#define ERROR      3
#define WAITING    4

#define TAPPRI (PZERO-1)
#define OPENPRI (PZERO+1)

/*
 * Tape Burst Commands
 */

#define WRITE      0x01            /* write data on tape                */
#define WRITETM    0x1F            /* write tape mark                   */
#define READ       0x02            /* read data from tape               */
#define READBKWD   0x0C            /* read data backwards               */

/*
 * Motion Control Commands
 */

#define REWIND     0x07            /* rewind tape                       */
#define REWUNL     0x0F            /* rewind/unload tape                */
#define BKSPBLK    0x27            /* backspace block                   */
#define FWDSPBLK   0x37            /* forward space block               */
#define FWDSPFL    0x3f            /* forward space file */
#define BKSPFL     0x2f            /* backspace file */
#define SENSE      0x04            /* sense */

#define ERG        0x17            /* erase gap */
#define TIC        0x08            /* transfer in channel */
#define TIE        0x1b            /* request track in error byte */
#define NOP        0x03            /* no operation */
#define DSE        0x97            /* data security erase */

/*
 * 9 track mode set commands
 */
#define M800       0xcb            /* 800 bpi */
#define M1600      0xc3            /* 1600 bpi */
#define M6250      0xd3            /* 6250 bpi */

/*
 * Tape motion masks for open and close
 */

#define REWOPEN     020             /* rewind on open                   */
#define REWCLOSE    040             /* rewind on close                  */
#define UNLCLOSE   0100             /* rewind/unload on close           */
#define REWOPCL     060             /* default - rewind on open & close */

/*
 * Bit definitions in sense bytes
 */

/* Sense byte 0 */
#define CMDREJ  0x80           /* command reject */
#define INTREQ  0x40           /* intervention required */
#define BUSOUT  0x20           /* bus out check */
#define EQCHK   0x10           /* equipment check */
#define DATACHK 0x08           /* data check */
#define OVERRUN 0x04           /* over run */
#define DCCK    0x01           /* data converter check */

/* Sense byte 1 */
#define NOISE   0x80           /* on if record is not noise */
#define STATUSA 0x40           /* tape unit status A */
#define STATUSB 0x20           /* tape unit status B */
#define TRACK7  0x10           /* seven track tape */
#define LDPNT   0x08           /* load point */
#define WRITESTS 0x04          /* write status */
#define FILPRT  0x02           /* file protect */
#define NOTCAP  0x01           /* not capable */

/* Sense byte 3 */
#define BACKWARD 0x02          /* backward status */

/* Sense byte 5 */
#define IDBCHK  0x10           /* ID burst check */

/* Sense byte 7 */
#define TPBTML  0x40           /* tape bottomed on left */
#define DSECHK  0x08           /* data security erase check */


#define READRETRY 41           /* Or so says IBM */

/*
 *   Tape error codes
 */

#define T_BUSOUT    1         /* busout check */
#define T_CHANERR   2         /* channel error */
#define T_CMDREJ    3         /* command reject */
#define T_DATACHK   4         /* data check */
#define T_DCCK      5         /* data converter check */
#define T_DSECHK    6         /* data security check */
#define T_EOREEL    7         /* went off the end of the reel */
#define T_EQCHK     8         /* equipment check */
#define T_IDBCHK    9         /* ID burst check */
#define T_INOP     10         /* not operational */
#define T_INVALID  11         /* invalid status information */
#define T_LDPNT    12         /* tape at loadpoint */
#define T_NODRIVE  13         /* no drive at this address */
#define T_NOTCAP   14         /* not capable */
#define T_OVERRUN  15         /* overrun */
#define T_PROGC    16         /* program check */
#define T_PROTC    17         /* protection check */
#define T_RDERR    18         /* permanent read error */
#define T_WRTERR   19         /* permanent write error */
#define T_WWP      20         /* write on write protected tape */
#define T_INTREQ   21         /* intervention required on drive */
#define T_ZCCW     22         /* zero ccw - shouldn't happen */

/*
 * Tape descriptor
 */

struct tapes {
	ccw_t   tp_ccw;            /* tape ccw - double aligned         */
	ccw_t   tp_rorccw;         /* read opposite recovery and other  */
	ccw_t   tp_tieccw;         /* mode set or TIE ccw               */
	ccw_t   tp_ticccw;         /* TIC or control ccw                */
	tod_t   tp_time;           /* for logging time of error */
	csw_t   tp_initcsw;        /* saved csw */
	csw_t   tp_curcsw;         /* current csw */
	char    tp_sense[24];      /* saved sense info */
	char    tp_cursense[24];   /* current sense info */
	caddr_t tp_idaws[32];      /* indirect address words            */
	int     tp_open;           /* determine if the tape is open     */
	int     tp_state;          /* state of the tape                 */
	int     tp_resid;          /* residual read count               */
	int     tp_addr;           /* device address                    */
	int     tp_mode;           /* density (modeset comand)          */
	int     tp_motion;         /* open/close tape motion flag       */
	int     tp_ucretry;        /* retry count */
	int     tp_error;          /* for later error reporting */
	ccw_t   *tp_address;       /* address of ccw in error */
	short   tp_back;           /* count for clean back spaces */
	short   tp_frwrd;          /* count for clean forward spaces */
	char    tp_ucflag;         /* flag for uc recovery */
	char    tp_tiebyte;        /* track in error byte */
	char    tp_seqflag;        /* flag to prevent illegal op sequences */
	char    tp_full;           /* flag is one if tape is full */
} tp_info[NTAPE];

/* Bits in ucflag */
#define UC_RETRY  0x80             /* currently in retry */
#define UC_ROR    0x40             /* currently in read opposite recovery */
#define UC_ISS    0x20             /* initial status saved */
#define UC_NOLOG  0x10             /* do not log this error */
#define UC_UXEG   0x08             /* unit exception on erase gap */
#define UC_ROR1   0x04             /* ROR read once successfully */
#define UC_CERR   0x02             /* channel detected error */
#define UC_WAIT   0x01             /* wait bit for uc i/o */

/* Bits in seqflag */
#define S_WRITE   0x80
#define S_READ    0x40
#define S_WTM     0x20
#define S_LDPNT   0x10

extern tod_t a_stck();

/*
 * Open a tape device.
 * If state not FREE,  returns error to calling program.
 */
/* ARGSUSED */
tapopen(dev, flag)
int dev, flag;
{
	struct tapes *tp;
	int tptiointr();

	tp = &tp_info[minor(dev)];
	if (tp->tp_open == 0) {     /* Never opened since ipl. */
		tp->tp_motion = REWOPCL;
		tp->tp_mode = M1600;
	} else if (tp->tp_open != CLOSED && tp->tp_open != OPENING) {
		u.u_error = EBUSY;
		return;
	}
	tp->tp_open = OPENING;
	tp->tp_error = 0;
	tp->tp_addr = cdevsw[major(dev)].d_addrs[minor(dev)];
	tio(tp->tp_addr, tptiointr, (int) tp);
	tp->tp_state = FREE;
	if (tp->tp_open == OPENING)
		u.u_error = ENXIO;
}

/*
 * Close a tape device.
 * Issues a write tapemark operation if the previous operation was
 * a write.  Issues rewind, rewind/unload or does nothing depending
 * on the setting of bits in the tape motion flag.  Sets state to
 * FREE, and zeroes the  mode.
 */

/* ARGSUSED */
tapclose(dev,flag)
int dev, flag;
{
       struct tapes *tp;

	tp = &tp_info[minor(dev)];
	tp->tp_ccw.cc_dblw = 0L;
	if (tp->tp_seqflag & S_WRITE) {
		tp->tp_ccw.cc_cmd = WRITETM;    /* write tapemark */
		tp->tp_ccw.cc_count = 1;
		tpsio(tp);
		tp->tp_seqflag = S_WTM;
		if (tp->tp_error) {
			tp->tp_state = FREE;
			tp->tp_open = CLOSED;
			return;
		}
	}
	if (tp->tp_motion & REWCLOSE) {
		if (tp->tp_seqflag != S_LDPNT) {
			tp->tp_ccw.cc_cmd = REWIND;
			tp->tp_ccw.cc_count = 1;
			tpsio(tp);
			tp->tp_seqflag = S_LDPNT;
			tp->tp_full = 0;
		}
	} else if (tp->tp_motion & UNLCLOSE) {
		tp->tp_ccw.cc_cmd = REWUNL;
		tp->tp_ccw.cc_count = 1;
		tpsio(tp);
		tp->tp_seqflg = 0;
	}
	tp->tp_state = FREE;         /* free up device */
	tp->tp_open = CLOSED;
}

/*
 * Read from tape.
 * Reads data directly from tape into the user process buffer.
 * The external routine, makidaw, is called to construct a channel
 * indirect address word list.  The variables u_base and u_count
 * are used to construct this list.  The variable u_error is set
 * by makidaw if construction of the list fails.  The read routine
 * calls the internal routine, tpsio, to issue the SIO request
 * for the read.
 */

tapread(dev)
int dev;
{
	struct tapes *tp;

	tp = &tp_info[minor(dev)];
	if (tp->tp_open == CLOSED) {
		u.u_error = EBADF;
		return;
	}
	if (tp->tp_seqflag & (S_WRITE | S_WTM)) {
		u.u_error = EINVAL;
		return;
	}
	if (tp->tp_open == JUSTOPEN) {
		if (tp->tp_motion & REWOPEN)
			t_io(tp, REWIND);
	        tp->tp_open = OPEN;
	}
	makidaw(0, tp->tp_idaws, u.u_base, u.u_count);
	if (u.u_error)
		return;             /* construction of word list failed */
	tp->tp_ccw.cc_dblw = 0;
	tp->tp_ccw.cc_cmd = READ;
	tp->tp_ccw.cc_addr = (int)tp->tp_idaws;
	tp->tp_ccw.cc_ida = 1;
	tp->tp_ccw.cc_count = u.u_count;
	tpsio(tp);                   /* issue read request */
	tp->tp_state = BUSY;
	tp->tp_seqflag = S_READ;     /* remember last operation */
	u.u_count = tp->tp_resid; /* set u_count to residual read count */
}

/*
 * Write to tape.
 * Writes data directly to tape from the user process buffer.
 * The external routine, makidaw, is called to construct a channel
 * indirect address word list.  The variables u_base and u_count
 * are used to construct this list.  The variable u_error is set
 * by makidaw if construction of the list fails.  The write routine
 * calls the internal routine, tpsio, to issue the SIO request
 * for the write.
 */

/* ARGSUSED */
tapwrite(dev,flag)
int dev;
{
	struct tapes *tp;

	tp = &tp_info[minor(dev)];
	if (tp->tp_open == CLOSED) {
		u.u_error = EBADF;
		return;
	}
	if (tp->tp_seqflag & S_READ) {
		u.u_error = EINVAL;
		return;
	}
	if (tp->tp_open == JUSTOPEN) {
		if (tp->tp_motion & REWOPEN)
			t_io(tp, REWIND);
	        tp->tp_open = OPEN;
	}
	if (tp->tp_full) {
		u.u_error = ENXIO;
		return;
	}
	makidaw(1, tp->tp_idaws, u.u_base, u.u_count);
	if (u.u_error)
		return;             /* construction of word list failed */
	tp->tp_ccw.cc_dblw = 0;
	tp->tp_ccw.cc_cmd = WRITE;
	tp->tp_ccw.cc_addr = (int)tp->tp_idaws;
	tp->tp_ccw.cc_ida = 1;
	tp->tp_ccw.cc_count = u.u_count;
	if (tp->tp_seqflag & S_LDPNT) {  /* the only time to do modeset */
		tp->tp_rorccw.cc_dblw = tp->tp_ccw.cc_dblw;
		tp->tp_rorccw.cc_sli = 1;
		tp->tp_ccw.cc_dblw = 0L;
		tp->tp_ccw.cc_cmd =tp->tp_mode;  /* use proper command */
		tp->tp_ccw.cc_count = 1;
		tp->tp_ccw.cc_cc = 1;       /* must be chained to write */
	}
	tpsio(tp);
	u.u_count = tp->tp_resid;     /* in case of errors */
	tp->tp_seqflag = S_WRITE;     /* remember last operation */
	tp->tp_state = BUSY;
}

/*
 * tio interrupt routine - used during open.
 */
/* ARGSUSED */
tptiointr(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	int tpaint();

	if (csw->cs_cc == 0)
		tp->tp_open = JUSTOPEN;
}

/*
 * Start I/O for the tape
 * Issues an SIO request for read or write to tape.
 * After the SIO request is issued, the process sleeps until
 * awakened by the interrupt routine, tpintr, and the state is
 * something other than BUSY.
 */

tpsio(tp)
struct tapes *tp;
{
	int tpintr();

	tp->tp_ccw.cc_sli = 1;
	tp->tp_state = BUSY;
	sio(tp->tp_addr, &tp->tp_ccw, tpintr, (int)tp);
	while (tp->tp_state == BUSY)
		sleep((caddr_t)tp, TAPPRI);
	while (tp->tp_state == ERROR)
		tapeuc(tp, &tp->tp_curcsw, tp->tp_cursense);
}


/*
 * Interrupt handlers for tape SIO requests.
 * It saves the status of the operation and wakes up tpsio
 * to handle any error conditions.
 * This means that the unit check recovery is done at a level
 * that allows waits.
 */

tpintr(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	int i;

	if (csw->cs_ce) {
	        tp->tp_address = (ccw_t *) csw->cs_addr;
	        tp->tp_resid = csw->cs_count;
	}
	if (csw->cs_de)
		tp->tp_state = DONE;
	/*
	 * Now check for errors and save status if any occurred.
	 */
	if ( (csw->cs_cc ==3) || (csw->cs_word2 & CHANCHEK) ||
	     csw->cs_uc || csw->cs_ue || csw->cs_progc || csw->cs_protc) {
		tp->tp_state = ERROR;  /* causes tpsio to call tapeuc */
		tp->tp_time = a_stck();
		tp->tp_curcsw.cs_dblw = tp->tp_initcsw.cs_dblw = csw->cs_dblw;
		for (i = 0; i < 24; i++)
			tp->tp_cursense[i] = tp->tp_sense[i] = sense[i];
		tp->tp_ucflag = UC_ISS;
		/*
		 * fix ccws for error recovery
		 */
		tp->tp_tieccw.cc_dblw = 0L;
		tp->tp_tieccw.cc_cmd = NOP;
		tp->tp_tieccw.cc_cc = 1;
		tp->tp_tieccw.cc_sli = 1;
		tp->tp_tieccw.cc_count = 1;
		tp->tp_ticccw.cc_dblw = 0L;
		tp->tp_ticccw.cc_cmd = NOP;
		tp->tp_ticccw.cc_sli = 1;
		tp->tp_ticccw.cc_count = 1;
	}
	wakeup((caddr_t) tp);     /* wakeup tpsio routine */
}

/*
 * Tape unit check (and other errors) handler
 */
tapeuc(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	int cmd;

	/*
	 * Some fatal errors first
	 */
	if (csw->cs_cc == 3) {
		if (tp->tp_open == OPEN)
			t_error(tp, T_INOP);
		else {                  /* tape not mounted yet */
                        u.u_error = ENXIO;
			tp->tp_error = T_INOP;
                        tp->tp_state = 0;
		}
		return;
	}
	if (csw->cs_progc) {
		t_error(tp, T_PROGC);
		return;
	}
	if (csw->cs_protc) {
		t_error(tp, T_PROTC);
		return;
	}
	/*
	 *  Check for unit exception
	 */
	if (csw->cs_ue) {
		tapeue(tp);
		return;
	}
	/*
	 * Get command in error
	 */
	if (tp->tp_address == 0) {
		t_error(tp, T_ZCCW);
		return;
	}
	tp->tp_address--;              /* back up one ccw */
	cmd = (int)tp->tp_address->cc_cmd;
	/*
	 * Look at the many bits in the correct order
	 */
	if (csw->cs_uc) {
		/*
		 * Equipment check
		 */
		if (sense[0] & EQCHK) {
			t_eqchk(tp, csw, sense);
			return;
		}
		/*
		 * Bus out check
		 */
		if (sense[0] & BUSOUT) {
			t_busout(tp, csw, sense, cmd);
			return;
		}
	       /*
		* Intervention required
		*/
	       if (sense[0] & INTREQ) {
			if (!csw->cs_de) {
				if (sense[1] & STATUSB) {
					t_error(tp, T_INTREQ);
					t_intreq(tp);        /* wait for de */
					return;
				}
				t_error(tp, T_NODRIVE);
				return;
			}
			if (cmd == REWUNL) {
				tp->tp_state = DONE;       /* ignore not ready */
				return;
			}
		}
	       /*
		* Command reject
		*/
	       if (sense[0] & CMDREJ) {
			t_cmdrej(tp, csw, sense, cmd);
			return;
	       }
	       /*
		* Overrun
		*/
	       if (sense[0] & OVERRUN) {
			t_datachk(tp, csw, sense, cmd);
			return;
	       }
	       /*
		* Load point
		*/
	       if (sense[1] & LDPNT) {
			t_ldpnt(tp, csw, sense);
			return;
	       }
	       /*
		* Data check
		*/
	       if (sense[0] & DATACHK) {
			t_datachk(tp, csw, sense, cmd);
			return;
	       }
	       /*
		* Data security erase
		*/
	       if (sense[7] & DSECHK) {
			t_error(tp, T_DSECHK);
			return;
		}
	       /*
		* Channel data check
		*/
	       if (csw->cs_cdc) {
			t_overrun(tp, csw, sense, cmd);
			return;
		}
	       /*
		* Data converter check
		*/
	       if (sense[0] & DCCK) {
			t_error(tp, T_DCCK);
			return;
		}
	       /*
		* Not capable
		*/
	       if (sense[1] & NOTCAP) {
			t_error(tp, T_NOTCAP);
			return;
		}
		/*
		 * Id burst check
		 */
		if (sense[5] & IDBCHK) {
			t_idbchk(tp, csw, sense, cmd);
			return;
		}
		/*
		 * Invalid sense or status
		 */
		t_error(tp, T_INVALID);
		return;
	}
	/*
	 * Look at channel errors
	 */
	if (csw->cs_cdc || csw->cs_chc) {
		/*
		 * treat channel data check or chaining check as overrun
		 */
		tp->tp_ucflag |= UC_CERR;
		sense[3] &= ~BACKWARD;            /* zero a bit */
		if (cmd == READBKWD)
			sense[3] |= BACKWARD;
		t_overrun(tp, csw, sense, cmd);
		return;
	}
	t_error(tp, T_CHANERR);
	return;
}

/*
 * Equipment check handler
 */
/* ARGSUSED */
t_eqchk(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	/*
	 * Check if we went off the end of the reel
	 */
	if (sense[0] & INTREQ) {
		t_error(tp, T_INTREQ);
		return;
	}
	t_error(tp, T_EQCHK);
	return;
}

/*
 * Busout handler
 */
t_busout(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	int t_count(), t_repos(), t_retry();

	if (t_count(tp, 6)) {         /* retry six times */
		if (csw->cs_de) {
		       if (sense[1] & WRITESTS) {
				/*
				 * if writing must first reposition tape
				 */
				if (t_repos(tp, csw, sense, cmd)) {
					t_retry(tp);
					return;
				}
			}
		}
		t_retry(tp);                /* retry operation */
		return;
	}
	t_error(tp, T_BUSOUT);
	return;
}

/*
 * Intervention required handler.
 * Set up exit for de, give operator message, wait for
 * interrupt, reset exit, and retry operation.
 */
t_intreq(tp)
struct tapes *tp;
{
	int t_intexit();

	setax(tp->tp_addr, t_intexit, (int)tp);
	printf("intervention required on tape drive %3x\n", tp->tp_addr);
	tp->tp_ucflag |= (UC_WAIT | UC_NOLOG);   /* don't log not ready */
	while (tp->tp_ucflag & UC_WAIT)
		sleep((caddr_t)&tp->tp_ucflag, TAPPRI);
	setax(tp->tp_addr, (int(*)())0, 0);
	t_retry(tp);
	return;
}

/*
 * Asynchronous interrupt hanlder for intervention required
 */
/* ARGSUSED */
t_intexit(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	if (csw->cs_de) {
		tp->tp_ucflag &= ~UC_WAIT;
		wakeup((caddr_t) &tp->tp_ucflag);
	}
}

/*
 * Command reject handler
 */
/* ARGSUSED */
t_cmdrej(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	/*
	 * Check if writing on file protected tape
	 */
	if (sense[1] & FILPRT) {
		switch (cmd) {
		case WRITE:
		case WRITETM:
		case ERG:
			t_error(tp, T_WWP);
			return;
		}
	}
	t_error(tp, T_CMDREJ);
}

/*
 * Overrun, channel data check, or chaining check routine
 */
t_overrun(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	int t_count(), t_repos(), t_retry();

	if (t_count(tp, 6)) {      /* retry six times */
		switch (cmd) {
		case READ:
		case READBKWD:
		case WRITE:
			if (t_repos(tp, csw, sense, cmd)) {
				t_retry(tp);
				return;
			} else
				goto error;
		default:
			t_retry(tp);
			return;
		}
	}
    error:
	switch (cmd) {
	case READ:
	case READBKWD:
		t_error(tp, T_RDERR);
		break;
	case WRITE:
		t_error(tp, T_WRTERR);
		break;
	default:
		if (tp->tp_ucflag & UC_CERR)
			t_error(tp, T_CHANERR);
		else
			t_error(tp, T_OVERRUN);
	}
}

/*
 * Load point routine
 */
/* ARGSUSED */
t_ldpnt(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	if (sense[3] & BACKWARD) {
		tp->tp_error = T_LDPNT;
		tp->tp_state = DONE;
		tp->tp_seqflag = S_LDPNT;   /* remember where we are */
		return;               /* don't give error return */
	}
	t_error(tp, T_LDPNT);
}

/*
 * Data check routine (the big one)
 */
t_datachk(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	int done;
	int t_clean(), t_count(), t_ergap(), t_noise(), t_repos(), t_retry();

	switch (cmd) {
	case READ:
	case READBKWD:
		if (t_count(tp, READRETRY)) {        /* many times */
			done = 1;
			if (t_noise(tp, csw, sense)) {  /* true if not noise */
				if (tp->tp_ucretry%4)
					done = t_repos(tp, csw, sense, cmd);
				else
					done = t_clean(tp, csw, sense); /* every fourth one */
			}
		        if (!done) {
		                t_error(tp, T_RDERR);
			        return;
		        }
			if (!(sense[1] & TRACK7)) {
				tp->tp_tiebyte = sense[2];  /* save track in error byte */
				tp->tp_tieccw.cc_addr = (int)&tp->tp_tiebyte;
				tp->tp_tieccw.cc_cmd = TIE;
			}
			t_retry(tp);
			return;
		}
		/*
		 * Check if it is possible to do read opposit recovery
		 */
		if ((tp->tp_ucflag & UC_ROR) ||
		    (tp->tp_address != &tp->tp_ccw) ||
		    tp->tp_ccw.cc_cd || tp->tp_ccw.cc_skip ||
		    (sense[1] & TRACK7) ) {
			t_error(tp, T_RDERR);
			return;
		}
	       /*
		* Only handle forward read case because that's all
		* the tape driver handles
		*/
	       flipidaws(tp);         /* fix idaws for backward read */
	       tp->tp_rorccw.cc_dblw = 0L;
	       tp->tp_rorccw.cc_cmd = READBKWD;
	       tp->tp_rorccw.cc_addr = tp->tp_ccw.cc_addr;
	       tp->tp_rorccw.cc_count = tp->tp_ccw.cc_count;
	       tp->tp_rorccw.cc_skip = 1;
	       tp->tp_rorccw.cc_ida = 1;
	       tp->tp_address = &tp->tp_rorccw;
	       tp->tp_ucretry = 0;       /* use same number of tries */
	       tp->tp_ucflag |= UC_ROR;
	       t_retry(tp);
	       return;

	case WRITE:
	case WRITETM:
		if (t_count(tp, 16)) {    /* retry sixteen times */
			if (t_repos(tp, csw, sense, cmd))
				if (t_ergap(tp)) {
					t_retry(tp);
					return;
				}
			t_error(tp, T_WRTERR);
			return;
		}
		t_error(tp, T_WRTERR);
		return;

	default:
		if (t_count(tp, 6)) {       /* retry the rest six times */
			t_retry(tp);
			return;
		}
		if (cmd == ERG)
			t_error(tp, T_WRTERR);
		else
			t_error(tp, T_DATACHK);
		return;
	}
}

/*
 * Routine to clean up after a successfull ROR
 */
t_ror(tp)
struct tapes *tp;
{
	int *idaw;
	int count, end, length;
	/*
	 * first check if finished skip read
	 */
	count = (int)tp->tp_curcsw.cs_count;
	if (tp->tp_rorccw.cc_skip) {
		if (tp->tp_curcsw.cs_il && (count == 0)) {
			t_error(tp, T_RDERR);
			return;           /* can't recover long records */
		}
		if (count) {        /* short records are more difficult */
			end = u.u_count - count;
			tp->tp_rorccw.cc_count = end;  /* set new count in ccw */
			end += (int) u.u_base;         /* point past end of record */
			end--;
			end &= 2047;          /* low order bit of last byte */
			idaw = (int *) tp->tp_idaws;
			while (count > 0) {
				length = *idaw;
				length &= 2047;     /* get low order bits */
				length++;
				count -= length;
				idaw++;
			}
			if (count < 0)
				idaw--;
			tp->tp_rorccw.cc_addr = (int)idaw;    /* set address */
			*idaw &= ~2047;                   /* zero low order bits */
			*idaw |= end;                     /* point to last byte */
		}
		tp->tp_rorccw.cc_skip = 0;          /* get the record */
		if (t_repos(tp, &tp->tp_curcsw, tp->tp_cursense, READBKWD)) {
			tp->tp_ucflag |= UC_ROR1;
			t_retry(tp);
			return;
		}
		t_error(tp, T_RDERR);
		return;
	}
	if (tp->tp_curcsw.cs_il) {
		t_error(tp, T_RDERR);
		return;                /* can't handle wrong length */
	}
	tp->tp_resid = u.u_count;
	tp->tp_resid -= tp->tp_rorccw.cc_count;
	tp->tp_resid += tp->tp_curcsw.cs_count; /* fixed resid count */
	if (t_repos(tp, &tp->tp_curcsw, tp->tp_cursense, READBKWD)) {
		tp->tp_ucretry += READRETRY;    /* add forward retries */
		t_tapelog(tp, 1);            /* log recovered error */
		tp->tp_ucflag &= ~UC_ROR;
		tp->tp_tieccw.cc_cmd = NOP;
		tp->tp_state = DONE;
		return;
	}
	t_error(tp, T_RDERR);
	return;
}

/*
 * ID burst check routine
 */
/* ARGSUSED */
t_idbchk(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	int t_count(), t_retry(), t_rewind();

	switch(cmd) {
	case WRITE:
	case ERG:
	case WRITETM:
		if (t_count(tp, 16)) {         /* retry sixteen times */
			if (t_rewind(tp)) {
				t_retry(tp);
				return;
			}
		}
	default:
		t_error(tp, T_IDBCHK);
		return;
	}
}

/*
 *  Routine to rewind tape
 */
t_rewind(tp)
struct tapes *tp;
{
	int t_ucsio();

	tp->tp_ticccw.cc_cmd = REWIND;
	if (t_ucsio(tp)) {
		tp->tp_seqflag = S_LDPNT;
		tp->tp_full = 0;
		return(1);
	} else
		return(0);
}

/*
 *  Routine to do sio for uc recovery. Returns 1 if no error.
 */
t_ucsio(tp)
struct tapes *tp;
{
       int t_ucintr();

       tp->tp_ucflag |= UC_WAIT;
       sio(tp->tp_addr, &tp->tp_tieccw, t_ucintr, (int)tp);
       while (tp->tp_ucflag & UC_WAIT)
		sleep((caddr_t)&tp->tp_ucflag, TAPPRI);
       /*
	*   check for errors
	*/
       if ((tp->tp_curcsw.cs_cc == 3) || tp->tp_curcsw.cs_uc ||
	   (tp->tp_curcsw.cs_word2 & CHANCHEK) ||
	    tp->tp_curcsw.cs_ue || tp->tp_curcsw.cs_progc ||
	    tp->tp_curcsw.cs_protc)
		return(0);
	else
		return(1);
}

/*
 * Interrupt handler for ucsio.  Just does a wakeup.
 */
t_ucintr(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	int i;

	tp->tp_curcsw.cs_dblw = csw->cs_dblw;
	for (i = 0; i < 24; i++)
		tp->tp_cursense[i] = sense[i];
	tp->tp_ucflag &= ~UC_WAIT;
	wakeup((caddr_t) &tp->tp_ucflag);
}

/*
 * Routine to keep track of retry count
 */
t_count(tp, max)
struct tapes *tp;
int max;
{
	if (!(tp->tp_ucflag & UC_RETRY)) {
		tp->tp_ucretry = 0;
		tp->tp_ucflag |= UC_RETRY;
	}
	if (max > tp->tp_ucretry) {
		tp->tp_ucretry++;
		return(1);
       } else
		return(0);
}

/*
 * Routine to reposition tape
 */
/* ARGSUSED */
t_repos(tp, csw, sense, cmd)
struct tapes *tp;
csw_t *csw;
char *sense;
int cmd;
{
	int t_ucsio();

	if (tp->tp_ucflag &UC_ROR) {   /* fix sense info */
		switch(cmd) {
		case READ:
			sense[3] &= ~BACKWARD;
			break;
		case READBKWD:
			sense[3] |= BACKWARD;
			break;
		}
	}
	if (sense[3] & BACKWARD)
		tp->tp_ticccw.cc_cmd = FWDSPBLK;
	else
		tp->tp_ticccw.cc_cmd = BKSPBLK;
	if (t_ucsio(tp))
		return(1);
	if (tp->tp_curcsw.cs_ue)
		return(1);      /* ignore tapemarks */
	if (tp->tp_curcsw.cs_uc &&
	    (tp->tp_cursense[1] & LDPNT) &&
	    (tp->tp_sense[1] & WRITESTS))
		return(1);                 /* ignore loadpoint on write */
	return(0);
}

/*
 * Routine to check if record is just noise
 */
t_noise(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	if (sense[1] & NOISE)
		return(1);
	if (tp->tp_address->cc_count > 12)
		return(1);
	if ((tp->tp_address->cc_count - csw->cs_count) > 11)
		return(1);
	return(0);
}

/*
 * Routine to do the cleaning sequence
 */
/* ARGSUSED */
t_clean(tp, csw, sense)
struct tapes *tp;
csw_t *csw;
char *sense;
{
	int t_ucsio();

	if (tp->tp_ucflag & UC_ROR)
		sense[3] |= BACKWARD; /* see repos */
	if (sense[3] & BACKWARD) {
		tp->tp_back = 4;
		tp->tp_frwrd = 5;
	} else {
		tp->tp_back = 5;
		tp->tp_frwrd = 4;
	}
	tp->tp_ticccw.cc_cmd = BKSPBLK;

	for ( ; tp->tp_back; tp->tp_back--) {
		if (!t_ucsio(tp)) {
			if (tp->tp_curcsw.cs_uc &&
			    (tp->tp_cursense[1] & LDPNT))
				goto loadpnt;
			if (!tp->tp_curcsw.cs_ue)
				return(0);
		}
	}

    loadpnt:
	tp->tp_frwrd -= tp->tp_back;   /* normally tp_back will be zero */
	tp->tp_ticccw.cc_cmd = FWDSPBLK; /* now we go forward */
	for ( ; tp->tp_frwrd; tp->tp_frwrd--) {
		if (!t_ucsio(tp)) {
			if (!tp->tp_curcsw.cs_ue)   /* ignore unit exception */
			return(0);
		}
	}
	return(1);
}

/*
 * Routine to do an erase gap
 */
t_ergap(tp)
struct tapes *tp;
{
	int t_ucsio();

	tp->tp_ticccw.cc_cmd = ERG;
	if (t_ucsio(tp)) return(1);
	if (tp->tp_curcsw.cs_ue) {
		tp->tp_ucflag |= UC_UXEG;
		return(1);
	}
	return(0);
}

/*
 * Routine which retries the failed operation.
 * If the retry is successful it logs that fact and returns.
 * Else it leaves the error flag on and causes tpsio to
 * call the uc routine again.
 */
t_retry(tp)
struct tapes *tp;
{
	int t_ucsio();

	if ((tp->tp_address != &tp->tp_tieccw) &&
	   (tp->tp_address != &tp->tp_ticccw)) {
		tp->tp_ticccw.cc_addr = (int)tp->tp_address;
		tp->tp_ticccw.cc_cmd = TIC;
	}
	if (t_ucsio(tp)) {
		if (tp->tp_ucflag & UC_ROR) {
			t_ror(tp);
			return;
		}
		t_tapelog(tp, 1);     /* log recovered error */
		if (tp->tp_ucflag & UC_UXEG) {
			tapeue(tp);
			return;
		}
		tp->tp_resid = (int)tp->tp_curcsw.cs_count;
		tp->tp_state = DONE;
		return;               /* finally done */
	}
	tp->tp_state = ERROR;
	return;                   /* let tapeuc handle the error */
}

/*
 * Unit exception handler
 */
tapeue(tp)
struct tapes *tp;
{
	switch(tp->tp_ccw.cc_cmd) {
	case WRITE:
	case ERG:
	case WRITETM:
		tp->tp_full = 1;
		u.u_error = ENOSPC;            /* full tape */
	default:
		tp->tp_state = DONE;
	}
	return;                   /* note ue on read gives zero bytes */
}

/*
 *  Error handler
 */
t_error(tp, flag)
struct tapes *tp;
int flag;
{
	tp->tp_error = flag;    /* allow user to look at this some time */
	if (tp->tp_ucflag & UC_ROR) {
		tp->tp_ucretry += READRETRY;
		if (flag == T_RDERR)
			t_repos(tp, &tp->tp_curcsw, tp->tp_cursense,READBKWD);
		/* this positions tape past bad record */
	}
        u.u_error = EIO;         /* really should be 'flag' */
	t_tapelog(tp, 0);        /* log the fatal error */
	tp->tp_state = DONE;
	return;
}

/*
 * Dummy error logging routine. Writes to operator console for now.
 */
t_tapelog(tp, flag)
struct tapes *tp;
int flag;
{
	int i;

	if (tp->tp_ucflag & UC_NOLOG)
		return;
	if (flag)
		printf("Recovered");
	else
		printf("Fatal");
	printf(" i/o error on tapedrive %3x.", tp->tp_addr);
	printf(" Csw: %8x", tp->tp_initcsw.cs_word1);
	printf(" %8x\nSense:", tp->tp_initcsw.cs_word2);
	for (i = 0; i < 24; i++)
		printf("%2x", tp->tp_sense[i]);
	printf(" retry count: %d", tp->tp_ucretry);
	if (!flag)
		printf(" error type: %d", tp->tp_error);
	printf("\n");
}

/*
 *  Routine to reverse the idaw list
 */
flipidaws(tp)
struct tapes *tp;
{
	caddr_t temp;
	int base, end, i;

	base = (int) u.u_base;
	end = base + u.u_count;
	for (i = 0; base < end; i++) {   /* find number of idaws */
		base += 2048;
		base -= base%2048;
		((int *) tp->tp_idaws)[i] |= 2047;   /* set all low order  bits */
	}
	end--;                  /* last character in buffer */
	end &= 2047;            /* save low order bits */
	base = i - 1;
	for (i = 0; i < base; i++) {
		temp = tp->tp_idaws[i];
		tp->tp_idaws[i] = tp->tp_idaws[base];
		tp->tp_idaws[base] = temp;
		base--;
	}
	((int *) tp->tp_idaws)[0] &= ~2047;
	((int *) tp->tp_idaws)[0] |= end;      /* set first one to end of  buffer */
}

/*
 * tape control routine
 */
/* ARGSUSED */
tapioctl(dev, cmd, argp, flag)
int dev, cmd, flag;
caddr_t argp;
{
	struct tapes *tp;
	char ringsense[24];

	tp = &tp_info[minor(dev)];

	if (tp->tp_open == OPEN || tp->tp_open == JUSTOPEN)  {
		switch(cmd)  {
		case TAPIO:      /* issue tape I/O function */
			t_io(tp, (int)argp);
			break;
		case TAPRING:    /* check for write ring */
			tp->tp_ccw.cc_dblw = 0;
			tp->tp_ccw.cc_cmd = SENSE;
			tp->tp_ccw.cc_addr = (int) ringsense;
			tp->tp_ccw.cc_count = 24;
			tpsio(tp);
			if(ringsense[1] & 0x02)
				suword(argp, 0);
			else
				suword(argp, 1);
			tp->tp_state = BUSY;
			break;
		case TAPMOTION:     /* set tape motion for open/close */
			tp->tp_motion = (int)argp;
			break;
		case TAPMODE:    /* issue 'modeset' command */
			/*
			 * can only do modeset chained to write at
			 * loadpoint
			 */
			if (((int)argp & 7) != 3) goto error;
			tp->tp_mode = (int)argp;
			break;
		case TAPGTTY:
			suword(argp, tp->tp_state);
			suword(argp+4, tp->tp_motion);
			break;
		default:         /* illegal option, error */
		error:
			u.u_error = EINVAL;     /* invalid request */
			break;
		}
	} else
		u.u_error = EBADF;        /* errror, tape not open */
	return;
}

/*
 * routine to do nonstandard tape i/o
 */
t_io(tp, cmd)
struct tapes *tp;
int cmd;
{
	tp->tp_ccw.cc_dblw = 0L;
	tp->tp_ccw.cc_cmd = cmd;
	tp->tp_ccw.cc_count = 1;
	switch(cmd) {
	case M800:
	case M1600:
	case M6250:
		/*
		 * can only set mode at loadpoint and
		 * chained to a write
		 */
		tp->tp_mode = cmd;
		return;
	case REWIND:
		if (tp->tp_seqflag & S_LDPNT)
			return;
	case REWUNL:
		tp->tp_full = 0;
	case BKSPBLK: case BKSPFL:
		if (tp->tp_seqflag & S_WRITE) {
			/*
			 * Backward motion after a write. Must first write a
			 * tapemark to prevent bad inter-record gaps.
			 */
			tp->tp_rorccw.cc_dblw = tp->tp_ccw.cc_dblw;
			tp->tp_rorccw.cc_sli = 1;
			tp->tp_ccw.cc_dblw = 0L;
			tp->tp_ccw.cc_cmd = WRITETM;
			tp->tp_ccw.cc_count = 1;
			tp->tp_ccw.cc_cc = 1;
		}
		break;
	case FWDSPFL:
	case FWDSPBLK:
		/*
		 * should not allow after a write
		 */
		if (tp->tp_seqflag & S_WRITE) {
			u.u_error = EINVAL;
			return;
		}
	case ERG:
	case WRITETM:
		break;
	case DSE:
		/*
		 * Data security erase must be chained from an erase gap
		 */
		tp->tp_rorccw.cc_dblw = tp->tp_ccw.cc_dblw;
		tp->tp_rorccw.cc_sli = 1;
		tp->tp_ccw.cc_dblw = 0L;
		tp->tp_ccw.cc_cmd = ERG;
		tp->tp_ccw.cc_count = 1;
		tp->tp_ccw.cc_cc = 1;
		break;
	default:
		u.u_error = EINVAL;
		return;
	}
	tpsio(tp);
	tp->tp_state = BUSY;
	switch(cmd) {
	case ERG:
		tp->tp_seqflag = S_WRITE;
		break;
	case WRITETM:
		tp->tp_seqflag = S_WTM;
		break;
	case REWIND:
		tp->tp_seqflag = S_LDPNT;
		tp->tp_full = 0;
		break;
	case FWDSPBLK:
	case FWDSPFL:
		tp->tp_seqflg = S_READ;
		break;
	case REWUNL:
	default:
		tp->tp_seqflag = 0;
	}
}

