/*
 * Description:
 *
 * Program to produce Unix I/O table "iotab.c" from a list of unit
 *     addreses.  There are two entry points.
 *         adddev(addr,name) to add another device to the configuation.
 *         geniotab()      to produce the c-code for iotab.c file.
 *
 */

#include <stdio.h>

#define  NCHANS 16

struct channels{
        struct  chblock  *chp[NCHANS];  /* pointers to 16 channel blocks */
} channels;

struct chblock{
        struct  cublock  *cup[16];  /* pointers to 16 control unit blocks */
} chblks[NCHANS];

struct cublock{
        char    *dev[16];           /* indicator for up to 16 devices */
} cublks[16*NCHANS];


int     namei;                        /* index of next available name location in names */
char    names[4096];                /* place to hold names */

/*
 * adddev(addr,name) - add a device to the structure
 *    addr   is the device address to add
 *    name   is a pointer to a null terminated name string
 *
 *
 *    returns 0 - ok
 *            1 - duplicate
 *            2 - invalid address
 */

int   adddev(addr,name)
int          addr;
char         *name;
{

        int   chi;         /* channel index */
        int   cui;         /* control unit index */
        int   dvi;         /* device index */
        struct chblock *lchp;  /* local channel block pointer */
        struct cublock *lcup;  /* local control unit block pointer */
        char  *pnm,*qnm;   /* pointers into name table */

        /*
         * first check argument.
         */

        if (addr > (256*NCHANS - 1)) return(2);
        chi = addr / 256;               /* compute channel index */
        cui = (addr % 256) / 16;        /* compute control unit index */
        dvi = (addr % 256) % 16;        /* compute device index */


        pnm = &names[namei];          /* where to put name */
        qnm = pnm;                      /* save start for later */

        while(*name != '\0') *pnm++ = *name++;


        /*
         * get a pointer to appropriate channel block, and also
         * make sure the channel table has the address of the chblock for
         * channel.
         */

        lchp = channels.chp[chi] = &chblks[chi];

        /*
         * get a pointer to appropriate control unit block, and also
         * make sure the chblock has the address of cublock for this
         * control unit.
         */

        lcup = lchp->cup[cui] = &cublks[chi*NCHANS + cui];

        if(lcup->dev[dvi] == 0){   /* if this is a new device */
                lcup->dev[dvi] = qnm; /* mark it with pointer to name */
                namei += (pnm - qnm) + 1;       /* update index into name table */
                return(0);            /* and return 0 */
        }
        else{                        /* otherwise    */
                return(1);            /* return 1     */


        }
}

/*
 * geniotab() - generates the iotab.c file from the data structure
 *              created by adddev
 */

geniotab()
{
        static  char    iotfile[] = "/usr/src/sys/conf/newiotab.c";
        FILE    *itf;
        int     chi;
        int     cui;
        int     dvi;
        struct  chblock *lchp;   /* local channel block ptr */
        struct  cublock *lcup;   /* local control unit block ptr */

        if ((itf = fopen(iotfile, "w")) == NULL) {
                fprintf(stderr,"ciotab: Unable to open iotab.c file.\n");
                exit(1);
        }
        fprintf(itf,"#include \"../h/io.h\"\n");   /* first header file */
        fprintf(itf,"#include \"../h/ios.h\"\n\n"); /* stuff */
        fprintf(itf,"/*\n");                      /* now some comments */
        fprintf(itf," * first the individual unit structures\n");
        fprintf(itf," */\n\n");
        /*
         * now tour the tree and output a line for each unit found
         */
        for(chi = 0; chi < NCHANS; chi++){
                lchp = channels.chp[chi];
                if(lchp == NULL) continue;
                for(cui = 0; cui < 16; cui++){
                        lcup = lchp->cup[cui];
                        if(lcup == NULL) continue;
                        for(dvi = 0; dvi < 16; dvi++){
                                if(lcup->dev[dvi] != NULL){
                                        fprintf(itf,"struct unit un%1x%1x%1x;   /* %s */\n",
                                                     chi,cui,dvi,lcup->dev[dvi]);
                                }
                        }
                }
        }


        /*
         * now we repeat all this for the control unit structures
         */

        fprintf(itf,"\n\n");
        fprintf(itf,"/*\n");
        fprintf(itf," * next the control unit structures\n");
        fprintf(itf," */\n\n");

        for(chi = 0; chi < NCHANS; chi++){
                lchp = channels.chp[chi];
                if(lchp == NULL) continue;
                for(cui = 0; cui < 16; cui++){
                        lcup = lchp->cup[cui];
                        if(lcup == NULL) continue;
                        fprintf(itf,"struct cu cu%1x%1xx = {",
                                chi,cui);
                        for(dvi = 0; dvi < 16; dvi++){
                                if(dvi != 0) fprintf(itf,",");
                                if(lcup->dev[dvi] != NULL){
                                        fprintf(itf,"&un%1x%1x%1x",
                                                     chi,cui,dvi);
                                }
                                else fprintf(itf,"0");
                        }
                        fprintf(itf,"};\n\n");
                }
        }


        /*
         * now we repeat partially for channel structures
         */

        fprintf(itf,"\n\n");
        fprintf(itf,"/*\n");
        fprintf(itf," * next the channel structures\n");
        fprintf(itf," */\n\n");

        for(chi = 0; chi < NCHANS; chi++){
                lchp = channels.chp[chi];
                if(lchp == NULL) continue;
                fprintf(itf,"struct chan ch%1x = {",chi);
                for(cui = 0; cui < 16; cui++){
                        lcup = lchp->cup[cui];
                        if(cui != 0) fprintf(itf,",");
                        if(lcup != NULL) fprintf(itf,"&cu%1x%1xx",
                                                 chi,cui);
                        else fprintf(itf,"0");
                }
                fprintf(itf,"};\n\n");
        }

        /*
         * now we make the chans structure
         */

        fprintf(itf,"\n\n");
        fprintf(itf,"/*\n");
        fprintf(itf," * finally the chans structure\n");
        fprintf(itf," */\n\n");

        fprintf(itf,"struct chan *chans[NCHANS] = {");
        for(chi = 0; chi < NCHANS; chi++){
                lchp = channels.chp[chi];
                if(chi != 0) fprintf(itf,",");
                if(lchp != NULL) fprintf(itf,"&ch%1x",chi);
                else fprintf(itf,"0");
        }
        fprintf(itf,"};\n");
        fclose(itf);
}
