/*
 * deroff -- strip troff, eqn, and tbl sequences from a file.
 *      -w argument causes output of one word per line
 *              rather than in the original format.
 *      -s causes the ms macro's to be interpreted
 *              so that just sentences are output.
 *      -l also gets rid of ms style lists.
 *
 * deroff follows .so and .nx commands, removes tbl command sequences,
 *      eqn equations (both .EQ ... .EN and $...$),
 *      nroff escape sequences and the contents of macro definitions.
 *
 * All input is through the C macro; the current input character is in c.
 */

#include <stdio.h>

#define SHIFT {Argc--; Argv++;}

#define C ( (c=getc(infile)) == EOF \
	  ? eof() \
	  : ((c==ldelim)&&(filesp==files)? skeqn(): c) \
	  )

#define C1 ((c=getc(infile)) == EOF? eof(): c)

#define SKIP while (C != '\n'); if (!wflag) putchar ('\n')

#define NO      0
#define YES     1
#define NOCHAR  -2
#define SPECIAL 0
#define PUNCT   1
#define APOS    2
#define DIGIT   3
#define LETTER  4

int     listlev  = 0;
int     wflag    = NO;
int     sflag    = NO;
int     lflag    = NO;
int     mm_mac   = NO;
int     inmacro  = NO;
int     intable  = NO;
int     indispl  = NO;
int     omit     = NO;
int     fillmode = YES;

char    chars[256];  /* SPECIAL, PUNCT, APOS, DIGIT, or LETTER */

char    line[512];
char    *lp;

int     c;
int     pc;
int     ldelim  = NOCHAR;
int     rdelim  = NOCHAR;

char    fname[50];
FILE    *files[15];
FILE    **filesp;
FILE    *infile;

char    *calloc();

int     Argc;
char    **Argv;

main (argc, argv)
int argc;
char *argv[];
{
	register int i;
	FILE     *opn();

	Argc = argc;
	Argv = argv;
	SHIFT;
	while (Argc > 0 && *Argv[0] == '-') {
		while (*++*Argv) switch (*Argv[0]) {
		case 'w':
			wflag = YES;
			break;
		case 's':
			sflag = YES;
			break;
		case 'l':
			lflag = sflag = YES;
			break;
		default:
			fprintf (stderr,
				"deroff: bad %s option\n", *Argv);
			fprintf (stderr,
				"usage: deroff [-wml] [file] ...\n");
			exit (1);
			break;
		}
		SHIFT;
	}

	if (Argc == 0) {
		infile = stdin;
	} else {
		infile = opn(*Argv);
		SHIFT;
	}

	files[0] = infile;
	filesp   = &files[0];

	for (i='a'; i<='z' ; ++i) {
		chars[i] = LETTER;
	}
	for (i='A'; i<='Z'; ++i) {
		chars[i] = LETTER;
	}
	for (i='0'; i<='9'; ++i) {
		chars[i] = DIGIT;
	}
	chars['\''] = APOS;
	chars['&']  = APOS;
	chars['-']  = APOS;
	chars['.']  = PUNCT;
	chars[',']  = PUNCT;
	chars[';']  = PUNCT;
	chars['?']  = PUNCT;
	chars[':']  = PUNCT;
	work ();
}

skeqn ()
{
	while ((c = getc(infile)) != rdelim) {
		if (c == EOF) {
			c = eof();
		} else if (c == '"') {
			while ((c = getc(infile)) != '"') {
				if (c == EOF) {
					c = eof();
				} else if (c == '\\') {
					if ((c = getc(infile)) == EOF) {
						c = eof();
					}
				}
			}
		}
	}
	c = (sflag? 'x': ' ');
	return (c);
}

FILE *opn (p)
register char *p;
{
	FILE    *fd;

	if ((fd = fopen(p, "r")) == NULL) {
		fprintf (stderr, "deroff: cannot open file %s\n", p);
		exit (1);
	}
	return (fd);
}

eof ()
{
	if (infile != stdin) {
		fclose (infile);
	}
	if (filesp > files) {
		infile = *--filesp;
	} else if (Argc > 0) {
		infile = opn(*Argv);
		SHIFT;
	} else {
		exit (0);
	}
	return (C);
}

getfname ()
{
	register char   *p;
	struct chain { 
		struct chain    *nextp;
		char        *datap;
	} *chainblock;
	register struct chain   *q;
	static struct chain     *namechain = NULL;
	char    *copys();

	while (C == ' ') {
	}
	for (p=fname; (*p=c)!='\n' && c!=' ' && c!='\t' && c!='\\'; ++p) {
		C;
	}
	*p = '\0';
	while (c != '\n') {
		C;
	}

	/* see if this name has already been used */
	for (q=namechain; q; q=q->nextp) {
		if (!strcmp(fname,q->datap)) {
			fname[0] = '\0';
			return;
		}
	}
	q        = (struct chain *) calloc(1,sizeof(*chainblock));
	q->nextp  = namechain;
	q->datap  = copys(fname);
	namechain = q;
}

work ()
{
	for (;;) {
		if (C=='.' || (!sflag && c=='\'')) {
			comline ();
		} else {
			regline (NO, 2);
		}
	}
}

regline (macline, const)
int macline;
int const;
{
	line[0] = c;
	lp = line;
	for (;;) {
		if (c == '\\') {
			*lp = ' ';
			backsl ();
		}
		if (c == '\n') {
			break;
		}
		if (intable && (c=='T'||c=='t')) {
			*++lp = C;
			if (c=='{' || c=='}') {
				lp[-1] = ' ';
				*lp = C;
			}
		} else {
			*++lp = C;
		}
	}

	*lp = '\0';

	if (line[0] == '\0' || (sflag && (!fillmode || (lflag && listlev>0) || omit))) {
		if (!wflag) {
			putchar ('\n');
		}
	} else {
		if (wflag) {
			putwords (macline);
		} else if (macline) {
			putmac (line, const);
		} else {
			puts (line);
		}
	}
}

putmac (s, const)
register char *s;
int const;
{
	register char   *t;
	register        found;

	found = 0;
	while (*s) {
		while (*s==' ' || *s=='\t') {
			putchar (*s++);
		}
		for (t=s; *t!=' ' && *t!='\t' && *t!='\0'; ++t) {
		}
		if (*s == '\"') {
			s++;
		}
		if (t>s+const && chars[s[0]]==LETTER && chars[s[1]]==LETTER) {
			while (s < t) {
				if (*s == '\"') {
					s++;
				} else {
					putchar (*s++);
				}
			}
			found++;
		} else if (found && chars[s[0]]==PUNCT && s[1]=='\0') {
			putchar (*s++);
		} else {
			s = t;
		}
	}
	putchar ('\n');
}

putwords (macline)       /* break into words for -w option */
int     macline;
{
	register char   *p, *p1;
	int          i, nlet;

	p1 = line;
	for (;;) {
		/* skip initial chars except alphabetics and numerics */
		while (chars[*p1] < DIGIT) {
			if (*p1++ == '\0') {
				return;
			}
		}
		nlet = 0;
		/* word may include letters, digits, or APOS chars only */
		for (p=p1; (i=chars[*p])>PUNCT; ++p) {
			if (i == LETTER) {
				++nlet;
			}
		}
		/* MDM definition of word */
		if ((!macline && nlet>1) ||
		    ( macline && chars[p1[0]]==LETTER &&
		      nlet>2  && chars[p1[1]]==LETTER
		    )
		   ) {
			/* delete trailing APOS chars */
			while (chars[p[-1]] <= APOS) {
				--p;
			}
			while (p1 < p) {
				putchar (*p1++);
			}
			putchar ('\n');
		} else {
			p1 = p;
		}
	}
}

macro ()
{
	if (sflag) {
		do { 
			SKIP; 
		} while (C!='.' || C!='.' || C=='.'); /* look for .. */
		if (c != '\n') {
			SKIP;
		}
		return;
	}
	SKIP;
	inmacro = YES;
}

stbl ()
{
	while (C != '.') {
	}
	for (;;) {
		skip_to_com();
		if (C=='T') {
			if (C=='E') {
				return;
			}
		} else if (c=='t') {
			if (C=='e') {
				return;
			}
		}
	}
}

skip_to_com()
{
	do {
		SKIP;
	} while (C!='.');
}

eqn ()
{
	register int    c1, c2;
	register int    dflg;
	char        last;

	last = 0;
	dflg = 1;
	SKIP;

	for (;;) {
		if (C1=='.' || c=='\'') {
			while (C1==' ' || c=='\t') {
			}
			if (c=='E' && C1=='N') {
				SKIP;
				if (sflag && dflg) {
					putchar ('x');
					putchar (' ');
					if (last) {
						putchar (last);
						putchar (' ');
					}
				}
				return;
			}
		} else if (c == 'd') {       /* look for delim */
			if (C1=='e' && C1=='l') {
				if (C1=='i' && C1=='m') {
					while (C1 == ' ') {
					}
					if ((c1=c)=='\n' || (c2=C1)=='\n' ||
					    (c1=='o' && c2=='f' && C1=='f')) {
						ldelim = NOCHAR;
						rdelim = NOCHAR;
					} else {
						ldelim = c1;
						rdelim = c2;
					}
				}
			}
			dflg = 0;
		}

		if (c != '\n') {
			while (C1 != '\n') {
				if (chars[c] == PUNCT) {
					last = c;
				} else {
					if (c != ' ') {
						last = 0;
					}
				}
			}
		}
	}
}

backsl () /* skip over a complete backslash construction */
{
	int bdelim;

	sw:

	switch (C) {
	case '.':
		if (sflag) {
			*lp = '\\';
			*++lp = '.';
		}
		return;
	case '"':
		SKIP;
		return;
	case 's':
		if (C == '\\') {
			backsl ();
		} else {
			while (C>='0' && c<='9') {
			}
			ungetc (c, infile);
			c = '0';
		}
		--lp;
		return;
	case 'n':
	case '*':
		if (sflag) {
			*lp = 'x';
		}
		/* fall through */
	case 'f':
		if (C != '(') {
			return;
		}
		/* fall through */
	case '(':
		if (C != '\n') {
			C;
		}
		return;
	case '$':
		C;      /* discard argument number */
		return;
	case 'b':
	case 'x':
	case 'v':
	case 'h':
	case 'w':
	case 'o':
	case 'l':
	case 'L':
		if ((bdelim=C) == '\n') {
			return;
		}
		while (C!='\n' && c!=bdelim) {
			if (c == '\\') {
				backsl ();
			}
		}
		return;
	case '\\':
		if (inmacro) {
			goto sw;
		}
	default:
		return;
	}
}

char *copys (s)
register char *s;
{
	register char *t, *t0;

	if ((t0=t=calloc((unsigned)(strlen(s)+1),sizeof(*t))) == NULL) {
		fprintf (stderr, "deroff: cannot allocate memory");
		exit (1);
	}
	while (*t++ = *s++) {
	}
	return (t0);
}

sce ()
{
	register char *ap;
	register int  n, i;

	char a[10];
	for (ap=a; C!='\n'; ap++) {
		*ap = c;
		if (ap == &a[9]) {
			SKIP;
			ap = a;
			break;
		}
	}
	if (ap != a) {
		n = atoi(a);
	} else {
		n = 1;
	}
	i = 0;
	while (i < n) {
		if (C == '.') {
			if (C == 'c') {
				if (C == 'e') {
					while (C == ' ') {
					}
					if (c == '0') {
						SKIP;
						break;
					} else {
						SKIP;
					}
				} else {
					SKIP;
				}
			} else if (c=='P' || C=='P') {
				if (c != '\n') {
					SKIP;
				}
				break;
			} else if (c != '\n') {
				SKIP;
			}
		} else {
			SKIP;
			i++;
		}
	}
}

refer (c1)
{
	register int c2;

	if (c1 != '\n') {
		SKIP;
	}
	for (;;) {
		if (C != '.') {
			SKIP;
		} else {
			if (C != ']') {
				SKIP;
			} else {
				while (C != '\n') {
					c2=c;
				}
				if (chars[c2] == PUNCT) {
					putchar (c2);
				}
				return;
			}
		}
	}
}

comline ()
{
	register int c1, c2;

	while (C==' ' || c=='\t') {
	}

	comx:

	if ((c1=c) == '\n') {
		return;
	}
	c2 = C;
	if (c1=='.' && c2!='.') {
		inmacro = NO;
	}
	if (sflag && c1=='[') {
		refer (c2);
		return;
	} else if (c2 == '\n') {
		return;
	} else if (c1=='\\' && c2=='\"') {
		SKIP;
	} else if (c1=='E' && c2=='Q' && filesp==files) {
		eqn ();
	} else if (((c1=='T' && (c2=='S'||c2=='C'||c2=='&')) ||
		    (c1=='t' && (c2=='s'||c2=='c'||c2=='&')))&&
		   filesp==files) {
		if (sflag) {
			stbl ();
		} else {
			while (C != '.') {
			}
			SKIP;
			intable = YES;
		}
	} else if ((c1=='T' && c2=='E') || (c1=='t' && c2=='e')) {
		intable = NO;
	} else if (!inmacro && c1=='d' && c2=='e') {
		macro ();
	} else if (!inmacro && c1=='i' && c2=='g') {
		macro ();
	} else if (!inmacro && c1=='a' && c2=='m') {
		macro ();
	} else if (c1=='s' && c2=='o') {
		getfname ();
		if (fname[0]) {
			infile = *++filesp = opn(fname);
		}
	} else if (c1=='n' && c2=='x') {
		getfname ();
		if (fname[0] == '\0') {
			exit (0);
		}
		if (infile != stdin) {
			fclose (infile);
		}
		infile = *filesp = opn(fname);
	} else if (c1=='d' && c2=='s') {
		SKIP;
	} else if (c1=='h' && c2=='w') {
		SKIP; 
	} else if (c1=='i' && c2=='n') {
		SKIP; 
	} else if (c1=='n' && c2=='e') {
		SKIP;
	} else if (c1=='r' && c2=='m') {
		SKIP;
	} else if (c1=='s' && c2=='p') {
		SKIP; 
	} else if (c1=='t' && c2=='i') {
		SKIP;
	} else if (c1=='t' && c2=='m') {
		SKIP;
	} else if (c1=='U' && c2=='X') {
		printf ("UNIX\n");
	} else if (sflag) {
		if        (c1=='n' && c2=='f') {
			fillmode = NO;
			SKIP;
		} else if ((c1=='f' && c2=='i') || (c1=='p' && c2=='p')) {
			fillmode = YES;
			SKIP;
		} else if (c1=='d' && c2=='p') {
			indispl  = YES;
			fillmode = NO;
			SKIP;
		} else if (c1=='e' && c2=='d') {
			indispl  = NO;
			fillmode = YES;
			SKIP;
		} else if (c2=='l' && (c1=='a'||c1=='b'||c1=='d'||c1=='i'||c1=='o'||c1=='r'||c1=='v')) {
			listlev++;
			fillmode = YES;
			SKIP;
		} else if (c1=='l' && c2=='x') {
			listlev--;
			fillmode = !indispl;
			SKIP;
		} else if ((c1=='f' && c2=='s') || (c1=='d' && c2=='(')) {
			omit = YES;
			SKIP;
		} else if ((c1=='f' && c2=='e') || (c1=='d' && c2==')')) {
			omit = NO;
			SKIP;
		} else if (c1=='a' && c2=='p') {
			SKIP;
		} else if (c1=='a' && c2=='u') {
			SKIP;
		} else if (c1=='b' && c2=='q') {
			SKIP;
		} else if (c1=='c' && c2=='m') {
			SKIP;
		} else if (c1=='c' && c2=='p') {
			SKIP;
		} else if (c1=='d' && c2=='a') {
			SKIP;
		} else if (c1=='d' && c2=='f') {
			SKIP;
		} else if (c1=='d' && c2=='s') {
			SKIP;
		} else if (c1=='d' && c2=='x') {
			SKIP;
		} else if (c1=='e' && c2=='q') {
			SKIP;
		} else if (c1=='f' && c2=='g') {
			SKIP;
		} else if (c1=='f' && c2=='o') {
			SKIP;
		} else if (c1=='f' && c2=='r') {
			SKIP;
		} else if (c1=='f' && c2=='t') {
			SKIP;
		} else if (c1=='g' && c2=='f') {
			SKIP;
		} else if (c1=='h' && c2=='d') {
			SKIP;
		} else if (c1=='l' && c2=='e') {
			SKIP;
		} else if (c1=='m' && c2=='t') {
			SKIP;
		} else if (c1=='n' && c2=='r') {
			SKIP;
		} else if (c1=='o' && c2=='p') {
			SKIP;
		} else if (c1=='p' && c2=='r') {
			SKIP;
		} else if (c1=='p' && c2=='t') {
			SKIP;
		} else if (c1=='r' && c2=='f') {
			SKIP;
		} else if (c1=='r' && c2=='h') {
			SKIP;
		} else if (c1=='s' && c2=='f') {
			SKIP;
		} else if (c1=='s' && c2=='h') {
			SKIP;
		} else if (c1=='s' && c2=='i') {
			SKIP;
		} else if (c1=='s' && c2=='k') {
			SKIP;
		} else if (c1=='s' && c2=='l') {
			SKIP;
		} else if (c1=='s' && c2=='u') {
			SKIP;
		} else if (c1=='t' && c2=='a') {
			SKIP;
		} else if (c1=='t' && c2=='f') {
			SKIP;
		} else if (c1=='t' && c2=='l') {
			SKIP;
		} else if (c1=='t' && c2=='o') {
			SKIP;
		} else if (c1=='A' && (c2=='U'||c2=='I')) {
			if (mm_mac) {
				SKIP;
			} else {
				skip_to_com();
				goto comx; 
			}
		} else if (c1=='F' && c2=='S') {
			skip_to_com();
			goto comx; 
		} else if (c1=='S' && c2=='H') {
			skip_to_com();
			goto comx; 
		} else if (c1=='N' && c2=='H') {
			skip_to_com();
			goto comx; 
		} else if (c1=='O' && c2=='K') {
			skip_to_com();
			goto comx; 
		} else if (c1=='N' && c2=='D') {
			SKIP;
		} else if (mm_mac && c1=='H' && (c2==' '||c2=='U')) {
			SKIP;
		} else if (mm_mac && c2=='L') {
			if (lflag || c1=='R') {
				sdis ('L', 'E');
			} else {
				SKIP;
				putchar ('.');
			}
		} else if ((c1=='D'||c1=='N'||c1=='K'||c1=='P') && c2=='S') {
			sdis (c1, 'E');
		} else if (c1=='K' && c2=='F') {
			sdis (c1, 'E');
		} else if (c1=='n' && c2=='f') {
			sdis ('f', 'i');
		} else if (c1=='c' && c2=='e') {
			sce ();
		}
	} else {
		if (c1=='.' && c2=='.') {
			if (sflag) {
				SKIP;
				return;
			}
			while (C == '.') {
			}
		}
		++inmacro;
		if (c1<='Z' && sflag) {
			regline (YES, 1);
		} else {
			regline (YES, 2);
		}
		--inmacro;
	}
}

sdis (a1, a2)
char a1, a2;
{
	register int c1, c2;
	register int eqnf;
	eqnf = 1;
	SKIP;
	for (;;) {
		while (C != '.') {
			if (c == '\n') {
				continue;
			} else {
				SKIP;
			}
		}
		if ((c1=C) == '\n') {
			continue;
		}
		if ((c2=C) == '\n') {
			continue;
		}
		if (c1==a1 && c2==a2) {
			SKIP;
			if (eqnf) {
				putchar ('.');
			}
			putchar ('\n');
			return;
		} else if (a1=='D' && c1=='E' && c2=='Q') {
			eqn ();
			eqnf = 0;
		} else if (a1=='f' && (c1=='P' || c2=='P')) {
			SKIP;
			return;
		} else {
			SKIP;
		}
	}
}
