/* This file is part of the KDE project
   Copyright 2005 Ariya Hidayat <ariya@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "inspector.h"

#include <tqlayout.h>
#include <tqlistview.h>
#include <tqtextstream.h>

#include <kdialogbase.h>

#include "kspread_cell.h"
#include "kspread_style.h"
#include "kspread_sheet.h"
#include "dependencies.h"

namespace KSpread
{

class Inspector::Private
{
public:
  Cell* cell;
  Format* format;
  Sheet* sheet;
  
  TQListView *cellView;
  TQListView *formatView;
  TQListView *sheetView;
  TQListView *styleView;
  TQListView* depView;
  
  void handleCell();
  void handleFormat();
  void handleSheet();
  void handleStyle();
  void handleDep();
};

}

using namespace KSpread;

static TQString boolAsString( bool b )
{
  if( b ) return TQString( "True" );
  else return TQString( "False" );
}

static TQString longAsHexstring( long l )
{
    return TQString("%1").arg(l, 8, 16);
}

static TQString dirAsString( Sheet::LayoutDirection dir )
{
  TQString str;
  switch( dir )
  {
    case Sheet::LeftToRight: str = "Left to Right"; break;
    case Sheet::RightToLeft: str = "Right to Left"; break;
    default: str = "Unknown"; break;
  }
  return str;
}

void Inspector::Private::handleCell()
{  
  TQString str;
  
  cellView->clear();
  
  new TQListViewItem( cellView, "Column", TQString::number( cell->column() ) );
  new TQListViewItem( cellView, "Row", TQString::number( cell->row() ) );
  new TQListViewItem( cellView, "Name", cell->name() );
  new TQListViewItem( cellView, "Full Name", cell->fullName() );

  new TQListViewItem( cellView, "Default", boolAsString( cell->isDefault() ) );
  new TQListViewItem( cellView, "Empty", boolAsString( cell->isEmpty() ) );
  new TQListViewItem( cellView, "Formula", boolAsString( cell->isFormula() ) );
  new TQListViewItem( cellView, "Format Properties", longAsHexstring( static_cast<long>( cell->format()->propertiesMask() ) ) );
  new TQListViewItem( cellView, "Style Properties", longAsHexstring( static_cast<long>( cell->format()->style()->features() ) ) );
  new TQListViewItem( cellView, "Text", cell->text() );
  new TQListViewItem( cellView, "Text (Displayed)", 
		     cell->strOutText().replace( TQChar('\n'), "\\n" ) );

  TQTextStream ts( &str, IO_WriteOnly );
  ts << cell->value();
  new TQListViewItem( cellView, "Value", str );

  new TQListViewItem( cellView, "Link", cell->link() );

  new TQListViewItem( cellView, "Width", TQString::number( cell->dblWidth() ) );
  new TQListViewItem( cellView, "Height", TQString::number( cell->dblHeight() ) );
}

void Inspector::Private::handleFormat()
{
  formatView->clear();
  int col = cell->column();
  int row = cell->row();

  new TQListViewItem( formatView, "Angle", TQString::number( format->getAngle(col, row) ) );
  new TQListViewItem( formatView, "Multirow", boolAsString( format->multiRow(col, row) ) );
  new TQListViewItem( formatView, "Protected", format->hasProperty( Format::PVerticalText ) 
    ? "Not specified" : boolAsString( format->isProtected(col, row) ) );
  new TQListViewItem( formatView, "Vertical Text", boolAsString( format->verticalText(col, row ) ) );

  Format::Currency currrency;
  bool valid = format->currencyInfo(currrency);
  new TQListViewItem( formatView, "Currency symbol", valid ? currrency.symbol : "Invalid" );
  bool ok = false;
  TQString currencyType;
  if (valid)
    currencyType = Currency::getChooseString(currrency.type, ok);
  new TQListViewItem( formatView, "Currency type", valid && ok ? currencyType : "Invalid" );

  TQListViewItem* flags = new TQListViewItem( formatView, "Flags" );
  new TQListViewItem( flags, "Border (left)",
                     boolAsString( format->hasProperty(Format::PLeftBorder, true) ) );
  new TQListViewItem( flags, "Border (right)",
                     boolAsString( format->hasProperty(Format::PRightBorder, true) ) );
  new TQListViewItem( flags, "Border (top)",
                     boolAsString( format->hasProperty(Format::PTopBorder, true) ) );
  new TQListViewItem( flags, "Border (bottom)",
                     boolAsString( format->hasProperty(Format::PBottomBorder, true) ) );

  new TQListViewItem( formatView, "Border pen width (bottom)",
                     TQString::number( format->bottomBorderPen(col,row).width() ) );
}

void Inspector::Private::handleStyle() // direct style access
{
  styleView->clear();
  const Style* style = cell->format()->style();

  TQListViewItem* flags = new TQListViewItem( styleView, "Flags" );
  new TQListViewItem( flags, "Border (left)",
                     boolAsString( style->hasFeature(Style::SLeftBorder, true) ) );
  new TQListViewItem( flags, "Border (right)",
                     boolAsString( style->hasFeature(Style::SRightBorder, true) ) );
  new TQListViewItem( flags, "Border (top)",
                     boolAsString( style->hasFeature(Style::STopBorder, true) ) );
  new TQListViewItem( flags, "Border (bottom)",
                     boolAsString( style->hasFeature(Style::SBottomBorder, true) ) );

  new TQListViewItem( styleView, "Border pen width (bottom)",
                     TQString::number( style->bottomBorderPen().width() ) );
}

void Inspector::Private::handleSheet()
{  
  sheetView->clear();
  
  new TQListViewItem( sheetView, "Name", sheet->sheetName() ) ;
  new TQListViewItem( sheetView, "Layout Direction", dirAsString( sheet->layoutDirection() ) );
}

void Inspector::Private::handleDep()
{  
  Point cellPoint;
  cellPoint.setSheet(sheet);
  cellPoint.setRow( cell->row() );
  cellPoint.setColumn( cell->column() );
  
  DependencyManager* manager = sheet->dependencies();
  TQValueList<Point> deps = manager->getDependants( cellPoint );
  
  depView->clear();
  for( unsigned i = 0; i < deps.count(); i++ )
  {
    TQString k1, k2;
    
    Point point = deps[i];
    int row = point.row();
    int column = point.column();
    k1 = Cell::fullName( point.sheet(), column, row );
    
    new TQListViewItem( depView, k1, k2 );
  }
  
}

Inspector::Inspector( Cell* cell ):
  KDialogBase( KDialogBase::Tabbed, "Inspector", KDialogBase::Close, 
  KDialogBase::Close )
{
  d = new Private;
  
  d->cell = cell;
  d->format = cell->format();
  d->sheet = cell->sheet();
  
  TQFrame* cellPage = addPage( TQString("Cell") );
  TQVBoxLayout* cellLayout = new TQVBoxLayout( cellPage, 0 );
  d->cellView = new TQListView( cellPage );
  cellLayout->addWidget( d->cellView );
  d->cellView->addColumn( "Key", 150 );
  d->cellView->addColumn( "Value" );
  
  TQFrame* formatPage = addPage( TQString("Format") );
  TQVBoxLayout* formatLayout = new TQVBoxLayout( formatPage, 0 );
  d->formatView = new TQListView( formatPage );
  formatLayout->addWidget( d->formatView );
  d->formatView->addColumn( "Key", 150 );
  d->formatView->addColumn( "Value" );
  
  TQFrame* stylePage = addPage( TQString("Style") );
  TQVBoxLayout* styleLayout = new TQVBoxLayout( stylePage, 0 );
  d->styleView = new TQListView( stylePage );
  styleLayout->addWidget( d->styleView );
  d->styleView->addColumn( "Key", 150 );
  d->styleView->addColumn( "Value" );
  
  TQFrame* sheetPage = addPage( TQString("Sheet") );
  TQVBoxLayout* sheetLayout = new TQVBoxLayout( sheetPage, 0 );
  d->sheetView = new TQListView( sheetPage );
  sheetLayout->addWidget( d->sheetView );
  d->sheetView->addColumn( "Key", 150 );
  d->sheetView->addColumn( "Value" );
  
  TQFrame* depPage = addPage( TQString("Dependencies") );
  TQVBoxLayout* depLayout = new TQVBoxLayout( depPage, 0 );
  d->depView = new TQListView( depPage );
  depLayout->addWidget( d->depView );
  d->depView->addColumn( "Cell", 150 );
  d->depView->addColumn( "Content" );
  
  d->handleCell();
  d->handleFormat();
  d->handleSheet();
  d->handleStyle();
  d->handleDep();
  
  resize( 350, 400 );
}  

Inspector::~Inspector()
{
  delete d;
}

#include "inspector.moc"


