/* ============================================================
 * File  : cameralist.cpp
 * Author: Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Date  : 2003-02-03
 * Description : 
 * 
 * Copyright 2003 by Renchi Raju <renchi@pooh.tam.uiuc.edu>

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt
#include <tqdom.h>
#include <tqfile.h>
#include <tqstring.h>
#include <tqtextstream.h>
// Local
#include "cameralist.h"
#include "cameratype.h"

namespace KIPIKameraKlientPlugin
{

CameraList* CameraList::instance_ = 0;

CameraList* CameraList::instance() {
    return instance_;    
}

class CameraListPrivate {
    public:
	TQPtrList<CameraType> clist;
	TQString file;
	bool modified;
};

CameraList::CameraList(TQObject *parent, const TQString& file) : TQObject(parent) {
    d = new CameraListPrivate;
    d->clist.setAutoDelete(true);
    d->file = file;
    d->modified = false;
    instance_ = this;
}

CameraList::~CameraList() {
    close();
    d->clist.clear();
    delete d;
    instance_ = 0;
}

bool CameraList::load() {
    d->modified = false;
    TQFile cfile(d->file);
    if(!cfile.open(IO_ReadOnly)) {
        return false;
    }
    TQDomDocument doc("cameralist");
    if(!doc.setContent(&cfile)) {
        return false;
    }
    TQDomElement docElem = doc.documentElement();
    if(docElem.tagName()!="cameralist") {
        return false;
    }
    for(TQDomNode n = docElem.firstChild(); !n.isNull(); n = n.nextSibling()) {
        TQDomElement e = n.toElement();
        if (e.isNull()) { 
	    continue;
	}
        if (e.tagName() != "item") {
	    continue;
	}
        TQString model  = e.attribute("model");
        TQString port   = e.attribute("port");
        CameraType *ctype = new CameraType(model, port);
        insertPrivate(ctype);
    }
    return true;
}

bool CameraList::close() {
    // If not modified don't save the file
    if (!d->modified) {
        return true;
    }
    TQDomDocument doc("cameralist");
    doc.setContent(TQString("<!DOCTYPE XMLCameraList><cameralist version=\"1.0\" client=\"kipiplugin_kameraklietnt\"/>"));
    TQDomElement docElem=doc.documentElement();
    for (CameraType *ctype = d->clist.first(); ctype; ctype = d->clist.next()) {
       TQDomElement elem = doc.createElement("item");
       elem.setAttribute("model", ctype->model());
       elem.setAttribute("port", ctype->port());
       docElem.appendChild(elem);
    }
    TQFile cfile(d->file);
    if (!cfile.open(IO_WriteOnly)) {
        return false;
    }
    TQTextStream stream(&cfile);
    stream.setEncoding(TQTextStream::UnicodeUTF8);
    stream << doc.toString();
    cfile.close();
    return true;    
}

void CameraList::insert(CameraType* ctype) {
    if(!ctype) {
	return;
    }
    d->modified = true;    
    insertPrivate(ctype);
}

void CameraList::remove(CameraType* ctype) {
    if (!ctype) {
	return;
    }
    d->modified = true;
    removePrivate(ctype);
}

void CameraList::insertPrivate(CameraType* ctype) {
    if (!ctype) {
	return;
    }
    d->clist.append(ctype);
    emit signalCameraListChanged();    
}

void CameraList::removePrivate(CameraType* ctype) {
    if (!ctype) {
	return;
    }
    d->clist.remove(ctype);
    emit signalCameraListChanged();
}

TQPtrList<CameraType>* CameraList::cameraList() {
    return &d->clist; 
}

CameraType* CameraList::find(const TQString& model) {
    for (CameraType *ctype = d->clist.first(); ctype; ctype = d->clist.next()) {
        if (ctype->model() == model) {
            return ctype;
	}
    }
    return 0;
}

void CameraList::clear() {
    CameraType *ctype = d->clist.first();
    while (ctype) {
        remove(ctype);
        ctype = d->clist.first();
    }
}

}  // NameSpace KIPIKameraKlientPlugin

#include "cameralist.moc"
