// (C) 2005 Max Howell (max.howell@methylblue.com)
// See COPYING file for licensing information

#include <tdeapplication.h> // XineConfigDialog::ctor -> to get the iconloader
#include <kcombobox.h>
#include <kiconloader.h>  // XineConfigDialog::ctor
#include <klineedit.h>
#include <kseparator.h>
#include <kstdguiitem.h>
#include <tqcheckbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqscrollview.h>
#include <tqspinbox.h>
#include <tqtabwidget.h>
#include <tqtooltip.h>
#include <tqvbox.h>
#include <xine.h>

#include "../debug.h"
#include "xineConfig.h"


TQString i18n(const char *text);


KDialogBase *XineConfigDialog::s_instance = nullptr;


namespace Codeine
{
   void
   showXineConfigurationDialog( TQWidget *parent, xine_t *xine )
   {
      XineConfigDialog d( xine, parent );
      if( d.exec() == TQDialog::Accepted )
         d.saveSettings();
   }
}

///@class XineConfigDialog

XineConfigDialog::XineConfigDialog( xine_t *xine, TQWidget *parent )
      : KDialogBase( parent, "xine_config_dialog",
               true, //modal
               i18n("Configure xine"), User1 | Stretch | Ok | Cancel,
               Ok, //default button
               false, //draw separator
               KStdGuiItem::reset() )
      , m_xine( xine )
{
   DEBUG_BLOCK

   s_instance = this;

   TQWidget *page = new TQWidget(this);
   setMainWidget(page);
   TQVBoxLayout *topLayout = new TQVBoxLayout(page, 0, spacingHint());

   {
      TQHBoxLayout *infoLayout = new TQHBoxLayout(topLayout, spacingHint());
      TQPixmap info = kapp->iconLoader()->loadIcon("messagebox_info", TDEIcon::NoGroup, TDEIcon::SizeMedium, TDEIcon::DefaultState, nullptr, true);
      TQLabel *label = new TQLabel(page);
      label->setPixmap(info);
      label->setSizePolicy(TQSizePolicy::Maximum, TQSizePolicy::Maximum);
      infoLayout->addWidget(label);
      label = new TQLabel(i18n(
            "Xine's defaults are usually sensible and should not require modification. "
            "However, full configurability is provided for your pleasure ;-)"), page);
      label->setAlignment(TQLabel::WordBreak | TQLabel::AlignVCenter);
      infoLayout->addWidget(label);
   }

   TQTabWidget *tabs = new TQTabWidget(page);

   class XineConfigEntryIterator {
      xine_t *m_xine;
      xine_cfg_entry_t m_entry;
      bool m_valid;
   public:
      XineConfigEntryIterator( xine_t *xine ) : m_xine( xine ) { m_valid = xine_config_get_first_entry( m_xine, &m_entry ); }
      inline XineConfigEntryIterator &operator++() { m_valid = xine_config_get_next_entry( m_xine, &m_entry ); return *this; }
      inline xine_cfg_entry_t *operator*() { return m_valid ? &m_entry : nullptr; }
   };

   TQGridLayout *grid = nullptr;
   TQString currentPageName;
   TQScrollView *view = nullptr;
   TQWidget *scrollWidget = nullptr;

   for (XineConfigEntryIterator it(m_xine); *it; ++it)
   {
      const TQString pageName = TQString::fromUtf8((*it)->key).section('.', 0, 0);

      if ((TQStringList() << "ui" << "effects" << "subtitles").contains(pageName)) {
         continue;
      }

      if (pageName != currentPageName) {
         currentPageName = pageName;

         TQString tabTitle = pageName;
         tabTitle[0] = tabTitle[0].upper();

         view = new TQScrollView(page);
         view->setHScrollBarMode(TQScrollView::ScrollBarMode::AlwaysOff);
         // TODO: It would be nice to leave VScrollBarMode on Auto, but
         //       there seems to be an issue when calculating the layout size.
         //       https://mirror.git.trinitydesktop.org/gitea/TDE/codeine/pulls/18
         view->setVScrollBarMode(TQScrollView::ScrollBarMode::AlwaysOn);
         view->setResizePolicy(TQScrollView::AutoOneFit);
         view->setFrameShape(TQFrame::NoFrame);
         tabs->addTab(view, tabTitle);

         scrollWidget = new TQWidget(view->viewport());
         view->addChild(scrollWidget);

         grid = new TQGridLayout(scrollWidget, 0, 2, marginHint(), spacingHint());
         grid->setColStretch(0, 3);
         grid->setColStretch(1, 2);
         grid->setAlignment(TQt::AlignTop | TQt::AlignAuto);
      }

      m_entrys.append(new XineConfigEntry(scrollWidget, grid, *it));
   }

   // finishing touches
   m_entrys.setAutoDelete(true);
   topLayout->addWidget(tabs, 1);

   enableButton( Ok, false );
   enableButton( User1, false );

   Q_ASSERT( !isUnsavedSettings() );
}

void
XineConfigDialog::slotHelp()
{
   /// HACK called when a widget's input value changes

   const bool b = isUnsavedSettings();
   enableButton( Ok, b );
   enableButton( User1, b );
}

void
XineConfigDialog::slotUser1()
{
   for (TQPtrListIterator<XineConfigEntry> it(m_entrys); *it != nullptr; ++it)
      (*it)->reset();

   slotHelp();
}

bool
XineConfigDialog::isUnsavedSettings() const
{
   for (TQPtrListIterator<XineConfigEntry> it(m_entrys); *it != nullptr; ++it)
      if ((*it)->isChanged())
         return true;

   return false;
}

#include <tqdir.h>
void
XineConfigDialog::saveSettings()
{
   for( XineConfigEntry *entry = m_entrys.first(); entry; entry = m_entrys.next() )
      if( entry->isChanged() )
         entry->save( m_xine );

   xine_config_save( m_xine, TQFile::encodeName( TQDir::homeDirPath() + "/.xine/config" ) );
}


///@class XineConfigEntry

XineConfigEntry::XineConfigEntry( TQWidget *parent, TQGridLayout *grid, xine_cfg_entry_t *entry )
      : m_widget( nullptr )
      , m_key( entry->key )
      , m_string( entry->str_value )
      , m_number( entry->num_value )
{
   TQWidget *&w = m_widget;
   const char *signal = nullptr;
   const int row = grid->numRows();

   TQString description_text = TQString::fromUtf8( entry->description );
   description_text[0] = description_text[0].upper();

   switch( entry->type )
   {
   case XINE_CONFIG_TYPE_STRING: {
      w = new KLineEdit( m_string, parent );
      signal = TQ_SIGNAL(textChanged( const TQString& ));
      break;
   }
   case XINE_CONFIG_TYPE_ENUM: {
      w = new KComboBox( parent );
      for( int i = 0; entry->enum_values[i]; ++i )
         ((KComboBox*)w)->insertItem( TQString::fromUtf8( entry->enum_values[i] ) );
      ((KComboBox*)w)->setCurrentItem( m_number );
      signal = TQ_SIGNAL(activated( int ));
      break;
   }
   case XINE_CONFIG_TYPE_RANGE:
   case XINE_CONFIG_TYPE_NUM: {
      w = new TQSpinBox(
               TQMIN( m_number, entry->range_min ), // xine bug, sometimes the min and max ranges
               TQMAX( m_number, entry->range_max ), // are both 0 even though this is bullshit
               1, parent );
      ((TQSpinBox*)w)->setValue( m_number );
      signal = TQ_SIGNAL(valueChanged( int ));
      break;
   }
   case XINE_CONFIG_TYPE_BOOL: {
      w = new TQCheckBox( description_text, parent );
      ((TQCheckBox*)w)->setChecked( m_number );

      connect( w, TQ_SIGNAL(toggled( bool )), XineConfigDialog::instance(), TQ_SLOT(slotHelp()) );
      TQToolTip::add( w, "<qt>" + TQString::fromUtf8( entry->help ) );
      grid->addMultiCellWidget( w, row, row, 0, 1 );
      return; //no need for a description label
   }
   default:
      ;
   }

   connect( w, signal, XineConfigDialog::instance(), TQ_SLOT(slotHelp()) );

   TQLabel *description = new TQLabel( description_text + ':', parent );
   description->setAlignment( TQLabel::WordBreak | TQLabel::AlignVCenter );

   const TQString tip = "<qt>" + TQString::fromUtf8( entry->help );
   TQToolTip::add( w, tip );
   TQToolTip::add( description, tip );

   grid->addWidget( description, row, 0, TQt::AlignVCenter );
   grid->addWidget( w, row, 1, TQt::AlignTop );
}

bool
XineConfigEntry::isChanged() const
{
   #define _( x ) static_cast<x*>(m_widget)

   switch( classType( m_widget->className() ) ) {
      case LineEdit: return _(KLineEdit)->text().utf8() != m_string;
      case ComboBox: return _(KComboBox)->currentItem() != m_number;
      case SpinBox:  return _(TQSpinBox)->value() != m_number;
      case CheckBox: return _(TQCheckBox)->isChecked() != m_number;
   }
   return false;
}

void
XineConfigEntry::reset()
{
   // this is because we only get called by the XineConfigDialog reset button
   // and we don't want to cause a check for Ok/Reset button enabled state for
   // every XineConfigEntry
   m_widget->blockSignals( true );

   switch( classType( m_widget->className() ) ) {
      case LineEdit: _(KLineEdit)->setText( m_string ); break;
      case ComboBox: _(KComboBox)->setCurrentItem( m_number ); break;
      case SpinBox:  _(TQSpinBox)->setValue( m_number ); break;
      case CheckBox: _(TQCheckBox)->setChecked( (bool)m_number ); break;
   }
   m_widget->blockSignals( false );
}

void
XineConfigEntry::save( xine_t *xine )
{
   xine_cfg_entry_t ent;

   if( xine_config_lookup_entry( xine, key(), &ent ) )
   {
      switch( classType( m_widget->className() ) ) {
         case LineEdit: m_string = _(KLineEdit)->text().utf8(); break;
         case ComboBox: m_number = _(KComboBox)->currentItem(); break;
         case SpinBox:  m_number = _(TQSpinBox)->value(); break;
         case CheckBox: m_number = _(TQCheckBox)->isChecked(); break;
      }

      ent.str_value = tqstrdup( m_string );
      ent.num_value = m_number;

      debug() << "Saving setting: " << key() << endl;
      xine_config_update_entry( xine, &ent );
   }
   else
      Debug::warning() << "Couldn't save: " << key() << endl;

   #undef _
}
