;;; magit-ido.el --- Support using Ido in Magit  -*- lexical-binding:t -*-

;; Copyright (C) 2025 The Magit Project Contributors

;; Author: Jonas Bernoulli <emacs.magit-ido@jonas.bernoulli.dev>
;; Maintainer: Jonas Bernoulli <emacs.magit-ido@jonas.bernoulli.dev>
;; Homepage: https://github.com/emacsorphanage/magit-ido
;; Package-Version: 1.0.0
;; Package-Revision: v1.0.0-0-g2b94abf65a20
;; Package-Requires: ((ido-completing-read+ "4.14") (magit "4.3.2"))

;; SPDX-License-Identifier: GPL-3.0-or-later

;; Magit is free software: you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.
;;
;; Magit is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with Magit.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; This package adds support for using Ido in Magit.  For the most part
;; it was extracted from code that was previously part of Magit itself.

;; To enable Ido support, you need to add something like this to your
;; init file:
;;
;; (with-eval-after-load 'magit
;;   (require 'magit-ido)
;;   (setq magit-completing-read-function
;;         'magit-ido-completing-read)
;;   ;; Optional:
;;   (keymap-set ido-common-completion-map
;;               "C-x g" 'magit-ido-enter-magit-status))

;;; Code:

(require 'magit)
(require 'ido-completing-read+)

(defun magit-ido-completing-read
    (prompt choices &optional predicate require-match initial-input hist def)
  "Ido-based `completing-read' almost-replacement.

Unfortunately `ido-completing-read' itself is not suitable as a drop-in
replacement for `completing-read'; instead we use `ido-completing-read+'
from the third-party package by the same name."
  (ido-completing-read+ prompt choices predicate require-match
                        initial-input hist
                        (or def (and require-match (car choices)))))

(define-advice magit-submodule-read-path
    (:override (prompt url) magit-ido)
  "Prevent use of Ido because it is broken for this use case.
`ido-read-file-name' does not work here, so force the use of
`read-file-name-default'."
  (directory-file-name
   (file-relative-name
    (let ((read-file-name-function #'read-file-name-default))
      (read-directory-name prompt nil nil nil
                           (and (string-match "\\([^./]+\\)\\(\\.git\\)?$" url)
                                (match-string 1 url)))))))

(define-advice magit-subtree-read-prefix
    (:override (prompt &optional default _history) magit-ido)
  "Prevent use of Ido because it is broken for this use case.
`ido-read-file-name' does not work here, so force the use of
`read-file-name-default'."
  (let* ((insert-default-directory nil)
         (topdir (magit-toplevel))
         (read-file-name-function #'read-file-name-default)
         (prefix (read-directory-name (concat prompt ": ") topdir default)))
    (if (file-name-absolute-p prefix)
        (if (string-prefix-p topdir prefix)
            (file-relative-name prefix topdir)
          (user-error "%s isn't inside the repository at %s" prefix topdir))
      prefix)))

(defvar ido-exit)
(defvar ido-fallback)

(defun magit-ido-enter-magit-status ()
  "Drop into the `magit-status' buffer from Ido file switching."
  (interactive)
  (setq ido-exit 'fallback)
  (setq ido-fallback #'magit-status)
  (exit-minibuffer))

(provide 'magit-ido)

;; Local Variables:
;; indent-tabs-mode: nil
;; End:
;;; magit-ido.el ends here
