// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_METAPROGS_H
#define POOMA_UTILITIES_METAPROGS_H

//-----------------------------------------------------------------------------
// Classes:
//   LoopUtils
// Meta-Programs:
//   LoopUtils<N>::copy
//   LoopUtils<N>::dot
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Overview:
//   LoopUtils: Template meta-programs for carrying out certain operations
//              on arrays at compile time.
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description:
// LoopUtils
//
// Template meta-program loop class with the following functions:
//
//   void LoopUtils<N>::copy(T *dst, const T *src)
//    - copies N contiguous T's from src to dst. Uses assignment
//      operator to perform the copy.
//   T LoopUtils<N>::dot(const T *a, const T *b)
//    - calculates dot product of a[N] and b[N] at compile time.
//
//-----------------------------------------------------------------------------

template <int N>
struct LoopUtils
{
  template <class T>
  static inline void copy(T* dest, const T* src)
  {
    dest[N-1] = src[N-1];
    LoopUtils<N-1>::copy(dest, src);
  }

  template <class T>
  static inline T dot(const T * a, const T * b)
  {
    return *a * *b + LoopUtils<N-1>::dot(a+1,b+1);
  }
};

template <>
struct LoopUtils<1>
{
  template <class T>
  static inline void copy(T* dest, const T* src)
    {
      dest[0] = src[0];
    }

  template <class T>
  static inline T dot(const T * a, const T * b)
    {
      return *a * *b;
    }
};


// } // namespace POOMA
///////////////////////////////////////////////////////////////////////////////

#endif // POOMA_UTILITIES_METAPROGS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: MetaProgs.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:18:24 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
