// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
//
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without
// charge, provided that this Notice and any statement of authorship are
// reproduced on all copies.  Neither the Government nor the University
// makes any warranty, express or implied, or assumes any liability or
// responsibility for the use of this SOFTWARE.
//
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
//
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Test of PatchSizeSyncer
//-----------------------------------------------------------------------------

// Include files

#include "Tulip/PatchSizeSyncer.h"
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Domain/Interval.h"
#include "Domain/Range.h"
#include "Domain/Grid.h"

#define BARRIER

#ifndef BARRIER
#if POOMA_CHEETAH
# define BARRIER Pooma::controller()->barrier()
#else
# define BARRIER
#endif
#endif

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  const int numContexts = Pooma::contexts();
  const int myContext   = Pooma::context();

  tester.out() << "Testing PatchSizeSyncer class . . .\n";
  tester.out() << "Running with " << numContexts << " contexts." << std::endl;

  int start = myContext * 10;
  int end   = (myContext + 1) * 10;

  Range<1> r(start, end, 2);

  Grid<1> foo(r);

  BARRIER;

  tester.out().setOutputContext(-1);
  tester.out() << "Initializing PatchSizeSyncer with grid " << foo 
	       << std::endl;

  Pooma::PatchSizeSyncer dls(myContext,foo);

  Grid<1> bar;

  dls.calcGlobalGrid(bar);

  BARRIER;

  tester.out().setOutputContext(0);
  tester.out() << "Here's the global grid (everyone's should be the same): " 
	       << std::endl;
  tester.out().setOutputContext(-1);
  tester.out() << bar << std::endl;

  BARRIER;

  tester.check(bar == Grid<1>(Range<1>(0,numContexts*10,2)));

  // Now we test one that mimics adding some particles:

  if (myContext == 1)
    {
      // Add 20 "elements" to the last domain on this context. 
      IndirectionList<int> tmp(foo.storage());
      tmp(foo.size()-1) += 20;
      foo = Grid<1>(tmp);
    }

  if (myContext == 2)
    {
      // Remove 1 "element" from all domains on this context. 
      foo = Grid<1>(Interval<1>(20,25));
    }

  tester.out().setOutputContext(0);
  tester.out() << "This test actually involves some changes..." << std::endl;
  tester.out().setOutputContext(-1);
  tester.out() << "Initializing PatchSizeSyncer with grid " << foo 
	       << std::endl;

  Pooma::PatchSizeSyncer dls2(myContext,foo);

  dls2.calcGlobalGrid(bar);

  BARRIER;

  tester.out().setOutputContext(0);
  tester.out() << "Here's the global grid: " << std::endl;
  tester.out().setOutputContext(-1);
  tester.out() << bar << std::endl;

  BARRIER;

  IndirectionList<int> idata(bar.size());

  int val = 0;
  for (int i = 0; i < bar.size(); ++i)
    {
      idata(i) = val;
      if (i < 9)  val+=2;
      if (i == 9) val+=22;
      if (i > 9 && i < 15) ++val;
      if (i >= 15) val+=2;
    }

  Grid<1> ans2(idata);

  tester.check(bar == ans2);

  int ret = tester.results("PatchSizeSyncer Test");
  Pooma::finalize();

  return ret;
}

