// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No.  W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE, and to allow others to do so.
// The public may copy and use this SOFTWARE, FOR NONCOMMERCIAL USE ONLY,
// without charge, provided that this Notice and any statement of
// authorship are reproduced on all copies.  Neither the Government nor
// the University makes any warranty, express or implied, or assumes any
// liability or responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// BisectionMapper
//-----------------------------------------------------------------------------

#ifndef POOMA_BISECTIONMAPPER_H
#define POOMA_BISECTIONMAPPER_H

#include <list>

// Recursive Bisection Mapper
template <int Dim>
class BisectionMapper
  : public ContextMapper<Dim>
{ 
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================
  typedef Interval<Dim>                       Domain_t;
  typedef Node<Domain_t>                      Value_t;
  typedef std::vector<Value_t *>              List_t;

  template<class Partitioner>
  BisectionMapper(const Partitioner & gp,
		   const Loc<Dim> &nblocks) 
    : blocks_m(gp.blocks())
  {
  }

  template<class Partitioner>  
  BisectionMapper(const Partitioner & gp)
     : blocks_m(gp.blocks())
  {
  }

  BisectionMapper(const Loc<Dim>& blocks)
     : blocks_m(blocks)
  {
  }

  // 

  void map(const List_t & templist) const 
  { 
    int ncontexts = Pooma::contexts();
    int npatch = 1; 
    for (int i =0;i<Dim; ++i) 
      npatch*=blocks_m[i].first();
    
    std::list<Domain_t> bvec;
    
    Domain_t allb;
    for (int i = 0; i<Dim; ++i)
      allb[i]=Interval<1>(0,blocks_m[i].first()-1);
    bvec.push_back(allb);
    
    while ( bvec.size() < ncontexts )
      {
	int s = 0;
	std::list<Domain_t>::iterator bstart = bvec.begin();
	std::list<Domain_t>::iterator bend = bvec.end();
	std::list<Domain_t>::iterator bpatch;
	// find the largest patch.
	for ( ; bstart != bend ; ++bstart)
	  {
	    if (s < (*bstart).size() )
	      {
		bpatch = bstart; 
		s = (*bstart).size();
	      }
	  }
	// now find the largest dimension on the largest patch
	int d = 0;
	int sd = 0;
	for (int i = 0; i<Dim; ++i)
	  {
	    if ( sd < (*bpatch)[i].size() )
	      {
		d = i;
		sd = (*bpatch)[i].size();
	      }
	  }
	Domain_t hi(*bpatch),lo(*bpatch);
	int lopoint = hi[d].first();
	int hipoint = hi[d].last();
	int mid     = lopoint + ( (hipoint - lopoint)/2);

	if (lopoint<=mid)
	  lo[d] = Interval<1>(lopoint,mid);
	else
	  lo[d] = Interval<1>(lopoint,lopoint);
	if ( hipoint>=mid+1)
	  hi[d] = Interval<1>(mid+1,hipoint);
	else
	  hi[d] = Interval<1>(hipoint,hipoint);
	bvec.erase(bpatch++);	
	bvec.insert(bpatch,lo);
	bvec.insert(bpatch,hi);
      }
    // now step through the intervals, using their elements as indexes into
    // all_m;
    int strides[Dim];
    strides[0] = 1;
    for ( int i=1; i<Dim; ++i)
      strides[i] = strides[i-1]*blocks_m[i-1].first();

    std::list<Domain_t>::iterator start = bvec.begin();
    std::list<Domain_t>::iterator end = bvec.end();
    int pcontext = 0;
    for ( ; start != end ; ++start)
      {
	int idx[Dim],mi[Dim],mx[Dim];
	for ( int  i = 0 ; i < Dim ; ++i)
	  {
	    idx[i] = mi[i] = (*start)[i].first();
	    mx[i]  = (*start)[i].last();
	  }

	while ( idx[Dim-1] <= mx[Dim-1] )
	  {
	    int allIdx = 0;
	    for ( int i = 0 ; i < Dim ; ++i)
	      allIdx += idx[i]*strides[i];
	    (*templist[allIdx]).context() = pcontext;
	    ++idx[0];
	    for ( int i = 0 ; i < Dim ; ++i)
	      {
		if ( idx[i] > mx[i] )
		  {
		    if ( i!=(Dim-1) ) 
		      {
			++idx[i+1];
			idx[i]=mi[i];
		      }
		    else
		      ++idx[i];
		  }
		else
		  break;
	      }
	  }
	++pcontext;
      }
    // set the affinity and local ID values
    setAffinity(templist);
  }

  // Memeber Data
  Loc<Dim> blocks_m;

};

#endif   // POOMA_BISECTIONMAPPER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BisectionMapper.h,v $   $Author: sa_smith $
// $Revision: 1.3 $   $Date: 2000/06/08 23:07:59 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
