// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "IO/ObjectSets.h"

//-----------------------------------------------------------------------------
// ObjectStorageSet member functions.
//-----------------------------------------------------------------------------
// Constructor.
//-----------------------------------------------------------------------------
ObjectStorageSet::ObjectStorageSet(const std::string& setName,
				   ObjectSetOpenMode mode){
	                 
  // Default state is closed until opened successfully.
  openState_m= Closed;
  
  // Store the set name.
  setName_m= setName;
	
  // Set the default flush parameters.
  flushCount_m=0;
  flushFreq_m=1;
  
  // Mode cases
  switch(mode){
  case Create:
    
    // Call the create() function to perform the initialization.
    create();
    break;
  case Restore:
    
    // Call the restore() function to restore the 
    //   existing information from the dictionary file.
    restore();
    break;
  default:
    
    // Bad mode. Leave uninstantiated and issue a warning.
    pwarn<<"Warning: Unknown mode specified. "
	 <<"ObjectStorageSet "<<setName<<" not instantiated"
	 <<std::endl;
    
    // Then reset the name.
    setName_m="";
    break;
  }              
}


//-----------------------------------------------------------------------------
// Destructor.
//-----------------------------------------------------------------------------
ObjectStorageSet::~ObjectStorageSet(){
  
  // Close the object set if not already closed.
  if(openState_m== Open){
    close();
  }
  
  // The clear() function will destroy pointers to maps.
  clear();
  
  // Delete the file pointers.
  delete dictFile_m.file;
  delete dataFile_m.file;
}

//-----------------------------------------------------------------------------
// Returns the type of an object.
//-----------------------------------------------------------------------------
std::string ObjectStorageSet::objectType(const ObjectID_t& id){
  // Check that the object exists.
  if(objectExists(id)){
    AttributeMap<std::string>* typeMap=
      (AttributeMap<std::string>*)
      attributeDict_m.findMap("Type");
    return typeMap->findID(id);
  }
  else{
    pwarn<<"Warning: No object with ID "<<id<<" exists."<<std::endl;
    return "";
  }
}

//-----------------------------------------------------------------------------
// Returns the label of an object.
//-----------------------------------------------------------------------------
std::string ObjectStorageSet::objectLabel(const ObjectID_t& id){
  // Check that the object exists.
  if(objectExists(id)){
    AttributeMap<std::string>* labelMap=
      (AttributeMap<std::string>*)
      attributeDict_m.findMap("Label");
    return labelMap->findID(id);
  }
  else{
    pwarn<<"Warning: No object with ID "<<id<<" exists."<<std::endl;
    return "";
  }
}


//-----------------------------------------------------------------------------
// Assigns a label attribute to an object already stored.
//-----------------------------------------------------------------------------
void ObjectStorageSet::objectLabel(ObjectID_t id,
				   const std::string& objectLabel){
  
  // Enter the label in the label map.
  if(objectExists(id)){
    AttributeMap<std::string>* labelMap=
      (AttributeMap<std::string>*)
      attributeDict_m.findMap("Label");
    labelMap->insert(objectLabel,id);
  }
  else{
    pwarn<<"Warning: No object with ID "<<id<<" exists."<<std::endl;
  }
}

//-----------------------------------------------------------------------------
// Assigns a label attribute to an object already stored.
//-----------------------------------------------------------------------------
ObjectID_t ObjectStorageSet::findLabel(const std::string& objectLabel){
  AttributeMap<std::string>* labelMap=
    (AttributeMap<std::string>*)
    attributeDict_m.findMap("Label");
  
  // Check that an object of the given label exists.
  if(labelMap->count(objectLabel)!=0){
    
    // Return the ID of the first instance stored; multimap containers
    // store duplicate key-value pairs in FIFO order, so the first
    // object stored is at the beginning of the range for a given key.
    return labelMap->findAttribute(objectLabel);
  }
  else{
    // Return a zero object ID.
    return 0;
  }
}

//-----------------------------------------------------------------------------
// Return the number of objects matching the given label.
//-----------------------------------------------------------------------------
int ObjectStorageSet::count(const std::string& objectLabel) {
  AttributeMap<std::string>* labelMap=
    (AttributeMap<std::string>*)
    attributeDict_m.findMap("Label");
  return labelMap->count(objectLabel);
}

//-----------------------------------------------------------------------------
// Return an object set whose objects match the given label.
//-----------------------------------------------------------------------------
ObjectSubset ObjectStorageSet::selectLabel(const std::string& objectLabel){
  ObjectSubset s;
  s.clear();
  AttributeMap<std::string>* labelMap=
    (AttributeMap<std::string>*)
    attributeDict_m.findMap("Label");
  s= labelMap->selectAttribute(objectLabel);
  return s;
}
	
//-----------------------------------------------------------------------------
// Close the object set.
//-----------------------------------------------------------------------------
void ObjectStorageSet::close(){
  // Write the dictionary to the end of the file.
  writeDict();
  // Seek to the dictionary reference location and
  //   write the value of the dictionary location.
  dataFile_m.file->seekp(dictRef_m,std::ios::beg);
  serialize(*dataFile_m.file,dictLoc_m);
  
  // Close the dictionary and data files.
  dictFile_m.file->close();
  dataFile_m.file->close();
  // Destroy the dictionary file.
  remove(dictFile_m.name.c_str());
  // Clear all the data members
  clear();
  // Set state to closed.
  openState_m= Closed;
}

//-----------------------------------------------------------------------------
// Test that an object with the given ID exists.
//-----------------------------------------------------------------------------
bool ObjectStorageSet::objectExists(ObjectID_t id) const{
  
  // Check that the set is not empty, and
  // then that the given id exists.
  if(numObjects()==0)
    return false;
  if(objectMap_m.count(id)==0)
    return false;
  return true;
}

//-----------------------------------------------------------------------------
// Generate an ID for a new object.
//-----------------------------------------------------------------------------
ObjectID_t ObjectStorageSet::newID(){
  
  // Generates a new ID. In this prototype, object IDs start at one. A zero
  //   object ID value is the standard invalid ID. The new object ID is
  //   equal to the current of number of objects plus one.
  return numObjects()+1;
}

//-----------------------------------------------------------------------------
// Clear all the data members of an object set.
//-----------------------------------------------------------------------------
void ObjectStorageSet::clear(){
  // Clear all the data members.
  setName_m=""; // clear() does not work in EGCS
  dataFile_m.name="";
  dataFile_m.file= NULL;
  dictFile_m.name="";
  dictFile_m.file= NULL;
  objectMap_m.clear();
  attributeDict_m.clear();
}

//-----------------------------------------------------------------------------
// Write an object location to the recovery file.
//-----------------------------------------------------------------------------
void ObjectStorageSet::writeLoc(ObjectLoc& loc){
  // Write a header to identify as a location.
  serialize(*dictFile_m.file,std::string("ObjectLoc"));
  // Then serialize the ObjectLoc.
  serialize(*dictFile_m.file,loc);
}


//-----------------------------------------------------------------------------
// Write the data file header.
//-----------------------------------------------------------------------------
void ObjectStorageSet::writeHeader(){
  
  // Write the cookie;
  // File cookie is standardized on this phrase
  //   length of 40 characters
  std::string cookie="POOMA OBJECT FILE VERSION 1.00 beta     ";
  serialize(*dataFile_m.file,cookie);
  
  // Write the set attributes.
  serialize(*dataFile_m.file,setName_m);
  serialize(*dataFile_m.file,dataFile_m.name);
  
  // Get the position of the dictionary location reference and
  // write a blank value;
  // Make sure this is the end of the file.
  dataFile_m.file->seekp(0,std::ios::end);
  
  // Record the location.
  dictRef_m= dataFile_m.file->tellp();
  dictLoc_m= 0;
  serialize(*dataFile_m.file,dictLoc_m);
}

//-----------------------------------------------------------------------------
// Read the data file header.
//-----------------------------------------------------------------------------
void ObjectStorageSet::readHeader(){

  // Seek to the beginning of the data file.
  dataFile_m.file->seekg(0,std::ios::beg);
  
  // Read the cookie;
  // File cookie is standardized on the phrase 
  // "POOMA OBJECT FILE VERSION N.nn XXXX"
  // and a fixed length of 40 characters
  std::string cookie;
  deserialize(cookie, *dataFile_m.file);
  
  // Check the cookie here if you want.
  // Read the set attributes.
  std::string setName;
  deserialize(setName, *dataFile_m.file);
  
  // Assert that the current set name and the stored set name match.
  PInsist(setName==setName_m,
	  "Error: stored set name does not match the name of this set.");
  
  // Read data file attributes
  std::string dataName;
  deserialize(dataName, *dataFile_m.file);
  
  // Check the stored datafile name against the current name.
  // Get the position of the dictionary location reference and
  // read the value;
  // Record the location.
  dictRef_m= dataFile_m.file->tellg();
  deserialize(dictLoc_m,*dataFile_m.file);
}

//-----------------------------------------------------------------------------
// Read the dictionary portion of the data file.
//-----------------------------------------------------------------------------
void ObjectStorageSet::readDict(){
  
  // Seek to the dictionary position.
  dataFile_m.file->seekg(dictLoc_m,std::ios::beg);
  
  // Read the object attributes.
  int num;
  deserialize(num, *dataFile_m.file);
  ObjectLoc object;
  for(int i=0;i<num;i++){
    deserialize(object, *dataFile_m.file);
    // Put the information in the map.
    objectMap_m[object.objectID]=object;
  }
  
  // Read the maps.
  AttributeMap<std::string>* stringMap;
  AttributeMap<long>* longMap;
  int numMaps;
  std::string mapType;
  
  deserialize(numMaps,*dataFile_m.file);
  for(int i=0;i<numMaps;i++){
    deserialize(mapType,*dataFile_m.file);
    if(mapType=="std::string"){
      stringMap= new AttributeMap<std::string>;
      deserialize(*stringMap, *dataFile_m.file);
      attributeDict_m.addAttribute(
				   (AttributeMapBase*) stringMap);
    }
    else if(mapType=="long"){
      longMap= new AttributeMap<long>;
      deserialize(*longMap, *dataFile_m.file);
      attributeDict_m.addAttribute(
				   (AttributeMapBase*) longMap);
    }
  }
}
	
//-----------------------------------------------------------------------------
// Write the dictionary portion of the data file.
//-----------------------------------------------------------------------------
void ObjectStorageSet::writeDict(){
  
  // Seek to the end of the data file.
  dataFile_m.file->seekp(0,std::ios::end);
  
  // Record the dictionary location
  dictLoc_m= dataFile_m.file->tellp();
  
  // Write the object attributes.
  int num=numObjects();
  serialize(*dataFile_m.file,num);
  ObjectLoc object;
  std::map<ObjectID_t, ObjectLoc>::iterator iter;
  for(iter= objectMap_m.begin();
      iter!=objectMap_m.end();iter++){
    object= (*iter).second;
    serialize(*dataFile_m.file,object);
  }
  
  // Write the maps.
  AttributeMap<std::string>* stringMap;
  AttributeMap<long>* longMap;
  AttributeMapBase* mapBase;
  AttributeDict::iterator diter;
  int numMaps= attributeDict_m.size();
  
  serialize(*dataFile_m.file,numMaps);
  std::string mapType;
  for(diter= attributeDict_m.begin();
      diter!= attributeDict_m.end(); diter++){
    mapBase= (*diter).second;
    mapType= mapBase->attributeType();
    serialize(*dataFile_m.file,mapType);
    if(mapType=="std::string"){
      stringMap= (AttributeMap<std::string>*)
	mapBase;
      serialize(*dataFile_m.file,*stringMap);
    }
    else if(mapType=="long"){
      longMap= (AttributeMap<long>*)
	mapBase;
      serialize(*dataFile_m.file,*longMap);
    }
  }
}

//-----------------------------------------------------------------------------
// Recover the dictionary from the recovery file.
//-----------------------------------------------------------------------------
void ObjectStorageSet::recoverDict(){
  // Seek to the beginning of the dictionary file.
  dictFile_m.file->seekg(0,std::ios::beg);
  // Read a record at a time until out of records.
  std::string recType;
  std::string attributeType;
  std::string attributeName;
  std::string stringValue;
  long longValue;
  AttributeMap<std::string>* stringMap;
  AttributeMap<long>* longMap;
  AttributeMapBase* baseMap;
  ObjectID_t objectID;
  ObjectLoc loc;
  
  for(;;){ // do forever
    deserialize(recType,*dictFile_m.file);
    if(dictFile_m.file->eof()){
      break;
    }
    if(recType=="ObjectLoc"){
      
      // Deserialize the subsequent ObjectLoc instance.
      deserialize(loc,*dictFile_m.file);
      
      // Record the object location in the object map.
      objectMap_m[loc.objectID]= loc;		
    }
    else if(recType=="ObjectAttribute"){
      
      // Read the attribute name.
      deserialize(attributeName,*dictFile_m.file);
      
      // Read the type.
      deserialize(attributeType,*dictFile_m.file);
      
      // Read the object ID.
      deserialize(objectID,*dictFile_m.file);
      
      // Check to see if an attribute map by this
      // name exists.
      if(attributeDict_m.count(attributeName)
	 ==0){
	
	// If not there, create one and
        //   put it in the dictionary.
	baseMap= attributeDict_m.newMap(attributeName,
					attributeType);
	if(baseMap!=NULL){
	  attributeDict_m.addAttribute(baseMap);
	}
	
      }
      else{
	
	// Otherwise, look up the map.
	baseMap= attributeDict_m.findMap(attributeName);
      }
			
      // Cast to the appropriate map type then read and place the value.
      if(attributeType=="std::string"){
	stringMap= (AttributeMap<std::string>*) baseMap;
	deserialize(stringValue,*dictFile_m.file);
	stringMap->insert(stringValue,objectID);
      }
      else if(attributeType=="long"){
	longMap= (AttributeMap<long>*) baseMap;
	deserialize(longValue,*dictFile_m.file);
	longMap->insert(longValue,objectID);
      }
      else{
	// Unrecognized attribute type.
	pwarn<<"Error. Unrecognized attribute type. "
	     <<"Dictionary not recovered."<<std::endl;
	break;
      }	
    }
    else{
      break;
    }
  }
  
}		

//-----------------------------------------------------------------------------
// Create a new object set.
//-----------------------------------------------------------------------------
void ObjectStorageSet::create(){
  
  // Generate file names.
  std::string dictName= "~" + setName_m;
  std::string datName= setName_m;
  
  // Open the dictionary file in in|out|trunc|binary mode.
  dictFile_m.file= new std::fstream(dictName.c_str(),
    std::ios::in|std::ios::out|std::ios::trunc|std::ios::binary);
  // Open the data file in out|trunc|binary mode.
  dataFile_m.file= new std::fstream(datName.c_str(),
    std::ios::out|std::ios::trunc|std::ios::binary);
		
  // Initialize file names.
  dataFile_m.name= datName;
  dictFile_m.name= dictName;
  
  // Create the standard attribute maps.
  attributeDict_m.addAttribute("Label","std::string");
  attributeDict_m.addAttribute("Type","std::string");
  
  // Write the file header.
  writeHeader();
  
  // Set state to open.
  openState_m= Open;
}

//-----------------------------------------------------------------------------
// Restore an existing object set.
//-----------------------------------------------------------------------------
void ObjectStorageSet::restore(){
  
  // Generate file names.
  std::string dictName= "~" + setName_m;
  std::string datName= setName_m;
  dataFile_m.name= datName;
  dictFile_m.name= dictName;
  
  // Open the data file in in|out|binary mode.
  dataFile_m.file= new std::fstream(datName.c_str(),
    std::ios::in|std::ios::out|std::ios::binary);
  
  // Read the header.
  readHeader();
  
  // Check to see if the file was closed properly
  // by looking at the dictionary pointer.
  
  if(dictLoc_m!=0){
    
    // Normal condition. File was properly closed.
    // Open the dictionary file in out|trunc|binary mode.
    
    dictFile_m.file= new std::fstream(dictName.c_str(),
      std::ios::out|std::ios::trunc|std::ios::binary);	
    
    // Read the dictionary.
    readDict();
  }
  else{
    
    // Abnormal condition. File was not closed properly.
    // Open the dictionary file in in|binary mode.
    dictFile_m.file= new std::fstream(dictName.c_str(),
      std::ios::in|std::ios::binary);
    
    // Recover the dictionary.
    recoverDict();
  }
  
  // Set state to open.
  openState_m= Open;
}


//-----------------------------------------------------------------------------
// AttributeDict member functions
//-----------------------------------------------------------------------------
// Constructor. No-op.
//-----------------------------------------------------------------------------
AttributeDict::AttributeDict(){}
	
//-----------------------------------------------------------------------------
// Destructor.
//-----------------------------------------------------------------------------
AttributeDict::~AttributeDict(){
  // Destroy all the AttributeMap instances.
  iterator iter;
  for(iter=map_m.begin(); iter!=map_m.end(); iter++){
    delete (*iter).second;
  }
}
	
//-----------------------------------------------------------------------------
// Add a new attribute map to the dictionary.
//-----------------------------------------------------------------------------
void AttributeDict::addAttribute(const std::string& attributeName,
				 const std::string& attributeType){
  
  // Check to see if the attribute already exists.
  if(map_m.count(attributeName)==0){
	
    // Use the factory to create a new map.
    AttributeMapBase* amap= newMap(attributeName,
				   attributeType);
    if(amap!=NULL){
      
      // If successful, enter in the dictionary.
      map_m[attributeName]=
	(AttributeMapBase*) amap;
    }
    else{
      
      // Otherwise complain.
      pwarn<<"Warning: Bad attribute map. "
	   <<"Attribute named "<<attributeName
	   <<" not added."<<std::endl;
    }
  }
  else{
    
    // The attribute already exists.
    pwarn<<"Warning: Attribute named "
	 <<attributeName<<" already defined; not added."
	 <<std::endl;
  }
}

//-----------------------------------------------------------------------------
// Add an externally created attribute map to the dictionary.
//-----------------------------------------------------------------------------
void AttributeDict::addAttribute(AttributeMapBase* amap){
  std::string attributeName= amap->attributeName();
  
  // Check to see if the attribute already exists.
  if(map_m.count(attributeName)==0){
    if(amap!=NULL){
      
      // Check to see that there's something there.
      map_m[attributeName]= amap;
    }
    else{
      // If not complain.
      pwarn<<"Warning: Bad attribute map. "
	   <<"Attribute named "<<attributeName
	   <<" not added."<<std::endl;
    }
  }
  else{
    // The attribute already exists.
    pwarn<<"Warning: Attribute named "
	 <<attributeName<<" already defined; not added."
	 <<std::endl;
  }
}

//-----------------------------------------------------------------------------
// Factory for a new attribute map.
//-----------------------------------------------------------------------------
AttributeMapBase* AttributeDict::newMap(const std::string& attributeName,
					const std::string& attributeType){
  AttributeMapBase* amap;
  
  // Maps of strings.
  if(attributeType=="string" ||
     attributeType=="std::string" ||
     attributeType=="char*"){
    amap= (AttributeMapBase*)
      new AttributeMap<std::string>(attributeName,
				    attributeType);
  }
  
  // Maps of integers.
  else if(attributeType=="int" ||
	  attributeType=="long"){
    amap= (AttributeMapBase*)
      new AttributeMap<long>(attributeName,
			     attributeType);
  }
  
  // Unsupported attribute types.
  else{
    amap= NULL;
  }
  return amap;
}

//-----------------------------------------------------------------------------
// Find the map with the given name.
//-----------------------------------------------------------------------------
AttributeMapBase* AttributeDict::findMap(const std::string& attributeName){
  if(map_m.count(attributeName)!=0){
    
    // Map exists.
    return map_m[attributeName];
  }
  else{
    
    // Map does not exist.
    pwarn<<"Warning: Attribute named "
	 <<attributeName<<" not found.";
    return NULL;
  }
}

//-----------------------------------------------------------------------------
// Return the size of the map with the given name.
//-----------------------------------------------------------------------------
int AttributeDict::size(const std::string& attributeName) {
  const AttributeMapBase* baseMap= findMap(attributeName);
  if(baseMap!=NULL){
    return baseMap->size();
  }
  else{
    return 0;
  }
}

//-----------------------------------------------------------------------------
// Clear the entire dictionary by destroying each map.
//-----------------------------------------------------------------------------
void AttributeDict::clear(){
  iterator iter;
  for(iter= map_m.begin();iter!= map_m.end();iter++){
    delete (*iter).second;
  }
  map_m.clear();
}
//-----------------------------------------------------------------------------

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ObjectSets.cmpl.cpp,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

