// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Classes: 
//   NoGeometry<Dim>
//-----------------------------------------------------------------------------

#ifndef POOMA_GEOMETRY_NOGEOMETRY_H
#define POOMA_GEOMETRY_NOGEOMETRY_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// NoGeometry<Dim>:
//
// A very stupid geometry that can stand in for a real geometry, but badly. It
// is useful for field operations, like views, where we essentially get out
// an array.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "Engine/IndexFunctionEngine.h"
#include "Domain/Interval.h"
#include "Domain/SliceInterval.h"
#include "Domain/SliceRange.h"
#include "Pooma/View.h"
#include "Tiny/Vector.h"

//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

struct NotAvailable;

//-----------------------------------------------------------------------------
// Full Description:
// 
// NoGeometry provides minimal geometry services to a discrete field.
//
// Exported typedefs:
//
//   Centering_t - not available.
//   CoordinateSystem_t - not available.
//   Domain_t - the type of the physical domain.
//   Mesh_t - not available.
//   PointType_t - not available.
//   PositionArray_t - a compute-based array used to supply something for
//     the x() function.
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the physical domain.
//   componentCentered - true if this field has different centerings
//     for each component.
//   coordinateDimensions - not available.
//
// Constructors:
//
//   NoGeometry(Domain) - constructs the physical domain.
//   NoGeometry(This_t) - copy constructor.
//
// Accessors:
//
//   physicalDomain() - returns this geometry's physical domain.
//   totalDomain() - returns this geometry's total domain.
//   x() - returns a "position" array consisting of the indices.
//-----------------------------------------------------------------------------

template<int Dim>
class NoGeometry
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef NoGeometry<Dim> This_t;
  
  // The centering type.
  
  typedef NotAvailable Centering_t;
  
  // The mesh type.
  
  typedef NotAvailable Mesh_t;
  
  // The coordinate system.
  
  typedef NotAvailable CoordinateSystem_t;
  
  // The domain type.
  
  typedef Interval<Dim> Domain_t;
  
  // Dimensions.
  
  enum { dimensions = Dim };
  enum { coordinateDimensions = Dim };

  // Are we component-centered?
  
  static const bool componentCentered = false;
  
  // The type used to represent a point.
  
  typedef Vector<coordinateDimensions, int> PointType_t;

  // Position computation functor. We just return a point consisting
  // of the indices. This exists so that we can have a non-templated
  // x() function for fields with no-geometry.
  
  class PositionFunctor
  {
    public:
    
      PositionFunctor()
      { }
      
      enum { dataObject = false };
      
      PointType_t operator()(int i0) const
      {
        return PointType_t(i0);
      }
      
      PointType_t operator()(int i0, int i1) const
      {
        return PointType_t(i0, i1);
      }

      PointType_t operator()(int i0, int i1, int i2) const
      {
        return PointType_t(i0, i1, i2);
      }
  };

  // Array types
  
  typedef Array<Dim, PointType_t, IndexFunction<PositionFunctor> > 
    PositionArray_t;
  
  //---------------------------------------------------------------------------
  // Constructors using a regular domain and a slice domain.
  
  template<class DT>
  NoGeometry(const Domain<Dim, DT> &dom)
  {
    // The domain's dimension should match ours.
    
    CTAssert(DT::dimensions == Dim);

    // Construct the physical domain.
    
    const typename DT::Domain_t &domain = dom.unwrap();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }

  template<int Dim2>
  NoGeometry(const SliceInterval<Dim2, Dim> &dom)
  {
    const typename SliceInterval<Dim2, Dim>::SliceDomain_t &domain = 
      dom.sliceDomain();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }

  template<int Dim2>
  NoGeometry(const SliceRange<Dim2, Dim> &dom)
  {
    const typename SliceInterval<Dim2, Dim>::SliceDomain_t &domain = 
      dom.sliceDomain();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }
  
  template<class OtherGeom, class DT>
  NoGeometry(const OtherGeom &, const Domain<Dim, DT> &dom)
  {
    // The domain's dimension should match ours.
    
    CTAssert(DT::dimensions == Dim);

    // Construct the physical domain.
    
    const typename DT::Domain_t &domain = dom.unwrap();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }

  template<class OtherGeom, int Dim2>
  NoGeometry(const OtherGeom &, const SliceInterval<Dim2, Dim> &dom)
  {
    const typename SliceInterval<Dim2, Dim>::SliceDomain_t &domain = 
      dom.sliceDomain();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }

  template<class OtherGeom, int Dim2>
  NoGeometry(const OtherGeom &, const SliceRange<Dim2, Dim> &dom)
  {
    const typename SliceInterval<Dim2, Dim>::SliceDomain_t &domain = 
      dom.sliceDomain();
    Domain_t zeroBasedDomain;
    for (int d = 0; d < Dim; ++d)
      zeroBasedDomain[d]  = Interval<1>(domain[d].length());
    positions_m.engine().setDomain(zeroBasedDomain);
  }

  //---------------------------------------------------------------------------
  // Copy constructor.
  
  NoGeometry(const This_t &model)
  : positions_m(model.positions_m)
  {
  }

  //---------------------------------------------------------------------------
  // Accessors.
  
  const Domain_t &physicalDomain() const 
  {
    return positions_m.engine().domain(); 
  }
  
  const Domain_t &totalDomain() const 
  {
    return positions_m.engine().domain(); 
  }

  // This x() function is not very useful to codes; however, it allows us to
  // have an x() function in Field, even using this geometry.
  
  const PositionArray_t &x() const
  {
    return positions_m;
  }
  
private:

  PositionArray_t positions_m;
  
};

#endif // POOMA_GEOMETRY_NOGEOMETRY_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: NoGeometry.h,v $   $Author: swhaney $
// $Revision: 1.13 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
