// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// test of RNGSimple
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Pooma.h"
#include "Pooma/Arrays.h"
#include "Utilities/Tester.h"
#include "Functions/RNGSimple.h"
#include "Functions/RNGComponent.h"

#include <iostream>

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);

  typedef Array<2, RNGSimple, MultiPatch<GridTag, Brick> > Array_t;
  typedef Array<2, double, MultiPatch<GridTag, Brick> > ArrayR_t;

  Interval<1> ii(20);
  Interval<2> dom(ii, ii);

  Loc<2> blocks(4,4);
  GridPartition<2> partition(blocks);
  GridLayout<2> layout(dom, partition,ReplicatedTag() );

  Array_t rng0(layout), rng1(layout);

  RNGValue value;
  RNGSeed seed;
  RNGAdvance advance;

  tester.out() << "----------------------------------" << std::endl;
  tester.out() << "some random numbers (all the same)" << std::endl;
  tester.out() << "----------------------------------" << std::endl;

  tester.out() << rng0.comp(value) << std::endl;
  tester.out() << rng1.comp(value) << std::endl;
  rng0.comp(advance) = 1;
  rng1.comp(advance) = 1;
  tester.out() << rng0.comp(value) << std::endl;
  tester.out() << rng1.comp(value) << std::endl;

  ArrayR_t a(layout);

  a = rng0.comp(value) - rng1.comp(value);

  tester.check("same values", sum(a) == 0.0);

  Vector<2, int> strides(1, dom[0].length());

  rng0.comp(seed) = dot(strides, iota(dom));
  rng1.comp(seed) = dot(strides, iota(dom));

  tester.out() << "-----------------------------------" << std::endl;
  tester.out() << "some random numbers (different now)" << std::endl;
  tester.out() << "-----------------------------------" << std::endl;

  tester.out() << rng0.comp(value) << std::endl;
  tester.out() << rng1.comp(value) << std::endl;

  a = rng0.comp(value) - rng1.comp(value);

  tester.check("same values", sum(a) == 0.0);

  rng0.comp(advance) = 20;
  rng1.comp(advance) = 10;

  tester.out() << "------------------------------------------" << std::endl;
  tester.out() << "some random numbers (completely different)" << std::endl;
  tester.out() << "------------------------------------------" << std::endl;

  tester.out() << rng0.comp(value) << std::endl;
  tester.out() << rng1.comp(value) << std::endl;

  a = rng0.comp(value) - rng1.comp(value);

  tester.check("different values", sum(a) != 0.0);

  tester.out() << "------------------------------------------" << std::endl;
  tester.out() << "finally the seeds:" << std::endl;
  tester.out() << "------------------------------------------" << std::endl;

  tester.out() << rng0.comp(seed) << std::endl;
  tester.out() << rng1.comp(seed) << std::endl;

  int ret = tester.results("rngArray");

  Pooma::finalize();

  return ret;
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rngArray.cpp,v $   $Author: luchini $
// $Revision: 1.4 $   $Date: 2000/06/12 18:37:07 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
