// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Non-inline functions related to Field.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview: 
// 
// Field: An Array-like container that adds the notions of geometry and
//        the management of automatic boundary conditions. Works closely
//        with its read-only cousin, ConstField.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/FieldBCondBase.h"

//----------------------------------------------------------------------
// Set up a little traits class that distinguishes between OpAssign and
// other assignment operators that read the LHS.
//----------------------------------------------------------------------

template<class Op>
struct AssignOpReadWriteTraits
{
  enum { readLHS = true };
};

template<>
struct AssignOpReadWriteTraits<OpAssign>
{
  enum { readLHS = false };
};

//---------------------------------------------------------------------------
// Implementation for assignment operators. We pack this assignment 
// expression into a PETE binary expression tree node and then use this 
// to construct a ConstField with an expression engine. We then pass this on 
// to an evaluator, which handles the computation. The first version 
// handles assigning Fields and ConstFields to Fields and the second one 
// handles assigning scalars.
//
// If you wish to have Field work with other types of objects on the right-
// hand side (for example, other classes that derive from Field), define
// extra assign() functions that take the following arguments:
//
//   assign(Field<Geom,T,EngineTag>, yourclass, Operator)
//
// where "yourclass" is the class that you would like to work on the
// right-hand side in an expression with a Field on the left-hand side.
//---------------------------------------------------------------------------

// assign for Field = Array

template<class Geom, class T, class EngineTag,
  int OtherDim, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const Array<OtherDim, OtherT, OtherEngineTag> &rhs,
       const Op &op)
{
  // The dimensions of the Field and Array must be the same.
  
  CTAssert(Geom::dimensions == OtherDim);
  
  // Some useful typedefs for this function

  typedef Field<Geom, T, EngineTag>                     LHS_t;
  typedef Array<OtherDim, OtherT, OtherEngineTag>       RHS_t;
  
  // In order to apply boundary conditions, we need to take views
  // of both the LHS and RHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef View1<RHS_t, typename RHS_t::Domain_t> ViewRHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  typedef typename ViewRHS_t::Type_t ViewedRHS_t;
  
  ViewedLHS_t vlhs = lhs();
  ViewedRHS_t vrhs = rhs();

  // Do a CreateLeaf to pull the PETE expression out of an ExpressionEngine
  // if the rhs is an expression.
  // This will allow subsequent functors to actually reach the leaves.

  CreateLeaf<ViewedRHS_t>::Leaf_t rhsExpr = 
    CreateLeaf<ViewedRHS_t>::make(vrhs);

  // Optionally, check conformance.
  
#if POOMA_BOUNDS_CHECK
  PInsist(forEach(rhsExpr, ConformTag<LHS_t::dimensions>
    (lhs.physicalDomain()), AndCombine()), 
    "Field <- Array assignment conformance error.");
#endif

  // For the LHS, we need to update the boundary conditions if
  // the operation is such that we're both reading and writing.

  if (AssignOpReadWriteTraits<Op>::readLHS)
    vlhs.notifyPreRead();
  
   // Just evaluate the sucker since it is impossible for an array to be
   // shifted relative to a field.
      
   Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, vrhs);

  // Having done the evaluation, we need to notify the LHS boundary conditions
  // that we've just written.
  
  vlhs.notifyPostWrite();

  return lhs;
}

// assign for Field = ConstField

template<class Geom, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const ConstField<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op)
{
  // Some useful typedefs for this function

  typedef Field<Geom, T, EngineTag>                     LHS_t;
  typedef ConstField<OtherGeom, OtherT, OtherEngineTag> RHS_t;
  
  // In order to apply boundary conditions, we need to take views
  // of both the LHS and RHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef View1<RHS_t, typename RHS_t::Domain_t> ViewRHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  typedef typename ViewRHS_t::Type_t ViewedRHS_t;
  
  ViewedLHS_t vlhs = lhs();
  ViewedRHS_t vrhs = rhs();

  // Do a CreateLeaf to pull the PETE expression out of an ExpressionEngine
  // if the rhs is an expression.
  // This will allow subsequent functors to actually reach the leaves.

  CreateLeaf<ViewedRHS_t>::Leaf_t rhsExpr = 
    CreateLeaf<ViewedRHS_t>::make(vrhs);

  // Optionally, check conformance.
  
#if POOMA_BOUNDS_CHECK
  PInsist(forEach(rhsExpr, ConformTag<LHS_t::dimensions>
    (lhs.physicalDomain()), AndCombine()), 
    "Field <- ConstField assignment conformance error.");
#endif

  // Make sure the boundary conditions on the rhs are up-to-date.
  
  forEach(rhsExpr, NotifyPreReadTag(), AndCombine());
    
  // For the LHS, we need to update the boundary conditions if
  // the operation is such that we're both reading and writing.
  // We also need to do this if we're shifted.

  if (AssignOpReadWriteTraits<Op>::readLHS)
    vlhs.notifyPreRead();
  
  // Just evaluate the sucker.
      
  Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, vrhs);

  // Having done the evaluation, we need to notify the LHS boundary conditions
  // that we've just written.
  
  vlhs.notifyPostWrite();

  return lhs;
}

// assign for Field = Field

template<class Geom, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const Field<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op)
{
  // Some useful typedefs for this function

  typedef Field<Geom, T, EngineTag>                     LHS_t;
  typedef Field<OtherGeom, OtherT, OtherEngineTag>      RHS_t;
  
  // In order to apply boundary conditions, we need to take views
  // of both the LHS and RHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef View1<RHS_t, typename RHS_t::Domain_t> ViewRHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  typedef typename ViewRHS_t::Type_t ViewedRHS_t;
  
  ViewedLHS_t vlhs = lhs();
  ViewedRHS_t vrhs = rhs();

  // Do a CreateLeaf to pull the PETE expression out of an ExpressionEngine
  // if the rhs is an expression.
  // This will allow subsequent functors to actually reach the leaves.

  CreateLeaf<ViewedRHS_t>::Leaf_t rhsExpr = 
    CreateLeaf<ViewedRHS_t>::make(vrhs);

  // Optionally, check conformance.
  
#if POOMA_BOUNDS_CHECK
  PInsist(forEach(rhsExpr, ConformTag<LHS_t::dimensions>
    (lhs.physicalDomain()), AndCombine()), 
    "Field <- Field assignment conformance error.");
#endif

  // Make sure the boundary conditions on the rhs are up-to-date.
  
  forEach(rhsExpr, NotifyPreReadTag(), AndCombine());
    
  // For the LHS, we need to update the boundary conditions if
  // the operation is such that we're both reading and writing.
  // We also need to do this if we're shifted.

  if (AssignOpReadWriteTraits<Op>::readLHS)
    vlhs.notifyPreRead();
  
  // Just evaluate the sucker.
      
  Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, vrhs);

  // Having done the evaluation, we need to notify the LHS boundary conditions
  // that we've just written.
  
  vlhs.notifyPostWrite();

  return lhs;
}

// assign for Array = ConstField

template<int Dim, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Array<Dim, T, EngineTag> &
assign(const Array<Dim, T, EngineTag> &lhs,
       const ConstField<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op)
{
  // The dimensions of the Array and Field must be the same.
  
  CTAssert(Dim == OtherGeom::dimensions);
  
  // Some useful typedefs for this function

  typedef Array<Dim, T, EngineTag>                      LHS_t;
  typedef ConstField<OtherGeom, OtherT, OtherEngineTag> RHS_t;
  
  // In order to apply boundary conditions, we need to take views
  // of both the LHS and RHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef View1<RHS_t, typename RHS_t::Domain_t> ViewRHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  typedef typename ViewRHS_t::Type_t ViewedRHS_t;
  
  ViewedLHS_t vlhs = lhs();
  ViewedRHS_t vrhs = rhs();

  // Do a CreateLeaf to pull the PETE expression out of an ExpressionEngine
  // if the rhs is an expression.
  // This will allow subsequent functors to actually reach the leaves.

  CreateLeaf<ViewedRHS_t>::Leaf_t rhsExpr = 
    CreateLeaf<ViewedRHS_t>::make(vrhs);

  // Optionally, check conformance.
  
#if POOMA_BOUNDS_CHECK
  PInsist(forEach(rhsExpr, ConformTag<LHS_t::dimensions>
    (lhs.domain()), AndCombine()), 
    "Array <- ConstField assignment conformance error.");
#endif

  // Make sure the boundary conditions on the rhs are up-to-date.
  
  forEach(rhsExpr, NotifyPreReadTag(), AndCombine());
    
  // Just evaluate the sucker since an array cannot be shifted relative to a field.
      
  Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, vrhs);

  return lhs;
}

// assign for Array = ConstField

template<int Dim, class T, class EngineTag,
  class OtherGeom, class OtherT, class OtherEngineTag, class Op>
const Array<Dim, T, EngineTag> &
assign(const Array<Dim, T, EngineTag> &lhs,
       const Field<OtherGeom, OtherT, OtherEngineTag> &rhs,
       const Op &op)
{
  // The dimensions of the Array and Field must be the same.
  
  CTAssert(Dim == OtherGeom::dimensions);
  
  // Some useful typedefs for this function

  typedef Array<Dim, T, EngineTag>                      LHS_t;
  typedef Field<OtherGeom, OtherT, OtherEngineTag>      RHS_t;
  
  // In order to apply boundary conditions, we need to take views
  // of both the LHS and RHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef View1<RHS_t, typename RHS_t::Domain_t> ViewRHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  typedef typename ViewRHS_t::Type_t ViewedRHS_t;
  
  ViewedLHS_t vlhs = lhs();
  ViewedRHS_t vrhs = rhs();

  // Do a CreateLeaf to pull the PETE expression out of an ExpressionEngine
  // if the rhs is an expression.
  // This will allow subsequent functors to actually reach the leaves.

  CreateLeaf<ViewedRHS_t>::Leaf_t rhsExpr = 
    CreateLeaf<ViewedRHS_t>::make(vrhs);

  // Optionally, check conformance.
  
#if POOMA_BOUNDS_CHECK
  PInsist(forEach(rhsExpr, ConformTag<LHS_t::dimensions>
    (lhs.domain()), AndCombine()), 
    "Array <- Field assignment conformance error.");
#endif

  // Make sure the boundary conditions on the rhs are up-to-date.
  
  forEach(rhsExpr, NotifyPreReadTag(), AndCombine());
    
  // Just evaluate the sucker since an array cannot be shifted relative to a field.
      
  Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, vrhs);

  return lhs;
}

// assign for Field = scalar

template<class Geom, class T, class EngineTag, class T1, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs, const T1 &rhs, const Op &op)
{
  // Some useful typedefs for this function

  typedef Field<Geom, T, EngineTag>                     LHS_t;

  // Make an expression out of the scalar.

  Array<LHS_t::dimensions, T1, ConstantFunction>
    rhsExpr(lhs.physicalDomain());
  rhsExpr.engine().setConstant(rhs);

  // In order to apply boundary conditions, we need to take a view
  // of the LHS. The reason is that we need to look
  // at the base-domain for where we're reading/writing and
  // views can naturally cough this up.
  
  typedef View1<LHS_t, typename LHS_t::Domain_t> ViewLHS_t;
  typedef typename ViewLHS_t::Type_t ViewedLHS_t;
  
  ViewedLHS_t vlhs = lhs();

  // We only need to run the boundary condition if the operator
  // is not a pure assignment (i.e., it reads values).
  
  if (AssignOpReadWriteTraits<Op>::readLHS)
    vlhs.notifyPreRead();
    
  // Evaluate.

  Evaluator<MainEvaluatorTag>().evaluateZeroBased(vlhs, op, rhsExpr());

  // Having done the evaluation, we need to notify the LHS boundary conditions
  // that we've just written.
  
  vlhs.notifyPostWrite();
    
  return lhs;
}

template<class Tree>
struct ConvertWhereProxy<ExpressionIsField, Tree>
{
  typedef MakeFieldReturn<Tree> Make_t;
};

// support for field = where(F,B);

template<class Geom, class T, class EngineTag,
  class F, class B, class Op>
const Field<Geom, T, EngineTag> &
assign(const Field<Geom, T, EngineTag> &lhs,
       const WhereProxy<F, B> &rhs,
       const Op &op)
{
  return assign(lhs, rhs.whereMask(), rhs.opMask(op));
}

//----------------------------------------------------------------------
// Instruct the field to make its own copy of its data.
//
// This is a little trickier for Fields than Arrays due to boundary 
// conditions since we have to retarget the boundary conditions to
// have the current field as the subject. 
//----------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
void Field<Geometry, T, EngineTag>::makeOwnCopy()
{
  // Make a distinct copy of the engine and BCondList.

  engine().makeOwnCopy();
  retargetBoundaryConditions();
}

//----------------------------------------------------------------------
// Retarget our boundary condition list to have this field as the
// subject.
//----------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
void Field<Geometry, T, EngineTag>::retargetBoundaryConditions() 
{
  // Make a distinct copy of the BCondList.
  
  bconds().makeOwnCopy();
  
  // Now, we need to replace the individual boundary conditions
  // with cloned versions with this Field as the subject.
  
  for (int i = 0; i < bconds().size(); i++)
    {
      FieldBCondBase<This_t> * bc = 
        dynamic_cast<FieldBCondBase<This_t> *>(bconds()(i));
      PAssert(bc != NULL);
      bconds()(i) = bc->retarget(*this);
    }
}
