// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_RANGE_ITERATOR_H
#define POOMA_DOMAIN_RANGE_ITERATOR_H

//-----------------------------------------------------------------------------
// Classes: 
//   RangeIterator
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
//   RangeIterator - Iterates through Range<1> points.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Range.h"
#include "Utilities/PAssert.h"

#include <iterator> // std::random_access_iterator_tag
#include <stddef.h> // ptrdiff_t

//-----------------------------------------------------------------------------
// Open POOMA namespace:
//-----------------------------------------------------------------------------

// namespace POOMA {

//-----------------------------------------------------------------------------
//
// Full Description:
// RangeIterator
//
// A random access iterator class that iterates through all of the points
// of an Range<1>, returning ints when dereferenced.
//
//-----------------------------------------------------------------------------

class RangeIterator
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================

  typedef RangeIterator                       This_t;
  typedef Range<1>                            Domain_t;
  typedef ptrdiff_t                           Value_t;

  typedef std::random_access_iterator_tag     iterator_category;
  typedef ptrdiff_t                           value_type;
  typedef ptrdiff_t                           difference_type;
  typedef const ptrdiff_t*                    pointer;
  typedef const ptrdiff_t&                    reference;
  
  //============================================================
  // Constructors
  //============================================================

  // The main RangeIterator constructor stores the given domain
  // and initializes the iterator to the beginning.

  RangeIterator(const Domain_t &d, int initial_pos = 0)
  : domain_m(d), 
    val_m(d.first() + d.stride() * initial_pos), 
    index_m(initial_pos)
  { }
    
  // Copy constructor.

  RangeIterator(const This_t &it)
  : domain_m(it.domain_m), val_m(it.val_m), index_m(it.index_m)
  { }

  // The default constructor constructs an end iterator for an empty
  // domain.

  RangeIterator() : index_m(1) { };

  //============================================================
  // Accessors
  //============================================================

  // Dereference operator. Returns const ref to internal Loc.

  inline const Value_t &operator*() const { PAssert(!done()); return val_m; }

  // Member selection operator. Not really useful (ints have no 
  // members to invoke), but part of the required interface. 

  inline const Value_t *operator->() const { PAssert(!done()); return &val_m; }

  // Equality tests.
  // This only tests that the iterators have the same value.
  // It does not test whether the underlying domains are the same.
  
  inline bool operator==(const This_t &i) const { return index_m == i.index_m; }
  inline bool operator!=(const This_t &i) const { return index_m != i.index_m; }
  inline bool operator< (const This_t &i) const { return index_m <  i.index_m; }
  inline bool operator<=(const This_t &i) const { return index_m <= i.index_m; }
  inline bool operator> (const This_t &i) const { return index_m >  i.index_m; }
  inline bool operator>=(const This_t &i) const { return index_m >= i.index_m; }
  
  inline Value_t operator[](int n) const { return *(*this + n); }
  
  This_t operator+(int n) const
  {
    RangeIterator ret(*this);
    ret += n;
    return ret;
  }

  This_t operator-(int n) const
  {
    RangeIterator ret(*this);
    ret -= n;
    return ret;
  }

  ptrdiff_t
  operator-(const This_t &it) const
  {
    PAssert(domain_m == it.domain_m);
    return index_m - it.index_m;
  }
    
  // At-end (false) test.
  // Returns true if this iterator is at-end.

  bool done() const { return (index_m > domain_m.size()); }

  //============================================================
  // Mutators
  //============================================================

  // Assignment operator.

  This_t &operator=(const This_t &it)
  {
    if (&it != this)
      {
        domain_m = it.domain_m;
        val_m    = it.val_m;
        index_m  = it.index_m;
      }
    return *this;
  }

  // Pre-increment operator. 

  This_t &operator++() { increment();   return *this; }
  This_t &operator--() { increment(-1); return *this; }

  // Post-increment operator.
  // This has to make a copy, so prefer the above if possible.

  This_t operator++(int)
  {
    RangeIterator save(*this);
    increment();
    return save;
  }

  This_t operator--(int)
  {
    RangeIterator save(*this);
    increment(-1);
    return save;
  }

  inline This_t operator+=(int n) { increment(n);  return *this; }
  inline This_t operator-=(int n) { increment(-n); return *this; }

private:

  //============================================================
  // Data members.
  //============================================================

  // The domain we're iterating over.

  Domain_t domain_m;

  // Our current value.
  
  Value_t val_m;
  
  // Our current position. 
  // Has to be tracked separately due to negative strides.
  
  int index_m;
  
  //============================================================
  // Implementation functions
  //============================================================

  // Increment iterator.

  inline void increment()
  {
    PAssert(!done());
    val_m += domain_m.stride();
    ++index_m;
  }
  
  inline void increment(int n)
  {
    PAssert(!done());
    val_m += n * domain_m.stride();
    index_m += n;
  }
};

inline RangeIterator operator+(int n, const RangeIterator &it)
{
  RangeIterator ret(it);
  ret += n;
  return ret;
}


// } // namespace POOMA

#endif // POOMA_DOMAIN_RANGE_ITERATOR_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RangeIterator.h,v $   $Author: jac $
// $Revision: 1.5 $   $Date: 2000/06/14 22:28:47 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
