// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// 
// LinearExtrapolateFaceBC
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_LINEAREXTRAPOLATEFACEBC_H
#define POOMA_BCONDS_LINEAREXTRAPOLATEFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// LinearExtrapolateFaceBC: BCondCategory class which takes the values of the
//                          last two physical elements, and linearly
//                          extrapolates from the line through them out to all
//                          the guard elements. As implemented, this is only
//                          for logically rectilinear meshes.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/BCond.h"
#include "BConds/FieldBCondBase.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Full Description:
// 
// LinearExtrapolateFaceBC is a specialization of BCondCategory.
// 
// Takes the values of the last two physical elements, and linearly
// extrapolates from the line through them out to all the guard elements.  This
// is independent of centering. The intended use is for tasks like filling
// global guard layers of a Field of Vektors holding the mesh node position
// values, for which this does the right thing at hi and lo faces (exactly
// right for UniformRectilinearMesh meshes, and a reasonable thing to do for
// nonuniform RectilinearMesh meshes)
//-----------------------------------------------------------------------------

class LinearExtrapolateFaceBC : public BCondCategory<LinearExtrapolateFaceBC>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  LinearExtrapolateFaceBC(int face) : 
    face_m(face) { }
  LinearExtrapolateFaceBC(const LinearExtrapolateFaceBC &model) : 
    face_m(model.face()) { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  LinearExtrapolateFaceBC &operator=(const LinearExtrapolateFaceBC &rhs)
  {
    face_m = rhs.face();

    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }

private:

  int face_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for LinearExtrapolateFaceBC (partial
// specialization of the BCond class)
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
class BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC> :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, LinearExtrapolateFaceBC> This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, const LinearExtrapolateFaceBC &bc);
  

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() {};


  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Store a copy of the BCondCategory object used to construct this.
  LinearExtrapolateFaceBC bc_m;

  // More extra domains, specified as Ranges. Need to set negative strides for
  // tidy single data-parallel assignment in applyBoundaryCondition(). Because
  // the Domain_t deduced from the Array may be Interval<Dim>, which doesn't
  // admit negative strides, have to copy source domain contents from the
  // BCondBase data member into this Ranges (arrange things so only the source
  // domain needs to have negative strides, not the dest domain):
  Range<Geom::dimensions> destRange_m;

  // The two one-element-in-active-direction-thick Range objects between which
  // the slop for the linear extrapolation is computed (using values from the
  // associated Array):
  Range<Geom::dimensions> srcLayer1_m;
  Range<Geom::dimensions> srcLayer2_m;

  // Need these one-element-in-active-direction-thick Range objects for the
  // assignment in applyBoundaryCondition(), which (for now) must loop over all
  // the guard layers one at a time and extrapolate values from the two source
  // layers into them.
  Range<Geom::dimensions> destLayer_m;

};

//-----------------------------------------------------------------------------
// Full Description:
// 
// AllLinearExtrapolateFaceBC is a functor class that sets all of a Field's BCs
// to be  LinearExtrapolateFaceBC.
//-----------------------------------------------------------------------------

class AllLinearExtrapolateFaceBC
{
public:

  template<class Geom, class T, class EngineTag>
  void operator()(Field<Geom, T, EngineTag> &f) const
    {
      for (int i = 0; i < 2 * Geom::dimensions; i++)
        f.addBoundaryCondition(LinearExtrapolateFaceBC(i));
    }
};

#include "BConds/LinExtrapFaceBC.cpp"

#endif // POOMA_BCONDS_LINEAREXTRAPOLATEFACEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: LinExtrapFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:16:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
