// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   UniformRectilinearMesh
//   UniformRectilinearMeshData
//-----------------------------------------------------------------------------

#ifndef POOMA_MESHES_UNIFORMRECTILINEARMESH_H
#define POOMA_MESHES_UNIFORMRECTILINEARMESH_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// UniformRectilinearMesh
//  - A rectilinear mesh with uniform spacing between vertices. 
//    URM is just an interface class. It is implemented as a 
//    RefCountedPtr to a UniformRectilinearMeshData object. 
//
// UniformRectilinearMeshData
//  - The actual implementation of URM.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "CoordinateSystems/Cartesian.h"
#include "Domain/Interval.h"
#include "Domain/Loc.h"
#include "Engine/ConstantFunctionEngine.h"
#include "Engine/IndexFunctionEngine.h"
#include "Layout/GuardLayers.h"
#include "Tiny/Vector.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"
#include "Utilities/Unique.h"
#include "Utilities/PAssert.h"

#include <iosfwd>
#include <math.h> // for floor

//-----------------------------------------------------------------------------
//
// Full Description:
// UniformRectilinearMeshData<Dim, CoordinateSystem, T>
//
// Holds the data for a UniformRectilinearMesh. That class has a ref-counted
// instance of this class
//-----------------------------------------------------------------------------

template <int Dim, class CoordinateSystem, class T>
class UniformRectilinearMeshData : public RefCounted
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef UniformRectilinearMeshData<Dim, CoordinateSystem, T> This_t;
  
  // The range of the coordinate axes.
  
  typedef T AxisType_t;
  
  // The coordinate system.
  
  typedef CoordinateSystem CoordinateSystem_t;
  
  // The domain type.
  
  typedef Interval<Dim> Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = Dim };
  
  // The dimensionality of the coordinate system.
  
  enum { coordinateDimensions = CoordinateSystem::dimensions };
  
  // The type used to represent a point in the mesh.
  
  typedef Vector<coordinateDimensions, T> PointType_t;
  
  // The type of our GuardLayers object.
  
  typedef GuardLayers<Dim> GuardLayers_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  // The default constructor simply calls the default constructor for all
  // data members. The object must be subsequently initialized using the
  // initialize() member.
  
  UniformRectilinearMeshData() : initialized_m(false) { }
  
  // This constructor fully constructs the object.
  
  UniformRectilinearMeshData(const Domain_t &pd, const PointType_t &origin,
    const PointType_t &spacings);
    
  // Copy constructor.

  UniformRectilinearMeshData(const This_t &model);
  
  //---------------------------------------------------------------------------
  // Copy assignment operator.
  
  This_t &operator=(const This_t &rhs);

  //---------------------------------------------------------------------------
  // Initialize function. Can be used when default constructor was invoked to
  // build the object.
  
  void initialize(const Domain_t &pd, const PointType_t &origin,
    const PointType_t &spacings);

  //---------------------------------------------------------------------------
  // Empty destructor is fine.

  ~UniformRectilinearMeshData() { }

  // Position computation functor.
  
  class PositionFunctor
  {
    public:
    
      PositionFunctor()
      : mesh_m(NULL) { }
      
      PositionFunctor(const This_t &mesh)
      : mesh_m(&mesh)
      { }
      
      PointType_t operator()(int i0) const
      {
        PointType_t i(i0);
        return mesh_m->origin() + i * mesh_m->meshSpacing();
      }
      
      PointType_t operator()(int i0, int i1) const
      {
        PointType_t i(i0, i1);
        return mesh_m->origin() + i * mesh_m->meshSpacing();
      }

      PointType_t operator()(int i0, int i1, int i2) const
      {
        PointType_t i(i0, i1, i2);
        return mesh_m->origin() + i * mesh_m->meshSpacing();
      }

    private:
    
      const This_t *mesh_m;
  };

  // Volume computation functor.
  
  class VolumeFunctor
  {
    public:
    
      VolumeFunctor()
      : mesh_m(NULL)
      { }
      
      VolumeFunctor(const This_t &mesh)
      : mesh_m(&mesh)
      { }

      T operator()(int i0) const
      {
        PointType_t start(mesh_m->vertexPositions().read(i0));
        Region<coordinateDimensions, T> cell;
        for (int d = 0; d < coordinateDimensions; d++)
          cell[d] = 
            Region<1, T>(start(d), start(d) + mesh_m->meshSpacing()(d));
          
        return mesh_m->coordinateSystem().volume(cell);
      }
      
      T operator()(int i0, int i1) const
      {
        PointType_t start(mesh_m->vertexPositions().read(i0, i1));
        Region<coordinateDimensions, T> cell;
        for (int d = 0; d < coordinateDimensions; d++)
          cell[d] = 
            Region<1, T>(start(d), start(d) + mesh_m->meshSpacing()(d));
          
        return mesh_m->coordinateSystem().volume(cell);
      }

      T operator()(int i0, int i1, int i2) const
      {
        PointType_t start(mesh_m->vertexPositions().read(i0, i1, i2));
        Region<coordinateDimensions, T> cell;
        for (int d = 0; d < coordinateDimensions; d++)
          cell[d] = 
            Region<1, T>(start(d), start(d) + mesh_m->meshSpacing()(d));
          
        return mesh_m->coordinateSystem().volume(cell);
      }

    private:
    
      const This_t *mesh_m;
  };

  // Types of internal arrays used to store volumes, positions, spacings, 
  // and surface normals.

  typedef Array<Dim, T, IndexFunction<VolumeFunctor> > 
    CellVolumesArray_t;

  typedef Array<Dim, PointType_t, IndexFunction<PositionFunctor> > 
    PositionsArray_t;

  typedef Array<Dim, PointType_t, ConstantFunction> 
    SpacingsArray_t;

  typedef Array<Dim, Vector<2*Dim, PointType_t>, ConstantFunction> 
    SurfaceNormalsArray_t;

  //---------------------------------------------------------------------------
  // General accessors.

  // The coordinate system.
  
  const CoordinateSystem_t &coordinateSystem() const { return coordSys_m; }

  // Our guard layers object.
  
  const GuardLayers_t &guardLayers() const
  {
    return guardLayers_m;
  }
  
  // Whether or not we're initialized.
  
  bool initialized() const { return initialized_m; }
  
  // The mesh spacing and origin.
  
  const PointType_t &meshSpacing() const { return spacings_m; }
  const PointType_t &origin() const { return origin_m; }
  
  //---------------------------------------------------------------------------
  // Domain functions.

  // The vertex domain, as the mesh was constructed with.

  const Domain_t &physicalDomain() const { return physicalDomain_m; }

  // Function that returns a domain adjusted to give the indices of the cells.

  Interval<Dim> physicalCellDomain() const
  {
    Interval<Dim> ret;
    for (int d = 0; d < Dim; d++)
      ret[d] = Interval<1>(physicalDomain()[d].first(), 
        physicalDomain()[d].last() - 1);
    
    return ret;
  }

  // The total vertex domain, including mesh guard vertices.

  const Domain_t &totalDomain() const { return totalDomain_m; }

  // The total cell domain, including mesh guard cells.
  
  const Domain_t &totalCellDomain() const { return totalCellDomain_m; }
  
  //---------------------------------------------------------------------------
  // Volume functions.

  // Returns the total volume of the mesh (physical zones only):
  
  T totalVolumeOfCells() const
  {
    return sum(cellVolumes_m(physicalCellDomain()));
  }

  // Uses input domain to subset member Array of cell volumes; returns 
  // total subset volume.

  template <class DomainType>
  T totalVolumeOfCells(const DomainType &domain) const
  {
    return sum(cellVolumes_m(domain));
  }

  // Return entire array of cell volumes.

  const CellVolumesArray_t &cellVolumes() const
  {
    return cellVolumes_m;
  }
          
  //---------------------------------------------------------------------------
  // Vertex functions.

  // Nearest vertex index.

  inline Loc<Dim> nearestVertex(const PointType_t &point) const
  {
    Loc<Dim> loc;
    for (int d = 0; d < Dim; ++d)
      {
	loc[d] = floor((point(d)-origin_m(d)) * invspacings_m(d) + 0.5);
      }

    return loc;
  }

  // Nearest vertex index with all vertex coordinates below point.

  inline Loc<Dim> vertexBelow(const PointType_t &point) const
  {
    Loc<Dim> loc;
    for (int d = 0; d < Dim; d++)
      {
	loc[d] = floor((point(d) - origin_m(d)) * invspacings_m(d));
      }

    return loc;
  }

  // Loc for cell in cell-ctrd Field containing the point (x,y,z).

  inline Loc<Dim> cellContaining(const PointType_t &point) const
  {
    return vertexBelow(point);
  }

  // Return coordinates (position vectors) of indexed vertex subsets.

  inline const PositionsArray_t &vertexPositions() const
  {
    return vertexPositions_m;
  }

  // Typical vertex-vertex grid spacings for indexed cell subsets, returned as
  // Vectors of spacings, with one component per coordinate dimension

  inline const SpacingsArray_t &vertexDeltas() const
  {
    return vertexDeltas_m;
  }

  // --------------------------------------------------------------------------
  // The surface normals (unit vectors), like the volumes, are constant
  // everywhere in this type of mesh, for cartesian coordinates. For
  // curvilinear coordinates, they would vary across the mesh. These functions
  // return elements from the compute-engine-based Array of surface
  // normals. Each element of the Array is an array of 2*Dim Vectors, one 
  // for each face along each dimension of the mesh. The ordering of the faces 
  // is the usual thing (fortran ordering, 0 means low face of 1st dim, 1 means
  // high face of 1st dim, 2 means low face of 2nd dim, 3 means high face of
  // 2nd dim, and so on.

  inline const SurfaceNormalsArray_t &cellSurfaceNormals() const
  {
    return cellSurfaceNormals_m;
  }

private:

  // Initialization flag.
  
  bool initialized_m;
  
  // Index domain of the vertices.

  Domain_t physicalDomain_m;

  // Number of guard elements on each face of each dimension (symmetric).

  GuardLayers_t guardLayers_m;

  // Index domain of the vertices plus mesh guard layers.

  Domain_t totalDomain_m;

  // Index domain of the cells plus mesh guard layers.

  Domain_t totalCellDomain_m;

  // Origin of mesh (coordinate vector of first vertex).

  PointType_t origin_m;

  // Spacings between vertices (vector), and 1/spacings

  PointType_t spacings_m;
  PointType_t invspacings_m;

  // Arrays of cell volumes, positions, spacings, and surface normals. 
  // These all have some kind of compute-based engine.

  CellVolumesArray_t cellVolumes_m;
  PositionsArray_t vertexPositions_m;
  SpacingsArray_t vertexDeltas_m;
  SurfaceNormalsArray_t cellSurfaceNormals_m;
  
  // The coordinate system object.
  
  CoordinateSystem_t coordSys_m;

  // Function that sets up compute-based arrays.

  void initializeFunctionArrays();
  
  // Returns a Loc<Dim> giving the guardLayer sizes.
  
  Loc<Dim> guardLayerSizes() const
  {
    Loc<Dim> loc(physicalDomain().lengths());
    for (int d = 0; d < Dim; d++)
      loc[d] = Loc<1>(loc[d].first() / 2);
    return loc;
  }
};


//-----------------------------------------------------------------------------
// Full Description:
// 
// UniformRectilinearMesh is the simplest form of rectilinear mesh in that it
// has uniform spacing between vertices. This spacing can be different in
// each coordinate direction.
//
// The only sensible mesh boundary condition type for this mesh type is a
// linear-extrapolation boundary condition that extends the vertices out beyond
// each end of the box by adding more cells with unit spacings. N.B.: Fields on
// this type of mesh with periodic Field BC should still be okay, as should
// implementations of on differential operators, as long as those operators
// only use mesh *spacings*, and not mesh-node positions.
//
// UniformRectilinearMesh has three template parameters:
//   o Dim - the number of parameters required to index a point in the mesh;
//   o CoordinateSystem - the coordinate system.
//   o T - the range of any one of the coordinate axes;
// Default template arguments are provided for T and CoordinateSystem so that
// one can declare the commonly used N-dimensional mesh of doubles in cartesian
// coordinates using UniformRectilinearMesh<N>.
//
// Exported typedefs:
//
//   AxisType_t - the type used to represent the range of a coordinate axes
//     (the exported value of the template parameter T).
//   CellVolumesArray_t - the type of Array returned by cellVolumes().
//   CoordinateSystem_t - the exported value of the template parameter
//     CoordinateSystem.
//   Domain_t - the type of the domains for this mesh.
//   GuardLayers_t - the type of our GuardLayers object.
//   MeshData_t - the type of our data object.
//   PointType_t - the type used to represent a point in the mesh.
//   PositionsArray_t - the type of Array returned by vertexPositions().
//   SpacingsArray_t - the type of Array returned by vertexDeltas().
//   SurfaceNormalsArray_t - the type of Array returned by 
//     cellSurfaceNormals().
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimension - the dimensionality of the mesh.
//   coordinateDimensions - the dimension of the coordinate system.
//
// Constructors:
//
//   UniformRectilinearMesh(Dom1,...Dom3[, Origin[, Spacing]]) - three 
//     constructors that take domain arguments and combine them according
//     to the same rules as are used to construct Arrays. Contains optional
//     arguments for specifying the origin (default (0,0,...)) and
//     mesh spacings (default (1,1,...)).
//
// General accessors:
//
//   coordinateSystem() - returns this mesh's coordinate system.
//   guardLayers() - return this mesh's guard layers.
//   intialized() - returns true if this mesh has been initialized.
//   meshSpacing() - returns the mesh spacings.
//   origin() - returns the mesh origin.
//   print() - prints the mesh on an output stream.
//
// Domain functions:
//
//   physicalCellDomain() - returns a domain representing the cells. 
//     This is simply physicalDomain() with 1 index shaved off at the 
//     end of each dimension.
//   physicalDomain() - returns this mesh's physical domain
//   totalDomain() - returns this mesh's domain, including mesh guard layers
//   totalCellDomain() - like physicalCellDomain(), but including mesh guard 
//     layers
//
// Volume functions:
//
//   cellVolumes() - returns a Array containing the cell volumes.
//   totalVolumeOfCells([domain]) - returns the total volume of the mesh or,
//     if a domain is specified, of a subset of the mesh.
//
// Vertex functions:
//
//   cellContaining(point) - returns a Loc giving the indices of the cell
//     containing a point in space.
//   cellSurfaceNormals() - returns a Array containing the 2*Dim surface 
//     normals for each cell.
//   nearestVertex(point) - returns a Loc giving the indices of the vertex
//     nearest to a point in space.
//   vertexBelow(point) - returns a Loc giving the indices of the vertex
//     nearest to a point in space, but with all vertex coordinates below.
//   vertexDeltas() - returns a Array containing the vertex deltas.
//   vertexPositions() - returns a Array containing the vertex positions.
//-----------------------------------------------------------------------------

template<int Dim, class CoordinateSystem = Cartesian<Dim>,
  class T = POOMA_DEFAULT_POSITION_TYPE>
class UniformRectilinearMesh 
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.

  // This class.
  
  typedef UniformRectilinearMesh<Dim, CoordinateSystem, T> This_t;
  
  // This class' data object type.
  
  typedef UniformRectilinearMeshData<Dim, CoordinateSystem, T> MeshData_t;
  
  // The range of the coordinate axes.
  
  typedef typename MeshData_t::AxisType_t AxisType_t;
  
  // The coordinate system.
  
  typedef typename MeshData_t::CoordinateSystem_t CoordinateSystem_t;
  
  // The domain type.
  
  typedef typename MeshData_t::Domain_t Domain_t;
  
  // The number of indices required to select a point in this mesh.
  
  enum { dimensions = MeshData_t::dimensions };
  
  // The dimensionality of the coordinate system.
  
  enum { coordinateDimensions = CoordinateSystem_t::dimensions };
  
  // The type used to represent a point in the mesh.
  
  typedef typename MeshData_t::PointType_t PointType_t;
  
  // The type of our GuardLayers object.
  
  typedef typename MeshData_t::GuardLayers_t GuardLayers_t;
  
  // Types of internal arrays used to store volumes, positions, spacings, 
  // and surface normals.

  typedef typename MeshData_t::CellVolumesArray_t CellVolumesArray_t;
  typedef typename MeshData_t::PositionsArray_t PositionsArray_t;
  typedef typename MeshData_t::SpacingsArray_t SpacingsArray_t;
  typedef typename MeshData_t::SurfaceNormalsArray_t SurfaceNormalsArray_t;

  //---------------------------------------------------------------------------
  // Constructors. We don't use default arguments for the origin and spacings
  // because that produces ambiguities.

  UniformRectilinearMesh() { }
  
  template<class Dom1>
  UniformRectilinearMesh(const Dom1 &d1)
  : data_m(new MeshData_t(NewDomain1<Dom1>::combine(d1),
    PointType_t(0.0), PointType_t(1.0)))
  { }

  template<class Dom1>
  UniformRectilinearMesh(const Dom1 &d1, const PointType_t &origin)
  : data_m(new MeshData_t(NewDomain1<Dom1>::combine(d1),
	                        origin, PointType_t(1.0)))
  { }

  template<class Dom1>
  UniformRectilinearMesh(const Dom1 &d1, 
    const PointType_t &origin, const PointType_t &spacings)
  : data_m(new MeshData_t(NewDomain1<Dom1>::combine(d1), origin, spacings))
  { }

  template<class Dom1, class Dom2>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2)
  : data_m(new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2),
      PointType_t(0.0), PointType_t(1.0)))
  { }

  template<class Dom1, class Dom2>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2, 
    const PointType_t &origin)
  : data_m(new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2),
      origin, PointType_t(1.0)))
  { }

  template<class Dom1, class Dom2>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2, 
    const PointType_t &origin, const PointType_t &spacings)
  : data_m(new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2), 
      origin, spacings))
  { }

  template<class Dom1, class Dom2, class Dom3>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3)
  : data_m(new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      PointType_t(0.0), PointType_t(1.0)))
  { }

  template<class Dom1, class Dom2, class Dom3>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3, 
    const PointType_t &origin)
  : data_m(new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      origin, PointType_t(1.0)))
  { }

  template<class Dom1, class Dom2, class Dom3>
  UniformRectilinearMesh(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3, 
    const PointType_t &origin, const PointType_t &spacings)
  : data_m(new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      origin, spacings))
  { }
    
  // Copy constructor.
  
  UniformRectilinearMesh(const This_t &model)
  : data_m(model.data_m)
  { }

  //---------------------------------------------------------------------------
  // Initialize functions. We don't use default arguments for the origin 
  // and spacings because that produces ambiguities.

  void initialize(const This_t &model)
  {
    data_m = model.data_m;
  }
  
  template<class Dom1>
  void initialize(const Dom1 &d1)
  {
    data_m = new MeshData_t(NewDomain1<Dom1>::combine(d1),
      PointType_t(0.0), PointType_t(1.0));
  }

  template<class Dom1>
  void initialize(const Dom1 &d1, const PointType_t &origin)
  {
    data_m = new MeshData_t(NewDomain1<Dom1>::combine(d1),
      origin, PointType_t(1.0));
  }

  template<class Dom1>
  void initialize(const Dom1 &d1, 
    const PointType_t &origin, const PointType_t &spacings)
  {
    data_m = new MeshData_t(NewDomain1<Dom1>::combine(d1), origin, spacings);
  }

  template<class Dom1, class Dom2>
  void initialize(const Dom1 &d1, const Dom2 &d2)
  {
    data_m = new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2),
      PointType_t(0.0), PointType_t(1.0));
  }

  template<class Dom1, class Dom2>
  void initialize(const Dom1 &d1, const Dom2 &d2, 
    const PointType_t &origin)
  {
    data_m = new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2),
      origin, PointType_t(1.0));
  }

  template<class Dom1, class Dom2>
  void initialize(const Dom1 &d1, const Dom2 &d2, 
    const PointType_t &origin, const PointType_t &spacings)
  {
    data_m = new MeshData_t(NewDomain2<Dom1, Dom2>::combine(d1, d2), 
      origin, spacings);
  }

  template<class Dom1, class Dom2, class Dom3>
  void initialize(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3)
  {
    data_m = new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      PointType_t(0.0), PointType_t(1.0));
  }

  template<class Dom1, class Dom2, class Dom3>
  void initialize(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3, 
    const PointType_t &origin)
  {
    data_m = new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      origin, PointType_t(1.0));
  }

  template<class Dom1, class Dom2, class Dom3>
  void initialize(const Dom1 &d1, const Dom2 &d2, const Dom3 &d3, 
    const PointType_t &origin, const PointType_t &spacings)
  {
    data_m = new MeshData_t(NewDomain3<Dom1, Dom2, Dom3>::combine(d1, d2, d3), 
      origin, spacings);
  }

  //---------------------------------------------------------------------------
  // Copy assignment operator (shallow).
  
  This_t &operator=(const This_t &rhs)
  {
    if (&rhs != this)
      data_m = rhs.data_m;
        
    return *this;
  }

  //---------------------------------------------------------------------------
  // Empty destructor is fine.
  
  ~UniformRectilinearMesh() { }

  //---------------------------------------------------------------------------
  // General accessors.

  // The coordinate system.
  
  const CoordinateSystem_t &coordinateSystem() const 
  {
    PAssert(initialized()); 
    return data_m->coordinateSystem(); 
  }

  // Our guard layers object.
  
  const GuardLayers_t &guardLayers() const
  {
    PAssert(initialized()); 
    return data_m->guardLayers(); 
  }
  
  // Whether or not we're initialized.
  
  bool initialized() const { return data_m.isValid(); }
  
  // The mesh spacing and origin.
  
  const PointType_t &meshSpacing() const
  {
    PAssert(initialized()); 
    return data_m->meshSpacing(); 
  }
  
  const PointType_t &origin() const
  {
    PAssert(initialized()); 
    return data_m->origin(); 
  }

  //---------------------------------------------------------------------------
  // Domain functions.
  
  // The vertex domain, as the mesh was constructed with:

  const Domain_t &physicalDomain() const
  {
    PAssert(initialized()); 
    return data_m->physicalDomain(); 
  }

  // Function that returns a domain adjusted to give the indices of the cells.

  Interval<Dim> physicalCellDomain() const
  {
    PAssert(initialized()); 
    return data_m->physicalCellDomain(); 
  }

  // The total vertex domain, including mesh guard vertices:

  const Domain_t &totalDomain() const
  {
    PAssert(initialized()); 
    return data_m->totalDomain(); 
  }

  // The total cell domain, including mesh guard cells:

  const Domain_t &totalCellDomain() const
  {
    PAssert(initialized()); 
    return data_m->totalCellDomain(); 
  }

  //---------------------------------------------------------------------------
  // Volume functions.

  // Returns the total volume of the mesh (physical zones only):
  
  T totalVolumeOfCells() const
  {
    PAssert(initialized()); 
    return data_m->totalVolumeOfCells(); 
  }

  // Uses input domain to subset member Array of cell volumes; returns 
  // total subset volume.

  template <class DomainType>
  T totalVolumeOfCells(const DomainType &domain) const
  {
    PAssert(initialized()); 
    return data_m->totalVolumeOfCells(domain); 
  }

  // Return entire array of cell volumes.

  const CellVolumesArray_t &cellVolumes() const
  {
    PAssert(initialized()); 
    return data_m->cellVolumes(); 
  }
          
  //---------------------------------------------------------------------------
  // Vertex functions.

  // Nearest vertex index.

  inline Loc<Dim> nearestVertex(const PointType_t &point) const
  {
    PAssert(initialized()); 
    return data_m->nearestVertex(point); 
  }

  // Nearest vertex index with all vertex coordinates below point.

  inline Loc<Dim> vertexBelow(const PointType_t &point) const
  {
    PAssert(initialized()); 
    return data_m->vertexBelow(point); 
  }

  // Loc for cell in cell-ctrd Field containing the point (x,y,z).

  inline Loc<Dim> cellContaining(const PointType_t &point) const
  {
    PAssert(initialized()); 
    return data_m->cellContaining(point); 
  }

  // Return coordinates (position vectors) of indexed vertex subsets.

  const PositionsArray_t &vertexPositions() const
  {
    PAssert(initialized()); 
    return data_m->vertexPositions(); 
  }

  // Typical vertex-vertex grid spacings for indexed cell subsets, returned as
  // Vectors of spacings, with one component per coordinate dimension

  const SpacingsArray_t &vertexDeltas() const
  {
    PAssert(initialized()); 
    return data_m->vertexDeltas(); 
  }

  // --------------------------------------------------------------------------
  // The surface normals (unit vectors), like the volumes, are constant
  // everywhere in this type of mesh, for cartesian coordinates. For
  // curvilinear coordinates, they would vary across the mesh. These functions
  // return elements from the compute-engine-based Array of surface
  // normals. Each element of the Array is an array of 2*Dim Vectors, one 
  // for each face along each dimension of the mesh. The ordering of the faces 
  // is the usual thing (fortran ordering, 0 means low face of 1st dim, 1 means
  // high face of 1st dim, 2 means low face of 2nd dim, 3 means high face of
  // 2nd dim, and so on.

  const SurfaceNormalsArray_t &cellSurfaceNormals() const
  {
    PAssert(initialized()); 
    return data_m->cellSurfaceNormals(); 
  }

  // --------------------------------------------------------------------------
  // Print a UniformRectilinearMesh on an output stream.
  
  template <class Ostream>
  void print(Ostream &ostr) const;
  
private:

  // Our data, stored as a ref-counted pointer to simplify memory management.
  
  RefCountedPtr<MeshData_t> data_m;
};

//-----------------------------------------------------------------------------
//
// ostream inserter for UniformRectilinearMesh
//
//-----------------------------------------------------------------------------

template<int Dim, class CoordinateSystem, class T>
std::ostream &operator<<(std::ostream &ostr, 
  const UniformRectilinearMesh<Dim, CoordinateSystem, T> &mesh)
{
  mesh.print(ostr);
  return ostr;
}

#include "Meshes/UniformRectilinearMesh.cpp"

#endif // POOMA_MESHES_UNIFORMRECTILINEARMESH_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UniformRectilinearMesh.h,v $   $Author: julianc $
// $Revision: 1.33 $   $Date: 2000/07/20 23:39:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

