// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support Remote Engines
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_REMOTE_ENGINE_IO_H
#define POOMA_IO_REMOTE_ENGINE_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include "Pooma/Arrays.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include "IO/DomainIO.h"
#include "IO/LayoutIO.h"
#include "IO/EngineIO.h"
#include "Engine/RemoteEngine.h"
#include <string>

//-----------------------------------------------------------
// Functions for Engine<Dim,T,Remote<Tag> >
//-----------------------------------------------------------
// CTTI functions for remote engine classes.
//-----------------------------------------------------------
template <int Dim, class T, class Tag>
std::string PoomaCTTI(const Engine<Dim, T, Remote<Tag> >&);
template <int Dim, class T, class Tag>
std::string PoomaCTTI(const Engine<Dim, T, Remote<Tag> >&){
  char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "Engine<"+
    std::string(numbuf)+ PoomaCTTI(T())
    + ",Remote<"+ PoomaCTTI(Tag())+ "> >";
  return strng;
}

//-----------------------------------------------------------
// Serializers for Engine<Dim,T,Remote<Tag> >
//-----------------------------------------------------------

template <class Stream, int Dim, class T, class Tag>
int serialize(Stream& s, const Engine<Dim, T, Remote<Tag> >& engine);
template <class Stream, int Dim, class T, class Tag>
int serialize(Stream& s, const Engine<Dim, T, Remote<Tag> >& engine){

  int nBytes=0;
  typedef Engine<Dim, T, Remote<Tag> > Engine_t;
  typedef typename Engine_t::Domain_t Domain_t;
  typedef typename Engine_t::LocalEngine_t LocalEngine_t;

  // Obtain the attributes of the engine.
  // Get the domain.

  Domain_t dom= engine.domain();

  // Get the owning context.
  int owningContext= engine.owningContext();

  // Serialize the attributes.

  // Serialize the domain.
  nBytes+= serialize(s,dom);

  // Serialize the owning context.
  nBytes+= serialize(s,owningContext);

  // Serialize the engine if the context is local.
  if(engine.engineIsLocal()){
    LocalEngine_t localEngine= engine.localEngine();
    nBytes+= serialize(s,localEngine);
  }

  return nBytes;

}

template <class Stream, int Dim, class T, class Tag>
int deserialize(Engine<Dim, T, Remote<Tag> >& engine, Stream& s);
template <class Stream, int Dim, class T, class Tag>
int deserialize(Engine<Dim, T, Remote<Tag> >& engine, Stream& s){

  int nBytes=0;
  typedef Engine<Dim, T, Remote<Tag> > Engine_t;
  typedef typename Engine_t::Domain_t Domain_t;
  typedef typename Engine_t::LocalEngine_t LocalEngine_t;

  // Obtain the attributes of the engine.

  // Get the domain.
  Domain_t dom;
  nBytes+= deserialize(dom,s);

  // Get the owning context.
  int owningContext;
  nBytes+= deserialize(owningContext,s);

  // Instantiate the target engine with the stored
  // context and domain.
  engine= Engine_t(owningContext, dom);

  // Read the engine if the context is local.
  if(engine.engineIsLocal()){
    LocalEngine_t* localEngine;
    localEngine= &engine.localEngine();
    nBytes+= deserialize((*localEngine),s);
  }
  
  return nBytes;
}

template <int Dim, class T, class Tag>
int serialSizeof(const Engine<Dim, T, Remote<Tag> >& engine);
template <int Dim, class T, class Tag>
int serialSizeof(const Engine<Dim, T, Remote<Tag> >& engine){

  int nBytes=0;
  typedef Engine<Dim, T, Remote<Tag> > Engine_t;
  typedef typename Engine_t::Domain_t Domain_t;
  typedef typename Engine_t::LocalEngine_t LocalEngine_t;

  // Obtain the attributes of the engine.
  // Get the domain.

  Domain_t dom= engine.domain();

  // Size the attributes.

  // Size the domain.
  nBytes+= serialSizeof(dom);

  // Size the owning context.
  nBytes+= sizeof(int);

  // Size the engine if the context is local.
  if(engine.engineIsLocal()){
    LocalEngine_t localEngine= engine.localEngine();
    nBytes+= serialSizeof(localEngine);
  }

  return nBytes;

}

#endif // POOMA_IO_REMOTE_ENGINE_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: EngineIO.h,v $   $Author: ambro $
// $Revision: 1.6 $   $Date: 2000/06/27 21:40:41 $
// ----------------------------------------------------------------------
// ACL:rcsinfo


