// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   ConstField
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_CONSTFIELD_H
#define POOMA_FIELD_CONSTFIELD_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// ConstField : A read-only version of Field.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Array/Array.h"
#include "BConds/BCondList.h"
#include "Domain/Loc.h"
#include "Domain/NewDomain.h"
#include "Engine/DataObject.h"
#include "Engine/Engine.h"
#include "Engine/BrickEngine.h"
#include "Engine/ExpressionEngine.h"
#include "Engine/ForwardingEngine.h"
#include "Engine/IndirectionEngine.h"
#include "Engine/EngineFunctor.h"
#include "Engine/EnginePatch.h"
#include "Evaluator/Evaluator.h"
#include "Field/PrintField.h"
#include "Geometry/NoGeometry.h"
#include "Geometry/DiscreteGeometry.RTComp.h"
#include "Layout/INode.h"
#include "Pooma/PETE/ExpressionTraits.h"
#include "Pooma/View.h"
#include "Utilities/Conform.h"
#include "Utilities/ElementProperties.h"
#include "Utilities/NotifyPreRead.h"
#include "Utilities/PAssert.h"
#include <iosfwd>

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag> class ConstField;

//-----------------------------------------------------------------------------
// Traits classes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Tag class to tell constructors not to fill boundary conditions.
//-----------------------------------------------------------------------------

struct DontCopyBoundaryConditions {};

//-----------------------------------------------------------------------------
// View specializations for ConstField.
//-----------------------------------------------------------------------------

// Create a  view.

template<class Subject, class Dom> struct ArrayView;

template<class Geometry, class T, class EngineTag, class Sub1>
struct ArrayView<ConstField<Geometry, T, EngineTag>, Sub1>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Engine_t Engine_t;
  typedef Sub1 SDomain_t;

  // Deduce the template parameters for the output type.
  
  typedef typename NewEngine<Engine_t, SDomain_t>::Type_t NewEngine_t;
  static const int newDim = NewEngine_t::dimensions;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;
  
  // The output type.
  
  typedef Array<newDim, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s)
    {
      typedef NewEngineEngine<Engine_t, SDomain_t> NewEE_t;
      typedef NewEngineDomain<Engine_t, SDomain_t> NewED_t;

#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField::arrayRead view bounds error.");
#endif

      return Type_t(
        NewEE_t::apply(f.engine(), s),
        NewED_t::apply(f.engine(), s));
    }
};

// These are all the views that are obtained by the syntax a(b).

// Single-valued version. Handles scalars and Locs.

template<class Subject, class Domain, bool SV>
struct View1Implementation;

// Single-valued version. Handles scalars and Locs.

template<class Geometry, class T, class EngineTag, class Domain>
struct View1Implementation<ConstField<Geometry, T, EngineTag>, Domain, true>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // The return type is pretty simple here.
  
  typedef typename Subject_t::Element_t Type_t;
  
#if defined(__MWERKS__)

  inline static 
  Type_t make(const Subject_t &f, const Domain &s)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s);
    }

#else // not metrowerks

  template<class S1, class Combine>
  inline static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s);
    }

  template<class S1, class S2, class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s);
    }

  template<class S1, class S2, class S3,
    class Combine>
  inline static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
    {
      Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s);
    }

#endif // not metrowerks
};

// Non-single-valued implementation. Works for general domains
// including Nodes and INodes.

template<class Geometry, class T, class EngineTag, class Domain>
struct View1Implementation<ConstField<Geometry, T, EngineTag>, Domain, false>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename NewEngine<Engine_t, Domain>::Type_t NewEngine_t;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;
  typedef typename NewGeometry<Geometry, Domain>::Type_t NewGeometry_t;
  
  // The output type.
  
  typedef ConstField<NewGeometry_t, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  typedef NewEngineEngine<Engine_t, Domain> NewEE_t;
  typedef NewEngineDomain<Engine_t, Domain> NewED_t;
  typedef NewGeometryGeometry<Geometry, Domain> NewGG_t;
  typedef NewGeometryDomain<Geometry, Domain> NewGD_t;

#if defined(__MWERKS__)

  inline static
  Type_t make(const Subject_t &f, const Domain &s)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), s),
        "ConstField view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
    }


#else // not metrowerks

  template<class S1, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const Combine &)
  {
    Domain s(Combine::make(f, s1));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.totalDomain(), s),
	    "ConstField view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

  template<class S1, class S2, class Combine>
  static 
  Type_t make(const Subject_t &f, const S1 &s1,
	      const S2 &s2, const Combine &)
  {
    Domain s(Combine::make(f, s1, s2));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.totalDomain(), s),
	    "ConstField view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

  template<class S1, class S2, class S3,
    class Combine>
  static 
  Type_t make(const Subject_t &f,
	      const S1 &s1, const S2 &s2, const S3 &s3,
	      const Combine &)
  {
    Domain s(Combine::make(f, s1, s2, s3));
#if POOMA_BOUNDS_CHECK
    PInsist(contains(f.dtotalDomain(), s),
	    "ConstField view bounds error.");
#endif

      return Type_t(
        NewGeometry_t(NewGG_t::apply(f.geometry(), s), 
          NewGD_t::apply(f.geometry(), s)),
        NewEngine_t(NewEE_t::apply(f.engine(), s),
          NewED_t::apply(f.engine(), s)),
        f.bconds());
  }

#endif
};

// General version.

template<class Geometry, class T, class EngineTag>
struct View0<ConstField<Geometry, T, EngineTag> >
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef typename Subject_t::Engine_t Engine_t;

  typedef typename NewEngine<Engine_t, Domain_t>::Type_t NewEngine_t;
  static const int newDim = NewEngine_t::dimensions;
  typedef typename NewEngine_t::Element_t NewT_t;
  typedef typename NewEngine_t::Tag_t NewEngineTag_t;

  
  // typedef TemporaryNewDomain1<Domain_t, Sub1> NewDomain_t;
  //typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  //static const bool sv = DomainTraits<Domain_t>::singleValued;
  //  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner:
  
  //typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  //  typedef typename Dispatch_t::Type_t Type_t;

  typedef ConstField<Geometry,NewT_t,NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f)
    {   
      typedef NewEngineEngine<Engine_t, Domain_t> NewEE_t;
      typedef NewEngineDomain<Engine_t, Domain_t> NewED_t;

      return Type_t(	  
		    NewEE_t::apply(f.engine(), f.engine().domain()),
		    NewED_t::apply(f.engine(), f.engine().domain()));
    }
};


template<class Geometry, class T, class EngineTag, class Sub1>
struct View1<ConstField<Geometry, T, EngineTag>, Sub1>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  // At some point, we need to fix NewDomain1; until then, use
  // the temporary version from .h.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef TemporaryNewDomain1<Domain_t, Sub1> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The optimized domain combiner:
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1)
    {
#if defined(__MWERKS__)
    SDomain_t s(Combine_t::make(f, s1));
    return Dispatch_t::make(f, s);
#else
    return Dispatch_t::make(f, s1, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View1<ConstField<Geometry, T, EngineTag>, int>
{
  // Convenience typedef for the thing we're taking a view of.

  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // The return type is pretty simple here.

  typedef typename Subject_t::Element_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<1>(s1)),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s1);
    }
};

// These for views that are obtained by the syntax a(b, c).

template<class Geometry, class T, class EngineTag, 
  class Sub1, class Sub2>
struct View2<ConstField<Geometry, T, EngineTag>, Sub1, Sub2>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain2<Sub1, Sub2> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2)
    {
#if defined(__MWERKS__)
      SDomain_t s(Combine_t::make(f, s1, s2));
      return Dispatch_t::make(f, s);
#else
      return Dispatch_t::make(f, s1, s2, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View2<ConstField<Geometry, T, EngineTag>, int, int>
{
  // Convenience typedef for the thing we're taking a view of.

  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // The return type is pretty simple here.

  typedef typename Subject_t::Element_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1, int s2)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<2>(s1, s2)),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s1, s2);
    }
};

// These ones are for views that are obtained by the syntax a(b, c, d).

template<class Geometry, class T, class EngineTag, 
  class Sub1, class Sub2, class Sub3>
struct View3<ConstField<Geometry, T, EngineTag>, Sub1, Sub2, Sub3>
{
  // Convenience typedef for the thing we're taking a view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce domains for the output type.
  
  typedef typename Subject_t::Domain_t Domain_t;
  typedef NewDomain3<Sub1, Sub2, Sub3> NewDomain_t;
  typedef typename NewDomain_t::SliceType_t SDomain_t;
  
  // Deduce appropriate version of implementation to dispatch to.
  
  static const bool sv = DomainTraits<SDomain_t>::singleValued;
  typedef View1Implementation<Subject_t, SDomain_t, sv> Dispatch_t;

  // The combiner.
  
  typedef CombineDomainOpt<NewDomain_t, sv> Combine_t;
  
  // The return type.
  
  typedef typename Dispatch_t::Type_t Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Sub1 &s1, const Sub2 &s2, 
    const Sub3 &s3)
    {
#if defined(__MWERKS__)
      SDomain_t s(Combine_t::make(f, s1, s2, s3));
      return Dispatch_t::make(f, s);
#else
      return Dispatch_t::make(f, s1, s2, s3, Combine_t());
#endif
    }
};

template<class Geometry, class T, class EngineTag>
struct View3<ConstField<Geometry, T, EngineTag>, int, int, int>
{
  // Convenience typedef for the thing we're taking a view of.

  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // The return type is pretty simple here.

  typedef typename Subject_t::Element_t Type_t;

  inline static
  Type_t make(const Subject_t &f, int s1, int s2, int s3)
    {
#if POOMA_BOUNDS_CHECK
      PInsist(contains(f.totalDomain(), Loc<3>(s1, s2, s3)),
        "ConstField view bounds error.");
#endif
      return f.engine().read(s1, s2, s3);
    }
};

//-----------------------------------------------------------------------------
// Patch specialization for ConstField.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
struct Patch<ConstField<Geometry, T, EngineTag> >
{
  typedef ConstField<Geometry, T, EngineTag> Subject_t;
  typedef typename Subject_t::Engine_t OldEngine_t;
  typedef typename EngineFunctor<OldEngine_t, EnginePatch>::Type_t Engine_t;

  // Should a patch view of a field be a field or an array?
  // or should there even be such a thing?

  typedef Array<Subject_t::dimensions, T, typename Engine_t::Tag_t> 
    Type_t;

  inline static
  Type_t make(const Subject_t &subject, int i)
  {
    return Type_t(engineFunctor(subject.engine(), EnginePatch(i)));
  }
};

//-----------------------------------------------------------------------------
// ComponentView specialization for ConstField.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
struct ComponentView<int, ConstField<Geometry, T, EngineTag> >
{
  typedef int Type_t;
};

template<class Components, class Geometry, class T, class EngineTag>
struct ComponentView<Components, ConstField<Geometry, T, EngineTag> >
{
  // Convenience typedef for the thing we're taking a component view of.
  
  typedef ConstField<Geometry, T, EngineTag> Subject_t;

  // Deduce the template parameters for the output type.
  
  typedef typename Subject_t::Element_t Element_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef typename ComponentAccess<Element_t, Components>::Element_t NewT_t;
  typedef CompFwd<Engine_t, Components> NewEngineTag_t;
  typedef Engine<Geometry::dimensions, NewT_t, NewEngineTag_t> NewEngine_t;
  typedef NewComponentGeometry<Components, Geometry> NewCompGeom_t;
  typedef typename NewCompGeom_t::Type_t NewGeometry_t;
  
  // The output type.
  
  typedef ConstField<NewGeometry_t, NewT_t, NewEngineTag_t> Type_t;

  // A function that creates the view.
  
  inline static
  Type_t make(const Subject_t &f, const Components &components)
  {
    return Type_t(NewCompGeom_t::make(f.geometry(), components),
		  NewEngine_t(f.engine(), components), 
		  f.bconds());
  }
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// ConstField is a read-only Field. It resembles a read-only array, but 
// adds a few extra capabilities:
//   o Geometry - in addition to being able to return values, it can also
//       return the position in physical coordinate space corresponding to the
//       value.
//   o Automatic boundary conditions - ConstField can automatically update
//        parts of its domain using boundary condition objects stored in
//        a list. Before viewing, these boundary condition objects can be
//        queried as to whether the domain they manage needs updating and
//        then told to update, in necessary.
//
// ConstField has three template parameters:
//   o Geometry - the geomtry object used to associate values with physical 
//       positions;
//   o T - the type of element stored in the field;
//   o EngineTag - the tag that selects the Engine used to turn indices int
//       values.
// Default template arguments are provided for T and EngineTag.
//
// Exported typedefs:
//
//   CoordinateSystem_t - the coordinate system for this field.
//   Domain_t - the type of physical/total domains for this field.
//   Element_t - the type of elements managed by this field.
//   Engine_t - the type of Engine running this field.
//   EngineTag_t - the tag used to select Engine_t.
//   Geometry_t - the type of this field's geometry.
//   Layout_t - the type of the engine's layout.
//   PointType_t - the type used to represent a point corresponding to a
//     field value.
//   PositionArray_t - the type of array returned by x().
//   This_t - this class.
//
// Exported enumerations and constants:
//
//   dimensions - the dimensionality of the field.
//   coordinateDimensions - the dimension of the coordinate system.
//
// Constructors:
//
//   ConstField() - Default constructor. Creates an uninitialized ConstField.
//     Use initialize(...) to complete initialization.
//   ConstField(Geometry [, Layout]) - Initialize the ConstField from the 
//     Geometry object or from a Geometry and a Layout.  Layouts must be 
//     supplied for multi-patch Engines.
//   ConstField(const ConstField & [,const DontCopyBoundaryConditions &]) - 
//     copy constructor - creates a shallow copy of the Engine and of the 
//     BConds; i.e. the BConds still refer to their original subject. If a 
//     DontCopyBoundaryConditions tag object is also specified, the BCs will
//     not be copied.
//   ConstField(Geometry_t, Engine_t [, BCondList]) - constructors used to
//     construct views.
//
// Initialize functions:
//
//   initialize(Geometry)
//   inititlize(Geometry, BCondInitFunctor)
//   initialize(Geometry, BCondList)
//   initialize(Geometry, Layout)
//   initialize(Geometry, Layout, BCondInitFunctor)
//   initialize(Geometry, Layout, BCondList)
//   initialize(Field)
//   initialize(Field, DontCopyBoundaryConditions)
//   initialize(Geometry, Engine, BCondList)
//   initialize(Geometry, Engine)
//     - See constructors with same signatures.
//     
// Indexing functions:
//
//   read(int1[,...,int7]) - returns the field value given the indices.
//   operator()(int1[,...,int7]) - returns the field value given the indices.
//   x(int1[,...,int7]) - returns the position value given the indices.
//
// View creation functions:
//
//   read(dom1[,...,dom7]) - returns a view of the field in the specified
//     domain (read only).
//   operator()(dom1[,...,dom7]) - returns a view of the field in the 
//     specified domain (read/write).
//   x(dom1[,...,dom7]) - returns a view of the position array in the 
//     specified domain.
//
// Accessors:
//
//   bconds() - returns the boundary condition list.
//   domain() - a synonym for physicalDomain().
//   engine() - returns this field's engine.
//   geometry() - returns this field's geometry.
//   physicalDomain() - returns this field's physical domain.
//   totalDomain() - returns this field's total domain.
//   x() - all of the positions in the physical domain.
//-----------------------------------------------------------------------------

template<class Geometry, 
         class T = POOMA_DEFAULT_ELEMENT_TYPE,
         class EngineTag = POOMA_DEFAULT_ENGINE_TYPE>
class ConstField {
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  // This_t is a convenience typedef referring to this class.

  typedef ConstField<Geometry, T, EngineTag> This_t;

  // The number of indices required to select a point in this ConstField.
  
  enum { dimensions = Geometry::dimensions };
  
  // The dimensionality of the coordinate system.
  
  enum { coordinateDimensions = Geometry::coordinateDimensions };

  // This field's coordinate system.
  
  typedef typename Geometry::CoordinateSystem_t CoordinateSystem_t;
  
  // EngineTag_t is a typedef that specifies the type of engine.
  // Engine_t is a typedef that maps the engine-tag to the Engine type.
  
  typedef EngineTag EngineTag_t;
  typedef Engine<dimensions, T, EngineTag_t> Engine_t;
  
  // Layout_t is the Engine's layout.
  
  typedef typename Engine_t::Layout_t Layout_t;

  // Geometry_t is this field's geometry.
  
  typedef Geometry Geometry_t;

  // Element_t is the type of elements managed by this field's engine. 
  
  typedef typename Engine_t::Element_t Element_t;

  // zeroBase tells if this field's engine is zero-based or not. 
  
  enum { zeroBased = Engine_t::zeroBased };

  // The types of the total/physical domains. 

  typedef typename Geometry_t::Domain_t Domain_t; 
  
  // The type of array returned by x().
  
  typedef typename Geometry_t::PositionArray_t PositionArray_t;
  
  // The type of points returned by the geometry.
  
  typedef typename Geometry_t::PointType_t PointType_t;
  
  //---------------------------------------------------------------------------
  // Default constructor. Creates default geometry and engine objects.
  
  ConstField() { }  

  //---------------------------------------------------------------------------
  // Geometry constructors. We use the specified Geometry object to supply the
  // Engine's domain. This is appropriate for single patch engines.
  
  ConstField(const Geometry_t &geom)
  : engine_m(geom.totalDomain()), geometry_m(geom)
  { } 

  //---------------------------------------------------------------------------
  // Geometry/layout constructors. We use the specified Geometry object to
  // initialize our geometry and the Layout to initialize the engine. Clearly,
  // these must be synchronized. This is appropriate for multi-patch engines.
  
  ConstField(const Geometry_t &geom, const Layout_t &layout)
  : engine_m(layout), geometry_m(geom)
  { 
    PAssert(layout.domain() == geom.totalDomain());
  } 

  //---------------------------------------------------------------------------
  // Copy constructor. Pass the model on to the base classes and initialize our
  // geometry and bconds list.
  
  // Note that this does not retarget the BCs since that can only be done
  // if the ConstField is actually a Field (this is done in Field's copy
  // constructor). We do store copies of Fields as ConstFields in expression
  // objects, and we have to be able to fill BCs in them, so this must be 
  // done this way.
  
  ConstField(const This_t &model)
  : engine_m(model.engine()), geometry_m(model.geometry()), 
    bc_m(model.bconds())
  { }  

  //---------------------------------------------------------------------------
  // Copy constructor, except do not copy boundary conditions. 
  // Pass the model on to the base classes and initialize our geometry only.
  
  ConstField(const This_t &model, const DontCopyBoundaryConditions &)
  : engine_m(model.engine()), geometry_m(model.geometry())
  { }  

  //---------------------------------------------------------------------------
  // View creation constructors.
  // Should these be checking some assertions???
  
  ConstField(const Geometry_t &g, const Engine_t &e, const BCondList &bc)
  : engine_m(e), geometry_m(g), bc_m(bc)
  { }

  ConstField(const Geometry_t &g, const Engine_t &e)
  : geometry_m(g), engine_m(e)
  { }

  //---------------------------------------------------------------------------
  // This constructor is used in the process of taking views of 
  // Expression-Engines. These engines are zero-based, but can have a
  // bunch of arrays at the leaves that are not zero-based. This constructor
  // therefore adjusts the zero-based domain passed in via the ExpressionView
  // functor by the starting point of the Field's physical domain.
    
  template<class OtherGeom, class OtherEngineTag, class OtherDomain>
  ConstField(const ConstField<OtherGeom, T, OtherEngineTag> &model,
             const OtherDomain &domain) 
    : engine_m(newEngineEngine(model.engine(), domain),
	       newEngineDomain(model.engine(), domain)),
      geometry_m(model.geometry(), engine().domain()),
      bc_m(model.bconds())
{ }

  //---------------------------------------------------------------------------
  // Initialize functions mirroring the basic constructors.
  
  void initialize(const Geometry_t &g)
  { 
    engine_m   = Engine_t(g.totalDomain());
    geometry_m = g;
    bc_m.erase();
  }

  void initialize(const Geometry_t &g, const Layout_t &layout)
  { 
    PAssert(layout.domain() == g.totalDomain());
    
    engine_m   = Engine_t(layout);
    geometry_m = g;
    bc_m.erase();
  }
   
  void initialize(const This_t &model)
  { 
    engine_m   = model.engine();
    geometry_m = model.geometry();
    bc_m       = model.bconds();
  }
   
  void initialize(const This_t &model, const DontCopyBoundaryConditions &tag)
  { 
    engine_m   = model.engine();
    geometry_m = model.geometry();
    bc_m.erase();
  }
 
  void initialize(const Geometry_t &g, const Engine_t &e)
  { 
    engine_m   = e;
    geometry_m = g;
    bc_m.erase();
  }

  //---------------------------------------------------------------------------
  // Destructor.
  
  ~ConstField() { }
    
  //---------------------------------------------------------------------------
  // Engine accessors.

  inline Engine_t &engine() 
  { return engine_m; }
  inline const Engine_t &engine() const 
  { return engine_m; }
  
  //---------------------------------------------------------------------------
  // Geometry accessor.

  const Geometry_t &geometry() const
  {
    return geometry_m;
  }
  
  //---------------------------------------------------------------------------
  // Domain accessors.
 
  Domain_t domain() const
  {
    return geometry().physicalDomain();
  }
 
  Domain_t physicalDomain() const
  {
    return geometry().physicalDomain();
  }
   
  Domain_t totalDomain() const
  {
    return geometry().totalDomain();
  }

  //---------------------------------------------------------------------------
  // Patch accessor function returns the i'th patch.
  //---------------------------------------------------------------------------

  inline typename Patch<This_t>::Type_t
  patchLocal(EnginePatch::PatchID_t i) const
  {
    return Patch<This_t>::make(*this,i);
  }

//   inline typename PatchView<This_t>::Type_t
//   patchView(EnginePatch::PatchID_t i) const
//   {
//     return PatchView<This_t>::make(*this,i);
//   }

  inline int
  numPatchesLocal() const
  {
    return engineFunctor(engine(), EngineNumPatches());
  }
  
  //---------------------------------------------------------------------------
  // Instruct the field to make its own copy of its data.
  //
  // This is a little trickier for Fields than Arrays due to boundary 
  // conditions. And, ConstField yet another wrinkle. One cannot explicitly
  // add boundary conditions to a ConstField; however, it can have them since
  // a Field can be used as a ConstField. The subject of a BC must be
  // a Field (since the BC modifies the values), so it would be impossible to 
  // execute a makeOwnCopy() for the new ConstField. Thus, instead of copying
  // the boundary conditions, we apply them to the original Field, and then
  // we make a deep copy of the engine and replace our boundary conditions
  // with an empty list.
  //
  // Note that, like all makeOwnCopy constructs, this probably happens in
  // the parse thread, and thus may a) be slow, and b) lead to bad data
  // locality. In the threaded environment, it is probably better to create
  // a new Field, assign to it, and make a ConstField copy of that Field.
  
  inline void makeOwnCopy()
  {
    notifyPreRead();
    engine_m.makeOwnCopy();
    bc_m = BCondList();
  }

  //---------------------------------------------------------------------------
  // Short-circuit functions that return information from domain.
  
  inline int first(int d) const
  {
#if POOMA_BOUNDS_CHECK
    PInsist2(d >= 0 && d < dimensions,
             "ConstField<%d,...>::first() bounds error, index = %d.", Dim, d);
#endif
    return engine().first(d);
  }
  
  inline int last(int d) const
  {
#if POOMA_BOUNDS_CHECK
    PInsist2(d >= 0 && d < dimensions,
             "ConstField<%d,...>::last() bounds error, index = %d.", Dim, d);
#endif
    return engine().domain()[d].last();
  }
  
  inline int length(int d) const
  {
#if POOMA_BOUNDS_CHECK
    PInsist2(d >= 0 && d < dimensions,
             "ConstField<%d,...>::length() bounds error, index = %d.", Dim, d);
#endif
    return engine().domain()[d].length();
  }
  
  inline Loc<dimensions> firsts() const
  {
    return engine().domain().firsts();
  }
  
  inline Loc<dimensions> lasts() const
  {
    return engine().domain().lasts();
  }
  
  inline Loc<dimensions> lengths() const
  {
    return engine().domain().lengths();
  }
  
  //---------------------------------------------------------------------------
  // Boundary condition list accessors.
   
  const BCondList &bconds() const
  {
    return bc_m;
  }

  BCondList &bconds()
  {
    return bc_m;
  }
  
  //---------------------------------------------------------------------------
  // Boundary condition update functions. We need to take action before
  // reading and after writing.
  
  void notifyPreRead() const
  {
    // We originally wrote "bconds().notifyPreRead(engine().baseDomain());".
    // We use the zero argument version of notifyPreRead() because not all
    // engines have baseDomains nor does the notion of base domain make sense
    // for all engines. We need to get a sense regarding the efficiency of
    // this and modify if necessary.
    
    bconds().notifyPreRead();
  }
  
  void notifyPostWrite() const
  {
    // We originally wrote "bconds().notifyPostWrite(engine().baseDomain());".
    // We use the zero argument version of notifyPostWrite() because not all
    // engines have baseDomains nor does the notion of base domain make sense
    // for all engines. We need to get a sense regarding the efficiency of
    // this and modify if necessary.
    
    bconds().notifyPostWrite();
  }

  //---------------------------------------------------------------------------
  // View-creation operations. These operator() functions take zero or more
  // sub-domains, which combine to form a domain with dimensionality identical
  // to the rank of the field. They return a ConstField with no geometry and
  // an engine capable of viewing the specified domain. Views based on up 
  // to 3 subdomains are supported. Since ConstFields are read-only, the 
  // operator() and read() functions are identical. The zero argument 
  // version returns a view of the physical domain. If the resulting domain
  // is formed from a bunch of ints or a loc, these views correspond to the
  // indexing operation. In this case an element is returned.
  //
  // Theory of operation:
  //
  // Is the view of a Field an Array or a Field? If it were an Array, we 
  // wouldn't be able to update the boundary conditions. Therefore, it makes 
  // sense that views, along with all the other field-related entities that 
  // can find themselves at a leaf in a PETE expression tree, be Fields of 
  // some sort. What sort of Field is it? This is a little tricky, because 
  // what is possible depends on the sort of view. For example, if a Field f 
  // has a rectilinear mesh we take the view f(I) where I is an 
  // Interval<1> and it is fully contained in the physical domain (PD) of f, 
  // one can imagine that the view can be as full-featured a field as any 
  // other, except for the fact that the view's total domain (TD) would be 
  // identical to its PD. In particular, one could conceivably take a view 
  // of the field's geometry/mesh and, importantly, retain connectivity 
  // information. However, if I is a more complicated domain like a Range<>
  // or indirection list, there is no plausible prescription for deducing 
  // connectivity. This means we lose the notions of a mesh and centering. 
  // Therefore, a view of a Field cannot be used in a differential operator. 
  // Can we retain the ability for the Field to return positions? This is
  // less clear since we clearly know which points correspond to which
  // Field values. We could just take a view of the position arrays over the
  // same domain we're using to view the field values. In POOMA, we follow 
  // the philosophy that we want to retain as much information as possible.
  // Therefore, if we take a view of a Field using an Interval we get
  // a full-featured field with a view of the mesh. If we use other domains,
  // we punt by making the a view of a Field a Field to ensure boundary 
  // conditions are run, but making the Geometry of the resulting Field 
  // a dumb NoGeometry<>.

  inline typename View1<This_t, Domain_t>::Type_t 
  read() const
  {
    typedef View1<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, physicalDomain());
  }

  inline typename View1<This_t, Domain_t>::Type_t 
  readAll() const
  {
    typedef View1<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, totalDomain());
  }

  template<class Sub1> 
  inline typename View1<This_t, Sub1>::Type_t 
  read(const Sub1 &s1) const
  {
    typedef View1<This_t, Sub1> Ret_t;
    return Ret_t::make(*this, s1);
  }

  template<class Sub1, class Sub2> 
  inline typename View2<This_t, Sub1, Sub2>::Type_t 
  read(const Sub1 &s1, const Sub2 &s2) const
  {
    typedef View2<This_t, Sub1, Sub2> Ret_t;
    return Ret_t::make(*this, s1, s2);
  }

  template<class Sub1, class Sub2, class Sub3> 
  inline typename View3<This_t, Sub1, Sub2, Sub3>::Type_t 
  read(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
  {
    typedef View3<This_t, Sub1, Sub2, Sub3> Ret_t;
    return Ret_t::make(*this, s1, s2, s3);
  }

  inline typename View1<This_t, Domain_t>::Type_t 
  operator()() const
  {
    return This_t::read();
  }

  inline typename View1<This_t, Domain_t>::Type_t 
  all() const
  {
    return This_t::readAll();
  }

  template<class Sub1> 
  inline typename View1<This_t, Sub1>::Type_t 
  operator()(const Sub1 &s1) const
  {
    return This_t::read(s1);
  }

  template<class Sub1, class Sub2> 
  inline typename View2<This_t, Sub1, Sub2>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2) const
  {
    return This_t::read(s1, s2);
  }

  template<class Sub1, class Sub2, class Sub3> 
  inline typename View3<This_t, Sub1, Sub2, Sub3>::Type_t 
  operator()(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
  {
    return This_t::read(s1, s2, s3);
  }
  
  //---------------------------------------------------------------------------
  // View-creation operations yielding s.
   
  inline typename ArrayView<This_t, Domain_t>::Type_t
  arrayRead() const
  {
    typedef ArrayView<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, physicalDomain());
  };
   
  inline typename ArrayView<This_t, Domain_t>::Type_t
  arrayReadAll() const
  {
    typedef ArrayView<This_t, Domain_t> Ret_t;
    return Ret_t::make(*this, totalDomain());
  };
  
  //---------------------------------------------------------------------------
  // View-creation/indexing operations for position array. The zero argument
  // version returns a view of the positions on the physical domain. The xAll()
  // function returns all of the positions.
   
  inline typename View1<PositionArray_t, Domain_t>::ReadType_t
  x() const
  {
    typedef View1<PositionArray_t, Domain_t> Ret_t;
    return Ret_t::makeRead(geometry().x(), physicalDomain());
  }

  inline typename View1<PositionArray_t, Domain_t>::ReadType_t
  xAll() const
  {
    typedef View1<PositionArray_t, Domain_t> Ret_t;
    return Ret_t::makeRead(geometry().x(), totalDomain());
  }
  
  template<class Sub1>
  inline typename View1<PositionArray_t, Sub1>::ReadType_t
  x(const Sub1 &s1) const
  {
    typedef View1<PositionArray_t, Sub1> Ret_t;
    return Ret_t::makeRead(geometry().x(), s1);
  }

  template<class Sub1, class Sub2>
  inline typename View2<PositionArray_t, Sub1, Sub2>::ReadType_t
  x(const Sub1 &s1, const Sub2 &s2) const
  {
    typedef View2<PositionArray_t, Sub1, Sub2> Ret_t;
    return Ret_t::makeRead(geometry().x(), s1, s2);
  }

  template<class Sub1, class Sub2, class Sub3>
  inline typename View3<PositionArray_t, Sub1, Sub2, Sub3>::ReadType_t
  x(const Sub1 &s1, const Sub2 &s2, const Sub3 &s3) const
  {
    typedef View3<PositionArray_t, Sub1, Sub2, Sub3> Ret_t;
    return Ret_t::makeRead(geometry().x(), s1, s2, s3);
  }

  //---------------------------------------------------------------------------
  // Component-forwarding functions for position array. In some sense, these
  // represent syntactic sugar since one can do f.x().comp(2) instead of
  // f.xComp(2). We include these for completeness.
  
  inline typename ComponentView<Loc<1>, PositionArray_t>::Type_t
  xComp(int i1) const
  {
    return ComponentView<Loc<1>, PositionArray_t>::make(geometry().x(), 
      Loc<1>(i1));
  }
  
  inline typename ComponentView<Loc<2>, PositionArray_t>::Type_t
  xComp(int i1, int i2) const
  {
    return ComponentView<Loc<2>, PositionArray_t>::make(geometry().x(), 
      Loc<2>(i1, i2));
  }

  template<class Components>
  inline typename ComponentView<Components, PositionArray_t>::Type_t
  xComp(const Components &loc) const
  {
    return ComponentView<Components, PositionArray_t>::make(geometry().x(), loc);
  }
 
  //---------------------------------------------------------------------------
  // Component-forwarding functions. These work quite similar to the ones from
  // Array except we produce a ConstField with the same Geometry
  // (unless the Field had componentwise-centering, see the ComponentView<>
  // specializations above).

  // Explicitly read-only functions.
  
  inline typename ComponentView<Loc<1>, This_t>::Type_t
  compRead(int i1) const
  {
    return ComponentView<Loc<1>, This_t>::make(*this, Loc<1>(i1));
  }

  inline typename ComponentView<Loc<2>, This_t>::Type_t
  compRead(int i1, int i2) const
  {
    return ComponentView<Loc<2>, This_t>::make(*this, Loc<2>(i1, i2));
  }

  template<class Components>
  inline typename ComponentView<Components, This_t>::Type_t
  compRead(const Components &loc) const
  {
    return ComponentView<Components, This_t>::make(*this, loc);
  }
   
  // Potential read-write versions. The same as compRead() for ConstFields.
  
  inline typename ComponentView<Loc<1>, This_t>::Type_t
  comp(int i1) const
  {
    return This_t::compRead(i1);
  }

  inline typename ComponentView<Loc<2>, This_t>::Type_t
  comp(int i1, int i2) const
  {
    return This_t::compRead(i1, i2);
  }

  template<class Components>
  inline typename ComponentView<Components, This_t>::Type_t
  comp(const Components &loc) const
  {
    return This_t::compRead(loc);
  }

protected:

  // These constructors and initialize functions take an BCondList.
  // These are used to construct new Fields that have the same BCs as
  // an existing Field. They are only to be used by Field. A ConstField 
  // cannot retarget its boundary conditions. Thus, calling these for a 
  // pure ConstField will leave the BCs referring to their original field.
    
  ConstField(const Geometry_t &geom, const BCondList &bc)
  : engine_m(geom.totalDomain()), geometry_m(geom), bc_m(bc)
  { } 
  
  ConstField(const Geometry_t &geom, const Layout_t &layout, 
             const BCondList &bc)
  : engine_m(layout), geometry_m(geom), bc_m(bc)
  { 
    PAssert(layout.domain() == geom.totalDomain());
  } 
  
  void initialize(const Geometry_t &g, const BCondList &bc)
  { 
    engine_m   = Engine_t(g.totalDomain());
    geometry_m = g;
    bc_m       = bc;
  }
 
  void initialize(const Geometry_t &g, const Layout_t &layout, 
                  const BCondList &bc)
  { 
    PAssert(layout.domain() == g.totalDomain());
    
    engine_m   = Engine_t(layout);
    geometry_m = g;
    bc_m       = bc;
  }
 
  void initialize(const Geometry_t &g, const Engine_t &e, const BCondList &bc)
  { 
    engine_m   = e;
    geometry_m = g;
    bc_m       = bc;
  }

private:

  //---------------------------------------------------------------------------
  // This field's engine, geometry, and boundary conditions. Vrroom.

  Engine_t engine_m;
  Geometry_t geometry_m;
  BCondList bc_m;
};

//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to get the domain type or the
// (zero-based) domain itself from a ConstField. Used only by Expression-Engine.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>, DomainFunctorTag>
{
  typedef typename ConstField<Geometry, T, EngineTag>::Domain_t Type_t;
  inline static Type_t apply(const ConstField<Geometry, T, EngineTag> &f, 
    const DomainFunctorTag &)
    {
      return f.physicalDomain();
    }
};

//-----------------------------------------------------------------------------
// This specialization of LeafFunctor is used to apply a view (subsetting) 
// operation to a ConstField. The domain will always be zero-based since this
// is used only by Expression-Engine. This is why we add the firsts() to
// the domain.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag, class Domain>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>, ViewFunctorTag<Domain> >
{
  typedef typename View1<ConstField<Geometry, T, EngineTag>, Domain>::Type_t Type_t;
  inline static Type_t apply(const ConstField<Geometry, T, EngineTag> &f, 
    const ViewFunctorTag<Domain> &t) 
    {
      typedef View1<ConstField<Geometry, T, EngineTag>, Domain> Ret_t;
      return Ret_t::make(f, t.domain_m);
    }
};

//-----------------------------------------------------------------------------
// This version of LeafFunctor is used by Expression-Engines to used to 
// evaluate a ConstField using indices. 
//-----------------------------------------------------------------------------

template<class Geometry, class T, class EngineTag, int Dim>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>, EvalLeaf<Dim> >
{
  typedef typename ConstField<Geometry, T, EngineTag>::Element_t Type_t;
  inline static
  Type_t apply(const ConstField<Geometry, T, EngineTag> &f, 
    const EvalLeaf<Dim> &t) 
  {
    return t.eval(f.engine());
  }
};

//-----------------------------------------------------------------------------
// EngineView functor acting on ConstField.
// This version is probably wrong in general, because you want to take a view
// of the geometry as well as the field.
// The commented out lines will produce correct code once the appropriate
// LeafFunctors are defined for Geometries and EngineView.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class E, class Tag>
struct LeafFunctor<ConstField<Geometry, T, E>, EngineView<Tag> >
{
  typedef ConstField<Geometry, T, E> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineView<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t NewEngine_t;

  // typedef LeafFunctor<Geometry, EngineView<Tag> > GeomFunctor_t;
  // typedef typename GeomFunctor_t::Type_t NewGeometry_t;

  typedef typename NewEngine_t::Tag_t NewTag_t;
  typedef ConstField<Geometry, T, NewTag_t> Type_t;
  //  typedef ConstField<NewGeometry_t, T, NewTag_t> Type_t;

  inline static
  Type_t apply(const Subject_t &field, 
	       const EngineView<Tag> &tag)
  {
    // return Type_t(GeomFunctor_t::apply(field.geometry(), tag),
    return Type_t(field.geometry(),
		  LeafFunctor_t::apply(field.engine(), tag));
  }
};

//-----------------------------------------------------------------------------
// EngineApply functor acting on ConstField.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class E, class Tag>
struct LeafFunctor<ConstField<Geometry, T, E>, EngineApply<Tag> >
{
  typedef ConstField<Geometry, T, E> Subject_t;
  typedef typename Subject_t::Engine_t Engine_t;
  typedef LeafFunctor<Engine_t, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  inline static
  Type_t apply(const Subject_t &field, 
	       const EngineApply<Tag> &tag)
  {
    return LeafFunctor_t::apply(field.engine(), tag);
  }
};

//----------------------------------------------------------------------
// Apply the ConformTag to the leaves of the tree.
// Check to see if a given ConstField conforms.
//----------------------------------------------------------------------

template<class Geometry, class T, class EngineTag, int Dim>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>, ConformTag<Dim> >
{
  typedef bool Type_t;
  inline static Type_t apply1(const Interval<Dim> &d, 
    const ConformTag<Dim> &ct)
    {
      return conforms(d, ct);
    }
  template<int Dim2>
  inline static Type_t apply1(const Interval<Dim2> &d, 
    const ConformTag<Dim> &ct)
    {
      return false;
    }
  inline static Type_t apply(const ConstField<Geometry, T, EngineTag> &f,
    const ConformTag<Dim> &ct)
    {
      return apply1(f.physicalDomain(), ct);
    }
};

//----------------------------------------------------------------------
// For Fields, we use DataObjectApply to pass the request on to array
// (or return the default, if the field has no data block).
//----------------------------------------------------------------------

template<class RequestType, class Geometry, class T, class EngineTag>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>,
  DataObjectRequest<RequestType> >
{
  typedef typename ConstField<Geometry, T, EngineTag>::Engine_t Engine_t;
  enum { dataObject = Engine_t::dataObject };
  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;
  inline static
  Type_t apply(const ConstField<Geometry, T, EngineTag> &f,
	       const DataObjectRequest<RequestType> &functor)
  {
    return DataObjectApply<dataObject>::apply(f.engine(), functor);
  }
};

//----------------------------------------------------------------------
// Do what needs to be done prior to reading. For ConstFields, this means
// telling the boundary conditions where we're going to read.
//----------------------------------------------------------------------

template<class Geometry, class T, class EngineTag>
struct LeafFunctor<ConstField<Geometry, T, EngineTag>, NotifyPreReadTag>
{
  typedef bool Type_t;
  static Type_t apply(const ConstField<Geometry, T, EngineTag> &f,
    const NotifyPreReadTag &)
    {
      f.notifyPreRead();
      return true;
    }
};

//-----------------------------------------------------------------------------
// Handle general engine functor requests.
//-----------------------------------------------------------------------------

template<class Geometry, class T, class E, class Tag>
struct LeafFunctor<ConstField<Geometry, T, E>, EngineFunctorTag<Tag> >
{
  typedef typename ConstField<Geometry,T,E>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t,Tag>::Type_t Type_t;
  inline static
  Type_t apply(const ConstField<Geometry, T, E> &field,
	       const EngineFunctorTag<Tag> &tag)
  {
    return EngineFunctor<Engine_t,Tag>::apply(field.engine(), tag.tag());
  }
};

//---------------------------------------------------------------------------
// A specialization of EngineFunctor for field.
//---------------------------------------------------------------------------

template<class Geom, class T, class E, class Tag>
struct EngineFunctor<ConstField<Geom, T, E>, Tag>
{
  typedef typename ConstField<Geom,T,E>::Engine_t Engine_t;
  typedef typename EngineFunctor<Engine_t, Tag>::Type_t Type_t;

  inline static 
  Type_t apply(const ConstField<Geom, T, E> &field,
	       const Tag &tag)
  {
    return engineFunctor(field.engine(), tag);
  }
};

//-----------------------------------------------------------------------------
// Traits class for expressions containing field.
//-----------------------------------------------------------------------------

struct ExpressionIsField { };

template<class Geometry, class T, class EngineTag>
struct ExpressionTraits<ConstField<Geometry, T, EngineTag> >
{
  typedef ExpressionIsField Type_t;
};

// Fields combine with anything to give field expressions.

template<>
struct CombineExpressionTraits<ExpressionIsField, ExpressionIsField>
{
  typedef ExpressionIsField Type_t;
};

template<>
struct CombineExpressionTraits<ExpressionIsField, ExpressionIsArray>
{
  typedef ExpressionIsField Type_t;
};

template<>
struct CombineExpressionTraits<ExpressionIsField, ExpressionIsScalar>
{
  typedef ExpressionIsField Type_t;
};

template<>
struct CombineExpressionTraits<ExpressionIsArray, ExpressionIsField>
{
  typedef ExpressionIsField Type_t;
};

template<>
struct CombineExpressionTraits<ExpressionIsScalar, ExpressionIsField>
{
  typedef ExpressionIsField Type_t;
};


//-----------------------------------------------------------------------------
// Overload the << operator to print a ConstField to a stream.  This
// uses the 'PrintField' class to perform the formatting of the data.
// It will create a default printer, print out the field with it, and
// return the stream object.
//-----------------------------------------------------------------------------

template <class Geometry, class T, class EngineTag>
std::ostream &operator<<(std::ostream &o, 
                         const ConstField<Geometry, T, EngineTag> &cf)
{
  Pooma::blockAndEvaluate();
  PrintField().print(o, cf);
  return o;
}

template <class Geometry, class T, class EngineTag>
std::fstream &operator<<(std::fstream &f, 
                         const ConstField<Geometry, T, EngineTag> &cf)
{
  Pooma::blockAndEvaluate();
  PrintField().print(f, cf);
  return f;
}

#endif // POOMA_FIELD_CONSTFIELD_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ConstField.h,v $   $Author: julianc $
// $Revision: 1.74 $   $Date: 2000/07/20 22:25:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
