// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// ConnectionBase
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Connect/Connection.h"
#include "Connect/Connector.h"
#include "Utilities/PAssert.h"
#include <string>
#include <vector>


//----------------------------------------------------------------------
//
// ConnectionBase Constructor
// The constructor takes the strings for the name and type.
//
//----------------------------------------------------------------------

ConnectionBase::ConnectionBase(const char *conname, const char *contype)
  : name_m(conname), type_m(contype)
{
}


//----------------------------------------------------------------------
//
// ConnectionBase Destructor
//
//----------------------------------------------------------------------

ConnectionBase::~ConnectionBase()
{
  // At this point, the subclasses should have removed all
  // connections

  PAssert(size() == 0);
}


//----------------------------------------------------------------------
//
// Return whether we are connected properly right now.  By default,
// we are.
//
//----------------------------------------------------------------------

bool ConnectionBase::connected() const
{
  return true;
}


//----------------------------------------------------------------------
//
// Connect a new Connector.  The Connector maps from a single data object
// (such as a Field) to this ConnectionBase.  The base class pointer is used
// to store the data, since Connector is a templated class.  The user must
// 'new' thie Connector object and give the allocated instance to this
// class, which will delete the instance when the connection closes.
// Return the ConnectorBase pointer just connected.
//
//----------------------------------------------------------------------

ConnectorBase *ConnectionBase::connect(ConnectorBase *cb)
{
  PAssert(connected());

  // make sure we do not already have this connector; if we do, just return

  for (iterator a = begin(); a != end(); ++a)
    if (cb == *a)
      return cb;

  // this is a new one; add it to our list

  PAssert(cb != 0);
  connectors_m.push_back(cb);
  return cb;
}


//----------------------------------------------------------------------
//
// Disconnect an existing Connector, but do not delete it.  This removes
// the connector from our list, and informs it it should not use the
// ConnectionBase anymore.
// ConnectorBase must implement a 'notify(int)' virtual function.
// Return the ConnectorBase pointer just disconnected.
//
//----------------------------------------------------------------------

ConnectorBase *ConnectionBase::disconnect(ConnectorBase *cb)
{
  PAssert(connected());

  // find the connection; if we do not have it, it is an error

  for (iterator a = begin(); a != end(); ++a)
    {
      if (cb == *a)
	{
	  // we have the item; notify it and remove it from our list

	  (*a)->notify(*this, ConnectionBase::disconnectEvent);
	  connectors_m.erase(a);

	  return cb;
	}
    }

  // if we're here, its an error

  PAssert(false);
  return 0;
}


//----------------------------------------------------------------------
//
// Update the connections - tell the connections to update, or just an
// individual one.
//
//----------------------------------------------------------------------

void ConnectionBase::update(ConnectorBase *cb)
{
  PAssert(connected());

  // loop through all the connectors; update them all, or just the
  // one mentioned.

  for (iterator a = begin(); a != end(); ++a)
    if (cb == 0 || cb == *a)
      (*a)->update();
}


//----------------------------------------------------------------------
//
// Allow for interaction with the connection.  An optional string
// can be provided to tell how to do the interaction.
//
//----------------------------------------------------------------------

void ConnectionBase::interact(const char *)
{
  // by default, this does nothing
}


//----------------------------------------------------------------------
//
// Disconnect any remaining connectors, and delete them.
//
//----------------------------------------------------------------------

void ConnectionBase::disconnectConnectors()
{
  while (size() > 0)
    {
      ConnectorBase *cb = *(begin());
      disconnect(cb);
      delete cb;
    }
}


//----------------------------------------------------------------------
//
// Completely close the connection and remove all connectors.  This is
// the equivalent of running the destructor, using this routine you can
// control when the connection is closed.  By default it just removes
// all connectors.
//
//----------------------------------------------------------------------

void ConnectionBase::close()
{
  disconnectConnectors();
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Connection.cmpl.cpp,v $   $Author: swhaney $
// $Revision: 1.4 $   $Date: 2000/03/07 13:16:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
