// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// BCondList: holds a list of boundary conditions.
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_BCONDLIST_H
#define POOMA_BCONDS_BCONDLIST_H

#include "BConds/BCondItem.h"
#include "Utilities/RefCountedPtr.h"
#include "Utilities/RefCounted.h"

#include <vector>

#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
#include <iostream>
#endif


//-----------------------------------------------------------------------------
// BCondListData is a ref-counted object that holds the data for a BC List.
//-----------------------------------------------------------------------------

class BCondListData : public RefCounted
{
public:

  //---------------------------------------------------------------------------
  // Constructors.

  // The default constructor simply calls the default constructor for the
  // contained vector.
  
  BCondListData() { }
    
  // Copy constructor.

  BCondListData(const BCondListData &model)
  : data_m(model.data_m) { }
  
  //---------------------------------------------------------------------------
  // Copy assignment operator.
  
  BCondListData &operator=(const BCondListData &rhs)
  {
    data_m = rhs.data_m;
    
    return *this;
  }

  //---------------------------------------------------------------------------
  // We need to delete the actual BCondItems.

  ~BCondListData() 
  {
    for (int i = 0; i < data_m.size(); i++)
      delete data_m[i]; 
  }

  //---------------------------------------------------------------------------
  // Return the number of BCondItems.
  
  int size() const
  {
    return data_m.size();
  }
  
  //---------------------------------------------------------------------------
  // Returns the ith element.
  
  BCondItem *elem(int i) const
  {
    return data_m[i];
  }
  
  BCondItem* &elem(int i)
  {
    return data_m[i];
  }

  //---------------------------------------------------------------------------
  // Pushes an item on the list.

  void push_back(BCondItem *item)
  {
    data_m.push_back(item);
  }
   
private:

  std::vector<BCondItem *> data_m;
};


//-----------------------------------------------------------------------------
// BCondList is a container that dispatches events to the list of boundary 
// conditions it contains. 
//-----------------------------------------------------------------------------

class BCondList {
public:

  //---------------------------------------------------------------------------
  // The default constructor simply makes an empty list. 
  
  BCondList() : list_m(new BCondListData) { }

  //---------------------------------------------------------------------------
  // The copy constructor makes a shallow copy of the data.
  
  BCondList(const BCondList &model)
  : list_m(model.list_m)
  {
  }

  //---------------------------------------------------------------------------
  // Destructor: trivial.
  
  ~BCondList()
  {
  }

  //---------------------------------------------------------------------------
  // Replaces the current list with a private copy of itself.
  
  void makeOwnCopy()
  {
    list_m = new BCondListData(*list_m);
  }

  //---------------------------------------------------------------------------
  // Replaces the current list with an empty list.
  
  void erase()
  {
   list_m = new BCondListData;
  }

  //---------------------------------------------------------------------------
  // Adds a boundary condition to the end of our list.

  template<class Subject, class Category>
  void addBoundaryCondition(const Subject &s, const Category &c)
  {
    // This is the only place I found in r2 where we require an engine
    // to know whether it is initialized or not.  Creeping requirements
    // are bad, and in this case I don't see a huge value in not initializing
    // the bc_m on startup.  This should be changed.

    BCondItem *bc = c.create(s);
    list_m->push_back(bc);

#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
    std::cout << "Adding BC " << (void *) bc << std::endl;
#endif
  }
  
  //---------------------------------------------------------------------------
  // Notify the boundary conditions about pre-read/post-write events.

  void notifyPreRead() const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->notifyPreRead();
  }
  
  template<class ReadDomain>
  void notifyPreRead(const ReadDomain &readDomain) const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->notifyPreRead(readDomain);
  }

  void notifyPostWrite() const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->notifyPostWrite();
  }

  template<class WriteDomain>
  void notifyPostWrite(const WriteDomain &writeDomain) const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->notifyPostWrite(writeDomain);
  }
  
  //---------------------------------------------------------------------------
  // Set/clear the dirty flags for all of the boundary conditions.

  void setDirty() const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->setDirty();
  }

  void clearDirty() const
  {
    for (int i = 0; i < list_m->size(); ++i)
      list_m->elem(i)->clearDirty();
  }
  
  //---------------------------------------------------------------------------
  // Give access to a specific boundary condition.

  BCondItem *operator()(int i) const
  {
#if POOMA_BOUNDS_CHECK
    PInsist2(i >= 0 && i < list_m->size(),
      "BCondList bounds error: index = %d, size = %d.",
      i, list_m->size());
#endif

    return list_m->elem(i);
  }

  BCondItem * &operator()(int i)
  {
#if POOMA_BOUNDS_CHECK
    PInsist2(i >= 0 && i < list_m->size(),
      "BCondList bounds error: index = %d, size = %d.",
      i, list_m->size());
#endif

    return list_m->elem(i);
  }

  //---------------------------------------------------------------------------
  // Return the number of boundary conditions.
  
  int size() const { return list_m->size(); }
  
private:

  RefCountedPtr<BCondListData> list_m;
};


#endif // POOMA_BCONDS_BCONDLIST_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BCondList.h,v $   $Author: swhaney $
// $Revision: 1.13 $   $Date: 2000/04/27 22:33:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
