﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/codecatalyst/CodeCatalystRequest.h>
#include <aws/codecatalyst/CodeCatalyst_EXPORTS.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CodeCatalyst {
namespace Model {

/**
 */
class ListEventLogsRequest : public CodeCatalystRequest {
 public:
  AWS_CODECATALYST_API ListEventLogsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListEventLogs"; }

  AWS_CODECATALYST_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the space.</p>
   */
  inline const Aws::String& GetSpaceName() const { return m_spaceName; }
  inline bool SpaceNameHasBeenSet() const { return m_spaceNameHasBeenSet; }
  template <typename SpaceNameT = Aws::String>
  void SetSpaceName(SpaceNameT&& value) {
    m_spaceNameHasBeenSet = true;
    m_spaceName = std::forward<SpaceNameT>(value);
  }
  template <typename SpaceNameT = Aws::String>
  ListEventLogsRequest& WithSpaceName(SpaceNameT&& value) {
    SetSpaceName(std::forward<SpaceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time when you want to start retrieving events, in coordinated
   * universal time (UTC) timestamp format as specified in <a
   * href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  ListEventLogsRequest& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time after which you do not want any events retrieved, in coordinated
   * universal time (UTC) timestamp format as specified in <a
   * href="https://www.rfc-editor.org/rfc/rfc3339#section-5.6">RFC 3339</a>.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  ListEventLogsRequest& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the event.</p>
   */
  inline const Aws::String& GetEventName() const { return m_eventName; }
  inline bool EventNameHasBeenSet() const { return m_eventNameHasBeenSet; }
  template <typename EventNameT = Aws::String>
  void SetEventName(EventNameT&& value) {
    m_eventNameHasBeenSet = true;
    m_eventName = std::forward<EventNameT>(value);
  }
  template <typename EventNameT = Aws::String>
  ListEventLogsRequest& WithEventName(EventNameT&& value) {
    SetEventName(std::forward<EventNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token returned from a call to this API to indicate the next batch of
   * results to return, if any.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListEventLogsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to show in a single call to this API. If the
   * number of results is larger than the number you specified, the response will
   * include a <code>NextToken</code> element, which you can use to obtain additional
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListEventLogsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_spaceName;

  Aws::Utils::DateTime m_startTime{};

  Aws::Utils::DateTime m_endTime{};

  Aws::String m_eventName;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_spaceNameHasBeenSet = false;
  bool m_startTimeHasBeenSet = false;
  bool m_endTimeHasBeenSet = false;
  bool m_eventNameHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CodeCatalyst
}  // namespace Aws
