package Affa::esxi;
use strict;

my $debug=0;

use Affa::lib qw(ExecCmd setlog lg dbg);
eval { require VMware::VIRuntime };
if( $@ )
	{
	my $s="VMware Infrastructure (VI) Perl Toolkit is not installed.\n";
	$s .= "  Download from http://www.vmware.com/support/developer/viperltoolkit/\n";
	$s .= "  Unpack the tarball and run vmware-install.pl";
	print "ERROR: $s\n";
	main::affaErrorExit( $s );
	}
require VMware::VIRuntime;
dbg( "Using VMware Infrastructure (VI) Perl Toolkit: Version " . $VMware::VIRuntime::VERSION );
use Data::Dumper;
$Data::Dumper::Indent = 3; 
use File::Path;
use Date::Format;
use Net::Domain qw (hostname hostfqdn hostdomain);
use Compress::Bzip2;

sub get_virtual_machines;
sub create_snapshot;
sub remove_bexi_snapshots;
sub get_vm;
sub checkAPIConnection($$);
sub syncESXi($);
sub ssh_cmd(@);
sub escape_filename($);

sub checkAPIConnection($$)
	{
	(my $username, my $password ) = @_;
	my $esxi_server = $main::job{'remoteHostName'};
	eval { Vim::login(service_url => "https://$esxi_server/sdk", user_name => $username, password => $password); };
	if ($@)
		{
		dbg( $@ );
		return 0;
		}
	eval { Vim::logout() }; if ($@) {main::affaErrorExit( $@ )}
	return 1;
	}

sub syncESXi($)
	{
	(my $cmd ) = @_;
	my $linkDest = "$main::job{'RootDir'}/$main::jobname/$cmd";
	my $username = $main::job{'ESXiUsername'};
	my $password = $main::job{'ESXiPassword'};

	my $RootDir = $main::job{'RootDir'} . "/" . $main::jobname . "/scheduled.running/";
	my $vmname = $main::job{'ESXiVMName'};
	my $esxi_server = $main::job{'remoteHostName'};
	my $status;

	dbg( "VI API connect to $esxi_server as user $username" );
	eval { Vim::login(service_url => "https://$esxi_server/sdk", user_name => $username, password => $password); };
	if ($@) {main::affaErrorExit( $@ )}
	
	my $current_vm = get_vm( $vmname );
	main::affaErrorExit( "Virtual machine '$vmname' does not exist." ) if not $current_vm;
	if( $current_vm->snapshot ) {
		remove_bexi_snapshots( $current_vm->snapshot->currentSnapshot, $current_vm->snapshot->rootSnapshotList );
		$current_vm = get_vm( $vmname );
	}
	not create_snapshot($current_vm) or main::affaErrorExit( "Creating snapshot failed." );
	$current_vm = get_vm( $vmname );
	dbg( "VI API disconnecting." );
	eval { Vim::logout() }; if ($@) {main::affaErrorExit( $@ )}
	
	
	# get datastore name mapping
	my %ds_map;
	my $ds_url=$current_vm->config->datastoreUrl;
	foreach my $d (@$ds_url) {
		$ds_map{$d->name}=$d->url;
	}
	if( $debug) {print "### get datastore name mapping\n"; print Dumper( \%ds_map);}

	dbg( "Datastore mapping:");
	while( (my $k, my $v ) = each( %ds_map )) {
		dbg( "  '$k' --> '$v'");
	}
	
	# VM Source Datastore
	my $datastore = $current_vm->datastore->[0]->value;
	print "datastore=$datastore\n" if $debug; 
	
	# VM Source Directory
	(my $vmpath = $current_vm->summary->config->vmPathName) =~ s/^\[(.*)\] (.*\/)(.*)$/$ds_map{$1}$2/;
	print "vmpath=$vmpath\n" if $debug; 
	$vmpath=~s/ /\\ /g;
	
	# find active diskfiles
	my $cdisks=$current_vm->config->hardware->device;
	my %cur_disks;
	foreach my $d (@$cdisks) {
		if( $d->deviceInfo->label =~ /^Hard Disk/ ) {
			(my $e = $d->backing->fileName) =~ s/^\[(.*)\] (.*)$/$ds_map{$1}$2/;
			$cur_disks{$e}=1;
		}
	}
	if( $debug) {print "### find active diskfiles\n"; print Dumper( \%cur_disks);}
	my %files_to_sync;
	my %datastore_path;
	
	# get snapshot file names
	my $snapshots=$current_vm->layout->snapshot;
	foreach my $d (@$snapshots) {
		my $a=$d->snapshotFile;
		foreach my $e (@$a) {
			$e =~ /^\[(.*)\] (.*\/)(.*)$/;
			$datastore_path{$ds_map{$1}.$2}=1;
			$files_to_sync{$ds_map{$1}.$2.$3}=$ds_map{$1}.$2 if not $cur_disks{$ds_map{$1}.$2.$3};
		}
	}
	
	# get disk file names
	my $disks=$current_vm->layout->disk;
	foreach my $d (@$disks) {
		my $a=$d->diskFile;
		foreach my $e (@$a) {
			$e =~ /^\[(.*)\] (.*\/)(.*\.vmdk)$/;
			$datastore_path{$ds_map{$1}.$2}=1;
		$files_to_sync{$ds_map{$1}.$2.$3}=$ds_map{$1}.$2 if not $cur_disks{$ds_map{$1}.$2.$3};
		}
	}
	
	if( $debug) {print "### datastore_path\n"; print Dumper( \%datastore_path );}
	
	
	# get diskfiles from datastores
	my @cmd=(
		"/usr/bin/ssh -o HostKeyAlias=$main::jobname -p $main::job{'sshPort'}",
		$esxi_server, 
		"\"find " . escape_filename($vmpath) . " -type f -name '*-delta.vmdk'\"" );
	ssh_cmd(@cmd);
	my $res = $Affa::lib::ExecCmdOutout;
	foreach my $path (keys %datastore_path) {
		$path=~s/ /\\ /g;
		@cmd=(
			"/usr/bin/ssh -o HostKeyAlias=$main::jobname -p $main::job{'sshPort'}",
			$esxi_server, 
			"\"find " . escape_filename($path) . " -type f -name '*-flat.vmdk'\"" );
		ssh_cmd(@cmd);
		$res .= $Affa::lib::ExecCmdOutout;
	}
	my @files_on_disk=split(/[\r\n]/, $res);
	if( $debug) {print "### Files on Disk\n"; print Dumper( \@files_on_disk );}
	
	foreach my $d (@files_on_disk ) {
		(my $file = $d ) =~ s/-(flat|delta)\.vmdk$/.vmdk/;
		$files_to_sync{$d}=$files_to_sync{$file} if $files_to_sync{$file};	
	}
	
	if( $debug) {print "### Files to sync\n";print Dumper( sort \%files_to_sync );}

	dbg( "Files to sync:" );
	foreach my $f (keys %files_to_sync) {
		dbg( "  $f");
	}
	
	
	# hier syncen

	# The dropbear ssh service on ESXi only accepts command strings < 1024 chars
	# Therefore we need to sync each file separately 
	
	my $rsync_out='';
	while( (my $k, my $v) = each( %files_to_sync )) 
		{
		$k=escape_filename($k);
		(my $dest="$RootDir/$v") =~ s:/+:/:g;
		File::Path::mkpath( $dest, 0, 0700 ) unless -d $dest;
		lg( "Syncing $k" );
	
		(my $destfile=$k)=~s/.*\///;

		if( $cmd && -f "$linkDest$k" )
			{
			if( (-s "$linkDest$k" > $main::ChunkThresholdSize) && ($cmd ne 'scheduled.0' or $main::job{'scheduledKeep'}>1) )
				{
				main::chunk( $cmd, $k, ".affa-chunked", $main::job{'ChunkSize'}, "$dest$destfile" );
				}
			else
				{
				dbg( "Copying $linkDest$k to $dest$destfile.");
				open( IF, "$linkDest$k" ) or main::affaErrorExit("Could not open $linkDest$k");
				open( OF, ">$dest$destfile" ) or main::affaErrorExit("Could not open $dest$destfile for writing");
				until( eof(IF) )
					{
					my $buffer;
					read( IF, $buffer, 1048576 ) or main::affaErrorExit("Reading from $linkDest$k failed.");
					print OF $buffer or main::affaErrorExit("Writing to $dest$destfile failed.");
					}
				close(OF);
				close(IF);
				}
			}
		@cmd=(
			$main::rsyncLocal,
			'-ti',
			'--stats',
			'--inplace',
			$main::job{'BandwidthLimit'} ? "--bwlimit=$main::job{'BandwidthLimit'}" : '',
			$main::job{'rsyncTimeout'} ? "--timeout=$main::job{'rsyncTimeout'}" : "",
			$main::job{'rsyncCompress'} eq 'yes' ? "-z" : "",
			"-e '/usr/bin/ssh -o HostKeyAlias=$main::jobname'",
			$main::rsyncOptions,
			"$esxi_server:'$k'",
			"\'$dest\'" );
		ssh_cmd(@cmd);
		$rsync_out .= $Affa::lib::ExecCmdOutout;

		print "Command: " . join( " ", @cmd) . "\n" if $debug;
		print "Result:\n$Affa::lib::ExecCmdOutout" if $debug;

		}

	dbg( "VI API connect to $esxi_server as user $username" );
	eval { Vim::login(service_url => "https://$esxi_server/sdk", user_name => $username, password => $password); };
	if ($@) {main::affaErrorExit( $@ )}
	remove_bexi_snapshots( $current_vm->snapshot->currentSnapshot, $current_vm->snapshot->rootSnapshotList );
	dbg( "VI API disconnecting.." );
	eval { Vim::logout() }; if ($@) {main::affaErrorExit( $@ )}
	

	# Syncing VM Configuration and Snapshot Configuration files
	lg( "Syncing VM Configuration (.vmx) and Snapshot Configuration (.vmsd) files" );
	(my $dest="$RootDir/$vmpath") =~ s:/+:/:g; 
	@cmd=(
		$main::rsyncLocal,
		'--archive',
		'--stats',
		'--partial',
		$main::job{'BandwidthLimit'} ? "--bwlimit=$main::job{'BandwidthLimit'}" : '',
		$main::job{'rsync--modify-window'} > 0 ? "--modify-window=$main::job{'rsync--modify-window'}" : '',
		$main::job{'rsync--inplace'} ne 'no' ? "--inplace" : "",
		$main::job{'rsyncTimeout'} ? "--timeout=$main::job{'rsyncTimeout'}" : "",
		$main::job{'rsyncCompress'} eq 'yes' ? "--compress" : "",
		"--rsync-path='cd " . escape_filename($vmpath) . " && rsync'", 
		( $linkDest ? "--link-dest='$linkDest'" : '' ),
		"--rsh='/usr/bin/ssh -o HostKeyAlias=$main::jobname -p $main::job{'sshPort'}'",
		$main::rsyncOptions,
		"$esxi_server:*.vmx",
		"$esxi_server:*.vmsd", 
		"\'$dest\'"  );
	ssh_cmd(@cmd);
	$rsync_out .= $Affa::lib::ExecCmdOutout;
	print "Command: " . join( " ", @cmd) . "\n" if $debug;
	print "Result:\n$Affa::lib::ExecCmdOutout" if $debug;

	# Syncing ESXi Configuration and root home directory"
	lg( "Syncing ESXi Configuration and root home directory" );
	($dest="$RootDir") =~ s:/+:/:g; 
	@cmd=(
		$main::rsyncLocal,
		'--archive',
		'--stats',
		'--partial',
		$main::job{'BandwidthLimit'} ? "--bwlimit=$main::job{'BandwidthLimit'}" : '',
		$main::job{'rsync--modify-window'} > 0 ? "--modify-window=$main::job{'rsync--modify-window'}" : '',
		$main::job{'rsync--inplace'} ne 'no' ? "--inplace" : "",
		$main::job{'rsyncTimeout'} ? "--timeout=$main::job{'rsyncTimeout'}" : "",
		$main::job{'rsyncCompress'} eq 'yes' ? "--compress" : "",
		( $linkDest ? "--link-dest='$linkDest'" : '' ),
		"--rsh='/usr/bin/ssh -o HostKeyAlias=$main::jobname -p $main::job{'sshPort'}'",
		$main::rsyncOptions,
		"$esxi_server:/bootbank/local.tgz*",
		"$esxi_server:/root", 
		$dest );
	ssh_cmd(@cmd);
	$rsync_out .= $Affa::lib::ExecCmdOutout;
	print "Command: " . join( " ", @cmd) . "\n" if $debug;
	print "Result:\n$Affa::lib::ExecCmdOutout" if $debug;

	return $rsync_out;
	}


sub get_vm {
	my $vmname = shift;
	my $vm = Vim::find_entity_views(view_type => "VirtualMachine");
	my $current_vm;
	foreach (@$vm) 
		{
		(my $decoded_name = $_->name) =~ s/%([0-9a-fA-F]{2})/pack("c",hex($1))/ge;
		if( $decoded_name eq $vmname )
			{
			$current_vm=$_;
			last;
			}
		}
	return  $current_vm;
}

sub remove_bexi_snapshots {
	my ($ref, $tree) = @_;
	my $fqhostname=hostfqdn();
	foreach my $node (@$tree) {
	if( $node->name =~ /^BEXI-[0-9]{8}-[0-9]{6}-$fqhostname/ ) {
	 	my $snapshot = Vim::get_view (mo_ref =>$node->snapshot);
		lg( "Removing snapshot " . $node->name );
	 	eval { $snapshot->RemoveSnapshot (removeChildren => 0) }; if ($@) {main::affaErrorExit( $@ )}
	}
	remove_bexi_snapshots( $ref, $node->childSnapshotList);
	}
}

sub create_snapshot {
   (my $vm) = @_;
	my $fqhostname=hostfqdn();
	my $snapshotname=Date::Format::time2str("BEXI-%Y%m%d-%H%M%S",time()) . "-$fqhostname" ;
	my $mor_host = $vm->runtime->host;
	my $hostname = Vim::get_view(mo_ref => $mor_host)->name;

	lg( "Creating snapshot $snapshotname" );
	my $status=0;
	eval {
		$vm->CreateSnapshot(name => $snapshotname,
			description => 'Snapshot created for virtual machine: ' . $vm->name ,
			memory => 0,
			quiesce => 0);
		lg( "Snapshot '$snapshotname' completed for VM '".$vm->name."' under host '$hostname'");
	};
	if ($@) {
		$status=1;
		lg( "Error creating snapshot of virtual Machine: " . $vm->name);
		if (ref($@) eq 'SoapFault') {
			if (ref($@->detail) eq 'InvalidName') {
				lg( "Specified snapshot name is invalid\n");
			}
			elsif (ref($@->detail) eq 'InvalidState') {
				lg( "The operation is not allowed in the current state.\n");
			}
			elsif (ref($@->detail) eq 'InvalidPowerState') {
				lg( "Current power state of the virtual machine '" . $vm->name . "' is invalid\n");
			}
			elsif (ref($@->detail) eq 'HostNotConnected') {
				lg( "Unable to communicate with the remote host, " . "since it is disconnected\n");
			}
			elsif (ref($@->detail) eq 'NotSupported') {
				lg( "Host does not support snapshots \n");
			}
			else {
				lg( "Error occurred: \n" . $@ . "\n");
			}
		}
		else {
			lg( "Error occurred: \n" . $@ . "\n");
		}
	}
	return $status;
}

sub ssh_cmd(@)
	{
	my @cmd = @_;
	my $error=ExecCmd( @cmd, 0);

	# catch rsync errors. Workaround for Dropbear < 0.51
	if ( $error==255 or (65535-int($error))==255 )
		{
		my @err_out=split( "\n", $Affa::lib::ExecCmdOutout );
		$error=0;
		foreach (@err_out) 
			{
			next if not /rsync error:.*code/;
			(my $e=$_) =~ s/.*\(code ([0-9]+)\).*/$1/;
			next if $e == 255;
			$error=$e;
			$Affa::lib::ExecCmdOutout .= "Effective exitstatus=$error (Dropbear workaround)\n";
			last;
			}
		}
	if( $error )
		{
		lg( $Affa::lib::ExecCmdOutout );
		my $username = $main::job{'ESXiUsername'};
		my $password = $main::job{'ESXiPassword'};
		my $esxi_server = $main::job{'remoteHostName'};
		dbg( "VI API connect to $esxi_server as user $username" );
		eval { Vim::login(service_url => "https://$esxi_server/sdk", user_name => $username, password => $password); };
		if (not $@) 
			{
			my $current_vm = get_vm( $main::job{'ESXiVMName'} );
			remove_bexi_snapshots( $current_vm->snapshot->currentSnapshot, $current_vm->snapshot->rootSnapshotList );
			dbg( "VI API disconnecting.." );
			eval { Vim::logout() };
			main::affaErrorExit( "Command returned error" );
			}
		}
	return $error;
	}

sub escape_filename($)
	{
	my $s=shift;
	$s =~ s/([ \+\(\)\[\]\{\}\;\?\*\$\!\"\'\&\%\<\>\|\:])/\\$1/g;
	return $s;
	}
	
1;
