;;; vimrc-mode.el --- Major mode for vimrc files  -*- lexical-binding: t; -*-

;; Copyright (C) 2013 Andrew Pennebaker
;; Copyright (C) 2011 Alpha Tan

;; Authors: Mark Oteiza
;;          Andrew Pennebaker
;;          Alpha Tan <alphatan.zh@gmail.com>
;; URL: https://github.com/mcandre/vimrc-mode
;; Package-Version: 20250128.635
;; Package-Revision: f594392a0834
;; Keywords: languages, vim
;; Package-Requires: ((emacs "24.3"))

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; The homepage for Alpha Tan's original vimrc mode is
;; <http://vimrc-mode.sf.net>

;;; Code:

(require 'font-lock)

(defgroup vimrc nil
  "Major mode for editing Vim script."
  :link '(url-link "https://github.com/mcandre/vimrc-mode")
  :group 'languages)

(defcustom vimrc-mode-hook nil
  "Normal hook run when entering `vimrc-mode'."
  :type 'hook
  :group 'vimrc-mode)

(defgroup vimrc-faces nil
  "Faces used for Vim script."
  :group 'faces)

(defface vimrc-option
  '((default :inherit font-lock-variable-name-face))
  "Face used for Vim's configuration options.")

(defface vimrc-function-builtin
  '((default :inherit font-lock-builtin-face))
  "Face used for Vim's built-in functions.")

(defface vimrc-command
  '((default :inherit font-lock-keyword-face))
  "Face used for Vim Ex commands.")

(defface vimrc-number
  '((((class color) (background light)) (:foreground "steel blue"))
    (((class color) (background dark)) (:foreground "sky blue"))
    (t nil))
  "Face used for Vim's numbers.")


;; Font lock linking

(defvar vimrc-font-lock-keywords
  `(
     ;; Line comment
     ("^[\t ]*\\(\"\\)\\(.*\\)$"
       (1 font-lock-comment-delimiter-face)
       (2 font-lock-comment-face))

     ;; Trailing comment
     ("[\t ]+\\(\"\\)\\([^\"\r\n]*\\)$"
       (1 font-lock-comment-delimiter-face)
       (2 font-lock-comment-face))

     ;; String start:
     ("\\(\"[^\n\r\"]*\"\\)\\|\\('[^\n\r]*'\\)"
       (0 font-lock-string-face)) ;; String end;

     ;; Function-name start:
     ("^[ \t]*\\(fun\\(?:ction\\)?\\)!?[ \t]+\\([a-zA-Z0-9_:#]+\\)?"
       (2 font-lock-function-name-face nil t)) ;; Function-name end;
     ("\\(\\([a-zA-Z]*:\\)?[a-zA-Z]*\\)("
       (1 font-lock-function-name-face nil t)) ;; Function-name end;

     ;; Variables
     ("\\<[bwglsav]:[a-zA-Z_][a-zA-Z0-9#_]*\\>"
       (0 font-lock-variable-name-face))
     ("\\(let[ \t]+\\)\\<\\([a-zA-Z_][a-zA-Z0-9#_]*\\)\\>"
       (2 font-lock-variable-name-face))

     ;; Options which can be prefixed with `no'
     (,(concat "[^_]\\<\\(\\(?:no\\)?"
               (regexp-opt '("autochdir" "acd"
                             "allowrevins" "ari"
                             "altkeymap" "akm"
                             "antialias" "anti"
                             "arabic" "arab"
                             "arabicshape" "arshape"
                             "autoindent" "ai"
                             "autoread" "ar"
                             "autowrite" "aw"
                             "autowriteall" "awa"
                             "backup" "bk"
                             "ballooneval" "beval"
                             "binary" "bin"
                             "bioskey" "biosk"
                             "bomb"
                             "buflisted" "bl"
                             "cindent" "cin"
                             "compatible" "cp"
                             "confirm" "cf"
                             "conskey" "consk"
                             "copyindent" "ci"
                             "cscopetag" "cst"
                             "cscopeverbose" "csverb"
                             "cursorbind" "crb"
                             "cursorcolumn" "cuc"
                             "cursorline" "cul"
                             "delcombine" "deco"
                             "diff"
                             "digraph" "dg"
                             "edcompatible" "ed"
                             "endofline" "eol"
                             "equalalways" "ea"
                             "errorbells" "eb"
                             "esckeys" "ek"
                             "expandtab" "et"
                             "exrc" "ex"
                             "fkmap" "fk"
                             "foldenable" "fen"
                             "gdefault" "gd"
                             "guipty"
                             "hidden" "hid"
                             "hkmap" "hk"
                             "hkmapp" "hkp"
                             "hlsearch" "hls"
                             "icon"
                             "ignorecase" "ic"
                             "imcmdline" "imc"
                             "imdisable" "imd"
                             "incsearch" "is"
                             "infercase" "inf"
                             "insertmode"  "im"
                             "joinspaces" "js"
                             "lazyredraw" "lz"
                             "linebreak" "lbr"
                             "lisp"
                             "list"
                             "loadplugins" "lpl"
                             "macatsui"
                             "magic"
                             "modeline" "ml"
                             "modifiable" "ma"
                             "modified" "mod"
                             "more"
                             "mousefocus" "mousef"
                             "mousehide" "mh"
                             "number" "nu"
                             "opendevice" "odev"
                             "paste"
                             "preserveindent" "pi"
                             "previewwindow" "pvw"
                             "prompt"
                             "readonly" "ro"
                             "relativenumber" "rnu"
                             "remap"
                             "restorescreen" "rs"
                             "revins" "ri"
                             "rightleft" "rl"
                             "ruler" "ru"
                             "scrollbind" "scb"
                             "secure"
                             "shellslash" "ssl"
                             "shelltemp" "stmp"
                             "shiftround" "sr"
                             "shortname" "sn"
                             "showcmd" "sc"
                             "showfulltag" "sft"
                             "showmatch" "sm"
                             "showmode" "smd"
                             "smartcase" "scs"
                             "smartindent" "si"
                             "smarttab" "sta"
                             "spell"
                             "splitbelow" "sb"
                             "splitright" "spr"
                             "startofline" "sol"
                             "swapfile" "swf"
                             "tagbsearch" "tbs"
                             "tagrelative" "tr"
                             "tagstack" "tgst"
                             "termbidi" "tbidi"
                             "terse"
                             "textauto" "ta"
                             "textmode" "tx"
                             "tildeop" "top"
                             "timeout" "to"
                             "title"
                             "ttimeout"
                             "ttybuiltin" "tbi"
                             "ttyfast" "tf"
                             "visualbell" "vb"
                             "warn"
                             "weirdinvert" "wiv"
                             "wildmenu" "wmnu"
                             "winfixheight" "wfh"
                             "winfixwidth" "wfw"
                             "wrap"
                             "wrapscan" "ws"
                             "write"
                             "writeany" "wa"
                             "writebackup" "wb") t)
               "\\)\\>[^_]" )
      1 '(face vimrc-option))

     ;; The rest of the options
     (,(concat "[^_]"
               (regexp-opt '("aleph" "al"
                             "ambiwidth" "ambw"
                             "arabicshape" "arshape"
                             "background" "bg"
                             "backspace" "bs"
                             "backupcopy" "bkc"
                             "backupdir" "bdir"
                             "backupext" "bex"
                             "backupskip" "bsk"
                             "balloondelay" "bdlay"
                             "balloonexpr" "bexpr"
                             "breakat" "brk"
                             "breakindent" "bri"
                             "breakindentopt" "briopt"
                             "browsedir" "bsdir"
                             "bufhidden" "bh"
                             "buftype" "bt"
                             "casemap" "cmp"
                             "cdpath" "cd"
                             "cedit"
                             "charconvert" "ccv"
                             "cinkeys" "cink"
                             "cinoptions" "cino"
                             "cinwords" "cinw"
                             "clipboard" "cb"
                             "cmdheight" "ch"
                             "cmdwinheight" "cwh"
                             "colorcolumn" "cc"
                             "columns" "co"
                             "comments" "com"
                             "commentstring" "cms"
                             "complete"
                             "completefunc" "cfu"
                             "completeopt" "cot"
                             "concealcursor" "cocu"
                             "conceallevel" "cole"
                             "cpoptions" "cpo"
                             "cryptmethod" "cm"
                             "cscopepathcomp" "cspc"
                             "cscopeprg" "csprg"
                             "cscopequickfix" "csqf"
                             "cscopetagorder" "csto"
                             "cscopeverbose" "csverb"
                             "debug"
                             "define" "def"
                             "dictionary" "dict"
                             "dex" "diffexpr"
                             "dip" "diffopt"
                             "directory" "dir"
                             "display" "dy"
                             "eadirection" "ead"
                             "encoding" "enc"
                             "equalprg" "ep"
                             "errorfile" "ef"
                             "errorformat" "efm"
                             "eventignore" "ei"
                             "fileencoding" "fenc"
                             "fe"
                             "fileencodings" "fencs"
                             "fileformat" "ff"
                             "fileformats" "ffs"
                             "filetype" "ft"
                             "fillchars" "fcs"
                             "fkmap" "fk"
                             "foldclose" "fcl"
                             "foldcolumn" "fdc"
                             "foldexpr" "fde"
                             "foldignore" "fdi"
                             "foldlevel" "fdl"
                             "foldlevelstart" "fdls"
                             "foldmarker" "fmr"
                             "foldmethod" "fdm"
                             "foldminlines" "fml"
                             "foldnestmax" "fdn"
                             "foldopen" "fdo"
                             "foldtext" "fdt"
                             "formatoptions" "fo"
                             "formatlistpat" "flp"
                             "formatprg" "fp"
                             "formatexpr" "fex"
                             "fsync" "fs"
                             "grepformat" "gfm"
                             "grepprg" "gp"
                             "guicursor" "gcr"
                             "guifont" "gfn"
                             "guifontset" "gfs"
                             "guifontwide" "gfw"
                             "guiheadroom" "ghr"
                             "guioptions" "go"
                             "guitablabel" "gtl"
                             "guitabtooltip" "gtt"
                             "helpfile" "hf"
                             "helpheight" "hh"
                             "helplang" "hlg"
                             "highlight" "hl"
                             "history" "hi"
                             "iconstring"
                             "imactivatefunc" "imaf"
                             "imactivatekey" "imak"
                             "iminsert" "imi"
                             "imsearch" "ims"
                             "imstatusfunc" "imsf"
                             "include" "inc"
                             "includeexpr" "inex"
                             "indentexpr" "inde"
                             "indentkeys" "indk"
                             "isfname" "isf"
                             "isident" "isi"
                             "iskeyword" "isk"
                             "isprint" "isp"
                             "key"
                             "keymap" "kmp"
                             "keymodel" "km"
                             "keywordprg" "kp"
                             "langmap" "lmap"
                             "langmenu" "lm"
                             "laststatus" "ls"
                             "lines"
                             "linespace" "lsp"
                             "lispwords" "lw"
                             "listchars" "lcs"
                             "makeef" "mef"
                             "makeprg" "mp"
                             "matchpairs" "mps"
                             "matchtime" "mat"
                             "maxcombine" "mco"
                             "maxfuncdepth" "mfd"
                             "maxmapdepth" "mmd"
                             "maxmem" "mm"
                             "maxmempattern" "mmp"
                             "maxmemtot" "mmt"
                             "menuitems" "mis"
                             "mkspellmem" "msm"
                             "modelines" "mls"
                             "mouse"
                             "mousemodel" "mousem"
                             "mouseshape" "mouses"
                             "mousetime" "mouset"
                             "mzquantum" "mzq"
                             "nrformats" "nf"
                             "numberwidth" "nuw"
                             "omnifunc" "ofu"
                             "operatorfunc" "opfunc"
                             "osfiletype" "oft"
                             "paragraphs" "para"
                             "pastetoggle" "pt"
                             "pex" "patchexpr"
                             "patchmode" "pm"
                             "path" "pa"
                             "previewheight" "pvh"
                             "printdevice" "pdev"
                             "printencoding" "penc"
                             "printexpr" "pexpr"
                             "printfont" "pfn"
                             "printheader" "pheader"
                             "printmbcharset" "pmbcs"
                             "printmbfont" "pmbfn"
                             "printoptions" "popt"
                             "pumheight" "ph"
                             "quoteescape" "qe"
                             "redrawtime" "rdt"
                             "regexpengine" "re"
                             "report"
                             "rightleftcmd" "rlc"
                             "rulerformat" "ruf"
                             "runtimepath" "rtp"
                             "scroll" "scr"
                             "scrolljump" "sj"
                             "scrolloff" "so"
                             "scrollopt" "sbo"
                             "sections" "sect"
                             "selection" "sel"
                             "selectmode" "slm"
                             "sessionoptions" "ssop"
                             "shell" "sh"
                             "shellcmdflag" "shcf"
                             "shellpipe" "sp"
                             "shellquote" "shq"
                             "shellredir" "srr"
                             "shelltype" "st"
                             "shellxescape" "sxe"
                             "shellxquote" "sxq"
                             "shiftwidth" "sw"
                             "shortmess" "shm"
                             "showbreak" "sbr"
                             "showtabline" "stal"
                             "sidescroll" "ss"
                             "sidescrolloff" "siso"
                             "softtabstop" "sts"
                             "spellcapcheck" "spc"
                             "spellfile" "spf"
                             "spelllang" "spl"
                             "spellsuggest" "sps"
                             "statusline" "stl"
                             "suffixes" "su"
                             "suffixesadd" "sua"
                             "swapsync" "sws"
                             "switchbuf" "swb"
                             "synmaxcol" "smc"
                             "syntax" "syn"
                             "tabline" "tal"
                             "tabpagemax" "tpm"
                             "tabstop" "ts"
                             "taglength" "tl"
                             "tags" "tag"
                             "term"
                             "termbidi" "tbidi"
                             "termencoding" "tenc"
                             "textwidth" "tw"
                             "thesaurus" "tsr"
                             "timeoutlen" "tm"
                             "ttimeoutlen" "ttm"
                             "titlelen"
                             "titleold"
                             "titlestring"
                             "toolbar" "tb"
                             "toolbariconsize" "tbis"
                             "ttymouse" "ttym"
                             "ttyscroll" "tsl"
                             "ttytype" "tty"
                             "undodir" "udir"
                             "undolevels" "ul"
                             "undoreload" "ur"
                             "updatecount" "uc"
                             "updatetime" "ut"
                             "verbose" "vbs"
                             "verbosefile" "vfile"
                             "viewdir" "vdir"
                             "viewoptions" "vop"
                             "viminfo" "vi"
                             "virtualedit" "ve"
                             "whichwrap" "ww"
                             "wildchar" "wc"
                             "wildcharm" "wcm"
                             "wildignore" "wig"
                             "wildmode" "wim"
                             "wildoptions" "wop"
                             "winaltkeys" "wak"
                             "window" "wi"
                             "winheight" "wh"
                             "winminheight" "wmh"
                             "winminwidth" "wmw"
                             "winwidth" "wiw"
                             "wrapmargin" "wm"
                             "wrapscan" "ws"
                             "writedelay" "wd") 'words)
               "[^_]")
      1 '(face vimrc-option))

     ;; Ex commands
     (,(concat "\\(^\\|[^_]\\)"
               (regexp-opt '("Next" "N"
                             "Print" "P"
                             "X"
                             "abclear" "abc"
                             "aboveleft" "abo"
                             "all" "al"
                             "am" "amenu" "an" "anoremenu"
                             "argadd" "arga"
                             "argdelete" "argd"
                             "argdo"
                             "argedit" "arge"
                             "argglobal" "argg"
                             "arglocal" "argl"
                             "args" "ar"
                             "argument" "argu"
                             "ascii" "as"
                             "augroup" "aug"
                             "autocmd" "au"
                             "bNext" "bN"
                             "badd" "bad"
                             "ball" "ba"
                             "bdelete" "bd"
                             "belowright" "bel"
                             "bfirst" "bf"
                             "blast" "bl"
                             "bmodified" "bm"
                             "bnext" "bn"
                             "botright" "bo"
                             "bprevious" "bp"
                             "break" "brea"
                             "breakadd" "breaka"
                             "breakdel" "breakd"
                             "breaklist" "breakl"
                             "brewind" "br"
                             "browse" "bro"
                             "bufdo"
                             "buffer" "b"
                             "bunload" "bun"
                             "bwipeout" "bw"
                             "cNext" "cN"
                             "cNfile" "cNf"
                             "cabclear" "cabc"
                             "caddbuffer" "caddb"
                             "caddexpr" "cad"
                             "caddfile" "caddf"
                             "call" "cal"
                             "catch" "cat"
                             "cbuffer" "cb"
                             "cc"
                             "cclose" "ccl"
                             "cd"
                             "center" "ce"
                             "cexpr" "cex"
                             "cfile" "cf"
                             "cfirst" "cfir"
                             "cgetbuffer" "cgetb"
                             "cgetexpr" "cgete"
                             "cgetfile" "cg"
                             "change" "c"
                             "changes"
                             "chdir" "chd"
                             "checkpath" "ckpath"
                             "checktime" "checkt"
                             "clast" "cla"
                             "clist" "cl"
                             "close" "clo"
                             "cm" "cmap"
                             "cmapc" "cmapclear"
                             "cme" "cmenu" "cnoreme" "cnoremenu"
                             "cnewer" "cnew"
                             "cnext" "cn"
                             "cnfile" "cnf"
                             "cno" "cnoremap"
                             "colder" "col"
                             "colo" "colorscheme"
                             "comclear" "comc"
                             "command" "com"
                             "compiler" "comp"
                             "confirm" "conf"
                             "continue" "con"
                             "copen" "cope"
                             "copy" "co"
                             "cpfile" "cpf"
                             "cprevious" "cp"
                             "cquit" "cq"
                             "crewind" "cr"
                             "cu" "cunmap"
                             "cunabbrev" "cuna"
                             "cwindow" "cw"
                             "debuggreedy" "debugg"
                             "delcommand" "delc"
                             "delete" "d"
                             "delfunction" "delf"
                             "delmarks" "delm"
                             "diffget" "diffg"
                             "diffoff"
                             "diffpatch"
                             "diffput" "diffpu"
                             "diffsplit"
                             "diffthis"
                             "diffupdate" "diffu"
                             "digraphs" "dig"
                             "display" "di"
                             "djump" "dj"
                             "dlist" "dl"
                             "drop" "dr"
                             "dsearch" "ds"
                             "dsplit" "dsp"
                             "earlier"
                             "echoerr" "echoe"
                             "echomsg" "echom"
                             "echon"
                             "edit" "e"
                             "else" "el"
                             "elseif" "elsei"
                             "em" "emenu"
                             "endf" "endfunction"
                             "enew" "ene"
                             "ex"
                             "execute" "exe"
                             "exit" "exi"
                             "file" "fi" "f"
                             "files" "buffers" "ls"
                             "filetype" "filet"
                             "finally" "fina"
                             "find" "fin"
                             "finish" "fini"
                             "first" "fir"
                             "fixdel" "fix"
                             "fold" "fo"
                             "foldclose" "foldc"
                             "folddoclosed" "folddoc"
                             "folddoopen" "foldd"
                             "foldopen" "foldo"
                             "for" "endfo" "endfor"
                             "fu" "fun" "function"
                             "goto" "go"
                             "grep" "gr"
                             "grepadd" "grepa"
                             "hardcopy" "ha"
                             "hide" "hid"
                             "history" "his"
                             "iabclear" "iabc"
                             "if" "endif" "en"
                             "ijump" "ij"
                             "ilist" "il"
                             "im" "imap"
                             "imapc" "imapclear"
                             "ime" "imenu" "inoreme" "inoremenu"
                             "ino" "inoremap"
                             "isearch" "is"
                             "isplit" "isp"
                             "iu" "iunmap"
                             "iunabbrev" "iuna"
                             "join" "j"
                             "jumps" "ju"
                             "k"
                             "keepalt"
                             "keepjumps" "keepj"
                             "keepmarks" "kee"
                             "lNext" "lN"
                             "lNfile" "lNf"
                             "laddbuffer" "laddb"
                             "laddexpr" "lad"
                             "laddfile" "laddf"
                             "language" "lan"
                             "last" "la"
                             "later"
                             "lbuffer" "lb"
                             "lcd" "lc"
                             "lchdir" "lch"
                             "lclose" "lcl"
                             "le" "left"
                             "leftabove" "lefta"
                             "let"
                             "lexpr" "lex"
                             "lfile" "lf"
                             "lfirst" "lfir"
                             "lgetbuffer" "lgetb"
                             "lgetexpr" "lgete"
                             "lgetfile" "lg"
                             "lgrep" "lgr"
                             "lgrepadd" "lgrepa"
                             "list" "l"
                             "ll"
                             "llast" "lla"
                             "llist" "lli"
                             "lm" "lmap"
                             "lmake" "lmak"
                             "lmapc" "lmapclear"
                             "ln" "lnoremap"
                             "lnewer" "lnew"
                             "lnext" "lne"
                             "lnfile" "lnf"
                             "loadview" "lo"
                             "lockmarks" "loc"
                             "lockvar" "lockv" "lockv"
                             "lolder" "lol"
                             "lopen" "lop"
                             "lpfile" "lpf"
                             "lprevious" "lp"
                             "lrewind" "lr"
                             "ltag" "lt"
                             "lu" "lunmap"
                             "lua" "lua"
                             "luado" "luado"
                             "luafile" "luafile"
                             "lvimgrep" "lv"
                             "lvimgrepadd" "lvimgrepa"
                             "lwindow" "lw"
                             "make" "mak"
                             "map"
                             "mapc" "mapclear"
                             "mark" "ma"
                             "marks"
                             "match" "mat"
                             "me" "menu" "noreme"  "noremenu"
                             "menutranslate" "menut"
                             "mkexrc" "mk"
                             "mksession" "mks"
                             "mkspell" "mksp"
                             "mkview" "mkvie"
                             "mkvimrc" "mkv"
                             "mode" "mod"
                             "move" "m"
                             "mzfile" "mzf"
                             "mzscheme" "mz"
                             "nbclose" "nbc"
                             "nbkey" "nb"
                             "nbstart" "nbs"
                             "new"
                             "next" "n"
                             "nm" "nmap"
                             "nmapc" "nmapclear"
                             "nme" "nmenu" "nnoreme" "nnoremenu"
                             "nn" "nnoremap"
                             "no" "noremap" "nor"
                             "nohlsearch" "noh"
                             "nore" "norem"
                             "number" "nu"
                             "nun" "nunmap"
                             "oldfiles" "ol"
                             "om" "omap"
                             "omapc" "omapclear"
                             "ome" "omenu" "onoreme" "onoremenu"
                             "only" "on"
                             "ono" "onoremap"
                             "open" "o"
                             "options" "opt"
                             "ou" "ounmap"
                             "pclose" "pc"
                             "pedit" "ped"
                             "perl" "pe"
                             "perldo" "perld"
                             "pop" "po"
                             "popu"
                             "popup" "popu"
                             "ppop" "pp"
                             "preserve" "pre"
                             "previous" "prev"
                             "print" "p"
                             "profdel" "profd"
                             "profile" "prof"
                             "promptfind" "promptf"
                             "promptrepl" "promptr"
                             "psearch" "ps"
                             "ptNext" "ptN"
                             "ptag" "pta"
                             "ptfirst" "ptf"
                             "ptjump" "ptj"
                             "ptlast" "ptl"
                             "ptnext" "ptn"
                             "ptprevious" "ptp"
                             "ptrewind" "ptr"
                             "ptselect" "pts"
                             "put" "pu"
                             "pwd" "pw"
                             "pyfile" "pyf"
                             "python" "py"
                             "qall" "qa"
                             "quit" "q"
                             "quitall" "quita"
                             "read" "r"
                             "recover" "rec"
                             "redir" "redi"
                             "redo" "red"
                             "redraw" "redr"
                             "redrawstatus" "redraws"
                             "registers" "reg"
                             "resize" "res"
                             "retab" "ret"
                             "return" "rn"
                             "rewind" "rew"
                             "right" "ri"
                             "rightbelow" "rightb"
                             "ruby" "rub"
                             "rubydo" "rubyd"
                             "rubyfile" "rubyf"
                             "rundo"
                             "runtime" "ru"
                             "rviminfo" "sa"
                             "sNext" "sN"
                             "sall" "sal"
                             "sandbox" "san"
                             "sargument" "sa"
                             "saveas" "sav"
                             "sbNext" "sbN"
                             "sball" "sba"
                             "sbfirst" "sbf"
                             "sblast" "sbl"
                             "sbmodified" "sbm"
                             "sbnext" "sbn"
                             "sbprevious" "sbp"
                             "sbrewind" "sbr"
                             "sbuffer" "sb"
                             "scriptencoding" "scripte"
                             "scriptnames" "scrip"
                             "set" "se"
                             "setfiletype" "setf"
                             "setglobal" "setg"
                             "setlocal" "setl"
                             "sfind" "sf"
                             "sfirst" "sfir"
                             "shell" "sh"
                             "sign"
                             "silent" "sil"
                             "simalt" "sim"
                             "slast" "sla"
                             "sleep" "sl"
                             "smagic" "sm"
                             "smap" "smap"
                             "smapc" "smapclear"
                             "sme" "smenu" "snoreme" "snoremenu"
                             "snext" "sn"
                             "sniff" "sni"
                             "snomagic" "sno"
                             "snor" "snoremap"
                             "sort" "sor"
                             "source" "so"
                             "spelldump" "spelld"
                             "spellgood" "spe"
                             "spellinfo" "spelli"
                             "spellrepall" "spellr"
                             "spellundo" "spellu"
                             "spellwrong" "spellw"
                             "split" "sp"
                             "sprevious" "spr"
                             "srewind" "sre"
                             "stag" "sta"
                             "startgreplace" "startg"
                             "startinsert" "star"
                             "startreplace" "startr"
                             "stjump" "stj"
                             "stop" "st"
                             "stopinsert" "stopi"
                             "stselect" "sts"
                             "sunhide" "sun"
                             "sunm" "sunmap"
                             "suspend" "sus"
                             "sview" "sv"
                             "syncbind"
                             "t"
                             "tNext" "tN"
                             "tab"
                             "tabNext" "tabN"
                             "tabclose" "tabc"
                             "tabdo" "tabd"
                             "tabedit" "tabe"
                             "tabfind" "tabf"
                             "tabfirst" "tabfir"
                             "tablast" "tabl"
                             "tabmove" "tabm"
                             "tabnew"
                             "tabnext" "tabn"
                             "tabonly" "tabo"
                             "tabprevious" "tabp"
                             "tabrewind" "tabr"
                             "tabs"
                             "tag" "ta"
                             "tags"
                             "tcl" "tc"
                             "tcldo" "tcld"
                             "tclfile" "tclf"
                             "tearoff" "te"
                             "tfirst" "tf"
                             "throw" "th"
                             "tjump" "tj"
                             "tlast" "tl"
                             "tmenu" "tm"
                             "tnext" "tn"
                             "topleft" "to"
                             "tprevious" "tp"
                             "trewind" "tr"
                             "try" "endt" "endtry"
                             "tselect" "ts"
                             "tunmenu" "tu"
                             "unabbreviate" "una"
                             "undo" "una"
                             "undojoin" "undoj"
                             "undolist" "undol"
                             "unhide" "unh"
                             "unlockvar" "unlo"
                             "unm" "unmap"
                             "unsilent" "uns"
                             "update" "up"
                             "verbose" "verb"
                             "version" "ve"
                             "vertical" "vert"
                             "view" "vie"
                             "vimgrep" "vim"
                             "vimgrepadd" "vimgrepa"
                             "visual" "vi"
                             "vm" "vmap"
                             "vmapc" "vmapclear"
                             "vme" "vmenu" "vnoreme" "vnoremenu"
                             "vn" "vnoremap"
                             "vnew" "vne"
                             "vsplit" "vs"
                             "vu" "vunmap"
                             "wNext" "wN"
                             "wall" "wa"
                             "while" "endwhile" "wh" "endw"
                             "wincmd" "winc"
                             "windo"
                             "winpos" "winp"
                             "winsize" "win"
                             "wnext" "wn"
                             "wprevious" "wp"
                             "wq"
                             "wqa" "wqall" "xa" "xall"
                             "write" "w"
                             "wsverb" "ws"
                             "wundo"
                             "wviminfo" "wv"
                             "x" "xit"
                             "xm" "xmap"
                             "xmapc" "xmapclear"
                             "xme" "xmenu" "xnoreme" "xnoremenu"
                             "xn" "xnoremap"
                             "xu" "xunmap"
                             "yank" "y") 'words)
               "\\([^_]\\|$\\)")
      2 '(face vimrc-command))

     ;; Built-in functions
     (,(concat "\\(^\\|[ \t]*\\)"
               (regexp-opt '("writefile"
                             "winwidth"
                             "winsaveview"
                             "winrestview"
                             "winrestcmd"
                             "winnr"
                             "winline"
                             "winheight"
                             "wincol"
                             "winbufnr"
                             "visualmode"
                             "virtcol"
                             "values"
                             "undotree"
                             "undofile"
                             "type"
                             "trunc"
                             "tr"
                             "toupper"
                             "tolower"
                             "tempname"
                             "tanh"
                             "tan"
                             "taglist"
                             "tagfiles"
                             "tabpagewinnr"
                             "tabpagenr"
                             "tabpagebuflist"
                             "system"
                             "synstack"
                             "synIDtrans"
                             "synIDattr"
                             "synID"
                             "synconcealed"
                             "substitute"
                             "submatch"
                             "strwidth"
                             "strtrans"
                             "strridx"
                             "strpart"
                             "strlen"
                             "string"
                             "stridx"
                             "strftime"
                             "strdisplaywidth"
                             "strchars"
                             "str2nr"
                             "str2float"
                             "sqrt"
                             "split"
                             "spellsuggest"
                             "spellbadword"
                             "soundfold"
                             "sort"
                             "sinh"
                             "sin"
                             "simplify"
                             "shellescape"
                             "setwinvar"
                             "settabwinvar"
                             "settabvar"
                             "setreg"
                             "setqflist"
                             "setpos"
                             "setmatches"
                             "setloclist"
                             "setline"
                             "setcmdpos"
                             "setbufvar"
                             "serverlist"
                             "server2client"
                             "searchpos"
                             "searchpairpos"
                             "searchpair"
                             "searchdecl"
                             "search"
                             "round"
                             "reverse"
                             "resolve"
                             "repeat"
                             "rename"
                             "remove"
                             "remote_send"
                             "remote_read"
                             "remote_peek"
                             "remote_foreground"
                             "remote_expr"
                             "reltimestr"
                             "reltime"
                             "readfile"
                             "range"
                             "pumvisible"
                             "printf"
                             "prevnonblank"
                             "pow"
                             "pathshorten"
                             "nr2char"
                             "nextnonblank"
                             "mzeval"
                             "mode"
                             "mkdir"
                             "min"
                             "max"
                             "matchstr"
                             "matchlist"
                             "matchend"
                             "matchdelete"
                             "matcharg"
                             "matchadd"
                             "match"
                             "mapcheck"
                             "maparg"
                             "map"
                             "log10"
                             "log"
                             "localtime"
                             "lispindent"
                             "line2byte"
                             "line"
                             "libcallnr"
                             "libcall"
                             "len"
                             "keys"
                             "join"
                             "items"
                             "islocked"
                             "isdirectory"
                             "insert"
                             "inputsecret"
                             "inputsave"
                             "inputrestore"
                             "inputlist"
                             "inputdialog"
                             "input"
                             "index"
                             "indent"
                             "iconv"
                             "hostname"
                             "hlID"
                             "hlexists"
                             "histnr"
                             "histget"
                             "histdel"
                             "histadd"
                             "hasmapto"
                             "haslocaldir"
                             "has_key"
                             "has"
                             "globpath"
                             "glob"
                             "getwinvar"
                             "getwinposy"
                             "getwinposx"
                             "gettabwinvar"
                             "gettabvar"
                             "getregtype"
                             "getreg"
                             "getqflist"
                             "getpos"
                             "getpid"
                             "getmatches"
                             "getloclist"
                             "getline"
                             "getftype"
                             "getftime"
                             "getfsize"
                             "getfperm"
                             "getfontname"
                             "getcwd"
                             "getcmdtype"
                             "getcmdpos"
                             "getcmdline"
                             "getcharmod"
                             "getchar"
                             "getbufvar"
                             "getbufline"
                             "get"
                             "garbagecollect"
                             "function"
                             "foreground"
                             "foldtextresult"
                             "foldtext"
                             "foldlevel"
                             "foldclosedend"
                             "foldclosed"
                             "fnamemodify"
                             "fnameescape"
                             "fmod"
                             "floor"
                             "float2nr"
                             "findfile"
                             "finddir"
                             "filter"
                             "filewritable"
                             "filereadable"
                             "feedkeys"
                             "extend"
                             "expr8"
                             "expand"
                             "exp"
                             "exists"
                             "eventhandler"
                             "eval"
                             "escape"
                             "empty"
                             "diff_hlID"
                             "diff_filler"
                             "did_filetype"
                             "delete"
                             "deepcopy"
                             "cursor"
                             "cscope_connection"
                             "count"
                             "cosh"
                             "cos"
                             "copy"
                             "contained"
                             "confirm"
                             "complete_check"
                             "complete_add"
                             "complete"
                             "col"
                             "clearmatches"
                             "cindent"
                             "char2nr"
                             "changenr"
                             "ceil"
                             "call"
                             "byteidx"
                             "byte2line"
                             "bufwinnr"
                             "bufnr"
                             "bufname"
                             "bufloaded"
                             "buflisted"
                             "bufexists"
                             "browsedir"
                             "browse"
                             "atan2"
                             "atan"
                             "asin"
                             "argv"
                             "argidx"
                             "argc"
                             "append"
                             "add"
                             "acos"
                             "abs") 'words)
               "\\([ \t]*(\\)")
      2 '(face vimrc-function-builtin))

     ;; Numbers
     ("\\<0[xX][[:xdigit:]]+"
      (0 '(face vimrc-number)))
     ("#[[:xdigit:]]\\{6\\}"
      (0 '(face vimrc-number)))
     (,(concat
        "\\(\\<\\|-\\)[[:digit:]]+"
        "\\(\\.[[:digit:]]+\\([eE][+-]?[[:digit:]]+\\)?\\)?")
      0 '(face vimrc-number))

     ;;
     ;; Operators start:
     (,(concat "\\("
                       ;; word char
                             "\\(\\<isnot\\>\\)"
                       "\\|" "\\(\\<is\\>\\)"

                       "\\|" "\\(![=~]?[#?]?\\)"
                       "\\|" "\\(>[#\\\\?=]?[#?]?\\)"
                       "\\|" "\\(<[#\\\\?=]?[#?]?\\)"
                       "\\|" "\\(\\+=?\\)"
                       "\\|" "\\(-=?\\)"
                       "\\|" "\\(=[=~]?[#?]?\\)"
                       "\\|" "\\(||\\)"
                       "\\|" "\\(&&\\)"

                       "\\|" "\\(\\.\\)"
                   "\\)")
           1 font-lock-constant-face) ;; Operators end;
     )
 "Default expressions to highlight in Vimrc mode.")


;; Support for Vim script

(defvar vimrc-imenu-generic-expression
  '((nil "^\\(fun\\(?:ction\\)?\\)!?[[:blank:]]+\\([[:alnum:]_:#]+\\)?" 2)
    (nil "^let[[:blank:]]+\\<\\([bwglsav]:[a-zA-Z_][[:alnum:]#_]*\\)\\>" 1)
    (nil "^let[[:blank:]]+\\<\\([a-zA-Z_][[:alnum:]#_]*\\)\\>[^:]" 1))
  "Value for `imenu-generic-expression' in Vimrc mode.

Create an index of the function and variable definitions in a
Vim file.")

(defun vimrc-beginning-of-defun (&optional arg)
  "Move backward to the beginning of the current function.

With argument, repeat ARG times."
  (interactive "p")
  (re-search-backward (concat  "^[ \t]*\\(fun\\(?:ction\\)?\\)\\b")
                      nil 'move (or arg 1)))

(defun vimrc-end-of-defun (&optional arg)
  "Move forward to the next end of a function.

With argument, repeat ARG times."
  (interactive "p")
  (re-search-forward (concat  "^[ \t]*\\(endf\\(?:unction\\)?\\)\\b")
                      nil 'move (or arg 1)))

(defvar vimrc-mode-syntax-table
  (let ((table (make-syntax-table)))
    (modify-syntax-entry ?'  "\"" table)
    (modify-syntax-entry ?\" "<"  table)
    (modify-syntax-entry ?\n ">"  table)
    (modify-syntax-entry ?#  "_"  table)
    table))

;;;###autoload (add-to-list 'auto-mode-alist '("\\.vim\\'" . vimrc-mode))
;;;###autoload (add-to-list 'auto-mode-alist '("[._]?g?vimrc\\'" . vimrc-mode))
;;;###autoload (add-to-list 'auto-mode-alist '("\\.exrc\\'" . vimrc-mode))

(defalias 'vimrc--parent-mode
  (if (fboundp 'prog-mode) #'prog-mode #'fundamental-mode))

;;;###autoload
(define-derived-mode vimrc-mode vimrc--parent-mode "Vimrc"
  "Major mode for editing `vimrc', `xxx.vim' and `.exrc' configuration files."
  :group 'vimrc-mode
  :syntax-table vimrc-mode-syntax-table
  (setq-local font-lock-defaults '(vimrc-font-lock-keywords))
  (setq-local comment-start "\"")
  (setq-local comment-end "")
  ;(set (make-local-variable 'comment-start-skip) "\"\\* +")
  (setq-local imenu-generic-expression vimrc-imenu-generic-expression)
  (setq-local beginning-of-defun-function 'vimrc-beginning-of-defun)
  (setq-local end-of-defun-function 'vimrc-end-of-defun)
  (run-hooks 'vimrc-mode-hook))

(provide 'vimrc-mode)

;;; vimrc-mode.el ends here
