;;; unicode-math-input.el --- Insert Unicode math symbols using TeX notation -*- lexical-binding: t; -*-

;; Copyright (C) 2018, 2021 Augusto Stoffel

;; Author: Augusto Stoffel
;; URL: https://github.com/astoff/unicode-math-input.el
;; Package-Version: 20251012.953
;; Package-Revision: d7ee1a963f14
;; Package-Requires: ((emacs "25"))

;; This file is not part of GNU Emacs.

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; For a full copy of the GNU General Public License
;; see <http://www.gnu.org/licenses/>.

;;; Commentary:

;; This package provides two methods to insert mathematical symbols:
;;
;; - A proper input method for efficient typing.  Activate with `C-u
;;   C-\ unicode-math RET'.  Then TeX commands (e.g., `\alpha') are
;;   replaced automatically as you type with the corresponding Unicode
;;   character.
;;
;; - An interactive command, `unicode-math-input', which reads the TeX
;;   name of a symbol with completion and inserts the corresponding
;;   Unicode character.
;;
;; In either case, all symbols provided by the unicode-math LaTeX
;; package are available, with the same names.
;;
;; The `unicode-math' input method is similar to Emacs's built-in
;; `TeX', but it differs in a couple of ways.  First, it has a much
;; larger collection of characters, including various alphabets
;; (Fraktur, script, etc.) and combining accents (note you have to
;; type, say, \pi\hat to get π̂).  Second, it does not include any
;; sequence not starting with a backslash, so it interferes less with
;; normal typing.
;;
;; The input method can be customized by the variables
;; `unicode-math-input-escape', `unicode-math-input-min-prefix' and
;; `unicode-math-input-deterministic' (but they must be set before
;; loading the package).

;;; Code:

(require 'quail)

(defgroup unicode-math-input nil
  "Insert Unicode math symbols using TeX notation"
  :group 'convenience
  :group 'quail)

(defcustom unicode-math-input-escape "\\"
  "The triggering character for the `unicode-math' input method.
Setting this variable has no effect after `unicode-math-input' is
loaded.  Restart Emacs if needed."
  :type 'string)

(defcustom unicode-math-input-min-prefix 2
  "Replace candidates at least this long by their match in the input method.
This should be a positive number, or nil to disable prefix
matching.

Setting this variable has no effect after `unicode-math-input' is
loaded.  Restart Emacs if needed."
  :type 'integer)

(defcustom unicode-math-input-deterministic nil
  "When nil, allows selecting candidate translations with numeric keys.
This sets the DETERMINISTIC option of the Quail input method.
See `quail-define-package' for more details.

Setting this variable has no effect after `unicode-math-input' is
loaded.  Restart Emacs if needed."
  :type 'boolean)

(defcustom unicode-math-input-annotation
  #(" %c" 1 3 (face font-lock-keyword-face))
  "Annotation displayed in the `unicode-math-input' query.
This is passed to `format' with the relevant character as an
additional argument."
  :type 'string)

(defcustom unicode-math-input-extra-symbols nil
  "Alist of user-defined symbols for `unicode-math-input'.
Entries should be of the form (NAME . CHARACTER).

You must set this variable before loading `unicode-math-input'
for it to affect the input method."
  :type '(alist :key-type string :value-type: character))

;; from unicode-math-table.tex
(defvar unicode-math-input-symbols
  '(("accurrent" . #x023E6)
    ("acidfree" . #x0267E)
    ("acute" . #x00301)
    ("acwcirclearrow" . #x02940)
    ("acwgapcirclearrow" . #x027F2)
    ("acwleftarcarrow" . #x02939)
    ("acwopencirclearrow" . #x021BA)
    ("acwoverarcarrow" . #x0293A)
    ("acwunderarcarrow" . #x0293B)
    ("adots" . #x022F0)
    ("aleph" . #x02135)
    ("Alpha" . #x00391)
    ("alpha" . #x003B1)
    ("amalg" . #x02A3F)
    ("angdnr" . #x0299F)
    ("angle" . #x02220)
    ("angles" . #x0299E)
    ("angleubar" . #x029A4)
    ("Angstrom" . #x0212B)
    ("annuity" . #x020E7)
    ("APLboxquestion" . #x02370)
    ("APLboxupcaret" . #x02353)
    ("APLnotbackslash" . #x02340)
    ("APLnotslash" . #x0233F)
    ("approx" . #x02248)
    ("approxeq" . #x0224A)
    ("approxeqq" . #x02A70)
    ("approxident" . #x0224B)
    ("arabichad" . #x1EEF1)
    ("arabicmaj" . #x1EEF0)
    ("arceq" . #x02258)
    ("assert" . #x022A6)
    ("ast" . #x02217)
    ("asteq" . #x02A6E)
    ("asteraccent" . #x020F0)
    ("astrosun" . #x02609)
    ("asymp" . #x0224D)
    ("awint" . #x02A11)
    ("backcong" . #x0224C)
    ("backdprime" . #x02036)
    ("backprime" . #x02035)
    ("backsim" . #x0223D)
    ("backsimeq" . #x022CD)
    ("backslash" . #x0005C)
    ("backtrprime" . #x02037)
    ("bagmember" . #x022FF)
    ("bar" . #x00304)
    ("barcap" . #x02A43)
    ("barcup" . #x02A42)
    ("bardownharpoonleft" . #x02961)
    ("bardownharpoonright" . #x0295D)
    ("barleftarrow" . #x021E4)
    ("barleftarrowrightarrowbar" . #x021B9)
    ("barleftharpoondown" . #x02956)
    ("barleftharpoonup" . #x02952)
    ("barovernorthwestarrow" . #x021B8)
    ("barrightarrowdiamond" . #x02920)
    ("barrightharpoondown" . #x0295F)
    ("barrightharpoonup" . #x0295B)
    ("baruparrow" . #x02912)
    ("barupharpoonleft" . #x02958)
    ("barupharpoonright" . #x02954)
    ("Barv" . #x02AE7)
    ("barV" . #x02AEA)
    ("barvee" . #x022BD)
    ("barwedge" . #x022BC)
    ("BbbA" . #x1D538)
    ("Bbba" . #x1D552)
    ("BbbB" . #x1D539)
    ("Bbbb" . #x1D553)
    ("BbbC" . #x02102)
    ("Bbbc" . #x1D554)
    ("BbbD" . #x1D53B)
    ("Bbbd" . #x1D555)
    ("BbbE" . #x1D53C)
    ("Bbbe" . #x1D556)
    ("Bbbeight" . #x1D7E0)
    ("BbbF" . #x1D53D)
    ("Bbbf" . #x1D557)
    ("Bbbfive" . #x1D7DD)
    ("Bbbfour" . #x1D7DC)
    ("BbbG" . #x1D53E)
    ("Bbbg" . #x1D558)
    ("Bbbgamma" . #x0213D)
    ("BbbGamma" . #x0213E)
    ("BbbH" . #x0210D)
    ("Bbbh" . #x1D559)
    ("BbbI" . #x1D540)
    ("Bbbi" . #x1D55A)
    ("BbbJ" . #x1D541)
    ("Bbbj" . #x1D55B)
    ("BbbK" . #x1D542)
    ("Bbbk" . #x1D55C)
    ("BbbL" . #x1D543)
    ("Bbbl" . #x1D55D)
    ("BbbM" . #x1D544)
    ("Bbbm" . #x1D55E)
    ("BbbN" . #x02115)
    ("Bbbn" . #x1D55F)
    ("Bbbnine" . #x1D7E1)
    ("BbbO" . #x1D546)
    ("Bbbo" . #x1D560)
    ("Bbbone" . #x1D7D9)
    ("BbbP" . #x02119)
    ("Bbbp" . #x1D561)
    ("Bbbpi" . #x0213C)
    ("BbbPi" . #x0213F)
    ("BbbQ" . #x0211A)
    ("Bbbq" . #x1D562)
    ("BbbR" . #x0211D)
    ("Bbbr" . #x1D563)
    ("BbbS" . #x1D54A)
    ("Bbbs" . #x1D564)
    ("Bbbseven" . #x1D7DF)
    ("Bbbsix" . #x1D7DE)
    ("Bbbsum" . #x02140)
    ("BbbT" . #x1D54B)
    ("Bbbt" . #x1D565)
    ("Bbbthree" . #x1D7DB)
    ("Bbbtwo" . #x1D7DA)
    ("BbbU" . #x1D54C)
    ("Bbbu" . #x1D566)
    ("BbbV" . #x1D54D)
    ("Bbbv" . #x1D567)
    ("BbbW" . #x1D54E)
    ("Bbbw" . #x1D568)
    ("BbbX" . #x1D54F)
    ("Bbbx" . #x1D569)
    ("BbbY" . #x1D550)
    ("Bbby" . #x1D56A)
    ("BbbZ" . #x02124)
    ("Bbbz" . #x1D56B)
    ("Bbbzero" . #x1D7D8)
    ("bbrktbrk" . #x023B6)
    ("bdtriplevdash" . #x02506)
    ("because" . #x02235)
    ("benzenr" . #x023E3)
    ("Beta" . #x00392)
    ("beta" . #x003B2)
    ("beth" . #x02136)
    ("between" . #x0226C)
    ("bigblacktriangledown" . #x025BC)
    ("bigblacktriangleup" . #x025B2)
    ("bigbot" . #x027D8)
    ("bigcap" . #x022C2)
    ("bigcup" . #x022C3)
    ("bigcupdot" . #x02A03)
    ("biginterleave" . #x02AFC)
    ("bigodot" . #x02A00)
    ("bigoplus" . #x02A01)
    ("bigotimes" . #x02A02)
    ("bigslopedvee" . #x02A57)
    ("bigslopedwedge" . #x02A58)
    ("bigsqcap" . #x02A05)
    ("bigsqcup" . #x02A06)
    ("bigstar" . #x02605)
    ("bigtalloblong" . #x02AFF)
    ("bigtimes" . #x02A09)
    ("bigtop" . #x027D9)
    ("bigtriangledown" . #x025BD)
    ("bigtriangleleft" . #x02A1E)
    ("bigtriangleup" . #x025B3)
    ("biguplus" . #x02A04)
    ("bigvee" . #x022C1)
    ("bigwedge" . #x022C0)
    ("bigwhitestar" . #x02606)
    ("blackcircledownarrow" . #x029ED)
    ("blackcircledrightdot" . #x02688)
    ("blackcircledtwodots" . #x02689)
    ("blackcircleulquadwhite" . #x025D5)
    ("blackdiamonddownarrow" . #x029EA)
    ("blackhourglass" . #x029D7)
    ("blackinwhitediamond" . #x025C8)
    ("blackinwhitesquare" . #x025A3)
    ("blacklefthalfcircle" . #x025D6)
    ("blackpointerleft" . #x025C4)
    ("blackpointerright" . #x025BA)
    ("blackrighthalfcircle" . #x025D7)
    ("blacksmiley" . #x0263B)
    ("blacktriangle" . #x025B4)
    ("blacktriangledown" . #x025BE)
    ("blacktriangleleft" . #x025C0)
    ("blacktriangleright" . #x025B6)
    ("blanksymbol" . #x02422)
    ("blkhorzoval" . #x02B2C)
    ("blkvertoval" . #x02B2E)
    ("blockfull" . #x02588)
    ("blockhalfshaded" . #x02592)
    ("blocklefthalf" . #x0258C)
    ("blocklowhalf" . #x02584)
    ("blockqtrshaded" . #x02591)
    ("blockrighthalf" . #x02590)
    ("blockthreeqtrshaded" . #x02593)
    ("blockuphalf" . #x02580)
    ("bNot" . #x02AED)
    ("bot" . #x022A5)
    ("botsemicircle" . #x025E1)
    ("bowtie" . #x022C8)
    ("boxast" . #x029C6)
    ("boxbar" . #x025EB)
    ("boxbox" . #x029C8)
    ("boxbslash" . #x029C5)
    ("boxcircle" . #x029C7)
    ("boxdiag" . #x029C4)
    ("boxdot" . #x022A1)
    ("boxminus" . #x0229F)
    ("boxonbox" . #x029C9)
    ("boxplus" . #x0229E)
    ("boxtimes" . #x022A0)
    ("breve" . #x00306)
    ("bsimilarleftarrow" . #x02B41)
    ("bsimilarrightarrow" . #x02B47)
    ("bsolhsub" . #x027C8)
    ("btimes" . #x02A32)
    ("bullseye" . #x025CE)
    ("Bumpeq" . #x0224E)
    ("bumpeq" . #x0224F)
    ("bumpeqq" . #x02AAE)
    ("candra" . #x00310)
    ("cap" . #x02229)
    ("Cap" . #x022D2)
    ("capbarcup" . #x02A49)
    ("capdot" . #x02A40)
    ("capovercup" . #x02A47)
    ("capwedge" . #x02A44)
    ("caretinsert" . #x02038)
    ("carriagereturn" . #x021B5)
    ("ccwundercurvearrow" . #x0293F)
    ("cdot" . #x022C5)
    ("cdotp" . #x000B7)
    ("check" . #x0030C)
    ("checkmark" . #x02713)
    ("Chi" . #x003A7)
    ("chi" . #x003C7)
    ("cirbot" . #x027DF)
    ("circ" . #x02218)
    ("circeq" . #x02257)
    ("circlebottomhalfblack" . #x025D2)
    ("circledast" . #x0229B)
    ("circledbullet" . #x029BF)
    ("circledcirc" . #x0229A)
    ("circleddash" . #x0229D)
    ("circledequal" . #x0229C)
    ("circledownarrow" . #x029EC)
    ("circledparallel" . #x029B7)
    ("circledrightdot" . #x02686)
    ("circledstar" . #x0272A)
    ("circledtwodots" . #x02687)
    ("circledvert" . #x029B6)
    ("circledwhitebullet" . #x029BE)
    ("circlehbar" . #x029B5)
    ("circlelefthalfblack" . #x025D0)
    ("circlellquad" . #x025F5)
    ("circlelrquad" . #x025F6)
    ("circleonleftarrow" . #x02B30)
    ("circleonrightarrow" . #x021F4)
    ("circlerighthalfblack" . #x025D1)
    ("circletophalfblack" . #x025D3)
    ("circleulquad" . #x025F4)
    ("circleurquad" . #x025F7)
    ("circleurquadblack" . #x025D4)
    ("circlevertfill" . #x025CD)
    ("cirE" . #x029C3)
    ("cirfnint" . #x02A10)
    ("cirmid" . #x02AEF)
    ("cirscir" . #x029C2)
    ("closedvarcap" . #x02A4D)
    ("closedvarcup" . #x02A4C)
    ("closedvarcupsmashprod" . #x02A50)
    ("closure" . #x02050)
    ("clubsuit" . #x02663)
    ("Colon" . #x02237)
    ("coloneq" . #x02254)
    ("Coloneq" . #x02A74)
    ("commaminus" . #x02A29)
    ("complement" . #x02201)
    ("concavediamond" . #x027E1)
    ("concavediamondtickleft" . #x027E2)
    ("concavediamondtickright" . #x027E3)
    ("cong" . #x02245)
    ("congdot" . #x02A6D)
    ("conictaper" . #x02332)
    ("conjquant" . #x02A07)
    ("coprod" . #x02210)
    ("csub" . #x02ACF)
    ("csube" . #x02AD1)
    ("csup" . #x02AD0)
    ("csupe" . #x02AD2)
    ("cuberoot" . #x0221B)
    ("cup" . #x0222A)
    ("Cup" . #x022D3)
    ("cupbarcap" . #x02A48)
    ("cupdot" . #x0228D)
    ("cupleftarrow" . #x0228C)
    ("cupovercap" . #x02A46)
    ("cupvee" . #x02A45)
    ("curlyeqprec" . #x022DE)
    ("curlyeqsucc" . #x022DF)
    ("curlyvee" . #x022CE)
    ("curlywedge" . #x022CF)
    ("curvearrowleft" . #x021B6)
    ("curvearrowleftplus" . #x0293D)
    ("curvearrowright" . #x021B7)
    ("curvearrowrightminus" . #x0293C)
    ("cwcirclearrow" . #x02941)
    ("cwgapcirclearrow" . #x027F3)
    ("cwopencirclearrow" . #x021BB)
    ("cwrightarcarrow" . #x02938)
    ("cwundercurvearrow" . #x0293E)
    ("dagger" . #x02020)
    ("daleth" . #x02138)
    ("danger" . #x02621)
    ("dashcolon" . #x02239)
    ("dashleftharpoondown" . #x0296B)
    ("dashrightharpoondown" . #x0296D)
    ("dashv" . #x022A3)
    ("dashV" . #x02AE3)
    ("Dashv" . #x02AE4)
    ("DashV" . #x02AE5)
    ("DashVDash" . #x027DA)
    ("dashVdash" . #x027DB)
    ("dbkarrow" . #x0290F)
    ("ddagger" . #x02021)
    ("ddddot" . #x020DC)
    ("dddot" . #x020DB)
    ("ddot" . #x00308)
    ("ddots" . #x022F1)
    ("ddotseq" . #x02A77)
    ("DDownarrow" . #x027F1)
    ("Ddownarrow" . #x0290B)
    ("Delta" . #x00394)
    ("delta" . #x003B4)
    ("diameter" . #x02300)
    ("diamondbotblack" . #x02B19)
    ("diamondcdot" . #x027D0)
    ("diamondleftarrow" . #x0291D)
    ("diamondleftarrowbar" . #x0291F)
    ("diamondleftblack" . #x02B16)
    ("diamondrightblack" . #x02B17)
    ("diamondsuit" . #x02662)
    ("diamondtopblack" . #x02B18)
    ("dicei" . #x02680)
    ("diceii" . #x02681)
    ("diceiii" . #x02682)
    ("diceiv" . #x02683)
    ("dicev" . #x02684)
    ("dicevi" . #x02685)
    ("dingasterisk" . #x0273D)
    ("disin" . #x022F2)
    ("disjquant" . #x02A08)
    ("div" . #x000F7)
    ("divideontimes" . #x022C7)
    ("divslash" . #x02215)
    ("dot" . #x00307)
    ("doteq" . #x02250)
    ("Doteq" . #x02251)
    ("dotequiv" . #x02A67)
    ("dotminus" . #x02238)
    ("dotplus" . #x02214)
    ("dotsim" . #x02A6A)
    ("dotsminusdots" . #x0223A)
    ("dottedcircle" . #x025CC)
    ("dottedsquare" . #x02B1A)
    ("dottimes" . #x02A30)
    ("doublebarvee" . #x02A62)
    ("doublebarwedge" . #x02A5E)
    ("doubleplus" . #x029FA)
    ("downarrow" . #x02193)
    ("Downarrow" . #x021D3)
    ("downarrowbar" . #x02913)
    ("downarrowbarred" . #x02908)
    ("downdasharrow" . #x021E3)
    ("downdownarrows" . #x021CA)
    ("downfishtail" . #x0297F)
    ("downharpoonleft" . #x021C3)
    ("downharpoonleftbar" . #x02959)
    ("downharpoonright" . #x021C2)
    ("downharpoonrightbar" . #x02955)
    ("downharpoonsleftright" . #x02965)
    ("downrightcurvedarrow" . #x02935)
    ("downtriangleleftblack" . #x029E8)
    ("downtrianglerightblack" . #x029E9)
    ("downuparrows" . #x021F5)
    ("downupharpoonsleftright" . #x0296F)
    ("downwhitearrow" . #x021E9)
    ("downzigzagarrow" . #x021AF)
    ("dprime" . #x02033)
    ("draftingarrow" . #x0279B)
    ("drbkarrow" . #x02910)
    ("droang" . #x0031A)
    ("dsol" . #x029F6)
    ("dsub" . #x02A64)
    ("dualmap" . #x029DF)
    ("egsdot" . #x02A98)
    ("eighthnote" . #x0266A)
    ("elinters" . #x023E7)
    ("ell" . #x02113)
    ("elsdot" . #x02A97)
    ("emptysetoarr" . #x029B3)
    ("emptysetoarrl" . #x029B4)
    ("emptysetobar" . #x029B1)
    ("emptysetocirc" . #x029B2)
    ("enclosecircle" . #x020DD)
    ("enclosediamond" . #x020DF)
    ("enclosesquare" . #x020DE)
    ("enclosetriangle" . #x020E4)
    ("enleadertwodots" . #x02025)
    ("eparsl" . #x029E3)
    ("Epsilon" . #x00395)
    ("epsilon" . #x003F5)
    ("eqcirc" . #x02256)
    ("eqcolon" . #x02255)
    ("eqdef" . #x0225D)
    ("eqdot" . #x02A66)
    ("eqeq" . #x02A75)
    ("eqeqeq" . #x02A76)
    ("eqgtr" . #x022DD)
    ("eqless" . #x022DC)
    ("eqqgtr" . #x02A9A)
    ("eqqless" . #x02A99)
    ("eqqplus" . #x02A71)
    ("eqqsim" . #x02A73)
    ("eqqslantgtr" . #x02A9C)
    ("eqqslantless" . #x02A9B)
    ("eqsim" . #x02242)
    ("eqslantgtr" . #x02A96)
    ("eqslantless" . #x02A95)
    ("equal" . #x0003D)
    ("equalleftarrow" . #x02B40)
    ("equalparallel" . #x022D5)
    ("equalrightarrow" . #x02971)
    ("equiv" . #x02261)
    ("Equiv" . #x02263)
    ("equivDD" . #x02A78)
    ("equivVert" . #x02A68)
    ("equivVvert" . #x02A69)
    ("eqvparsl" . #x029E5)
    ("errbarblackcircle" . #x029F3)
    ("errbarblackdiamond" . #x029F1)
    ("errbarblacksquare" . #x029EF)
    ("errbarcircle" . #x029F2)
    ("errbardiamond" . #x029F0)
    ("errbarsquare" . #x029EE)
    ("Eta" . #x00397)
    ("eta" . #x003B7)
    ("Eulerconst" . #x02107)
    ("euro" . #x020AC)
    ("Exclam" . #x0203C)
    ("exists" . #x02203)
    ("fallingdotseq" . #x02252)
    ("fbowtie" . #x029D3)
    ("fcmp" . #x02A3E)
    ("fdiagovnearrow" . #x0292F)
    ("fdiagovrdiag" . #x0292C)
    ("female" . #x02640)
    ("fint" . #x02A0F)
    ("Finv" . #x02132)
    ("fisheye" . #x025C9)
    ("flat" . #x0266D)
    ("fltns" . #x023E5)
    ("forall" . #x02200)
    ("forks" . #x02ADC)
    ("forksnot" . #x02ADD)
    ("forkv" . #x02AD9)
    ("fourthroot" . #x0221C)
    ("fourvdots" . #x02999)
    ("fracslash" . #x02044)
    ("frown" . #x02322)
    ("fullouterjoin" . #x027D7)
    ("Game" . #x02141)
    ("Gamma" . #x00393)
    ("gamma" . #x003B3)
    ("ge" . #x02265) ; plain.tex alias
    ("geq" . #x02265)
    ("geqq" . #x02267)
    ("geqqslant" . #x02AFA)
    ("geqslant" . #x02A7E)
    ("gescc" . #x02AA9)
    ("gesdot" . #x02A80)
    ("gesdoto" . #x02A82)
    ("gesdotol" . #x02A84)
    ("gesles" . #x02A94)
    ("gets" . #x02190) ; plain.tex alias
    ("gg" . #x0226B)
    ("ggg" . #x022D9)
    ("gggnest" . #x02AF8)
    ("gimel" . #x02137)
    ("gla" . #x02AA5)
    ("glE" . #x02A92)
    ("gleichstark" . #x029E6)
    ("glj" . #x02AA4)
    ("gnapprox" . #x02A8A)
    ("gneq" . #x02A88)
    ("gneqq" . #x02269)
    ("gnsim" . #x022E7)
    ("grave" . #x00300)
    ("greater" . #x0003E)
    ("gsime" . #x02A8E)
    ("gsiml" . #x02A90)
    ("Gt" . #x02AA2)
    ("gtcc" . #x02AA7)
    ("gtcir" . #x02A7A)
    ("gtlpar" . #x029A0)
    ("gtquest" . #x02A7C)
    ("gtrapprox" . #x02A86)
    ("gtrarr" . #x02978)
    ("gtrdot" . #x022D7)
    ("gtreqless" . #x022DB)
    ("gtreqqless" . #x02A8C)
    ("gtrless" . #x02277)
    ("gtrsim" . #x02273)
    ("harrowextender" . #x023AF)
    ("hat" . #x00302)
    ("hatapprox" . #x02A6F)
    ("heartsuit" . #x02661)
    ("Hermaphrodite" . #x026A5)
    ("hermitmatrix" . #x022B9)
    ("hexagon" . #x02394)
    ("hexagonblack" . #x02B23)
    ("hknearrow" . #x02924)
    ("hknwarrow" . #x02923)
    ("hksearrow" . #x02925)
    ("hkswarrow" . #x02926)
    ("hookleftarrow" . #x021A9)
    ("hookrightarrow" . #x021AA)
    ("horizbar" . #x02015)
    ("hourglass" . #x029D6)
    ("house" . #x02302)
    ("hrectangle" . #x025AD)
    ("hrectangleblack" . #x025AC)
    ("hslash" . #x0210F)
    ("hyphenbullet" . #x02043)
    ("hzigzag" . #x03030)
    ("iiiint" . #x02A0C)
    ("iiint" . #x0222D)
    ("iinfin" . #x029DC)
    ("iint" . #x0222C)
    ("Im" . #x02111)
    ("imageof" . #x022B7)
    ("imath" . #x1D6A4)
    ("in" . #x02208)
    ("increment" . #x02206)
    ("infty" . #x0221E)
    ("int" . #x0222B)
    ("intbar" . #x02A0D)
    ("intBar" . #x02A0E)
    ("intbottom" . #x02321)
    ("intcap" . #x02A19)
    ("intclockwise" . #x02231)
    ("intcup" . #x02A1A)
    ("intercal" . #x022BA)
    ("interleave" . #x02AF4)
    ("intextender" . #x023AE)
    ("intlarhk" . #x02A17)
    ("intprod" . #x02A3C)
    ("intprodr" . #x02A3D)
    ("inttop" . #x02320)
    ("intx" . #x02A18)
    ("inversebullet" . #x025D8)
    ("inversewhitecircle" . #x025D9)
    ("invlazys" . #x0223E)
    ("invnot" . #x02310)
    ("invwhitelowerhalfcircle" . #x025DB)
    ("invwhiteupperhalfcircle" . #x025DA)
    ("Iota" . #x00399)
    ("iota" . #x003B9)
    ("isindot" . #x022F5)
    ("isinE" . #x022F9)
    ("isinobar" . #x022F7)
    ("isins" . #x022F4)
    ("isinvb" . #x022F8)
    ("jmath" . #x1D6A5)
    ("Join" . #x02A1D)
    ("Kappa" . #x0039A)
    ("kappa" . #x003BA)
    ("kernelcontraction" . #x0223B)
    ("Lambda" . #x0039B)
    ("lambda" . #x003BB)
    ("land" . #x02227) ; plain.tex alias
    ("langle" . #x027E8)
    ("lAngle" . #x027EA)
    ("langledot" . #x02991)
    ("laplac" . #x029E0)
    ("lat" . #x02AAB)
    ("late" . #x02AAD)
    ("lbag" . #x027C5)
    ("lblkbrbrak" . #x02997)
    ("lbrace" . #x0007B)
    ("lBrace" . #x02983)
    ("lbracelend" . #x023A9)
    ("lbracemid" . #x023A8)
    ("lbraceuend" . #x023A7)
    ("lbrack" . #x0005B)
    ("lBrack" . #x027E6)
    ("lbrackextender" . #x023A2)
    ("lbracklend" . #x023A3)
    ("lbracklltick" . #x0298F)
    ("lbrackubar" . #x0298B)
    ("lbrackuend" . #x023A1)
    ("lbrackultick" . #x0298D)
    ("lbrbrak" . #x02772)
    ("Lbrbrak" . #x027EC)
    ("lceil" . #x02308)
    ("lcurvyangle" . #x029FC)
    ("Ldsh" . #x021B2)
    ("le" . #x02264) ; plain.tex alias
    ("leftarrow" . #x02190)
    ("Leftarrow" . #x021D0)
    ("leftarrowapprox" . #x02B4A)
    ("leftarrowbackapprox" . #x02B42)
    ("leftarrowbsimilar" . #x02B4B)
    ("leftarrowless" . #x02977)
    ("leftarrowonoplus" . #x02B32)
    ("leftarrowplus" . #x02946)
    ("leftarrowshortrightarrow" . #x02943)
    ("leftarrowsimilar" . #x02973)
    ("leftarrowsubset" . #x0297A)
    ("leftarrowtail" . #x021A2)
    ("leftarrowtriangle" . #x021FD)
    ("leftarrowx" . #x02B3E)
    ("leftbkarrow" . #x0290C)
    ("leftcurvedarrow" . #x02B3F)
    ("leftdasharrow" . #x021E0)
    ("leftdbkarrow" . #x0290E)
    ("leftdbltail" . #x0291B)
    ("leftdotarrow" . #x02B38)
    ("leftdowncurvedarrow" . #x02936)
    ("leftfishtail" . #x0297C)
    ("leftharpoonaccent" . #x020D0)
    ("leftharpoondown" . #x021BD)
    ("leftharpoondownbar" . #x0295E)
    ("leftharpoonsupdown" . #x02962)
    ("leftharpoonup" . #x021BC)
    ("leftharpoonupbar" . #x0295A)
    ("leftharpoonupdash" . #x0296A)
    ("leftleftarrows" . #x021C7)
    ("leftmoon" . #x0263E)
    ("leftouterjoin" . #x027D5)
    ("leftrightarrow" . #x02194)
    ("Leftrightarrow" . #x021D4)
    ("leftrightarrowcircle" . #x02948)
    ("leftrightarrows" . #x021C6)
    ("leftrightarrowtriangle" . #x021FF)
    ("leftrightharpoondowndown" . #x02950)
    ("leftrightharpoondownup" . #x0294B)
    ("leftrightharpoons" . #x021CB)
    ("leftrightharpoonsdown" . #x02967)
    ("leftrightharpoonsup" . #x02966)
    ("leftrightharpoonupdown" . #x0294A)
    ("leftrightharpoonupup" . #x0294E)
    ("leftrightsquigarrow" . #x021AD)
    ("leftsquigarrow" . #x021DC)
    ("lefttail" . #x02919)
    ("leftthreearrows" . #x02B31)
    ("leftthreetimes" . #x022CB)
    ("leftwavearrow" . #x0219C)
    ("leftwhitearrow" . #x021E6)
    ("leq" . #x02264)
    ("leqq" . #x02266)
    ("leqqslant" . #x02AF9)
    ("leqslant" . #x02A7D)
    ("lescc" . #x02AA8)
    ("lesdot" . #x02A7F)
    ("lesdoto" . #x02A81)
    ("lesdotor" . #x02A83)
    ("lesges" . #x02A93)
    ("less" . #x0003C)
    ("lessapprox" . #x02A85)
    ("lessdot" . #x022D6)
    ("lesseqgtr" . #x022DA)
    ("lesseqqgtr" . #x02A8B)
    ("lessgtr" . #x02276)
    ("lesssim" . #x02272)
    ("lfbowtie" . #x029D1)
    ("lfloor" . #x0230A)
    ("lftimes" . #x029D4)
    ("lgblkcircle" . #x02B24)
    ("lgblksquare" . #x02B1B)
    ("lgE" . #x02A91)
    ("lgroup" . #x027EE)
    ("lgwhtcircle" . #x025EF)
    ("lgwhtsquare" . #x02B1C)
    ("linefeed" . #x021B4)
    ("ll" . #x0226A)
    ("llangle" . #x02989)
    ("llarc" . #x025DF)
    ("llblacktriangle" . #x025E3)
    ("llcorner" . #x0231E)
    ("Lleftarrow" . #x021DA)
    ("LLeftarrow" . #x02B45)
    ("lll" . #x022D8)
    ("lllnest" . #x02AF7)
    ("llparenthesis" . #x02987)
    ("lltriangle" . #x025FA)
    ("lmoustache" . #x023B0)
    ("lnapprox" . #x02A89)
    ("lneq" . #x02A87)
    ("lneqq" . #x02268)
    ("lnot" . #x000AC) ; plain.tex alias
    ("lnsim" . #x022E6)
    ("longdashv" . #x027DE)
    ("longdivision" . #x027CC)
    ("longleftarrow" . #x027F5)
    ("Longleftarrow" . #x027F8)
    ("longleftrightarrow" . #x027F7)
    ("Longleftrightarrow" . #x027FA)
    ("longleftsquigarrow" . #x02B33)
    ("longmapsfrom" . #x027FB)
    ("Longmapsfrom" . #x027FD)
    ("longmapsto" . #x027FC)
    ("Longmapsto" . #x027FE)
    ("longrightarrow" . #x027F6)
    ("Longrightarrow" . #x027F9)
    ("longrightsquigarrow" . #x027FF)
    ("looparrowleft" . #x021AB)
    ("looparrowright" . #x021AC)
    ("lor" . #x02228) ; plain.tex alias
    ("lowint" . #x02A1C)
    ("lozengeminus" . #x027E0)
    ("lparen" . #x00028)
    ("lParen" . #x02985)
    ("lparenextender" . #x0239C)
    ("Lparengtr" . #x02995)
    ("lparenlend" . #x0239D)
    ("lparenless" . #x02993)
    ("lparenuend" . #x0239B)
    ("lrarc" . #x025DE)
    ("lrblacktriangle" . #x025E2)
    ("lrcorner" . #x0231F)
    ("lrtriangle" . #x025FF)
    ("lrtriangleeq" . #x029E1)
    ("Lsh" . #x021B0)
    ("lsime" . #x02A8D)
    ("lsimg" . #x02A8F)
    ("lsqhook" . #x02ACD)
    ("Lt" . #x02AA1)
    ("ltcc" . #x02AA6)
    ("ltcir" . #x02A79)
    ("ltimes" . #x022C9)
    ("ltlarr" . #x02976)
    ("ltquest" . #x02A7B)
    ("ltrivb" . #x029CF)
    ("lvboxline" . #x023B8)
    ("lvzigzag" . #x029D8)
    ("Lvzigzag" . #x029DA)
    ("male" . #x02642)
    ("maltese" . #x02720)
    ("mapsdown" . #x021A7)
    ("mapsfrom" . #x021A4)
    ("Mapsfrom" . #x02906)
    ("mapsto" . #x021A6)
    ("Mapsto" . #x02907)
    ("mapsup" . #x021A5)
    ("mathampersand" . #x00026)
    ("mathatsign" . #x00040)
    ("mathcolon" . #x0003A)
    ("mathcomma" . #x0002C)
    ("mathdollar" . #x00024)
    ("matheth" . #x000F0)
    ("mathexclam" . #x00021)
    ("mathhyphen" . #x02010)
    ("mathoctothorpe" . #x00023)
    ("mathparagraph" . #x000B6)
    ("mathpercent" . #x00025)
    ("mathperiod" . #x0002E)
    ("mathplus" . #x0002B)
    ("mathquestion" . #x0003F)
    ("mathratio" . #x02236)
    ("mathsection" . #x000A7)
    ("mathsemicolon" . #x0003B)
    ("mathslash" . #x0002F)
    ("mathsterling" . #x000A3)
    ("mathunderbar" . #x00332)
    ("mathvisiblespace" . #x02423)
    ("mathyen" . #x000A5)
    ("mbfA" . #x1D400)
    ("mbfa" . #x1D41A)
    ("mbfAlpha" . #x1D6A8)
    ("mbfalpha" . #x1D6C2)
    ("mbfB" . #x1D401)
    ("mbfb" . #x1D41B)
    ("mbfBeta" . #x1D6A9)
    ("mbfbeta" . #x1D6C3)
    ("mbfC" . #x1D402)
    ("mbfc" . #x1D41C)
    ("mbfChi" . #x1D6BE)
    ("mbfchi" . #x1D6D8)
    ("mbfD" . #x1D403)
    ("mbfd" . #x1D41D)
    ("mbfDelta" . #x1D6AB)
    ("mbfdelta" . #x1D6C5)
    ("mbfDigamma" . #x1D7CA)
    ("mbfdigamma" . #x1D7CB)
    ("mbfE" . #x1D404)
    ("mbfe" . #x1D41E)
    ("mbfeight" . #x1D7D6)
    ("mbfEpsilon" . #x1D6AC)
    ("mbfepsilon" . #x1D6DC)
    ("mbfEta" . #x1D6AE)
    ("mbfeta" . #x1D6C8)
    ("mbfF" . #x1D405)
    ("mbff" . #x1D41F)
    ("mbffive" . #x1D7D3)
    ("mbffour" . #x1D7D2)
    ("mbffrakA" . #x1D56C)
    ("mbffraka" . #x1D586)
    ("mbffrakB" . #x1D56D)
    ("mbffrakb" . #x1D587)
    ("mbffrakC" . #x1D56E)
    ("mbffrakc" . #x1D588)
    ("mbffrakD" . #x1D56F)
    ("mbffrakd" . #x1D589)
    ("mbffrakE" . #x1D570)
    ("mbffrake" . #x1D58A)
    ("mbffrakF" . #x1D571)
    ("mbffrakf" . #x1D58B)
    ("mbffrakG" . #x1D572)
    ("mbffrakg" . #x1D58C)
    ("mbffrakH" . #x1D573)
    ("mbffrakh" . #x1D58D)
    ("mbffrakI" . #x1D574)
    ("mbffraki" . #x1D58E)
    ("mbffrakJ" . #x1D575)
    ("mbffrakj" . #x1D58F)
    ("mbffrakK" . #x1D576)
    ("mbffrakk" . #x1D590)
    ("mbffrakL" . #x1D577)
    ("mbffrakl" . #x1D591)
    ("mbffrakM" . #x1D578)
    ("mbffrakm" . #x1D592)
    ("mbffrakN" . #x1D579)
    ("mbffrakn" . #x1D593)
    ("mbffrakO" . #x1D57A)
    ("mbffrako" . #x1D594)
    ("mbffrakP" . #x1D57B)
    ("mbffrakp" . #x1D595)
    ("mbffrakQ" . #x1D57C)
    ("mbffrakq" . #x1D596)
    ("mbffrakR" . #x1D57D)
    ("mbffrakr" . #x1D597)
    ("mbffrakS" . #x1D57E)
    ("mbffraks" . #x1D598)
    ("mbffrakT" . #x1D57F)
    ("mbffrakt" . #x1D599)
    ("mbffrakU" . #x1D580)
    ("mbffraku" . #x1D59A)
    ("mbffrakV" . #x1D581)
    ("mbffrakv" . #x1D59B)
    ("mbffrakW" . #x1D582)
    ("mbffrakw" . #x1D59C)
    ("mbffrakX" . #x1D583)
    ("mbffrakx" . #x1D59D)
    ("mbffrakY" . #x1D584)
    ("mbffraky" . #x1D59E)
    ("mbffrakZ" . #x1D585)
    ("mbffrakz" . #x1D59F)
    ("mbfG" . #x1D406)
    ("mbfg" . #x1D420)
    ("mbfGamma" . #x1D6AA)
    ("mbfgamma" . #x1D6C4)
    ("mbfH" . #x1D407)
    ("mbfh" . #x1D421)
    ("mbfI" . #x1D408)
    ("mbfi" . #x1D422)
    ("mbfIota" . #x1D6B0)
    ("mbfiota" . #x1D6CA)
    ("mbfitA" . #x1D468)
    ("mbfita" . #x1D482)
    ("mbfitAlpha" . #x1D71C)
    ("mbfitalpha" . #x1D736)
    ("mbfitB" . #x1D469)
    ("mbfitb" . #x1D483)
    ("mbfitBeta" . #x1D71D)
    ("mbfitbeta" . #x1D737)
    ("mbfitC" . #x1D46A)
    ("mbfitc" . #x1D484)
    ("mbfitChi" . #x1D732)
    ("mbfitchi" . #x1D74C)
    ("mbfitD" . #x1D46B)
    ("mbfitd" . #x1D485)
    ("mbfitDelta" . #x1D71F)
    ("mbfitdelta" . #x1D739)
    ("mbfitE" . #x1D46C)
    ("mbfite" . #x1D486)
    ("mbfitEpsilon" . #x1D720)
    ("mbfitepsilon" . #x1D750)
    ("mbfitEta" . #x1D722)
    ("mbfiteta" . #x1D73C)
    ("mbfitF" . #x1D46D)
    ("mbfitf" . #x1D487)
    ("mbfitG" . #x1D46E)
    ("mbfitg" . #x1D488)
    ("mbfitGamma" . #x1D71E)
    ("mbfitgamma" . #x1D738)
    ("mbfitH" . #x1D46F)
    ("mbfith" . #x1D489)
    ("mbfitI" . #x1D470)
    ("mbfiti" . #x1D48A)
    ("mbfitIota" . #x1D724)
    ("mbfitiota" . #x1D73E)
    ("mbfitJ" . #x1D471)
    ("mbfitj" . #x1D48B)
    ("mbfitK" . #x1D472)
    ("mbfitk" . #x1D48C)
    ("mbfitKappa" . #x1D725)
    ("mbfitkappa" . #x1D73F)
    ("mbfitL" . #x1D473)
    ("mbfitl" . #x1D48D)
    ("mbfitLambda" . #x1D726)
    ("mbfitlambda" . #x1D740)
    ("mbfitM" . #x1D474)
    ("mbfitm" . #x1D48E)
    ("mbfitMu" . #x1D727)
    ("mbfitmu" . #x1D741)
    ("mbfitN" . #x1D475)
    ("mbfitn" . #x1D48F)
    ("mbfitnabla" . #x1D735)
    ("mbfitNu" . #x1D728)
    ("mbfitnu" . #x1D742)
    ("mbfitO" . #x1D476)
    ("mbfito" . #x1D490)
    ("mbfitOmega" . #x1D734)
    ("mbfitomega" . #x1D74E)
    ("mbfitOmicron" . #x1D72A)
    ("mbfitomicron" . #x1D744)
    ("mbfitP" . #x1D477)
    ("mbfitp" . #x1D491)
    ("mbfitpartial" . #x1D74F)
    ("mbfitPhi" . #x1D731)
    ("mbfitphi" . #x1D753)
    ("mbfitPi" . #x1D72B)
    ("mbfitpi" . #x1D745)
    ("mbfitPsi" . #x1D733)
    ("mbfitpsi" . #x1D74D)
    ("mbfitQ" . #x1D478)
    ("mbfitq" . #x1D492)
    ("mbfitR" . #x1D479)
    ("mbfitr" . #x1D493)
    ("mbfitRho" . #x1D72C)
    ("mbfitrho" . #x1D746)
    ("mbfitS" . #x1D47A)
    ("mbfits" . #x1D494)
    ("mbfitsansA" . #x1D63C)
    ("mbfitsansa" . #x1D656)
    ("mbfitsansAlpha" . #x1D790)
    ("mbfitsansalpha" . #x1D7AA)
    ("mbfitsansB" . #x1D63D)
    ("mbfitsansb" . #x1D657)
    ("mbfitsansBeta" . #x1D791)
    ("mbfitsansbeta" . #x1D7AB)
    ("mbfitsansC" . #x1D63E)
    ("mbfitsansc" . #x1D658)
    ("mbfitsansChi" . #x1D7A6)
    ("mbfitsanschi" . #x1D7C0)
    ("mbfitsansD" . #x1D63F)
    ("mbfitsansd" . #x1D659)
    ("mbfitsansDelta" . #x1D793)
    ("mbfitsansdelta" . #x1D7AD)
    ("mbfitsansE" . #x1D640)
    ("mbfitsanse" . #x1D65A)
    ("mbfitsansEpsilon" . #x1D794)
    ("mbfitsansepsilon" . #x1D7C4)
    ("mbfitsansEta" . #x1D796)
    ("mbfitsanseta" . #x1D7B0)
    ("mbfitsansF" . #x1D641)
    ("mbfitsansf" . #x1D65B)
    ("mbfitsansG" . #x1D642)
    ("mbfitsansg" . #x1D65C)
    ("mbfitsansGamma" . #x1D792)
    ("mbfitsansgamma" . #x1D7AC)
    ("mbfitsansH" . #x1D643)
    ("mbfitsansh" . #x1D65D)
    ("mbfitsansI" . #x1D644)
    ("mbfitsansi" . #x1D65E)
    ("mbfitsansIota" . #x1D798)
    ("mbfitsansiota" . #x1D7B2)
    ("mbfitsansJ" . #x1D645)
    ("mbfitsansj" . #x1D65F)
    ("mbfitsansK" . #x1D646)
    ("mbfitsansk" . #x1D660)
    ("mbfitsansKappa" . #x1D799)
    ("mbfitsanskappa" . #x1D7B3)
    ("mbfitsansL" . #x1D647)
    ("mbfitsansl" . #x1D661)
    ("mbfitsansLambda" . #x1D79A)
    ("mbfitsanslambda" . #x1D7B4)
    ("mbfitsansM" . #x1D648)
    ("mbfitsansm" . #x1D662)
    ("mbfitsansMu" . #x1D79B)
    ("mbfitsansmu" . #x1D7B5)
    ("mbfitsansN" . #x1D649)
    ("mbfitsansn" . #x1D663)
    ("mbfitsansnabla" . #x1D7A9)
    ("mbfitsansNu" . #x1D79C)
    ("mbfitsansnu" . #x1D7B6)
    ("mbfitsansO" . #x1D64A)
    ("mbfitsanso" . #x1D664)
    ("mbfitsansOmega" . #x1D7A8)
    ("mbfitsansomega" . #x1D7C2)
    ("mbfitsansOmicron" . #x1D79E)
    ("mbfitsansomicron" . #x1D7B8)
    ("mbfitsansP" . #x1D64B)
    ("mbfitsansp" . #x1D665)
    ("mbfitsanspartial" . #x1D7C3)
    ("mbfitsansPhi" . #x1D7A5)
    ("mbfitsansphi" . #x1D7C7)
    ("mbfitsansPi" . #x1D79F)
    ("mbfitsanspi" . #x1D7B9)
    ("mbfitsansPsi" . #x1D7A7)
    ("mbfitsanspsi" . #x1D7C1)
    ("mbfitsansQ" . #x1D64C)
    ("mbfitsansq" . #x1D666)
    ("mbfitsansR" . #x1D64D)
    ("mbfitsansr" . #x1D667)
    ("mbfitsansRho" . #x1D7A0)
    ("mbfitsansrho" . #x1D7BA)
    ("mbfitsansS" . #x1D64E)
    ("mbfitsanss" . #x1D668)
    ("mbfitsansSigma" . #x1D7A2)
    ("mbfitsanssigma" . #x1D7BC)
    ("mbfitsansT" . #x1D64F)
    ("mbfitsanst" . #x1D669)
    ("mbfitsansTau" . #x1D7A3)
    ("mbfitsanstau" . #x1D7BD)
    ("mbfitsansTheta" . #x1D797)
    ("mbfitsanstheta" . #x1D7B1)
    ("mbfitsansU" . #x1D650)
    ("mbfitsansu" . #x1D66A)
    ("mbfitsansUpsilon" . #x1D7A4)
    ("mbfitsansupsilon" . #x1D7BE)
    ("mbfitsansV" . #x1D651)
    ("mbfitsansv" . #x1D66B)
    ("mbfitsansvarepsilon" . #x1D7AE)
    ("mbfitsansvarkappa" . #x1D7C6)
    ("mbfitsansvarphi" . #x1D7BF)
    ("mbfitsansvarpi" . #x1D7C9)
    ("mbfitsansvarrho" . #x1D7C8)
    ("mbfitsansvarsigma" . #x1D7BB)
    ("mbfitsansvarTheta" . #x1D7A1)
    ("mbfitsansvartheta" . #x1D7C5)
    ("mbfitsansW" . #x1D652)
    ("mbfitsansw" . #x1D66C)
    ("mbfitsansX" . #x1D653)
    ("mbfitsansx" . #x1D66D)
    ("mbfitsansXi" . #x1D79D)
    ("mbfitsansxi" . #x1D7B7)
    ("mbfitsansY" . #x1D654)
    ("mbfitsansy" . #x1D66E)
    ("mbfitsansZ" . #x1D655)
    ("mbfitsansz" . #x1D66F)
    ("mbfitsansZeta" . #x1D795)
    ("mbfitsanszeta" . #x1D7AF)
    ("mbfitSigma" . #x1D72E)
    ("mbfitsigma" . #x1D748)
    ("mbfitT" . #x1D47B)
    ("mbfitt" . #x1D495)
    ("mbfitTau" . #x1D72F)
    ("mbfittau" . #x1D749)
    ("mbfitTheta" . #x1D723)
    ("mbfittheta" . #x1D73D)
    ("mbfitU" . #x1D47C)
    ("mbfitu" . #x1D496)
    ("mbfitUpsilon" . #x1D730)
    ("mbfitupsilon" . #x1D74A)
    ("mbfitV" . #x1D47D)
    ("mbfitv" . #x1D497)
    ("mbfitvarepsilon" . #x1D73A)
    ("mbfitvarkappa" . #x1D752)
    ("mbfitvarphi" . #x1D74B)
    ("mbfitvarpi" . #x1D755)
    ("mbfitvarrho" . #x1D754)
    ("mbfitvarsigma" . #x1D747)
    ("mbfitvarTheta" . #x1D72D)
    ("mbfitvartheta" . #x1D751)
    ("mbfitW" . #x1D47E)
    ("mbfitw" . #x1D498)
    ("mbfitX" . #x1D47F)
    ("mbfitx" . #x1D499)
    ("mbfitXi" . #x1D729)
    ("mbfitxi" . #x1D743)
    ("mbfitY" . #x1D480)
    ("mbfity" . #x1D49A)
    ("mbfitZ" . #x1D481)
    ("mbfitz" . #x1D49B)
    ("mbfitZeta" . #x1D721)
    ("mbfitzeta" . #x1D73B)
    ("mbfJ" . #x1D409)
    ("mbfj" . #x1D423)
    ("mbfK" . #x1D40A)
    ("mbfk" . #x1D424)
    ("mbfKappa" . #x1D6B1)
    ("mbfkappa" . #x1D6CB)
    ("mbfL" . #x1D40B)
    ("mbfl" . #x1D425)
    ("mbfLambda" . #x1D6B2)
    ("mbflambda" . #x1D6CC)
    ("mbfM" . #x1D40C)
    ("mbfm" . #x1D426)
    ("mbfMu" . #x1D6B3)
    ("mbfmu" . #x1D6CD)
    ("mbfN" . #x1D40D)
    ("mbfn" . #x1D427)
    ("mbfnabla" . #x1D6C1)
    ("mbfnine" . #x1D7D7)
    ("mbfNu" . #x1D6B4)
    ("mbfnu" . #x1D6CE)
    ("mbfO" . #x1D40E)
    ("mbfo" . #x1D428)
    ("mbfOmega" . #x1D6C0)
    ("mbfomega" . #x1D6DA)
    ("mbfOmicron" . #x1D6B6)
    ("mbfomicron" . #x1D6D0)
    ("mbfone" . #x1D7CF)
    ("mbfP" . #x1D40F)
    ("mbfp" . #x1D429)
    ("mbfpartial" . #x1D6DB)
    ("mbfPhi" . #x1D6BD)
    ("mbfphi" . #x1D6DF)
    ("mbfPi" . #x1D6B7)
    ("mbfpi" . #x1D6D1)
    ("mbfPsi" . #x1D6BF)
    ("mbfpsi" . #x1D6D9)
    ("mbfQ" . #x1D410)
    ("mbfq" . #x1D42A)
    ("mbfR" . #x1D411)
    ("mbfr" . #x1D42B)
    ("mbfRho" . #x1D6B8)
    ("mbfrho" . #x1D6D2)
    ("mbfS" . #x1D412)
    ("mbfs" . #x1D42C)
    ("mbfsansA" . #x1D5D4)
    ("mbfsansa" . #x1D5EE)
    ("mbfsansAlpha" . #x1D756)
    ("mbfsansalpha" . #x1D770)
    ("mbfsansB" . #x1D5D5)
    ("mbfsansb" . #x1D5EF)
    ("mbfsansBeta" . #x1D757)
    ("mbfsansbeta" . #x1D771)
    ("mbfsansC" . #x1D5D6)
    ("mbfsansc" . #x1D5F0)
    ("mbfsansChi" . #x1D76C)
    ("mbfsanschi" . #x1D786)
    ("mbfsansD" . #x1D5D7)
    ("mbfsansd" . #x1D5F1)
    ("mbfsansDelta" . #x1D759)
    ("mbfsansdelta" . #x1D773)
    ("mbfsansE" . #x1D5D8)
    ("mbfsanse" . #x1D5F2)
    ("mbfsanseight" . #x1D7F4)
    ("mbfsansEpsilon" . #x1D75A)
    ("mbfsansepsilon" . #x1D78A)
    ("mbfsansEta" . #x1D75C)
    ("mbfsanseta" . #x1D776)
    ("mbfsansF" . #x1D5D9)
    ("mbfsansf" . #x1D5F3)
    ("mbfsansfive" . #x1D7F1)
    ("mbfsansfour" . #x1D7F0)
    ("mbfsansG" . #x1D5DA)
    ("mbfsansg" . #x1D5F4)
    ("mbfsansGamma" . #x1D758)
    ("mbfsansgamma" . #x1D772)
    ("mbfsansH" . #x1D5DB)
    ("mbfsansh" . #x1D5F5)
    ("mbfsansI" . #x1D5DC)
    ("mbfsansi" . #x1D5F6)
    ("mbfsansIota" . #x1D75E)
    ("mbfsansiota" . #x1D778)
    ("mbfsansJ" . #x1D5DD)
    ("mbfsansj" . #x1D5F7)
    ("mbfsansK" . #x1D5DE)
    ("mbfsansk" . #x1D5F8)
    ("mbfsansKappa" . #x1D75F)
    ("mbfsanskappa" . #x1D779)
    ("mbfsansL" . #x1D5DF)
    ("mbfsansl" . #x1D5F9)
    ("mbfsansLambda" . #x1D760)
    ("mbfsanslambda" . #x1D77A)
    ("mbfsansM" . #x1D5E0)
    ("mbfsansm" . #x1D5FA)
    ("mbfsansMu" . #x1D761)
    ("mbfsansmu" . #x1D77B)
    ("mbfsansN" . #x1D5E1)
    ("mbfsansn" . #x1D5FB)
    ("mbfsansnabla" . #x1D76F)
    ("mbfsansnine" . #x1D7F5)
    ("mbfsansNu" . #x1D762)
    ("mbfsansnu" . #x1D77C)
    ("mbfsansO" . #x1D5E2)
    ("mbfsanso" . #x1D5FC)
    ("mbfsansOmega" . #x1D76E)
    ("mbfsansomega" . #x1D788)
    ("mbfsansOmicron" . #x1D764)
    ("mbfsansomicron" . #x1D77E)
    ("mbfsansone" . #x1D7ED)
    ("mbfsansP" . #x1D5E3)
    ("mbfsansp" . #x1D5FD)
    ("mbfsanspartial" . #x1D789)
    ("mbfsansPhi" . #x1D76B)
    ("mbfsansphi" . #x1D78D)
    ("mbfsansPi" . #x1D765)
    ("mbfsanspi" . #x1D77F)
    ("mbfsansPsi" . #x1D76D)
    ("mbfsanspsi" . #x1D787)
    ("mbfsansQ" . #x1D5E4)
    ("mbfsansq" . #x1D5FE)
    ("mbfsansR" . #x1D5E5)
    ("mbfsansr" . #x1D5FF)
    ("mbfsansRho" . #x1D766)
    ("mbfsansrho" . #x1D780)
    ("mbfsansS" . #x1D5E6)
    ("mbfsanss" . #x1D600)
    ("mbfsansseven" . #x1D7F3)
    ("mbfsansSigma" . #x1D768)
    ("mbfsanssigma" . #x1D782)
    ("mbfsanssix" . #x1D7F2)
    ("mbfsansT" . #x1D5E7)
    ("mbfsanst" . #x1D601)
    ("mbfsansTau" . #x1D769)
    ("mbfsanstau" . #x1D783)
    ("mbfsansTheta" . #x1D75D)
    ("mbfsanstheta" . #x1D777)
    ("mbfsansthree" . #x1D7EF)
    ("mbfsanstwo" . #x1D7EE)
    ("mbfsansU" . #x1D5E8)
    ("mbfsansu" . #x1D602)
    ("mbfsansUpsilon" . #x1D76A)
    ("mbfsansupsilon" . #x1D784)
    ("mbfsansV" . #x1D5E9)
    ("mbfsansv" . #x1D603)
    ("mbfsansvarepsilon" . #x1D774)
    ("mbfsansvarkappa" . #x1D78C)
    ("mbfsansvarphi" . #x1D785)
    ("mbfsansvarpi" . #x1D78F)
    ("mbfsansvarrho" . #x1D78E)
    ("mbfsansvarsigma" . #x1D781)
    ("mbfsansvarTheta" . #x1D767)
    ("mbfsansvartheta" . #x1D78B)
    ("mbfsansW" . #x1D5EA)
    ("mbfsansw" . #x1D604)
    ("mbfsansX" . #x1D5EB)
    ("mbfsansx" . #x1D605)
    ("mbfsansXi" . #x1D763)
    ("mbfsansxi" . #x1D77D)
    ("mbfsansY" . #x1D5EC)
    ("mbfsansy" . #x1D606)
    ("mbfsansZ" . #x1D5ED)
    ("mbfsansz" . #x1D607)
    ("mbfsanszero" . #x1D7EC)
    ("mbfsansZeta" . #x1D75B)
    ("mbfsanszeta" . #x1D775)
    ("mbfscrA" . #x1D4D0)
    ("mbfscra" . #x1D4EA)
    ("mbfscrB" . #x1D4D1)
    ("mbfscrb" . #x1D4EB)
    ("mbfscrC" . #x1D4D2)
    ("mbfscrc" . #x1D4EC)
    ("mbfscrD" . #x1D4D3)
    ("mbfscrd" . #x1D4ED)
    ("mbfscrE" . #x1D4D4)
    ("mbfscre" . #x1D4EE)
    ("mbfscrF" . #x1D4D5)
    ("mbfscrf" . #x1D4EF)
    ("mbfscrG" . #x1D4D6)
    ("mbfscrg" . #x1D4F0)
    ("mbfscrH" . #x1D4D7)
    ("mbfscrh" . #x1D4F1)
    ("mbfscrI" . #x1D4D8)
    ("mbfscri" . #x1D4F2)
    ("mbfscrJ" . #x1D4D9)
    ("mbfscrj" . #x1D4F3)
    ("mbfscrK" . #x1D4DA)
    ("mbfscrk" . #x1D4F4)
    ("mbfscrL" . #x1D4DB)
    ("mbfscrl" . #x1D4F5)
    ("mbfscrM" . #x1D4DC)
    ("mbfscrm" . #x1D4F6)
    ("mbfscrN" . #x1D4DD)
    ("mbfscrn" . #x1D4F7)
    ("mbfscrO" . #x1D4DE)
    ("mbfscro" . #x1D4F8)
    ("mbfscrP" . #x1D4DF)
    ("mbfscrp" . #x1D4F9)
    ("mbfscrQ" . #x1D4E0)
    ("mbfscrq" . #x1D4FA)
    ("mbfscrR" . #x1D4E1)
    ("mbfscrr" . #x1D4FB)
    ("mbfscrS" . #x1D4E2)
    ("mbfscrs" . #x1D4FC)
    ("mbfscrT" . #x1D4E3)
    ("mbfscrt" . #x1D4FD)
    ("mbfscrU" . #x1D4E4)
    ("mbfscru" . #x1D4FE)
    ("mbfscrV" . #x1D4E5)
    ("mbfscrv" . #x1D4FF)
    ("mbfscrW" . #x1D4E6)
    ("mbfscrw" . #x1D500)
    ("mbfscrX" . #x1D4E7)
    ("mbfscrx" . #x1D501)
    ("mbfscrY" . #x1D4E8)
    ("mbfscry" . #x1D502)
    ("mbfscrZ" . #x1D4E9)
    ("mbfscrz" . #x1D503)
    ("mbfseven" . #x1D7D5)
    ("mbfSigma" . #x1D6BA)
    ("mbfsigma" . #x1D6D4)
    ("mbfsix" . #x1D7D4)
    ("mbfT" . #x1D413)
    ("mbft" . #x1D42D)
    ("mbfTau" . #x1D6BB)
    ("mbftau" . #x1D6D5)
    ("mbfTheta" . #x1D6AF)
    ("mbftheta" . #x1D6C9)
    ("mbfthree" . #x1D7D1)
    ("mbftwo" . #x1D7D0)
    ("mbfU" . #x1D414)
    ("mbfu" . #x1D42E)
    ("mbfUpsilon" . #x1D6BC)
    ("mbfupsilon" . #x1D6D6)
    ("mbfV" . #x1D415)
    ("mbfv" . #x1D42F)
    ("mbfvarepsilon" . #x1D6C6)
    ("mbfvarkappa" . #x1D6DE)
    ("mbfvarphi" . #x1D6D7)
    ("mbfvarpi" . #x1D6E1)
    ("mbfvarrho" . #x1D6E0)
    ("mbfvarsigma" . #x1D6D3)
    ("mbfvarTheta" . #x1D6B9)
    ("mbfvartheta" . #x1D6DD)
    ("mbfW" . #x1D416)
    ("mbfw" . #x1D430)
    ("mbfX" . #x1D417)
    ("mbfx" . #x1D431)
    ("mbfXi" . #x1D6B5)
    ("mbfxi" . #x1D6CF)
    ("mbfY" . #x1D418)
    ("mbfy" . #x1D432)
    ("mbfZ" . #x1D419)
    ("mbfz" . #x1D433)
    ("mbfzero" . #x1D7CE)
    ("mbfZeta" . #x1D6AD)
    ("mbfzeta" . #x1D6C7)
    ("mdblkcircle" . #x026AB)
    ("mdblkdiamond" . #x02B25)
    ("mdblklozenge" . #x02B27)
    ("mdblksquare" . #x025FC)
    ("mdlgblkcircle" . #x025CF)
    ("mdlgblkdiamond" . #x025C6)
    ("mdlgblklozenge" . #x029EB)
    ("mdlgblksquare" . #x025A0)
    ("mdlgwhtcircle" . #x025CB)
    ("mdlgwhtdiamond" . #x025C7)
    ("mdlgwhtlozenge" . #x025CA)
    ("mdlgwhtsquare" . #x025A1)
    ("mdsmblkcircle" . #x02981)
    ("mdsmblksquare" . #x025FE)
    ("mdsmwhtcircle" . #x026AC)
    ("mdsmwhtsquare" . #x025FD)
    ("mdwhtcircle" . #x026AA)
    ("mdwhtdiamond" . #x02B26)
    ("mdwhtlozenge" . #x02B28)
    ("mdwhtsquare" . #x025FB)
    ("measangledltosw" . #x029AF)
    ("measangledrtose" . #x029AE)
    ("measangleldtosw" . #x029AB)
    ("measanglelutonw" . #x029A9)
    ("measanglerdtose" . #x029AA)
    ("measanglerutone" . #x029A8)
    ("measangleultonw" . #x029AD)
    ("measangleurtone" . #x029AC)
    ("measeq" . #x0225E)
    ("measuredangle" . #x02221)
    ("measuredangleleft" . #x0299B)
    ("measuredrightangle" . #x022BE)
    ("medblackstar" . #x02B51)
    ("medwhitestar" . #x02B50)
    ("mfrakA" . #x1D504)
    ("mfraka" . #x1D51E)
    ("mfrakB" . #x1D505)
    ("mfrakb" . #x1D51F)
    ("mfrakC" . #x0212D)
    ("mfrakc" . #x1D520)
    ("mfrakD" . #x1D507)
    ("mfrakd" . #x1D521)
    ("mfrakE" . #x1D508)
    ("mfrake" . #x1D522)
    ("mfrakF" . #x1D509)
    ("mfrakf" . #x1D523)
    ("mfrakG" . #x1D50A)
    ("mfrakg" . #x1D524)
    ("mfrakH" . #x0210C)
    ("mfrakh" . #x1D525)
    ("mfraki" . #x1D526)
    ("mfrakJ" . #x1D50D)
    ("mfrakj" . #x1D527)
    ("mfrakK" . #x1D50E)
    ("mfrakk" . #x1D528)
    ("mfrakL" . #x1D50F)
    ("mfrakl" . #x1D529)
    ("mfrakM" . #x1D510)
    ("mfrakm" . #x1D52A)
    ("mfrakN" . #x1D511)
    ("mfrakn" . #x1D52B)
    ("mfrakO" . #x1D512)
    ("mfrako" . #x1D52C)
    ("mfrakP" . #x1D513)
    ("mfrakp" . #x1D52D)
    ("mfrakQ" . #x1D514)
    ("mfrakq" . #x1D52E)
    ("mfrakr" . #x1D52F)
    ("mfrakS" . #x1D516)
    ("mfraks" . #x1D530)
    ("mfrakT" . #x1D517)
    ("mfrakt" . #x1D531)
    ("mfrakU" . #x1D518)
    ("mfraku" . #x1D532)
    ("mfrakV" . #x1D519)
    ("mfrakv" . #x1D533)
    ("mfrakW" . #x1D51A)
    ("mfrakw" . #x1D534)
    ("mfrakX" . #x1D51B)
    ("mfrakx" . #x1D535)
    ("mfrakY" . #x1D51C)
    ("mfraky" . #x1D536)
    ("mfrakZ" . #x02128)
    ("mfrakz" . #x1D537)
    ("mho" . #x02127)
    ("mid" . #x02223)
    ("midbarvee" . #x02A5D)
    ("midbarwedge" . #x02A5C)
    ("midcir" . #x02AF0)
    ("minus" . #x02212)
    ("minusdot" . #x02A2A)
    ("minusfdots" . #x02A2B)
    ("minusrdots" . #x02A2C)
    ("mitA" . #x1D434)
    ("mita" . #x1D44E)
    ("mitAlpha" . #x1D6E2)
    ("mitalpha" . #x1D6FC)
    ("mitB" . #x1D435)
    ("mitb" . #x1D44F)
    ("mitBbbD" . #x02145)
    ("mitBbbd" . #x02146)
    ("mitBbbe" . #x02147)
    ("mitBbbi" . #x02148)
    ("mitBbbj" . #x02149)
    ("mitBeta" . #x1D6E3)
    ("mitbeta" . #x1D6FD)
    ("mitC" . #x1D436)
    ("mitc" . #x1D450)
    ("mitChi" . #x1D6F8)
    ("mitchi" . #x1D712)
    ("mitD" . #x1D437)
    ("mitd" . #x1D451)
    ("mitDelta" . #x1D6E5)
    ("mitdelta" . #x1D6FF)
    ("mitE" . #x1D438)
    ("mite" . #x1D452)
    ("mitEpsilon" . #x1D6E6)
    ("mitepsilon" . #x1D716)
    ("mitEta" . #x1D6E8)
    ("miteta" . #x1D702)
    ("mitF" . #x1D439)
    ("mitf" . #x1D453)
    ("mitG" . #x1D43A)
    ("mitg" . #x1D454)
    ("mitGamma" . #x1D6E4)
    ("mitgamma" . #x1D6FE)
    ("mitH" . #x1D43B)
    ("mitI" . #x1D43C)
    ("miti" . #x1D456)
    ("mitIota" . #x1D6EA)
    ("mitiota" . #x1D704)
    ("mitJ" . #x1D43D)
    ("mitj" . #x1D457)
    ("mitK" . #x1D43E)
    ("mitk" . #x1D458)
    ("mitKappa" . #x1D6EB)
    ("mitkappa" . #x1D705)
    ("mitL" . #x1D43F)
    ("mitl" . #x1D459)
    ("mitLambda" . #x1D6EC)
    ("mitlambda" . #x1D706)
    ("mitM" . #x1D440)
    ("mitm" . #x1D45A)
    ("mitMu" . #x1D6ED)
    ("mitmu" . #x1D707)
    ("mitN" . #x1D441)
    ("mitn" . #x1D45B)
    ("mitnabla" . #x1D6FB)
    ("mitNu" . #x1D6EE)
    ("mitnu" . #x1D708)
    ("mitO" . #x1D442)
    ("mito" . #x1D45C)
    ("mitOmega" . #x1D6FA)
    ("mitomega" . #x1D714)
    ("mitOmicron" . #x1D6F0)
    ("mitomicron" . #x1D70A)
    ("mitP" . #x1D443)
    ("mitp" . #x1D45D)
    ("mitpartial" . #x1D715)
    ("mitPhi" . #x1D6F7)
    ("mitphi" . #x1D719)
    ("mitPi" . #x1D6F1)
    ("mitpi" . #x1D70B)
    ("mitPsi" . #x1D6F9)
    ("mitpsi" . #x1D713)
    ("mitQ" . #x1D444)
    ("mitq" . #x1D45E)
    ("mitR" . #x1D445)
    ("mitr" . #x1D45F)
    ("mitRho" . #x1D6F2)
    ("mitrho" . #x1D70C)
    ("mitS" . #x1D446)
    ("mits" . #x1D460)
    ("mitsansA" . #x1D608)
    ("mitsansa" . #x1D622)
    ("mitsansB" . #x1D609)
    ("mitsansb" . #x1D623)
    ("mitsansC" . #x1D60A)
    ("mitsansc" . #x1D624)
    ("mitsansD" . #x1D60B)
    ("mitsansd" . #x1D625)
    ("mitsansE" . #x1D60C)
    ("mitsanse" . #x1D626)
    ("mitsansF" . #x1D60D)
    ("mitsansf" . #x1D627)
    ("mitsansG" . #x1D60E)
    ("mitsansg" . #x1D628)
    ("mitsansH" . #x1D60F)
    ("mitsansh" . #x1D629)
    ("mitsansI" . #x1D610)
    ("mitsansi" . #x1D62A)
    ("mitsansJ" . #x1D611)
    ("mitsansj" . #x1D62B)
    ("mitsansK" . #x1D612)
    ("mitsansk" . #x1D62C)
    ("mitsansL" . #x1D613)
    ("mitsansl" . #x1D62D)
    ("mitsansM" . #x1D614)
    ("mitsansm" . #x1D62E)
    ("mitsansN" . #x1D615)
    ("mitsansn" . #x1D62F)
    ("mitsansO" . #x1D616)
    ("mitsanso" . #x1D630)
    ("mitsansP" . #x1D617)
    ("mitsansp" . #x1D631)
    ("mitsansQ" . #x1D618)
    ("mitsansq" . #x1D632)
    ("mitsansR" . #x1D619)
    ("mitsansr" . #x1D633)
    ("mitsansS" . #x1D61A)
    ("mitsanss" . #x1D634)
    ("mitsansT" . #x1D61B)
    ("mitsanst" . #x1D635)
    ("mitsansU" . #x1D61C)
    ("mitsansu" . #x1D636)
    ("mitsansV" . #x1D61D)
    ("mitsansv" . #x1D637)
    ("mitsansW" . #x1D61E)
    ("mitsansw" . #x1D638)
    ("mitsansX" . #x1D61F)
    ("mitsansx" . #x1D639)
    ("mitsansY" . #x1D620)
    ("mitsansy" . #x1D63A)
    ("mitsansZ" . #x1D621)
    ("mitsansz" . #x1D63B)
    ("mitSigma" . #x1D6F4)
    ("mitsigma" . #x1D70E)
    ("mitT" . #x1D447)
    ("mitt" . #x1D461)
    ("mitTau" . #x1D6F5)
    ("mittau" . #x1D70F)
    ("mitTheta" . #x1D6E9)
    ("mittheta" . #x1D703)
    ("mitU" . #x1D448)
    ("mitu" . #x1D462)
    ("mitUpsilon" . #x1D6F6)
    ("mitupsilon" . #x1D710)
    ("mitV" . #x1D449)
    ("mitv" . #x1D463)
    ("mitvarepsilon" . #x1D700)
    ("mitvarkappa" . #x1D718)
    ("mitvarphi" . #x1D711)
    ("mitvarpi" . #x1D71B)
    ("mitvarrho" . #x1D71A)
    ("mitvarsigma" . #x1D70D)
    ("mitvarTheta" . #x1D6F3)
    ("mitvartheta" . #x1D717)
    ("mitW" . #x1D44A)
    ("mitw" . #x1D464)
    ("mitX" . #x1D44B)
    ("mitx" . #x1D465)
    ("mitXi" . #x1D6EF)
    ("mitxi" . #x1D709)
    ("mitY" . #x1D44C)
    ("mity" . #x1D466)
    ("mitZ" . #x1D44D)
    ("mitz" . #x1D467)
    ("mitZeta" . #x1D6E7)
    ("mitzeta" . #x1D701)
    ("mlcp" . #x02ADB)
    ("models" . #x022A7)
    ("modtwosum" . #x02A0A)
    ("mp" . #x02213)
    ("msansA" . #x1D5A0)
    ("msansa" . #x1D5BA)
    ("msansB" . #x1D5A1)
    ("msansb" . #x1D5BB)
    ("msansC" . #x1D5A2)
    ("msansc" . #x1D5BC)
    ("msansD" . #x1D5A3)
    ("msansd" . #x1D5BD)
    ("msansE" . #x1D5A4)
    ("msanse" . #x1D5BE)
    ("msanseight" . #x1D7EA)
    ("msansF" . #x1D5A5)
    ("msansf" . #x1D5BF)
    ("msansfive" . #x1D7E7)
    ("msansfour" . #x1D7E6)
    ("msansG" . #x1D5A6)
    ("msansg" . #x1D5C0)
    ("msansH" . #x1D5A7)
    ("msansh" . #x1D5C1)
    ("msansI" . #x1D5A8)
    ("msansi" . #x1D5C2)
    ("msansJ" . #x1D5A9)
    ("msansj" . #x1D5C3)
    ("msansK" . #x1D5AA)
    ("msansk" . #x1D5C4)
    ("msansL" . #x1D5AB)
    ("msansl" . #x1D5C5)
    ("msansM" . #x1D5AC)
    ("msansm" . #x1D5C6)
    ("msansN" . #x1D5AD)
    ("msansn" . #x1D5C7)
    ("msansnine" . #x1D7EB)
    ("msansO" . #x1D5AE)
    ("msanso" . #x1D5C8)
    ("msansone" . #x1D7E3)
    ("msansP" . #x1D5AF)
    ("msansp" . #x1D5C9)
    ("msansQ" . #x1D5B0)
    ("msansq" . #x1D5CA)
    ("msansR" . #x1D5B1)
    ("msansr" . #x1D5CB)
    ("msansS" . #x1D5B2)
    ("msanss" . #x1D5CC)
    ("msansseven" . #x1D7E9)
    ("msanssix" . #x1D7E8)
    ("msansT" . #x1D5B3)
    ("msanst" . #x1D5CD)
    ("msansthree" . #x1D7E5)
    ("msanstwo" . #x1D7E4)
    ("msansU" . #x1D5B4)
    ("msansu" . #x1D5CE)
    ("msansV" . #x1D5B5)
    ("msansv" . #x1D5CF)
    ("msansW" . #x1D5B6)
    ("msansw" . #x1D5D0)
    ("msansX" . #x1D5B7)
    ("msansx" . #x1D5D1)
    ("msansY" . #x1D5B8)
    ("msansy" . #x1D5D2)
    ("msansZ" . #x1D5B9)
    ("msansz" . #x1D5D3)
    ("msanszero" . #x1D7E2)
    ("mscrA" . #x1D49C)
    ("mscra" . #x1D4B6)
    ("mscrB" . #x0212C)
    ("mscrb" . #x1D4B7)
    ("mscrC" . #x1D49E)
    ("mscrc" . #x1D4B8)
    ("mscrD" . #x1D49F)
    ("mscrd" . #x1D4B9)
    ("mscre" . #x0212F)
    ("mscrE" . #x02130)
    ("mscrF" . #x02131)
    ("mscrf" . #x1D4BB)
    ("mscrg" . #x0210A)
    ("mscrG" . #x1D4A2)
    ("mscrH" . #x0210B)
    ("mscrh" . #x1D4BD)
    ("mscrI" . #x02110)
    ("mscri" . #x1D4BE)
    ("mscrJ" . #x1D4A5)
    ("mscrj" . #x1D4BF)
    ("mscrK" . #x1D4A6)
    ("mscrk" . #x1D4C0)
    ("mscrL" . #x02112)
    ("mscrl" . #x1D4C1)
    ("mscrM" . #x02133)
    ("mscrm" . #x1D4C2)
    ("mscrN" . #x1D4A9)
    ("mscrn" . #x1D4C3)
    ("mscro" . #x02134)
    ("mscrO" . #x1D4AA)
    ("mscrP" . #x1D4AB)
    ("mscrp" . #x1D4C5)
    ("mscrQ" . #x1D4AC)
    ("mscrq" . #x1D4C6)
    ("mscrR" . #x0211B)
    ("mscrr" . #x1D4C7)
    ("mscrS" . #x1D4AE)
    ("mscrs" . #x1D4C8)
    ("mscrT" . #x1D4AF)
    ("mscrt" . #x1D4C9)
    ("mscrU" . #x1D4B0)
    ("mscru" . #x1D4CA)
    ("mscrV" . #x1D4B1)
    ("mscrv" . #x1D4CB)
    ("mscrW" . #x1D4B2)
    ("mscrw" . #x1D4CC)
    ("mscrX" . #x1D4B3)
    ("mscrx" . #x1D4CD)
    ("mscrY" . #x1D4B4)
    ("mscry" . #x1D4CE)
    ("mscrZ" . #x1D4B5)
    ("mscrz" . #x1D4CF)
    ("mttA" . #x1D670)
    ("mtta" . #x1D68A)
    ("mttB" . #x1D671)
    ("mttb" . #x1D68B)
    ("mttC" . #x1D672)
    ("mttc" . #x1D68C)
    ("mttD" . #x1D673)
    ("mttd" . #x1D68D)
    ("mttE" . #x1D674)
    ("mtte" . #x1D68E)
    ("mtteight" . #x1D7FE)
    ("mttF" . #x1D675)
    ("mttf" . #x1D68F)
    ("mttfive" . #x1D7FB)
    ("mttfour" . #x1D7FA)
    ("mttG" . #x1D676)
    ("mttg" . #x1D690)
    ("mttH" . #x1D677)
    ("mtth" . #x1D691)
    ("mttI" . #x1D678)
    ("mtti" . #x1D692)
    ("mttJ" . #x1D679)
    ("mttj" . #x1D693)
    ("mttK" . #x1D67A)
    ("mttk" . #x1D694)
    ("mttL" . #x1D67B)
    ("mttl" . #x1D695)
    ("mttM" . #x1D67C)
    ("mttm" . #x1D696)
    ("mttN" . #x1D67D)
    ("mttn" . #x1D697)
    ("mttnine" . #x1D7FF)
    ("mttO" . #x1D67E)
    ("mtto" . #x1D698)
    ("mttone" . #x1D7F7)
    ("mttP" . #x1D67F)
    ("mttp" . #x1D699)
    ("mttQ" . #x1D680)
    ("mttq" . #x1D69A)
    ("mttR" . #x1D681)
    ("mttr" . #x1D69B)
    ("mttS" . #x1D682)
    ("mtts" . #x1D69C)
    ("mttseven" . #x1D7FD)
    ("mttsix" . #x1D7FC)
    ("mttT" . #x1D683)
    ("mttt" . #x1D69D)
    ("mttthree" . #x1D7F9)
    ("mtttwo" . #x1D7F8)
    ("mttU" . #x1D684)
    ("mttu" . #x1D69E)
    ("mttV" . #x1D685)
    ("mttv" . #x1D69F)
    ("mttW" . #x1D686)
    ("mttw" . #x1D6A0)
    ("mttX" . #x1D687)
    ("mttx" . #x1D6A1)
    ("mttY" . #x1D688)
    ("mtty" . #x1D6A2)
    ("mttZ" . #x1D689)
    ("mttz" . #x1D6A3)
    ("mttzero" . #x1D7F6)
    ("Mu" . #x0039C)
    ("mu" . #x003BC)
    ("multimap" . #x022B8)
    ("multimapinv" . #x027DC)
    ("mupAlpha" . #x00391)
    ("mupalpha" . #x003B1)
    ("mupBeta" . #x00392)
    ("mupbeta" . #x003B2)
    ("mupChi" . #x003A7)
    ("mupchi" . #x003C7)
    ("mupDelta" . #x00394)
    ("mupdelta" . #x003B4)
    ("mupEpsilon" . #x00395)
    ("mupepsilon" . #x003F5)
    ("mupEta" . #x00397)
    ("mupeta" . #x003B7)
    ("mupGamma" . #x00393)
    ("mupgamma" . #x003B3)
    ("mupIota" . #x00399)
    ("mupiota" . #x003B9)
    ("mupKappa" . #x0039A)
    ("mupkappa" . #x003BA)
    ("mupLambda" . #x0039B)
    ("muplambda" . #x003BB)
    ("mupMu" . #x0039C)
    ("mupmu" . #x003BC)
    ("mupNu" . #x0039D)
    ("mupnu" . #x003BD)
    ("mupOmega" . #x003A9)
    ("mupomega" . #x003C9)
    ("mupOmicron" . #x0039F)
    ("mupomicron" . #x003BF)
    ("mupPhi" . #x003A6)
    ("mupphi" . #x003D5)
    ("mupPi" . #x003A0)
    ("muppi" . #x003C0)
    ("mupPsi" . #x003A8)
    ("muppsi" . #x003C8)
    ("mupRho" . #x003A1)
    ("muprho" . #x003C1)
    ("mupSigma" . #x003A3)
    ("mupsigma" . #x003C3)
    ("mupTau" . #x003A4)
    ("muptau" . #x003C4)
    ("mupTheta" . #x00398)
    ("muptheta" . #x003B8)
    ("mupUpsilon" . #x003A5)
    ("mupupsilon" . #x003C5)
    ("mupvarepsilon" . #x003B5)
    ("mupvarkappa" . #x003F0)
    ("mupvarphi" . #x003C6)
    ("mupvarpi" . #x003D6)
    ("mupvarrho" . #x003F1)
    ("mupvarsigma" . #x003C2)
    ("mupvartheta" . #x003D1)
    ("mupvarTheta" . #x003F4)
    ("mupXi" . #x0039E)
    ("mupxi" . #x003BE)
    ("mupZeta" . #x00396)
    ("mupzeta" . #x003B6)
    ("nabla" . #x02207)
    ("napprox" . #x02249)
    ("nasymp" . #x0226D)
    ("natural" . #x0266E)
    ("ncong" . #x02247)
    ("ne" . #x02260)
    ("nearrow" . #x02197)
    ("Nearrow" . #x021D7)
    ("neg" . #x000AC)
    ("neovnwarrow" . #x02931)
    ("neovsearrow" . #x0292E)
    ("neq" . #x02260) ; plain.tex alias
    ("nequiv" . #x02262)
    ("neswarrow" . #x02922)
    ("neuter" . #x026B2)
    ("nexists" . #x02204)
    ("ngeq" . #x02271)
    ("ngtr" . #x0226F)
    ("ngtrless" . #x02279)
    ("ngtrsim" . #x02275)
    ("nHdownarrow" . #x021DF)
    ("nhpar" . #x02AF2)
    ("nHuparrow" . #x021DE)
    ("nhVvert" . #x02AF5)
    ("ni" . #x0220B)
    ("niobar" . #x022FE)
    ("nis" . #x022FC)
    ("nisd" . #x022FA)
    ("nleftarrow" . #x0219A)
    ("nLeftarrow" . #x021CD)
    ("nleftrightarrow" . #x021AE)
    ("nLeftrightarrow" . #x021CE)
    ("nleq" . #x02270)
    ("nless" . #x0226E)
    ("nlessgtr" . #x02278)
    ("nlesssim" . #x02274)
    ("nmid" . #x02224)
    ("nni" . #x0220C)
    ("Not" . #x02AEC)
    ("notaccent" . #x00338)
    ("notin" . #x02209)
    ("nparallel" . #x02226)
    ("npolint" . #x02A14)
    ("nprec" . #x02280)
    ("npreccurlyeq" . #x022E0)
    ("nrightarrow" . #x0219B)
    ("nRightarrow" . #x021CF)
    ("nsim" . #x02241)
    ("nsime" . #x02244)
    ("nsimeq" . #x02244)
    ("nsqsubseteq" . #x022E2)
    ("nsqsupseteq" . #x022E3)
    ("nsubset" . #x02284)
    ("nsubseteq" . #x02288)
    ("nsucc" . #x02281)
    ("nsucccurlyeq" . #x022E1)
    ("nsupset" . #x02285)
    ("nsupseteq" . #x02289)
    ("ntrianglelefteq" . #x022EC)
    ("ntrianglerighteq" . #x022ED)
    ("Nu" . #x0039D)
    ("nu" . #x003BD)
    ("nvartriangleleft" . #x022EA)
    ("nvartriangleright" . #x022EB)
    ("nvdash" . #x022AC)
    ("nvDash" . #x022AD)
    ("nVdash" . #x022AE)
    ("nVDash" . #x022AF)
    ("nvinfty" . #x029DE)
    ("nvleftarrow" . #x021F7)
    ("nVleftarrow" . #x021FA)
    ("nvLeftarrow" . #x02902)
    ("nvleftarrowtail" . #x02B39)
    ("nVleftarrowtail" . #x02B3A)
    ("nvleftrightarrow" . #x021F9)
    ("nVleftrightarrow" . #x021FC)
    ("nvLeftrightarrow" . #x02904)
    ("nvrightarrow" . #x021F8)
    ("nVrightarrow" . #x021FB)
    ("nvRightarrow" . #x02903)
    ("nvrightarrowtail" . #x02914)
    ("nVrightarrowtail" . #x02915)
    ("nvtwoheadleftarrow" . #x02B34)
    ("nVtwoheadleftarrow" . #x02B35)
    ("nvtwoheadleftarrowtail" . #x02B3C)
    ("nVtwoheadleftarrowtail" . #x02B3D)
    ("nvtwoheadrightarrow" . #x02900)
    ("nVtwoheadrightarrow" . #x02901)
    ("nvtwoheadrightarrowtail" . #x02917)
    ("nVtwoheadrightarrowtail" . #x02918)
    ("nwarrow" . #x02196)
    ("Nwarrow" . #x021D6)
    ("nwovnearrow" . #x02932)
    ("nwsearrow" . #x02921)
    ("obar" . #x0233D)
    ("obot" . #x029BA)
    ("obrbrak" . #x023E0)
    ("obslash" . #x029B8)
    ("ocirc" . #x0030A)
    ("ocommatopright" . #x00315)
    ("odiv" . #x02A38)
    ("odot" . #x02299)
    ("odotslashdot" . #x029BC)
    ("ogreaterthan" . #x029C1)
    ("oiiint" . #x02230)
    ("oiint" . #x0222F)
    ("oint" . #x0222E)
    ("ointctrclockwise" . #x02233)
    ("olcross" . #x029BB)
    ("olessthan" . #x029C0)
    ("Omega" . #x003A9)
    ("omega" . #x003C9)
    ("Omicron" . #x0039F)
    ("omicron" . #x003BF)
    ("ominus" . #x02296)
    ("operp" . #x029B9)
    ("oplus" . #x02295)
    ("opluslhrim" . #x02A2D)
    ("oplusrhrim" . #x02A2E)
    ("origof" . #x022B6)
    ("oslash" . #x02298)
    ("otimes" . #x02297)
    ("Otimes" . #x02A37)
    ("otimeshat" . #x02A36)
    ("otimeslhrim" . #x02A34)
    ("otimesrhrim" . #x02A35)
    ("oturnedcomma" . #x00312)
    ("overbar" . #x00305)
    ("overbrace" . #x023DE)
    ("overbracket" . #x023B4)
    ("overleftarrow" . #x020D6)
    ("overleftharpoon" . #x020D0)
    ("overleftrightarrow" . #x020E1)
    ("overparen" . #x023DC)
    ("overrightarrow" . #x020D7)
    ("overrightharpoon" . #x020D1)
    ("ovhook" . #x00309)
    ("owns" . #x0220B) ; plain.tex alias
    ("parallel" . #x02225)
    ("parallelogram" . #x025B1)
    ("parallelogramblack" . #x025B0)
    ("parsim" . #x02AF3)
    ("partial" . #x02202)
    ("partialmeetcontraction" . #x02AA3)
    ("pentagon" . #x02B20)
    ("pentagonblack" . #x02B1F)
    ("perp" . #x027C2)
    ("perps" . #x02AE1)
    ("Phi" . #x003A6)
    ("phi" . #x003D5)
    ("Pi" . #x003A0)
    ("pi" . #x003C0)
    ("pitchfork" . #x022D4)
    ("Planckconst" . #x0210E)
    ("plusdot" . #x02A25)
    ("pluseqq" . #x02A72)
    ("plushat" . #x02A23)
    ("plussim" . #x02A26)
    ("plussubtwo" . #x02A27)
    ("plustrif" . #x02A28)
    ("pm" . #x000B1)
    ("pointint" . #x02A15)
    ("postalmark" . #x03012)
    ("prec" . #x0227A)
    ("Prec" . #x02ABB)
    ("precapprox" . #x02AB7)
    ("preccurlyeq" . #x0227C)
    ("preceq" . #x02AAF)
    ("preceqq" . #x02AB3)
    ("precnapprox" . #x02AB9)
    ("precneq" . #x02AB1)
    ("precneqq" . #x02AB5)
    ("precnsim" . #x022E8)
    ("precsim" . #x0227E)
    ("prime" . #x02032)
    ("prod" . #x0220F)
    ("profline" . #x02312)
    ("profsurf" . #x02313)
    ("PropertyLine" . #x0214A)
    ("propto" . #x0221D)
    ("prurel" . #x022B0)
    ("Psi" . #x003A8)
    ("psi" . #x003C8)
    ("pullback" . #x027D3)
    ("pushout" . #x027D4)
    ("QED" . #x0220E)
    ("qprime" . #x02057)
    ("quarternote" . #x02669)
    ("questeq" . #x0225F)
    ("Question" . #x02047)
    ("rangle" . #x027E9)
    ("rAngle" . #x027EB)
    ("rangledot" . #x02992)
    ("rangledownzigzagarrow" . #x0237C)
    ("rbag" . #x027C6)
    ("rblkbrbrak" . #x02998)
    ("rbrace" . #x0007D)
    ("rBrace" . #x02984)
    ("rbracelend" . #x023AD)
    ("rbracemid" . #x023AC)
    ("rbraceuend" . #x023AB)
    ("rbrack" . #x0005D)
    ("rBrack" . #x027E7)
    ("rbrackextender" . #x023A5)
    ("rbracklend" . #x023A6)
    ("rbracklrtick" . #x0298E)
    ("rbrackubar" . #x0298C)
    ("rbrackuend" . #x023A4)
    ("rbrackurtick" . #x02990)
    ("rbrbrak" . #x02773)
    ("Rbrbrak" . #x027ED)
    ("rceil" . #x02309)
    ("rcurvyangle" . #x029FD)
    ("rdiagovfdiag" . #x0292B)
    ("rdiagovsearrow" . #x02930)
    ("Rdsh" . #x021B3)
    ("Re" . #x0211C)
    ("revangle" . #x029A3)
    ("revangleubar" . #x029A5)
    ("revemptyset" . #x029B0)
    ("revnmid" . #x02AEE)
    ("rfbowtie" . #x029D2)
    ("rfloor" . #x0230B)
    ("rftimes" . #x029D5)
    ("rgroup" . #x027EF)
    ("Rho" . #x003A1)
    ("rho" . #x003C1)
    ("rightangle" . #x0221F)
    ("rightanglemdot" . #x0299D)
    ("rightanglesqr" . #x0299C)
    ("rightarrow" . #x02192)
    ("Rightarrow" . #x021D2)
    ("rightarrowapprox" . #x02975)
    ("rightarrowbackapprox" . #x02B48)
    ("rightarrowbar" . #x021E5)
    ("rightarrowbsimilar" . #x02B4C)
    ("rightarrowdiamond" . #x0291E)
    ("rightarrowgtr" . #x02B43)
    ("rightarrowonoplus" . #x027F4)
    ("rightarrowplus" . #x02945)
    ("rightarrowshortleftarrow" . #x02942)
    ("rightarrowsimilar" . #x02974)
    ("rightarrowsupset" . #x02B44)
    ("rightarrowtail" . #x021A3)
    ("rightarrowtriangle" . #x021FE)
    ("rightarrowx" . #x02947)
    ("rightbkarrow" . #x0290D)
    ("rightcurvedarrow" . #x02933)
    ("rightdasharrow" . #x021E2)
    ("rightdbltail" . #x0291C)
    ("rightdotarrow" . #x02911)
    ("rightdowncurvedarrow" . #x02937)
    ("rightfishtail" . #x0297D)
    ("rightharpoonaccent" . #x020D1)
    ("rightharpoondown" . #x021C1)
    ("rightharpoondownbar" . #x02957)
    ("rightharpoonsupdown" . #x02964)
    ("rightharpoonup" . #x021C0)
    ("rightharpoonupbar" . #x02953)
    ("rightharpoonupdash" . #x0296C)
    ("rightimply" . #x02970)
    ("rightleftarrows" . #x021C4)
    ("rightleftharpoons" . #x021CC)
    ("rightleftharpoonsdown" . #x02969)
    ("rightleftharpoonsup" . #x02968)
    ("rightmoon" . #x0263D)
    ("rightouterjoin" . #x027D6)
    ("rightpentagon" . #x02B54)
    ("rightpentagonblack" . #x02B53)
    ("rightrightarrows" . #x021C9)
    ("rightsquigarrow" . #x021DD)
    ("righttail" . #x0291A)
    ("rightthreearrows" . #x021F6)
    ("rightthreetimes" . #x022CC)
    ("rightwavearrow" . #x0219D)
    ("rightwhitearrow" . #x021E8)
    ("ringplus" . #x02A22)
    ("risingdotseq" . #x02253)
    ("rmoustache" . #x023B1)
    ("rparen" . #x00029)
    ("rParen" . #x02986)
    ("rparenextender" . #x0239F)
    ("rparengtr" . #x02994)
    ("rparenlend" . #x023A0)
    ("Rparenless" . #x02996)
    ("rparenuend" . #x0239E)
    ("rppolint" . #x02A12)
    ("rrangle" . #x0298A)
    ("Rrightarrow" . #x021DB)
    ("RRightarrow" . #x02B46)
    ("rrparenthesis" . #x02988)
    ("Rsh" . #x021B1)
    ("rsolbar" . #x029F7)
    ("rsqhook" . #x02ACE)
    ("rsub" . #x02A65)
    ("rtimes" . #x022CA)
    ("rtriltri" . #x029CE)
    ("ruledelayed" . #x029F4)
    ("rvboxline" . #x023B9)
    ("rvzigzag" . #x029D9)
    ("Rvzigzag" . #x029DB)
    ("sansLmirrored" . #x02143)
    ("sansLturned" . #x02142)
    ("scpolint" . #x02A13)
    ("scurel" . #x022B1)
    ("searrow" . #x02198)
    ("Searrow" . #x021D8)
    ("seovnearrow" . #x0292D)
    ("setminus" . #x029F5)
    ("sharp" . #x0266F)
    ("shortdowntack" . #x02ADF)
    ("shortlefttack" . #x02ADE)
    ("shortrightarrowleftarrow" . #x02944)
    ("shortuptack" . #x02AE0)
    ("shuffle" . #x029E2)
    ("Sigma" . #x003A3)
    ("sigma" . #x003C3)
    ("sim" . #x0223C)
    ("sime" . #x02243)
    ("simeq" . #x02243)
    ("simgE" . #x02AA0)
    ("simgtr" . #x02A9E)
    ("similarleftarrow" . #x02B49)
    ("similarrightarrow" . #x02972)
    ("simlE" . #x02A9F)
    ("simless" . #x02A9D)
    ("simminussim" . #x02A6C)
    ("simneqq" . #x02246)
    ("simplus" . #x02A24)
    ("simrdots" . #x02A6B)
    ("sinewave" . #x0223F)
    ("smallblacktriangleleft" . #x025C2)
    ("smallblacktriangleright" . #x025B8)
    ("smallin" . #x0220A)
    ("smallni" . #x0220D)
    ("smallsetminus" . #x02216)
    ("smalltriangleleft" . #x025C3)
    ("smalltriangleright" . #x025B9)
    ("smashtimes" . #x02A33)
    ("smblkcircle" . #x02022)
    ("smblkdiamond" . #x02B29)
    ("smblklozenge" . #x02B2A)
    ("smblksquare" . #x025AA)
    ("smeparsl" . #x029E4)
    ("smile" . #x02323)
    ("smt" . #x02AAA)
    ("smte" . #x02AAC)
    ("smwhitestar" . #x02B52)
    ("smwhtcircle" . #x025E6)
    ("smwhtdiamond" . #x022C4)
    ("smwhtlozenge" . #x02B2B)
    ("smwhtsquare" . #x025AB)
    ("spadesuit" . #x02660)
    ("sphericalangle" . #x02222)
    ("sphericalangleup" . #x029A1)
    ("sqcap" . #x02293)
    ("Sqcap" . #x02A4E)
    ("sqcup" . #x02294)
    ("Sqcup" . #x02A4F)
    ("sqint" . #x02A16)
    ("sqlozenge" . #x02311)
    ("sqrt" . #x0221A)
    ("sqrtbottom" . #x023B7)
    ("sqsubset" . #x0228F)
    ("sqsubseteq" . #x02291)
    ("sqsubsetneq" . #x022E4)
    ("sqsupset" . #x02290)
    ("sqsupseteq" . #x02292)
    ("sqsupsetneq" . #x022E5)
    ("squarebotblack" . #x02B13)
    ("squarecrossfill" . #x025A9)
    ("squarehfill" . #x025A4)
    ("squarehvfill" . #x025A6)
    ("squareleftblack" . #x025E7)
    ("squarellblack" . #x02B15)
    ("squarellquad" . #x025F1)
    ("squarelrblack" . #x025EA)
    ("squarelrquad" . #x025F2)
    ("squareneswfill" . #x025A8)
    ("squarenwsefill" . #x025A7)
    ("squarerightblack" . #x025E8)
    ("squaretopblack" . #x02B12)
    ("squareulblack" . #x025E9)
    ("squareulquad" . #x025F0)
    ("squareurblack" . #x02B14)
    ("squareurquad" . #x025F3)
    ("squarevfill" . #x025A5)
    ("squoval" . #x025A2)
    ("sslash" . #x02AFD)
    ("star" . #x022C6)
    ("stareq" . #x0225B)
    ("strns" . #x023E4)
    ("subedot" . #x02AC3)
    ("submult" . #x02AC1)
    ("subrarr" . #x02979)
    ("subset" . #x02282)
    ("Subset" . #x022D0)
    ("subsetapprox" . #x02AC9)
    ("subsetcirc" . #x027C3)
    ("subsetdot" . #x02ABD)
    ("subseteq" . #x02286)
    ("subseteqq" . #x02AC5)
    ("subsetneq" . #x0228A)
    ("subsetneqq" . #x02ACB)
    ("subsetplus" . #x02ABF)
    ("subsim" . #x02AC7)
    ("subsub" . #x02AD5)
    ("subsup" . #x02AD3)
    ("succ" . #x0227B)
    ("Succ" . #x02ABC)
    ("succapprox" . #x02AB8)
    ("succcurlyeq" . #x0227D)
    ("succeq" . #x02AB0)
    ("succeqq" . #x02AB4)
    ("succnapprox" . #x02ABA)
    ("succneq" . #x02AB2)
    ("succneqq" . #x02AB6)
    ("succnsim" . #x022E9)
    ("succsim" . #x0227F)
    ("sum" . #x02211)
    ("sumbottom" . #x023B3)
    ("sumint" . #x02A0B)
    ("sumtop" . #x023B2)
    ("sun" . #x0263C)
    ("supdsub" . #x02AD8)
    ("supedot" . #x02AC4)
    ("suphsol" . #x027C9)
    ("suphsub" . #x02AD7)
    ("suplarr" . #x0297B)
    ("supmult" . #x02AC2)
    ("supset" . #x02283)
    ("Supset" . #x022D1)
    ("supsetapprox" . #x02ACA)
    ("supsetcirc" . #x027C4)
    ("supsetdot" . #x02ABE)
    ("supseteq" . #x02287)
    ("supseteqq" . #x02AC6)
    ("supsetneq" . #x0228B)
    ("supsetneqq" . #x02ACC)
    ("supsetplus" . #x02AC0)
    ("supsim" . #x02AC8)
    ("supsub" . #x02AD4)
    ("supsup" . #x02AD6)
    ("surd" . #x0221A)
    ("swarrow" . #x02199)
    ("Swarrow" . #x021D9)
    ("talloblong" . #x02AFE)
    ("Tau" . #x003A4)
    ("tau" . #x003C4)
    ("therefore" . #x02234)
    ("thermod" . #x029E7)
    ("Theta" . #x00398)
    ("theta" . #x003B8)
    ("threedangle" . #x027C0)
    ("threedotcolon" . #x02AF6)
    ("threeunderdot" . #x020E8)
    ("tieconcat" . #x02040)
    ("tieinfty" . #x029DD)
    ("tilde" . #x00303)
    ("times" . #x000D7)
    ("timesbar" . #x02A31)
    ("tminus" . #x029FF)
    ("to" . #x02192) ; plain.tex alias
    ("toea" . #x02928)
    ("tona" . #x02927)
    ("top" . #x022A4)
    ("topbot" . #x02336)
    ("topcir" . #x02AF1)
    ("topfork" . #x02ADA)
    ("topsemicircle" . #x025E0)
    ("tosa" . #x02929)
    ("towa" . #x0292A)
    ("tplus" . #x029FE)
    ("trapezium" . #x023E2)
    ("trianglecdot" . #x025EC)
    ("triangledown" . #x025BF)
    ("triangleleft" . #x025C1)
    ("triangleleftblack" . #x025ED)
    ("trianglelefteq" . #x022B4)
    ("triangleminus" . #x02A3A)
    ("triangleodot" . #x029CA)
    ("triangleplus" . #x02A39)
    ("triangleq" . #x0225C)
    ("triangleright" . #x025B7)
    ("trianglerightblack" . #x025EE)
    ("trianglerighteq" . #x022B5)
    ("triangles" . #x029CC)
    ("triangleserifs" . #x029CD)
    ("triangletimes" . #x02A3B)
    ("triangleubar" . #x029CB)
    ("tripleplus" . #x029FB)
    ("trprime" . #x02034)
    ("trslash" . #x02AFB)
    ("turnangle" . #x029A2)
    ("turnediota" . #x02129)
    ("turnednot" . #x02319)
    ("twocaps" . #x02A4B)
    ("twocups" . #x02A4A)
    ("twoheaddownarrow" . #x021A1)
    ("twoheadleftarrow" . #x0219E)
    ("twoheadleftarrowtail" . #x02B3B)
    ("twoheadleftdbkarrow" . #x02B37)
    ("twoheadmapsfrom" . #x02B36)
    ("twoheadmapsto" . #x02905)
    ("twoheadrightarrow" . #x021A0)
    ("twoheadrightarrowtail" . #x02916)
    ("twoheaduparrow" . #x0219F)
    ("twoheaduparrowcircle" . #x02949)
    ("twolowline" . #x02017)
    ("twonotes" . #x0266B)
    ("typecolon" . #x02982)
    ("ubrbrak" . #x023E1)
    ("ularc" . #x025DC)
    ("ulblacktriangle" . #x025E4)
    ("ulcorner" . #x0231C)
    ("ultriangle" . #x025F8)
    ("uminus" . #x02A41)
    ("underbrace" . #x023DF)
    ("underbracket" . #x023B5)
    ("underleftarrow" . #x020EE)
    ("underleftharpoondown" . #x020ED)
    ("underleftrightarrow" . #x0034D)
    ("underparen" . #x023DD)
    ("underrightarrow" . #x020EF)
    ("underrightharpoondown" . #x020EC)
    ("unicodecdots" . #x022EF)
    ("unicodeellipsis" . #x02026)
    ("upand" . #x0214B)
    ("uparrow" . #x02191)
    ("Uparrow" . #x021D1)
    ("uparrowbarred" . #x02909)
    ("uparrowoncircle" . #x029BD)
    ("upbackepsilon" . #x003F6)
    ("updasharrow" . #x021E1)
    ("upDigamma" . #x003DC)
    ("updigamma" . #x003DD)
    ("updownarrow" . #x02195)
    ("Updownarrow" . #x021D5)
    ("updownarrowbar" . #x021A8)
    ("updownarrows" . #x021C5)
    ("updownharpoonleftleft" . #x02951)
    ("updownharpoonleftright" . #x0294D)
    ("updownharpoonrightleft" . #x0294C)
    ("updownharpoonrightright" . #x0294F)
    ("updownharpoonsleftright" . #x0296E)
    ("upfishtail" . #x0297E)
    ("upharpoonleft" . #x021BF)
    ("upharpoonleftbar" . #x02960)
    ("upharpoonright" . #x021BE)
    ("upharpoonrightbar" . #x0295C)
    ("upharpoonsleftright" . #x02963)
    ("upin" . #x027D2)
    ("upint" . #x02A1B)
    ("uplus" . #x0228E)
    ("uprightcurvearrow" . #x02934)
    ("Upsilon" . #x003A5)
    ("upsilon" . #x003C5)
    ("upuparrows" . #x021C8)
    ("upwhitearrow" . #x021E7)
    ("urarc" . #x025DD)
    ("urblacktriangle" . #x025E5)
    ("urcorner" . #x0231D)
    ("urtriangle" . #x025F9)
    ("UUparrow" . #x027F0)
    ("Uuparrow" . #x0290A)
    ("varbarwedge" . #x02305)
    ("varcarriagereturn" . #x023CE)
    ("varclubsuit" . #x02667)
    ("vardiamondsuit" . #x02666)
    ("vardoublebarwedge" . #x02306)
    ("varepsilon" . #x003B5)
    ("varheartsuit" . #x02665)
    ("varhexagon" . #x02B21)
    ("varhexagonblack" . #x02B22)
    ("varhexagonlrbonds" . #x0232C)
    ("varisinobar" . #x022F6)
    ("varisins" . #x022F3)
    ("varkappa" . #x003F0)
    ("varlrtriangle" . #x022BF)
    ("varniobar" . #x022FD)
    ("varnis" . #x022FB)
    ("varnothing" . #x02205)
    ("varointclockwise" . #x02232)
    ("varphi" . #x003C6)
    ("varpi" . #x003D6)
    ("varrho" . #x003F1)
    ("varsigma" . #x003C2)
    ("varspadesuit" . #x02664)
    ("varstar" . #x02736)
    ("vartheta" . #x003D1)
    ("varTheta" . #x003F4)
    ("vartriangle" . #x025B5)
    ("vartriangleleft" . #x022B2)
    ("vartriangleright" . #x022B3)
    ("varVdash" . #x02AE6)
    ("varveebar" . #x02A61)
    ("vBar" . #x02AE8)
    ("Vbar" . #x02AEB)
    ("vBarv" . #x02AE9)
    ("vbraceextender" . #x023AA)
    ("vbrtri" . #x029D0)
    ("vdash" . #x022A2)
    ("vDash" . #x022A8)
    ("Vdash" . #x022A9)
    ("VDash" . #x022AB)
    ("vDdash" . #x02AE2)
    ("vdots" . #x022EE)
    ("vec" . #x020D7)
    ("vectimes" . #x02A2F)
    ("vee" . #x02228)
    ("Vee" . #x02A54)
    ("veebar" . #x022BB)
    ("veedot" . #x027C7)
    ("veedoublebar" . #x02A63)
    ("veeeq" . #x0225A)
    ("veemidvert" . #x02A5B)
    ("veeodot" . #x02A52)
    ("veeonvee" . #x02A56)
    ("veeonwedge" . #x02A59)
    ("vert" . #x0007C)
    ("Vert" . #x02016)
    ("vertoverlay" . #x020D2)
    ("viewdata" . #x02317)
    ("vlongdash" . #x027DD)
    ("vrectangle" . #x025AF)
    ("vrectangleblack" . #x025AE)
    ("Vvdash" . #x022AA)
    ("Vvert" . #x02980)
    ("vysmblkcircle" . #x02219)
    ("vysmblksquare" . #x02B1D)
    ("vysmwhtcircle" . #x02218)
    ("vysmwhtsquare" . #x02B1E)
    ("vzigzag" . #x0299A)
    ("wedge" . #x02227)
    ("Wedge" . #x02A53)
    ("wedgebar" . #x02A5F)
    ("wedgedot" . #x027D1)
    ("wedgedoublebar" . #x02A60)
    ("wedgemidvert" . #x02A5A)
    ("wedgeodot" . #x02A51)
    ("wedgeonwedge" . #x02A55)
    ("wedgeq" . #x02259)
    ("whitearrowupfrombar" . #x021EA)
    ("whiteinwhitetriangle" . #x027C1)
    ("whitepointerleft" . #x025C5)
    ("whitepointerright" . #x025BB)
    ("whitesquaretickleft" . #x027E4)
    ("whitesquaretickright" . #x027E5)
    ("whthorzoval" . #x02B2D)
    ("whtvertoval" . #x02B2F)
    ("wideangledown" . #x029A6)
    ("wideangleup" . #x029A7)
    ("widebreve" . #x00306)
    ("widebridgeabove" . #x020E9)
    ("widecheck" . #x0030C)
    ("widehat" . #x00302)
    ("wideoverbar" . #x00305)
    ("widetilde" . #x00303)
    ("wideutilde" . #x00330)
    ("wp" . #x02118)
    ("wr" . #x02240)
    ("xbsol" . #x029F9)
    ("Xi" . #x0039E)
    ("xi" . #x003BE)
    ("xsol" . #x029F8)
    ("Yup" . #x02144)
    ("Zbar" . #x001B5)
    ("zcmp" . #x02A1F)
    ("Zeta" . #x00396)
    ("zeta" . #x003B6)
    ("zpipe" . #x02A20)
    ("zproject" . #x02A21))
  "Alist of Unicode math symbols.")

(quail-define-package
 "unicode-math" "UTF-8" "Ω" t
 "Input math symbols in TeX notation.

\\alpha ~> α, \\sum ~> ∑, \\pi\\hat ~> π̂, etc.

Use the `unicode-math-input' command to display a complete list
of symbols."
 nil nil unicode-math-input-deterministic)

(let ((min-prefix (or unicode-math-input-min-prefix 1e+INF)))
  (dolist (it (reverse (append unicode-math-input-extra-symbols
                               unicode-math-input-symbols)))
    (cl-loop with name = (car it)
             with len = (length name)
             for n from (min min-prefix len) to len
             do (quail-defrule (concat unicode-math-input-escape
                                       (substring name 0 n))
                               (cdr it)))))

;;;###autoload
(register-input-method
 "unicode-math"
 "UTF-8"
 'quail-use-package
 "Ω"
 "Input math symbols in TeX notation."
 "unicode-math-input")

(defun unicode-math-input--completion-table ()
  "Completion table for `unicode-math-input'."
  (let* ((cands (append unicode-math-input-extra-symbols
                        unicode-math-input-symbols))
         (metadata `((annotation-function . ,(lambda (name)
                                               (format unicode-math-input-annotation
                                                       (cdr (assoc name cands)))))
                     (category . unicode-math-symbol))))
    (lambda (string predicate action)
      (if (eq action 'metadata)
          `(metadata ,@metadata)
        (complete-with-action action cands string predicate)))))

;;;###autoload
(defun unicode-math-input (&optional name)
  "Insert the Unicode character with the given TeX NAME.
Can also be used as an item in `completion-at-point-functions'."
  (interactive (list (completing-read "Insert math symbol: "
                                      (unicode-math-input--completion-table)
                                      nil t)))
  (if name
      (insert (cdr (or (assoc name unicode-math-input-extra-symbols)
                       (assoc name unicode-math-input-symbols)
                       (user-error "No symbol named `%s'" name))))
    (when (thing-at-point-looking-at
           (concat (regexp-quote unicode-math-input-escape)
                   "\\([A-Za-z]+\\)"))
      `(,(match-beginning 1) ,(match-end 1)
        ,(unicode-math-input--completion-table)
        :exclusive no))))

(provide 'unicode-math-input)
;;; unicode-math-input.el ends here
