;;; org-mpv-notes.el --- Take notes in org mode while watching videos in mpv -*- lexical-binding: t -*-

;; Copyright (C) 2021-2024 Bibek Panthi

;; Author: Bibek Panthi <bpanthi977@gmail.com>
;; Maintainer: Bibek Panthi <bpanthi977@gmail.com>
;; URL: https://github.com/bpanthi977/org-mpv-notes
;; Package-Version: 20241222.1958
;; Package-Revision: 1d8db9ff8031
;; Package-Requires: ((emacs "28.1"))
;; Kewords: mpv, org

;; This file in not part of GNU Emacs

;;; SPDX-License-Identifier: MIT

;;; Commentary:

;; org-mpv-notes allows you to control mpv and take notes from videos
;; playing in mpv.  You can control mpv (play, pause, seek, ...) while
;; in org buffer and insert heading or notes with timestamps to the
;; current playing position.  Later you can revist the notes and seek
;; to the inserted timestamp with a few keystrokes.  Also, it can
;; insert screenshots as org link, run ocr (if ocr program is
;; installed) and insert the ocr-ed text to the org buffer.

;;; Code:
(require 'cl-lib)
(require 'mpv nil 'noerror)
(require 'empv nil 'noerror)
(require 'org-attach)
(require 'org-element)
(require 'org-timer)
(require 'org-mpv-notes-compat)
(require 'org-mpv-notes-subtitles)

;;;;;
;;; Config
;;;;;

(defgroup org-mpv-notes nil
  "Options concerning mpv links in Org mode."
  :group 'org-link
  :prefix "org-mpv-notes-")

(defcustom org-mpv-notes-empv-wait-interval 0.1
  "How many seconds to wait for mpv to settle.
This may be necessary because much of the empv library runs
asynchronously."
  :type '(float
          :validate
          (lambda (w)
            (when (> 0 (floor (widget-value w)))
              (widget-put w :error "Must be a positive number")
              w))))

(defcustom org-mpv-notes-preferred-backend 'mpv
  "The preferred mpv library to open new media with."
  :type 'symbol
  :options '(mpv empv))

(defcustom org-mpv-notes-mpv-args '("--no-terminal"
                                    "--idle"
                                    "--video=auto"
                                    "--focus-on=never"
                                    "--volume=40"
                                    "--sub-delay=-1"
                                    "--ontop=yes"
                                    "--geometry=100%:100%"
                                    "--autofit=35%"
                                    "--autofit-larger=50%")
  "Args used while starting mpv.
This will over-ride the settings of your chosen mpv
backend (variable `mpv-default-options' for mpv.el, or variable
`empv-mpv-args' for empv.el) for just this use-case. See man(1)
mpv for details."
  :type '(repeat
          (string
           :validate
           (lambda (w)
             (let ((val (widget-value w)))
               (when (or (not (stringp val))
                         (not (string-match "^--" val)))
                 (widget-put w :error "All elements must be command line option strings, eg. --foo")
                 w))))))

(defcustom org-mpv-notes-seek-step 5
  "Step size in seconds used when seeking."
  :type 'number)

(defcustom org-mpv-notes-export-path-translation #'identity
  "Function that applies path translation to mpv: file link's when exporting to html.
Function takes one argument `path' which is mpv: link's path without search string and returns translated path.

This is useful to presever links if you have configured org mode to export html to different directory.")

;;;;;
;;; Opening Link & Interface with org link
;;;;;

;; from https://github.com/kljohann/mpv.el/wiki
;;  To create a mpv: link type that is completely analogous to file: links but opens using mpv-play instead

(defun org-mpv-notes-complete-link (&optional arg)
  "Provide completion to mpv: link in `org-mode'.
ARG is passed to `org-link-complete-file'."
  (replace-regexp-in-string
   "file:" "mpv:"
   (org-link-complete-file arg)
   t t))

(defun org-mpv-notes-setup-link ()
  "Setup mpv: link to be recognized by `org-mode'."
  (org-link-set-parameters "mpv"
                           :complete #'org-mpv-notes-complete-link
                           :follow #'org-mpv-notes--open
                           :export #'org-mpv-notes-export))

(org-mpv-notes-setup-link)

;; adapted from https://bitspook.in/blog/extending-org-mode-to-handle-youtube-links/
(defun org-mpv-notes-export (path desc backend info)
  "Format mpv link while exporing.
For html exports, YouTube links are converted to thumbnails.
`PATH' and `DESC' are the mpv link and description.
`BACKEND' is the export backend (html, latex, ...)
`INFO' is the export communication plist"
  (when (eq backend 'html)
    (cl-multiple-value-bind (path secs) (org-mpv-notes--parse-link path)
      (cond ((string-search "youtube.com/" path)
             (cond ((or (not desc) (string-equal desc ""))
                    (let* ((video-id (cadar (url-parse-query-string path)))
                           (url (if (string-empty-p video-id) path
                                  (format "https://www.youtube.com/embed/%s" video-id))))
                      (when url
                        (format "<p style=\"text-align:center; width:100%%\"><iframe width=\"560\" height=\"315\" src=\"%s\" title=\"%s\" frameborder=\"0\" allow=\"accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share\" allowfullscreen></iframe></p>"
                                url desc))))
                   (secs
                    (format "<a href=\"%s&t=%ds\">%s</a>" path secs (substring-no-properties desc)))))
            ((and (file-exists-p path) (not desc))
             (let* ((hash (sxhash path))
                    (seek (format "
<script>
{
  const func = () => {
    const video = document.getElementById('video-%d');
    video.currentTime = %d;
    if (video.parentElement?.tagName == \"P\") {
      video.setAttribute(\"width\", video.parentElement.getAttribute(\"width\"));
      video.setAttribute(\"height\", video.parentElement.getAttribute(\"height\"));
      if (video.parentElement.getAttribute(\"loop\"))
           video.setAttribute(\"loop\", true);
      if (video.parentElement.getAttribute(\"autoplay\"))
           video.setAttribute(\"autoplay\", true);
    }
  }

  if (document.getElementById('video-%d'))
    func();
  else
    document.addEventListener('DOMContentLoaded', func);
}
</script>
"
                                  hash (or secs 0) hash)))
               (format "
<video controls id=\"video-%s\">
  <source src=\"%s\">
  Your browser does not support the video tag.
</video>
%s"
                       hash
                       (funcall org-mpv-notes-export-path-translation path)
                       (or seek ""))))))))

(defvar org-mpv-notes-timestamp-regex "[0-9]+:[0-9]+:[0-9]+")

(defun org-mpv-notes--parse-link (path)
  "Parse the org-link `PATH' to extract the media path and timestamp.
Timestamp can exist at the end of `PATH' with `::' as separator.
Timestamp can be in two formats: hh:mm:ss or ss

Returns path (string)
        timestamp in seconds (integer or nil)."
  (let (search-option
        (secs nil))
    ;; 1. Find the timestamp
    (when (string-match "::\\(.*\\)\\'" path)
      (setq search-option (match-string 1 path))
      (setq path (replace-match "" nil nil path)))
    ;; 2. Parse the timestamp
    (cond ((null search-option) nil)
          ((string-match (concat "^" org-mpv-notes-timestamp-regex) search-option)
           (setf secs (org-timer-hms-to-secs search-option)))
          ((string-match "^\\([0-9]+\\)$" search-option)
           (setf secs (string-to-number search-option))))
    (cl-values path (and secs (> secs 0) secs))))

(defun org-mpv-notes--open (path &optional arg)
  "Open the mpv `PATH'.
`ARG' is required by org-follow-link but is ignored here."
  (cl-multiple-value-bind (path secs) (org-mpv-notes--parse-link path)
    ;; Enable Minor mode
    (org-mpv-notes-mode t)
    (let* ((org-mpv-notes-mpv-args (if secs
                                       (cons (format "--start=%d" secs)
                                             org-mpv-notes-mpv-args)
                                     org-mpv-notes-mpv-args))
           (mpv-default-options org-mpv-notes-mpv-args)
           (empv-mpv-args (when (boundp 'empv-mpv-args)
                            (append empv-mpv-args org-mpv-notes-mpv-args))))
      (cl-flet ((start (path)
                  (let ((path (if (string-prefix-p "http" path)
                                  path
                                (file-truename path))))
                    (message "org-mpv-notes: Opening %s" path)
                    (if (eql (org-mpv-notes--backend) 'mpv)
                        (mpv-start path)
                      (empv-start path)))))

        ;; Open mpv player
        (cond ((not (org-mpv-notes--active-backend t))
               (start path))
              ((not (string-equal (org-mpv-notes--get-property "path") path))
               (org-mpv-notes-kill)
               (sleep-for org-mpv-notes-empv-wait-interval)
               (start path))
              (t
               (when secs
                 (sleep-for org-mpv-notes-empv-wait-interval)
                 (org-mpv-notes--seek secs))))))))

;;;###autoload
(defun org-mpv-notes-open (&optional path)
  "Open a media file or URL and insert mpv: link for it in the bfufer.
If `PATH' is provided, the path is used for the media,
otherwise user is prompted for a File path or URL link."
  (interactive)
  (if path
      (org-mpv-notes--open path)
    (let ((choice (completing-read "Choose media source:" (list "File" "URL") nil t)))
      (cond ((string-equal-ignore-case choice "File")
             (let ((open-file (lambda (path)
                                (interactive "fMedia File:" org-mode)
                                (insert (concat "[[mpv:" path "]]"))
                                (org-mpv-notes--open path))))
               (call-interactively open-file)))
            ((string-equal-ignore-case choice "URL")
             (let ((open-url (lambda (url)
                               (interactive "sURL: " url)
                               (insert (concat "[[mpv:" url "]]"))
                               (org-mpv-notes--open url))))
               (call-interactively open-url)))))))

;;;;;
;;; Screenshot
;;;;;

(defcustom org-mpv-notes-save-image-function
  #'org-mpv-notes-save-as-attach
  "Function that saves screenshot image file to org buffer.
Filename is passed as first argument.  The function has to copy
the file to proper location and insert a link to that file."
  :type '(function)
  :options '(#'org-mpv-notes-save-as-attach
             #'org-download-image))

(defun org-mpv-notes-save-as-attach (file)
  "Save image FILE to org file using `org-attach'."
  ;; attach it
  (let ((org-attach-method 'mv))
    (org-attach-attach file))
  ;; insert the link
  (insert "[[attachment:" (file-name-base file) "." (file-name-extension file) "]]"))

;; save screenshot as attachment
(defun org-mpv-notes-save-screenshot ()
  "Save screenshot of current frame as attachment."
  (interactive)
  (let ((filename (format "%s.png" (make-temp-file "mpv-screenshot"))))
    ;; take screenshot
    (org-mpv-notes--cmd "screenshot-to-file"
                        filename
                        "video")
    (funcall org-mpv-notes-save-image-function filename)
    (org-display-inline-images)))

;;;;;
;;; OCR on screenshot
;;;;;

(defcustom org-mpv-notes-ocr-command "tesseract"
  "OCR program to extract text from mpv screenshot."
  :type '(string))

(defcustom org-mpv-notes-ocr-command-args "-"
  "Extra arguments to pass to ocr-command after the input image file."
  :type '(string))

(defun org-mpv-notes--ocr-on-file (file)
  "Run tesseract OCR on the screenshot FILE."
  (unless (executable-find org-mpv-notes-ocr-command)
    (user-error "OCR program %S not found" org-mpv-notes-ocr-command))
  (with-temp-buffer
    (if (zerop (apply #'call-process org-mpv-notes-ocr-command nil t nil
                      (file-truename file)
                      (split-string-shell-command org-mpv-notes-ocr-command-args)))
        (remove ? (buffer-string))
      (error "OCR command failed: %S" (buffer-string)))))

(defun org-mpv-notes-screenshot-ocr ()
  "Take screenshot, run OCR on it and insert the text to org buffer."
  (interactive)
  (let ((filename (format "%s.png" (make-temp-file "mpv-screenshot"))))
    ;; take screenshot
    (org-mpv-notes--cmd "screenshot-to-file"
                        filename
                        "video")
    (let ((string (org-mpv-notes--ocr-on-file filename)))
      (insert "\n"
              string
              "\n"))))
;;;;;
;;; Motion (jump to next, previous, ... link)
;;;;;

(defcustom org-mpv-notes-narrow-timestamp-navigation nil
  "Restrict timestamp navigation to within the current heading.
This affects functions `org-mpv-notes-next-timestamp' and
`org-mpv-notes-previous-timestamp'."
  :type 'boolean)

(defun org-mpv-notes--timestamp-p ()
  "Return non-NIL if POINT is on a timestamp."
  (string-match "mpv" (or (org-element-property :type (org-element-context)) "")))

(defun org-mpv-notes-next-timestamp (&optional reverse)
  "Seek to next timestamp in the notes file.
`REVERSE' searches in backwards direction."
  (interactive)
  (let ((p (point))
        success)
    (save-excursion
      (when org-mpv-notes-narrow-timestamp-navigation
        (org-narrow-to-subtree))
      (while (and (not success)
                  (org-next-link reverse)
                  (not (eq p (point))))
        (when (and (org-mpv-notes--timestamp-p)
                   (not (eq p (point))))
          (setq success t))
        (setq p (point)))
      (when org-mpv-notes-narrow-timestamp-navigation
        (widen)))
    (if (not success)
        (error "Error: No %s link" (if reverse "prior" "next"))
      (goto-char p)
      (org-open-at-point)
      (org-fold-show-entry)
      (recenter))))

(defun org-mpv-notes-previous-timestamp ()
  "Seek to previous timestamp in the notes file."
  (interactive)
  (org-mpv-notes-next-timestamp t))

(defun org-mpv-notes-this-timestamp ()
  "Seek to the timestamp at POINT or previous.
If there is no timestamp at POINT, consider the previous one as this one."
  (interactive)
  (cond
   ((org-mpv-notes--timestamp-p)
    (org-mpv-notes--open (org-element-property :path (org-element-context)))
    (org-fold-show-entry)
    (recenter))
   (t
    (save-excursion (org-mpv-notes-previous-timestamp)))))

;;;;;
;;; Creating Links
;;;;;

(defcustom org-mpv-notes-pause-on-link-create nil
  "Whether to automatically pause mpv when creating a link or note."
  :type 'boolean)

(defcustom org-mpv-notes-timestamp-lag 0
  "Number of seconds to subtract when setting timestamp.

This variable acknowledges that many of us may sometimes be slow
to create a note or link."
  :type '(integer
          :validate (lambda (w)
                      (let ((val (widget-value w)))
                        (when (> 0 val)
                          (widget-put w :error "Must be a positive integer")
                          w)))))

(cl-defun org-mpv-notes--create-link (&optional (read-description t))
  "Create a link with timestamp to insert in org file.
If `READ-DESCRIPTION' is true, ask for a link description from user."
  (let* ((path (org-link-escape (org-mpv-notes--get-property "path")))
         (time (max 0 (- (org-mpv-notes--get-property "playback-time")
                         org-mpv-notes-timestamp-lag)))
         (h (floor (/ time 3600)))
         (m (floor (/ (mod time 3600) 60)))
         (s (floor (mod time 60)))
         (timestamp (format "%02d:%02d:%02d" h m s))
         (description ""))
    (when org-mpv-notes-pause-on-link-create
      (org-mpv-notes-pause))
    (when read-description
      (setq description (read-string "Description: ")))
    (when (string-equal description "")
      (setf description timestamp))
    (concat "[[mpv:" path "::" timestamp "][" description "]]")))

(defun org-mpv-notes-insert-note ()
  "Insert a heading with link & timestamp."
  (interactive)
  (let ((link (org-mpv-notes--create-link nil)))
    (when link
      (org-insert-heading)
      (insert link))))

(defun org-mpv-notes-insert-link ()
  "Insert link with timestamp."
  (interactive)
  (insert (org-mpv-notes--create-link t)))

(defun org-mpv-notes-replace-timestamp-with-link (begin end link)
  "Convert hh:mm:ss text within region to link with timestamp.
Region is between `BEGIN' and `END' points,
`LINK' is the new media url/path."
  (interactive "r\nsLink:")
  (save-excursion
    (let (timestamp)
      (setq link (org-link-escape link))
      (goto-char end)
      (while (re-search-backward "[^0-9]\\([0-9]+:[0-9]+:[0-9]+\\)" begin t)
        (setq timestamp (match-string 1))
        (replace-region-contents (match-beginning 1) (match-end 1)
                                 (lambda () (concat "[[mpv:" link "::" timestamp "][" timestamp "]]")))
        (search-backward "[[" begin t)))))

(defun org-mpv-notes-change-link-reference (all-occurences)
  "Change a link to reflect a moved or renamed media file.
With a PREFIX-ARG (`ALL-OCCURENCES'), apply the change to all similar references
within the current buffer."
  (interactive "P")
  (unless (org-mpv-notes--timestamp-p)
    ;; We could always look to the timestamp link prior to POINT, but
    ;; this is a decent trade-off between convenience and preventing
    ;; accidental changes.
    (error "Error: POINT is not within a timestamp link"))
  (let* ((target (org-link-escape
                  (read-file-name "Correct target path?: " nil nil t)))
         (context (org-element-context))
         (old-link-path (split-string
                         (or (org-element-property :path context)
                             (error "Error: Failed to extract old path-name"))
                         "::"))
         (old-path (if (/= 2 (length old-link-path))
                       (error "Error: Failed to parse the old link")
                     (org-link-escape (car old-link-path))))
         (p (point))
         here
         (replace-it
          (lambda ()
            (setq context (org-element-context))
            (replace-string-in-region old-path target
                                      (org-element-property :begin context)
                                      (org-element-property :end context)))))
    (org-toggle-link-display)
    (cond
     (all-occurences
      (goto-char (point-min))
      (setq here (point))
      (while (and (org-next-link)
                  (> (point) here))
        (when (org-mpv-notes--timestamp-p)
          (funcall replace-it))
        (setq here (point)))
      (goto-char p))
     (t ; ie. (not all-occurences)
      (funcall replace-it)))
    (org-toggle-link-display)))

;;;;;
;;; Minor Mode and Keymap
;;;;;
(defvar org-mpv-notes-key-bindings
  `(;; Inserting links
    ("i" . org-mpv-notes-insert-link)
    ("M-i" . org-mpv-notes-insert-note)

    ;; Traversing links
    ("=" . org-mpv-notes-this-timestamp)
    ("<left>" . org-mpv-notes-previous-timestamp)
    ("p" . org-mpv-notes-previous-timestamp)
    ("<right>" . org-mpv-notes-next-timestamp)
    ("n" . org-mpv-notes-next-timestamp)

    ;; Screenshot and OCR
    ("s" .  org-mpv-notes-save-screenshot)
    ("M-s" .  org-mpv-notes-screenshot-ocr)

    ;; Seeking
    ("f" . org-mpv-notes-seek-forward)
    ("b" . org-mpv-notes-seek-backward)
    ("a" . org-mpv-notes-halve-seek-step)
    ("d" . org-mpv-notes-double-seek-step)

    ;; other mpv controls
    ("q" . keyboard-quit)
    ("F" . org-mpv-notes-toggle-fullscreen)
    ("SPC" .  org-mpv-notes-pause)
    ("]" . org-mpv-notes-speed-up)
    ("[" . org-mpv-notes-speed-down)
    ("k" . org-mpv-notes-kill)))

(define-prefix-command 'org-mpv-notes-prefix-map)
(cl-loop for (key . command) in org-mpv-notes-key-bindings do
         (define-key org-mpv-notes-prefix-map (kbd key) command))

;;;###autoload
(define-minor-mode org-mpv-notes-mode
  "Org minor mode for Note taking alongside audio and video.
Uses mpv.el to control mpv process"
  :keymap `((,(kbd "M-n") . org-mpv-notes-prefix-map)))

(provide 'org-mpv-notes)

;;; org-mpv-notes.el ends here
