;;; js-pkg-mode.el --- Minor mode for working with javascript projects

;; Package-Version: 20251017.936
;; Package-Revision: 4d7248b829b6
;; Author: Ovi Stoica <ovidiu.stoica1094@gmail.com>
;; Url: https://github.com/ovistoica/js-pkg-mode
;; Keywords: convenience, project, javascript, package-manager
;; Package-Requires: ((emacs "25.1"))

;; This file is NOT part of GNU Emacs.

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 3, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Commentary:

;; This package allows you to easily work with javascript projects.It provides
;; a minor mode for convenient interactive use of API with a mode-specific
;; command keymap.
;;
;; | command                       | keymap       | description                         |
;; |-------------------------------|--------------|-------------------------------------|
;; | js-pkg-init                   | <kbd>n</kbd> | Initialize new project              |
;; | js-pkg-install                | <kbd>i</kbd> | Install all project dependencies    |
;; | js-pkg-install-save           | <kbd>s</kbd> | Add new project dependency          |
;; | js-pkg-install-save-dev       | <kbd>d</kbd> | Add new project dev dependency      |
;; | js-pkg-uninstall              | <kbd>u</kbd> | Remove project dependency           |
;; | js-pkg-list                   | <kbd>l</kbd> | List installed project dependencies |
;; | js-pkg-run                    | <kbd>r</kbd> | Run project script                  |
;; | js-pkg-visit-project-file     | <kbd>v</kbd> | Visit project package.json file     |
;; |                               | <kbd>?</kbd> | Display keymap commands             |

;;; Credit:

;; This package began as a fork of the npm-mode package, and its repository
;; history has been preserved.  Many thanks to Allen Gooch for his contribution.
;; https://github.com/mojochao/npm-mode repo.


;;; Code:

(require 'json)
(require 'project)

(defgroup js-pkg nil
  "Customization group for js-pkg."
  :group 'tools)

(defface js-pkg-mode-line-on-face '((t :inherit success))
  "Face used in mode line to indicate that js-pkg is in effect.")

(defface js-pkg-mode-line-error-face '((t :inherit error))
  "Face used in mode line to indicate that js-pkg failed.")

(defface js-pkg-mode-line-none-face '((t :inherit warning))
  "Face used in mode line to indicate that js-pkg is not active.")


(defcustom js-pkg-project-file-name "package.json"
  "The name of npm project files."
  :type 'string)

(defcustom js-pkg-package-manager-type 'npm
  "Package manager to use (npm, yarn, pnpm, bun or deno).
It is automatically infered based on the lockfile but you can overwrite."
  :type '(choice (const npm)
                 (const yarn)
                 (const pnpm)
                 (const deno)
                 (const bun))
  :group 'js-pkg)

(defun js-pkg--default-lighter-formatter (package-manager-sym)
  "Format PACKAGE-MANAGER-SYM for modeline."
  (concat " JS["
          (propertize (symbol-name package-manager-sym)
                      'face 'js-pkg-mode-line-on-face)
          "]"))

(defcustom js-pkg-lighter-formatter #'js-pkg--default-lighter-formatter
  "Spec for the modeline based on the symbol."
  :type 'function
  :group 'js-pkg)

(defcustom js-pkg-lighter '(:eval (funcall js-pkg-lighter-formatter js-pkg-package-manager-type))
  "The mode line lighter for `envrc-mode'.
You can set this to nil to disable the lighter."
  :type 'sexp
  :risky t)


(defun js-pkg--ensure-npm-module ()
  "Asserts that you're currently inside an npm module."
  (js-pkg--project-file))

(defun js-pkg--project-file ()
  "Return path to the project file, or nil.
If project file exists in the current working directory, or a
parent directory recursively, return its path.  Otherwise, return
nil."
  (let ((dir (locate-dominating-file default-directory js-pkg-project-file-name)))
    (unless dir
      (error (concat "Error: cannot find " js-pkg-project-file-name)))
    (concat dir js-pkg-project-file-name)))


(defun js-pkg--lock-file ()
  "Return path to the package lock file, or nil."
  (let* ((dir (locate-dominating-file default-directory js-pkg-project-file-name))
         (lock-files '("package-lock.json" "deno.lock" "yarn.lock" "pnpm-lock.yaml" "bun.lock")))
    (when dir
      (cl-find-if (lambda (file)
                    (file-exists-p (concat dir file)))
                  lock-files))))

(defun js-pkg-lockfile->package-manager (lock-file_)
  "Given LOCK-FILE_ type, output package-manager used in the project."
  (cond
   ((string= lock-file_ "package-lock.json") 'npm)
   ((string= lock-file_ "yarn.lock") 'yarn)
   ((string= lock-file_ "pnpm-lock.yaml") 'pnpm)
   ((string= lock-file_ "bun.lock") 'bun)
   ((string= lock-file_ "deno.lock") 'deno)))


(defun js-pkg-package-manager ()
  "Get the package manager for the current buffer."
  (js-pkg-lockfile->package-manager (js-pkg--lock-file)))


(defun js-pkg--get-project-property (prop)
  "Get the given PROP from the current project file."
  (let* ((project-file (js-pkg--project-file))
         (json-object-type 'hash-table)
         (json-contents (with-temp-buffer
                          (insert-file-contents project-file)
                          (buffer-string)))
         (json-hash (json-read-from-string json-contents))
         (value (gethash prop json-hash))
         (commands (list)))
    (cond ((hash-table-p value)
           (maphash (lambda (key value)
                      (setq commands
                            (append commands
                                    (list (list key (format "%s %s" "npm" key))))))
                    value)
           commands)
          (t value))))

(defun js-pkg--get-project-scripts ()
  "Get a list of project scripts."
  (js-pkg--get-project-property "scripts"))

(defun js-pkg--get-project-dependencies ()
  "Get a list of project dependencies."
  (js-pkg--get-project-property "dependencies"))

(defun js-pkg--exec-process (cmd &optional comint)
  "Execute a process running CMD.
Optional argument COMINT when non-nil runs the command in comint mode."
  (let* ((pm-type (symbol-name js-pkg-package-manager-type))
         (compilation-buffer-name-function
          (lambda (mode)
            (format "*%s:%s - %s*"
                    pm-type
                    (js-pkg--get-project-property "name")
                    cmd))))
    (message (concat "Running " cmd))
    (compile cmd comint)))

(defun js-pkg-clean ()
  "Remove node_modules."
  (interactive)
  (let ((dir (concat (file-name-directory (js-pkg--ensure-npm-module)) "node_modules")))
    (if (file-directory-p dir)
        (when (yes-or-no-p (format "Are you sure you wish to delete %s?" dir))
          (js-pkg--exec-process (format "rm -rf %s" dir)))
      (message "%s has already been cleaned" dir))))

(defun js-pkg-init ()
  "Initialize a new javascript project.  Prompt for package manager choice."
  (interactive)
  (let* ((pm-choices '(("npm" . npm)
                       ("yarn" . yarn)
                       ("pnpm" . pnpm)
                       ("bun" . bun)
                       ("deno" . deno)))
         (choice (completing-read "Choose package manager: " pm-choices nil t))
         (pm-symbol (cdr (assoc choice pm-choices))))
    (setq js-pkg-package-manager-type pm-symbol)
    (js-pkg--exec-process (format "%s init" (symbol-name pm-symbol)))))

(defun js-pkg-install ()
  "Run the install command."
  (interactive)
  (let ((pm-name (symbol-name js-pkg-package-manager-type)))
    (js-pkg--exec-process (format "%s install" pm-name))))

(defun js-pkg-install-save (dep)
  "Install and save DEP as a dependency."
  (interactive "sEnter package name: ")
  (let* ((pm-name (symbol-name js-pkg-package-manager-type))
         (cmd (pcase pm-name
                ("npm" (format "npm install %s --save" dep))
                ("yarn" (format "yarn add %s" dep))
                ("pnpm" (format "pnpm add %s" dep))
                ("bun" (format "bun add %s" dep)))))
    (js-pkg--exec-process cmd)))

(defun js-pkg-install-save-dev (dep)
  "Install and save DEP as a dev dependency."
  (interactive "sEnter package name: ")
  (let* ((pm-name (symbol-name js-pkg-package-manager-type))
         (cmd (pcase pm-name
                ("npm" (format "npm install %s --save-dev" dep))
                ("yarn" (format "yarn add %s --dev" dep))
                ("pnpm" (format "pnpm add -D %s" dep))
                ("bun" (format "bun add -d %s" dep)))))
    (js-pkg--exec-process cmd)))

(defun js-pkg-uninstall ()
  "Uninstall a dependency."
  (interactive)
  (let* ((pm-name (symbol-name js-pkg-package-manager-type))
         (dep (completing-read "Uninstall dependency: " (js-pkg--get-project-dependencies)))
         (cmd (pcase pm-name
                ("npm" (format "npm uninstall %s" dep))
                ("yarn" (format "yarn remove %s" dep))
                ("pnpm" (format "pnpm remove %s" dep))
                ("bun" (format "bun remove %s" dep)))))
    (js-pkg--exec-process cmd)))

(defun js-pkg-list ()
  "List installed dependencies."
  (interactive)
  (let* ((pm-name (symbol-name js-pkg-package-manager-type))
         (cmd (pcase pm-name
                ("npm" "npm list --depth=0")
                ("yarn" "yarn list --depth=0")
                ("pnpm" "pnpm list --depth=0")
                ("bun" "bun pm ls"))))
    (js-pkg--exec-process cmd)))


(defun js-pkg-run--read-command ()
  "Prompt user to select a script from package.json scripts or install.
Returns the selected script name as a string."
  (let* ((scripts (js-pkg--get-project-scripts))
         (choices (cons "install" (mapcar #'car scripts))))
    (completing-read "Run script: " choices)))


(defun js-pkg-run (script &optional comint)
  "Run the package manager's run command on a project script.
SCRIPT is the npm script to run.
Optional argument COMINT when non-nil runs the command in comint mode."
  (interactive
   (list (js-pkg-run--read-command)
         (consp current-prefix-arg)))
  (if (string= script "install")
      (js-pkg-install)
    (let ((pm-name (symbol-name js-pkg-package-manager-type)))
      (js-pkg--exec-process
       (format "%s run %s" pm-name script)
       comint))))

(defun js-pkg-visit-project-file ()
  "Visit the project file."
  (interactive)
  (find-file (js-pkg--project-file)))

(defun js-pkg--update-keymap ()
  "Update the js-pkg-mode keymap based on the current prefix setting."
  (setq js-pkg-mode-keymap (make-sparse-keymap))
  (when js-pkg-mode-keymap-prefix
    (define-key js-pkg-mode-keymap (kbd js-pkg-mode-keymap-prefix) js-pkg-command-keymap)))

(defcustom js-pkg-mode-keymap-prefix nil
  "Prefix for js-pkg keybindings.
When nil, no keybindings are active. Set this to a key sequence
like \"C-c n\" to enable keybindings with that prefix."
  :type '(choice (const :tag "No keybindings" nil)
                 key-sequence)
  :set (lambda (symbol value)
         (set-default symbol value)
         (js-pkg--update-keymap))
  :group 'js-pkg)

(defvar js-pkg-command-keymap
  (let ((map (make-sparse-keymap)))
    (define-key map "n" 'js-pkg-init)
    (define-key map "i" 'js-pkg-install)
    (define-key map "s" 'js-pkg-install-save)
    (define-key map "d" 'js-pkg-install-save-dev)
    (define-key map "u" 'js-pkg-uninstall)
    (define-key map "l" 'js-pkg-list)
    (define-key map "r" 'js-pkg-run)
    (define-key map "v" 'js-pkg-visit-project-file)
    (define-key map "c" 'js-pkg-clean)
    map)
  "Keymap for js-pkg commands.")

(defvar js-pkg-mode-keymap (make-sparse-keymap)
  "Keymap for `js-pkg-mode'.")

;;;###autoload
(define-minor-mode js-pkg-mode
  "Minor mode for working with javascript projects."
  :lighter js-pkg-lighter
  :keymap js-pkg-mode-keymap
  :group 'js-pkg
  (when js-pkg-mode
    (setq js-pkg-package-manager-type
          (js-pkg-package-manager))))

;;;###autoload
(define-globalized-minor-mode js-pkg-global-mode
  js-pkg-mode
  js-pkg-mode)

(provide 'js-pkg-mode)
;;; js-pkg-mode.el ends here
