;;; hypb.el --- Miscellaneous GNU Hyperbole support features  -*- lexical-binding: t; -*-
;;
;; Author:       Bob Weiner
;;
;; Orig-Date:     6-Oct-91 at 03:42:38
;; Last-Mod:      9-Feb-26 at 00:21:58 by Bob Weiner
;;
;; SPDX-License-Identifier: GPL-3.0-or-later
;;
;; Copyright (C) 1991-2025  Free Software Foundation, Inc.
;; See the "HY-COPY" file for license information.
;;
;; This file is part of GNU Hyperbole.

;;; Commentary:

;;; Code:
;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

;; Load Org here for `org-fold-show-all'.
(eval-and-compile (mapc #'require '(compile hversion hact locate
				    cl-lib org package seq)))

;;; ************************************************************************
;;; Public declarations
;;; ************************************************************************

(defvar mh-e-RCS-id)
(defvar pm-version)

(declare-function helm-info "ext:helm")
(declare-function helm-apropos "ext:helm")
(declare-function devdocs-lookup "ext:devdocs")
(declare-function native-comp-available-p "comp.c")

;; interaction-log
(defvar ilog-buffer-name)
(defvar ilog-display-state)
(defvar ilog-idle-time)
(defvar ilog-insertion-timer)
(defvar ilog-print-lambdas)
(defvar ilog-self-insert-command-regexps)
(defvar ilog-truncation-timer)
(defvar interaction-log-mode)
(defvar interaction-log-mode-hook)

(declare-function ilog-note-buffer-change "ext:interaction-log")
(declare-function ilog-post-command "ext:interaction-log")
(declare-function ilog-record-this-command "ext:interaction-log")
(declare-function ilog-show-in-other-frame "ext:interaction-log")
(declare-function ilog-timer-function "ext:interaction-log")
(declare-function ilog-toggle-view "ext:interaction-log")
(declare-function ilog-truncate-log-buffer "ext:interaction-log")
(declare-function interaction-log-mode "ext:interaction-log")

(defvar hyperb:user-email)              ; "hinit.el"

(declare-function hkey-either "hmouse-drv")
(declare-function hycontrol-frame-to-right-center "hycontrol")

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defconst hypb:help-buf-prefix "*Help: Hyperbole "
  "Prefix attached to all native Hyperbole help buffer names.
This should end with a space.")

(defcustom hypb:in-string-modes-regexps
  '(let ((open-regexp "\\(^\\|[^\\]\\)\\(%s\\)")
	 (close-regexp "\\(^\\|[^\\]\\)\\(%s\\)"))
     (cond ((derived-mode-p 'texinfo-mode)
	    (list (format open-regexp "``\\|\"")
		  (format close-regexp "''\\|\"")))
	   (t
	    (list (format open-regexp "'''\\|\"\"\"\\|\"\\|'")
		  (format close-regexp "'''\\|\"\"\"\\|\"\\|'")))))
  "Return a list of open/close string delimiter regexps for `hypb:in-string-p'.
Or clauses in regexps must be arranged from longest match to shortest match."
  :type 'sexp
  :group 'hyperbole-commands)

(defvar hypb:mail-address-mode-list
  '(fundamental-mode prog-mode text-mode)
  "List of major modes in which mail address implicit buttons are active.
Also active in any Decendent modes of those listed.")

(defconst hypb:mail-address-tld-regexp
  (format "\\.%s\\'"
          (regexp-opt
           '("aero" "arpa" "asia" "biz" "cat" "com" "coop" "edu" "gov" "info"
             "int" "jobs" "mil" "mobi" "museum" "name" "net" "org" "pro" "tel"
             "travel" "uucp"
             "ac" "ad" "ae" "af" "ag" "ai" "al" "am" "an" "ao" "aq"
             "ar" "as" "at" "au" "aw" "ax" "az" "ba" "bb" "bd" "be" "bf" "bg" "bh"
             "bi" "bj" "bl" "bm" "bn" "bo" "br" "bs" "bt" "bv" "bw" "by" "bz" "ca"
             "cc" "cd" "cf" "cg" "ch" "ci" "ck" "cl" "cm" "cn" "co" "cr" "cu" "cv"
             "cx" "cy" "cz" "de" "dj" "dk" "dm" "do" "dz" "ec" "ee" "eg" "eh" "er"
             "es" "et" "eu" "fi" "fj" "fk" "fm" "fo" "fr" "ga" "gb" "gd" "ge" "gf"
             "gg" "gh" "gi" "gl" "gm" "gn" "gp" "gq" "gr" "gs" "gt" "gu" "gw" "gy"
             "hk" "hm" "hn" "hr" "ht" "hu" "id" "ie" "il" "im" "in" "io" "iq" "ir"
             "is" "it" "je" "jm" "jo" "jp" "ke" "kg" "kh" "ki" "km" "kn" "kp" "kr"
             "kw" "ky" "kz" "la" "lb" "lc" "li" "lk" "lr" "ls" "lt" "lu" "lv" "ly"
             "ma" "mc" "md" "me" "mf" "mg" "mh" "mk" "ml" "mm" "mn" "mo" "mp" "mq"
             "mr" "ms" "mt" "mu" "mv" "mw" "mx" "my" "mz" "na" "nc" "ne" "nf" "ng"
             "ni" "nl" "no" "np" "nr" "nu" "nz" "om" "pa" "pe" "pf" "pg" "ph" "pk"
             "pl" "pm" "pn" "pr" "ps" "pt" "pw" "py" "qa" "re" "ro" "rs" "ru" "rw"
             "sa" "sb" "sc" "sd" "se" "sg" "sh" "si" "sj" "sk" "sl" "sm" "sn" "so"
             "sr" "st" "su" "sv" "sy" "sz" "tc" "td" "tf" "tg" "th" "tj" "tk" "tl"
             "tm" "tn" "to" "tp" "tr" "tt" "tv" "tw" "tz" "ua" "ug" "uk" "um" "us"
             "uy" "uz" "va" "vc" "ve" "vg" "vi" "vn" "vu" "wf" "ws" "ye" "yt" "yu"
             "za" "zm" "zw")
           t))
  "Regular expression of most common Internet top level domain names.")

(defconst hypb:mail-address-regexp
  "\\([_a-zA-Z0-9][-_a-zA-Z0-9.!@+%]*@[-_a-zA-Z0-9.!@+%]+\\.[a-zA-Z0-9][-_a-zA-Z0-9]+\\)\\($\\|[^a-zA-Z0-9@%]\\)"
  "Regexp with group 1 matching an Internet email address.")

;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

;; Adapted from "subr.el" but doesn't add if ELEMENT already exists
;; Used in `kotl-mode', so autoload.
;;;###autoload
(defun hypb:add-to-invisibility-spec (element)
  "Add ELEMENT to `buffer-invisibility-spec'.
See documentation for `buffer-invisibility-spec' for the kind of elements
that can be added.

If `buffer-invisibility-spec' isn't a list before calling this
function, `buffer-invisibility-spec' will afterwards be a list
with the value `(t ELEMENT)'.  This means that if text exists
that invisibility values that aren't either t or ELEMENT, that
text will become visible."
  (if (eq buffer-invisibility-spec t)
      (setq buffer-invisibility-spec (list t)))
  (unless (member element buffer-invisibility-spec)
    (setq buffer-invisibility-spec
	  (cons element buffer-invisibility-spec))))

;;;###autoload
(defun hypb:activate-interaction-log-mode ()
  "Configure and enable the interaction-log package for use with Hyperbole.
This displays a clean log of Emacs keys used and commands executed."
  (interactive)
  ;; Ensure package is installed
  (unless (package-installed-p 'interaction-log)
    (package-install 'interaction-log))

  ;; Ensure interaction-log-mode is disabled to removes its command
  ;; hooks which are replaced below.
  (require 'interaction-log)
  (interaction-log-mode 0)

  ;; Optional binding you can enable to display the ilog buffer
  ;; (global-set-key
  ;;          (kbd "C-h C-l")
  ;;          (lambda () (interactive) (display-buffer ilog-buffer-name)))

  ;; Display source code lambdas only
  (setq ilog-print-lambdas 'not-compiled)

  ;; Omit display of some lower-level Hyperbole commands for cleaner logs
  (mapc (lambda (cmd-str) (cl-pushnew (format "^%s$" cmd-str) ilog-self-insert-command-regexps))
        '("hyperbole" "hui:menu-enter"))

  ;; Redefine the mode to display commands on post-command-hook rather
  ;; than pre-command-hook since Hyperbole rewrites some command names
  ;; and key sequences.
  (define-minor-mode interaction-log-mode
    "Global minor mode logging keys, commands, file loads and messages.
	   Logged stuff goes to the *Emacs Log* buffer."
    :group 'interaction-log
    :lighter nil
    :global t
    :after-hook interaction-log-mode-hook
    (if interaction-log-mode
	(progn
	  (add-hook 'after-change-functions #'ilog-note-buffer-change)
	  (add-hook 'post-command-hook      #'ilog-record-this-command)
	  (add-hook 'post-command-hook      #'ilog-post-command)
	  (setq ilog-truncation-timer (run-at-time 30 30 #'ilog-truncate-log-buffer))
	  (setq ilog-insertion-timer (run-with-timer ilog-idle-time ilog-idle-time
						     #'ilog-timer-function))
	  (message "Interaction Log: started logging in %s" ilog-buffer-name))
      (remove-hook 'after-change-functions #'ilog-note-buffer-change)
      (remove-hook 'post-command-hook      #'ilog-record-this-command)
      (remove-hook 'post-command-hook      #'ilog-post-command)
      (when (timerp ilog-truncation-timer) (cancel-timer ilog-truncation-timer))
      (setq ilog-truncation-timer nil)
      (when (timerp ilog-insertion-timer) (cancel-timer ilog-insertion-timer))
      (setq ilog-insertion-timer nil)))

  ;; Define this function to display a 41 character wide ilog frame
  ;; at the right of the screen with other frame parameters that match
  ;; the frame selected when this is called.
  (defun ilog-show-in-other-frame ()
    "Display ilog in a separate frame of width 41 with parameters of selected frame.
Raise and reuse any existing single window frame displaying ilog."
    (interactive)
    (require 'hycontrol)
    (with-selected-window (selected-window)
      (let* ((win (get-buffer-window ilog-buffer-name t))
	     (frame (when win (window-frame win))))
	(if (and frame (= (with-selected-frame frame (count-windows)) 1))
	    (raise-frame frame)
	  (unless interaction-log-mode (interaction-log-mode 1))
	  (let ((params (frame-parameters)))
	    (setcdr (assq 'width params) 41)
	    (setq win (display-buffer-pop-up-frame
		       (get-buffer ilog-buffer-name)
		       (list (cons 'pop-up-frame-parameters params))))
	    (set-window-dedicated-p win t)
	    (with-selected-frame (window-frame win)
	      (hycontrol-frame-to-right-center))
	    win)))))

  ;; Enable the mode
  (interaction-log-mode 1)

  ;; Limit display to commands executed
  (with-current-buffer (get-buffer-create ilog-buffer-name)
    (setq ilog-display-state 'messages)
    ;; Changes to command-only mode
    (ilog-toggle-view)
    (message ""))

  (ilog-show-in-other-frame))

(defmacro hypb:assert-same-start-and-end-buffer (&rest body)
  "Assert that current buffer does not change following execution of BODY.
Trigger an error with traceback if the buffer is not live or its
name differs at the start and end of BODY."
  (declare (indent 0) (debug t))
  `(let ((debug-on-error t)
	 (start-buffer (current-buffer)))
     (unless (buffer-live-p start-buffer)
       (error "Start buffer, '%s', is not live" (current-buffer)))
     ;; `kill-buffer' can change current-buffer in some odd cases.
     (unwind-protect
	 (progn ,@body)
       (unless  (eq start-buffer (current-buffer))
	 (error "Start buffer, '%s', differs from end buffer, '%s'" start-buffer (current-buffer)))
       (unless (buffer-live-p start-buffer)
	 (error "End buffer, '%s', is not live" (current-buffer))))))

;;;###autoload
(defun hypb:buffer-file-name (&optional buffer)
  "Return name of file BUFFER or current buffer is visiting; nil if none.
No argument or nil as argument means use the current buffer.
Produces the correct file name for indirect buffers as well."
   (buffer-file-name (or (buffer-base-buffer buffer) buffer)))

(defun hypb:call-process-p (program &optional infile predicate &rest args)
  "Call an external PROGRAM with INFILE for input.
If PREDICATE is given, it is evaluated in a buffer with the PROGRAM's
output and the result returned.  If PREDICATE is nil, returns t iff
program has no output or just a 0-valued output.
Rest of ARGS are passed as arguments to PROGRAM."
  (let ((buf (get-buffer-create "*test-output*"))
	(found))
    (with-current-buffer buf
      (setq buffer-read-only nil)
      (erase-buffer)
      (apply 'call-process program infile buf nil args)
      (setq found
	    (if predicate
		(eval predicate)
	      (or (= (point-max) 1) ;; No output, consider cmd a success.
		  (and (< (point-max) 4)
		       (string= (buffer-substring 1 2) "0")))))
      (set-buffer-modified-p nil)
      (kill-buffer buf))
    found))

(defun hypb:char-count (char array)
  "Return count of occurrences of CHAR in ARRAY."
  (let ((i 0) (c 0) (l (length array)))
    (while (< i l)
      (when (= char (aref array i)) (setq c (1+ c)))
      (setq i (1+ i)))
    c))

(defun hypb:chmod (op octal-permissions file)
  "Use OP and OCTAL-PERMISSIONS integer to set FILE permissions.
OP may be +, -, xor, or default =."
  (let ((func (cond ((eq op '+)   #'logior)
		    ((eq op '-)   (lambda (p1 p2) (logand (lognot p1) p2)))
		    ((eq op 'xor) #'logxor)
		    (t            (lambda (p1 p2) p2 p1)))))
    (set-file-modes file (funcall func (hypb:oct-to-int octal-permissions)
				  (file-modes file)))))

(defun hypb:cmd-key-series (cmd-sym &optional keymap)
  "Return a brace-delimited, human readable key sequence string bound to CMD-SYM.
Global keymap is used unless optional KEYMAP is given.

Trigger an error if CMD-SYM is not bound."
  (if (and cmd-sym (symbolp cmd-sym) (fboundp cmd-sym))
      (let* ((get-keys (lambda (cmd-sym keymap)
		         (key-description (where-is-internal
				           cmd-sym keymap 'first))))
	     (keys (funcall get-keys cmd-sym keymap)))
        (concat "{"
	        (if (string= keys "")
		    (concat (funcall get-keys 'execute-extended-command nil)
			    " " (symbol-name cmd-sym) " RET")
	          keys)
	        "}"))
    (error "(hypb:cmd-key-series): Invalid cmd-sym arg: %s" cmd-sym)))

(defun hypb:cmd-key-vector (cmd-sym &optional keymap)
  "Return as a vector the first key sequence bound to CMD-SYM.
Search global keymap or optional KEYMAP.  Return nil if no valid
key binding is found.

The returned value may be compared with `equal' to `this-single-command-keys'.
Use `key-description' to make it human readable."
  (where-is-internal cmd-sym keymap t))

;;;###autoload
(defun hypb:count-visible-windows ()
  "Return the number of visible, non-minibuffer windows across all frames."
  (apply '+ (mapcar (lambda (frm) (length (window-list frm)))
		    (visible-frame-list))))
;;;###autoload
(defun hypb:configuration (&optional out-buf)
  "Insert Emacs configuration information at the end of a buffer.
Use optional OUT-BUF if present, else the current buffer."
  (save-excursion
    (and out-buf (set-buffer out-buf))
    (goto-char (point-min))
    (if (re-search-forward mail-header-separator nil t)
	(forward-line 1)
      (goto-char (point-max)))
    (delete-blank-lines) (delete-blank-lines)
    (let ((start (point)))
      (insert (format "I use:\tEditor:      GNU Emacs %s %s\n\tHyperbole:   %s\n"
		      emacs-version
                      (if (and (fboundp #'native-comp-available-p)
                               (native-comp-available-p))
                          "with native comp"
                        "")
                      hyperb:version))
      (when (and (boundp 'br-version) (stringp br-version))
	(insert (format "\tOO-Browser:  %s\n" br-version)))
      (when (and (boundp 'system-configuration) (stringp system-configuration))
	(insert (format "\tSys Type:    %s\n" system-configuration)))
      (insert (format "\tOS Type:     %s\n\tWindow Sys:  %s\n"
                      system-type (or window-system (hyperb:window-system)
				      "None")))
      (when (and (boundp 'hmail:reader) hmail:reader)
        (insert (format "\tMail Reader: %s\n"
                        (cond ((eq hmail:reader 'rmail-mode) "RMAIL")
                              ((and (eq hmail:reader 'mh-show-mode)
                                    (string-match "v ?\\([0-9]+.[0-9]+\\)"
                                                  mh-e-RCS-id))
                               (concat "MH-e "
                                       (substring mh-e-RCS-id
                                                  (match-beginning 1)
                                                  (match-end 1))))
                              ((eq hmail:reader 'pm-fdr-mode)
                               (concat "PIEmail " pm-version))))))
      (when (and (boundp 'hnews:reader) (boundp 'gnus-version) hnews:reader)
        (insert (format "\tNews Reader: %s\n" gnus-version)))
      (let ((install-type (hypb:installation-type)))
        (when install-type
          (insert (format "\tInstall:     %s, %s" (car install-type) (cadr install-type)))))
      (insert "\n")
      ;; Insert recent Hyperbole debugging messages if any.
      (when (messages-buffer)
	(let ((opoint (point)))
	  (insert-buffer-substring (buffer-name (messages-buffer)))
	  (keep-lines "^(HyDebug)" opoint (point))))
      (untabify start (point)))))

(defun hypb:debug ()
  "Load Hyperbole hbut.el source file and set debugging traceback flag."
  (interactive)
  (or (featurep 'hinit) (load "hyperbole"))
  (or (and (featurep 'hbut)
	   (let ((func (hypb:indirect-function 'ebut:create)))
	     (not (or (subrp func)
		      (byte-code-function-p func)
		      (eq 'byte-code
			  (car (car (nthcdr 3 (hypb:indirect-function
					       'ebut:create)))))))))
      (load "hbut.el"))
  (setq debug-on-error t))

;; Copied from eww.el, eww-decode-url-file-name, so as to not require
;; that package.
(defun hypb:decode-url (url-file-name)
  "Decode a URL-FILE-NAME."
  (let* ((binary (url-unhex-string url-file-name))
         (decoded
          (decode-coding-string
           binary
           ;; Possibly set by `universal-coding-system-argument'.
           (or coding-system-for-read
               ;; RFC 3986 says that %AB stuff is utf-8.
               (if (equal (decode-coding-string binary 'utf-8)
                          '(unicode))
                   'utf-8
                 ;; But perhaps not.
                 (car (detect-coding-string binary))))))
         (encodes (find-coding-systems-string decoded)))
    (if (or (equal encodes '(undecided))
            (memq (coding-system-base (or file-name-coding-system
                                          default-file-name-coding-system))
                  encodes))
        decoded
      ;; If we can't encode the decoded file name (due to language
      ;; environment settings), then we return the original, hexified
      ;; string.
      url-file-name)))

;; Similar keyboard macro to next function, but less flexible: {C-x 1 M-o F M-o a C-x b *scratch* RET M-< M-o s C-@ C-M-h M-o t a C-u C-@ C-u C-@ M-o a C-M-p}

;;;###autoload
(defun hypb:def-to-buffer (&optional arg buffer)
  "Copy next optional ARG code definitions to the start of optional BUFFER.
Default ARG is 1 and default BUFFER is \"*scratch*\".  Leave
point at the start of the inserted text."
  (interactive "p\nbDef insertion buffer (default *scratch*): ")
  (let ((def (save-excursion
	       (mark-defun arg)
	       (deactivate-mark)
	       (buffer-substring (region-beginning) (region-end)))))
    (pop-to-buffer (or buffer "*scratch*"))
    (goto-char (point-min))
    (insert def)
    (goto-char (point-min))
    (forward-line 1)))

;;;###autoload
(defun hypb:devdocs-lookup ()
  "Prompt for and display a devdocs.io docset section within Emacs.
This will install the Emacs devdocs package if not yet installed."
  (interactive)
  (hypb:require-package 'devdocs)
  ;; (call-interactively #'devdocs-install)
  (devdocs-lookup))

(defun hypb:domain-name ()
  "Return current Internet domain name with '@' prepended or nil if none."
  (let* ((dname-cmd (or (file-exists-p "/usr/bin/domainname")
			(file-exists-p "/bin/domainname")))
	 (dname (or (and (boundp 'message-user-fqdn) (stringp message-user-fqdn)
			 (string-match "\\." message-user-fqdn)
			 message-user-fqdn)
		    (getenv "DOMAINNAME")
		    (when dname-cmd
		      (hypb:call-process-p
		       "domainname" nil
		       '(substring (buffer-string) 0 -1)))))
	 host-and-domain)
    (when (or (and dname (string-match "\\." dname))
	      (and (setq host-and-domain (hypb:call-process-p
					  "hostname" nil '(substring (buffer-string) 0 -1) "-f"))
		   (setq dname (when (string-match "\\`[^.]+\\." host-and-domain)
				 (substring host-and-domain (match-end 0)))))
	      (let* ((src "/etc/resolv.conf")
		     (src-buf-exists-p (get-file-buffer src)))
	        (and (file-exists-p src) (file-readable-p src)
		     (with-temp-buffer
		       (insert-file-contents-literally src)
		       (goto-char (point-min))
		       (when (re-search-forward  "^domain[ \t]+\\([^ \t\n\r]+\\)" nil t)
			 (setq dname (match-string 1)))
		       (or src-buf-exists-p (kill-buffer nil))
		       dname))))
      (concat "@" dname))))

(defun hypb:empty-file-p ()
  "Return non-nil if the current buffer has an attached file of zero size."
  (when (and (hypb:buffer-file-name) (file-readable-p (hypb:buffer-file-name)))
    (= (file-attribute-size (file-attributes buffer-file-name)) 0)))

(defun hypb:error (&rest args)
  "Signal an error typically to be caught by `hyperbole'.
The error message is formatted passing the rest of the ARGS to
the `format' function."
  (let ((msg (if (< (length args) 2)
		 (car args)
	       (apply 'format (cons (car args)
				    (mapcar #'hypb:format-quote (cdr args)))))))
    (put 'error 'error-message msg)
    (error msg)))

(defun hypb:eval (sexp &rest rest)
  "Apply SEXP to REST of arguments and maintain the current buffer."
  (let ((buf (current-buffer))
	(cmd (cond ((symbolp sexp)
		    sexp)
		   ((listp sexp)
		    (if (eq 'quote (car sexp))
			;; Unquote the expression so it is evaluated
			(cadr sexp)
		      sexp)))))
    (setq last-command this-command
	  this-command (if (and (listp cmd) (symbolp (car cmd)))
			   (car cmd)
			 cmd))
    (run-hooks 'pre-command-hook)
    (unwind-protect
	(command-execute
	 (lambda () (interactive)
	   (if rest
	       (apply cmd rest)
	     (eval cmd t))))
      ;; Ensure point remains in the same buffer before and after SEXP
      ;; evaluation.  This prevents false switching to the *ert* test
      ;; buffer when debugging.
      (set-buffer buf)
      ;; Comment this out as it triggered an error in CI/CD
      ;; (when (memq this-command (list 'self-insert-command
      ;; 				     (key-binding [remap self-insert-command])))
      ;; 	(run-hooks 'post-self-insert-hook))
      (run-hooks 'post-command-hook))))

(defun hypb:eval-debug (sexp)
  "Eval SEXP and on error, show a debug backtrace of the problem."
  (let ((debug-on-error t)
	(debug-on-quit t))
    (eval sexp)))

(defun hypb:fgrep-git-log (string)
  "List git log entries whose changesets include STRING for selection and display.
Listing is asynchronous.  A press of RET, the Action Key or the
Assist Key on any log line will display its committed changes."
  (interactive "sFgrep git commits containing: ")
  (compile (format "git log -S'%s' --line-prefix='commit ' --oneline" string)
	   #'hypb:fgrep-git-log-mode))

(defun hypb:fgrep-git-log-activate (_ignore1 &optional _ignore2)
  "Display git commit for the current line when `compile-goto-error' {RET} is used.
Does not support use of next and previous error; simply displays
the current one."
  (interactive '(nil))
  (hkey-either nil))

(define-derived-mode hypb:fgrep-git-log-mode compilation-mode "Fgrep-Git-Log"
  "Major mode for listing a matching set of git commits for selection and display.
Mode is derived from `compilation-mode'.  Turning on
Fgrep-Git-Log mode runs the normal hook `compilation-mode-hook'."
  (setq-local next-error-function #'hypb:fgrep-git-log-activate))

(defun hypb:file-major-mode (file)
  "Return the major mode used by FILE.
FILE is temporarily read into a buffer to determine the major mode if necessary."
  (let ((existing-flag (get-file-buffer file))
	(buf (find-file-noselect file)))
    (prog1 (when buf (save-excursion (with-current-buffer buf
				       major-mode)))
      (unless (or existing-flag (null buf))
	(kill-buffer buf)))))

(defun hypb:filter-directories (file-regexp &rest dirs)
  "Filter files to those matching FILE-REGEXP from rest of DIRS (recursively).
Also filters out any files matching `completion-ignored-extensions' or
ending with # or ~.
Return a flattened list of all matching files."
  (setq file-regexp (hypb:glob-to-regexp file-regexp))
  (setq dirs (hypb:readable-directories dirs))
  (delq nil (mapcar (lambda (f)
		      (unless (string-match-p
			       (concat (regexp-opt (append completion-ignored-extensions '("#" "~"))
						   'paren) "$") f)
			f))
		    (apply #'nconc (mapcar (lambda (dir) (directory-files-recursively dir file-regexp))
					   dirs)))))

(defun hypb:format-args (args)
  "Return a space-separated string of quoted ARGS without surrounding parentheses."
  (if args (mapconcat (lambda (a) (format "%S" a)) args " ") ""))

(defun hypb:format-quote (arg)
  "Replace all single % with %% in any string ARG.
This is so that a call to `format' or `message' ignores them.
Return either the modified string or the original ARG when not
modified."
  (if (stringp arg)
      (replace-regexp-in-string
       "@@@" "%%" (replace-regexp-in-string
		   "%" "%%" (replace-regexp-in-string "%%" "@@@" arg nil t)
		   nil t)
       nil t)
    arg))

;;;###autoload
(defun hypb:function-p (func)
  "Return non-nil if FUNC is a valid function, subroutine, or closure."
  (or (subrp func) (byte-code-function-p func)
      (and (symbolp func) (fboundp func))
      (and (listp func) (memq (car func) '(closure lambda)))
      (and (fboundp 'closurep) (closurep func))))

;; Extracted from part of `choose-completion' in "simple.el"
(defun hypb:get-completion (&optional event)
  "Return the completion at point.
If EVENT, use EVENT's position to determine the starting position."
  (interactive (list last-nonmenu-event))
  ;; In case this is run via the mouse, give temporary modes such as
  ;; isearch a chance to turn off.
  (run-hooks 'mouse-leave-buffer-hook)
  (with-current-buffer (window-buffer (posn-window (event-start event)))
    (save-excursion
      (goto-char (posn-point (event-start event)))
      (let (beg end)
        (cond
         ((and (not (smart-eobp)) (get-text-property (point) 'mouse-face))
          (setq end (point) beg (1+ (point))))
         ((and (not (bobp))
               (get-text-property (1- (point)) 'mouse-face))
          (setq end (1- (point)) beg (point)))
         (t (error "No completion here")))
        (setq beg (previous-single-property-change beg 'mouse-face))
        (setq end (or (next-single-property-change end 'mouse-face)
                      (point-max)))
        (buffer-substring-no-properties beg end)))))

(defun hypb:get-raw-syntax-descriptor (char &optional syntax-table)
  "Return the raw syntax descriptor for CHAR.
Use the current syntax table or optional SYNTAX-TABLE."
  (aref (or syntax-table (syntax-table)) char))

(defun hypb:glob-to-regexp (str)
  "Convert any file glob syntax in STR to Emacs regexp syntax."
  (when (stringp str)
    (setq str (replace-regexp-in-string "\\`\\*" ".*" str nil t)
	  str (replace-regexp-in-string "\\([^\\.]\\)\\*" "\\1.*" str))
    (when (and (not (string-match-p "\\(\\$\\|\\\\'\\)\\'" str))
	       (string-match-p "\\.\\S-+\\'" str))
      (setq str (concat str "$"))))
    str)

;; Derived from pop-global-mark of "simple.el" in GNU Emacs.
(defun hypb:goto-marker (marker)
  "Make MARKER's buffer and position current.
If MARKER is invalid signal an error."
  (cond ((not (markerp marker))
	 (error "Invalid marker: %s" marker))
	((not (marker-buffer marker))
	 (error "Invalid marker buffer: %s" marker))
	(t (let* ((buffer (marker-buffer marker))
		  (position (marker-position marker)))
	     (set-buffer buffer)
	     (unless (and (>= position (point-min))
			  (<= position (point-max)))
	       (if widen-automatically
		   (widen)
		 (error "Marker position is outside accessible part of buffer: %s" marker)))
	     (goto-char position)
	     (switch-to-buffer buffer)))))

(defun hypb:grep-git-log (regexp)
  "List git log entries whose changesets include REGEXP for selection and display.
Listing is asynchronous.  A press of RET, the Action Key or the
Assist Key on any log line will display its committed changes."
  (interactive "sGrep git commits containing: ")
  (compile (format "git log -G'%s' --line-prefix='commit ' --oneline" regexp)))

(defun hypb:help-buf-name (&optional suffix)
  "Return a Hyperbole help buffer name for current buffer.
With optional SUFFIX string, uses it rather than buffer name."
  (let ((bn (or suffix (buffer-name))))
    (if (string-match (regexp-quote hypb:help-buf-prefix) bn)
	(buffer-name (generate-new-buffer bn))
      (concat hypb:help-buf-prefix bn "*"))))

;;;###autoload
(defun hypb:helm-apropos (&optional symbol-name)
  "Prompt for and display the doc for a command, function, variable or face.
With optional SYMBOL-NAME non-nil, display the doc for that.
This will this install the Emacs helm package when needed."
  (interactive "P")
  (hypb:require-package 'helm)
  (helm-apropos symbol-name))

;;;###autoload
(defun hypb:helm-info (&optional refresh)
  "Prompt across all Info manuals and display the node selected.
With optional prefix arg REFRESH non-nil, refresh the cache of Info manuals.
This will this install the Emacs helm package when needed."
  (interactive "P")
  (hypb:require-package 'helm)
  (helm-info refresh))

(defun hypb:hkey-help-file ()
  "Return the full path to the Hyperbole mouse key help file."
  (cond ((and (fboundp 'locate-data-file)
	      (locate-data-file "hkey-help.txt")))
	(t (let* ((hypb-man (expand-file-name "man/" hyperb:dir))
		  (help-file (expand-file-name "hkey-help.txt" hypb-man)))
	     (if (or (file-exists-p help-file)
		     (file-exists-p
		      (setq help-file (expand-file-name
				       "hkey-help.txt" data-directory))))
		 help-file
	       (error "(hypb:hkey-help-file): Non-existent file: \"%s\""
		      help-file))))))

(defun hypb:in-string-p (&optional max-lines range-flag)
  "Return non-nil iff point is within a string and not on the closing quote.

With optional MAX-LINES, an integer, match only within that many
lines from point.  With optional RANGE-FLAG when there is a
match, return list of (string-matched start-pos end-pos), where
the positions exclude the delimiters.

To prevent searching back to the buffer start and producing slow
performance, this limits its count of quotes found prior to point
to the beginning of the first line prior to point that contains a
non-quoted quote mark.

Quoting conventions recognized are:
  double-quotes:                 \"str\";
  Markdown triple backticks:     ```str```;
  Python single-quotes:          \\='str\\=';
  Python triple single-quotes:   '''str''';
  Python triple double-quotes:   \"\"\"str\"\"\";
  Texinfo open and close quotes: ``str''."
  (save-excursion
    (save-restriction
      (when (integerp max-lines)
	(if (zerop max-lines)
	    (narrow-to-region (point) (point)) ;; Empty range
	  ;; Allow for +/- (+ 1 max-lines) including current line so start
	  ;; and end delimiters can be on separate lines.  Before returning,
	  ;; this function checks that any matched string has <= max-lines.
	  (narrow-to-region (line-beginning-position
			     (when max-lines (1+ (- max-lines))))
			    (line-end-position (1+ max-lines)))))
      ;; Don't use `syntax-ppss' here as it fails to ignore backquoted
      ;; double quote characters in strings and doesn't work in
      ;; `change-log-mode' due to its syntax-table.
      (let ((opoint (point))
	    (start (point-min))
	    (open-match-string "")
	    str
	    str-start
	    str-end)
	(cl-destructuring-bind (open-regexp close-regexp)
	    (eval hypb:in-string-modes-regexps)
	  (save-match-data
	    (when (and (re-search-backward open-regexp nil t)
		       (setq open-match-string (match-string 2)
			     str-start (match-end 2))
		       ;; If this is the start of a string, it must be
		       ;; at the start of line, preceded by whitespace
		       ;; or preceded by another string end sequence.
		       ;; (save-match-data
		       ;; 	 (or (string-empty-p (match-string 1))
		       ;; 	     (string-search (match-string 1) " \t\n\r\f")
		       ;; 	     (progn (goto-char (1+ (point)))
		       ;; 		    (looking-back close-regexp nil))))
		       )
	      (forward-line 0)
	      (setq start (point))
	      (goto-char opoint)
	      (if (and (derived-mode-p 'texinfo-mode)
		       (string-equal open-match-string texinfo-open-quote))
		  (and (cl-oddp (- (count-matches (regexp-quote open-match-string)
						  start (point))
				   ;; Subtract any backslash quoted delimiters
				   (count-matches
				    (format "[\\]\\(%s\\)"
					    (regexp-quote open-match-string))
				    start (point))
				   (count-matches (regexp-quote texinfo-close-quote)
						  start (point))
				   ;; Subtract any backslash quoted delimiters
				   (count-matches
				    (format "[\\]\\(%s\\)"
					    (regexp-quote texinfo-close-quote))
				    start (point))))
		       (search-forward texinfo-close-quote nil t)
		       (setq str-end (match-beginning 0)
			     str (buffer-substring-no-properties str-start str-end)))
		(and (cl-oddp (- (count-matches (regexp-quote open-match-string)
						start (point))
				 ;; Subtract any backslash quoted delimiters
				 (count-matches
				  (format "[\\]\\(%s\\)"
					  (regexp-quote open-match-string))
				  start (point))))
		     ;; Move back one char in case point is on a
		     ;; closing delimiter char to ensure it is not
		     ;; backslash quoted and so the right delimiter is matched.
		     (if (/= (1- (point)) (line-beginning-position))
			 (goto-char (1- (point)))
		       t)
		     (re-search-forward close-regexp nil t)
		     (setq str-end (match-beginning 2)
			   str (buffer-substring-no-properties str-start str-end))))

	      ;; Ignore if more than `max-lines' matched
	      (when (and str
			 (or (null max-lines)
			     (and (integerp max-lines)
				  ;; When computing the number of lines in
				  ;; the string match, ignore any leading and
				  ;; trailing newlines.  This allows for
				  ;; opening and closing quotes to be on
				  ;; separate lines, useful with multi-line
				  ;; strings.
				  (< (hypb:string-count-matches
				      "\n" (string-trim str))
				     max-lines))))
		(if range-flag
		    (list str str-start str-end)
		  t)))))))))

(defun hypb:indirect-function (obj)
  "Return the function at the end of OBJ's function chain.
Resolves autoloadable function symbols properly."
  (let ((func (indirect-function obj)))
    ;; Handle functions with autoload bodies.
    (if (and (symbolp obj) (listp func) (eq (car func) 'autoload))
	(let ((load-file (car (cdr func))))
	  (load load-file)
	  ;; Prevent infinite recursion
	  (if (equal func (symbol-function obj))
	      (error "(hypb:indirect-function): Autoload of '%s' failed" obj)
	    (hypb:indirect-function obj)))
      func)))

(defun hypb:insert-region (buffer start end invisible-flag)
  "Insert into BUFFER the contents of the region from START to END.
Contents come from the current buffer.  INVISIBLE-FLAG, if
non-nil, means invisible text in an outline region is copied,
otherwise, it is omitted."
  (if invisible-flag
      ;; Skip hidden blank lines between cells but include hidden outline text.
      (while (< start end)
	(if (not (get-text-property start 'invisible))
	    (append-to-buffer buffer start (1+ start)))
	(setq start (1+ start)))
    ;; Skip both hidden blank lines between cells and hidden outline text.
    (while (< start end)
      (or (kview:char-invisible-p start) (append-to-buffer buffer start (1+ start)))
      (setq start (1+ start)))))

(defun hypb:installation-type ()
  "Return type of installation and version number.
Is a list of (hyperbole-installation-type-string
hyperbole-install-version-number-string).  If no matching
installation type is found, return a list of (\"unknown\"
`hyperb:dir')."
  (let ((hypb-dir-name (file-name-nondirectory (directory-file-name hyperb:dir)))
	(dir-sep-string (substring (file-name-as-directory ".") -1)))
    (cond
     ;; straight.el package install -- hyperbole gnu-elpa-mirror master 56cd3d8 2022-02-05
     ((string-match (concat dir-sep-string "straight" dir-sep-string
			    "build" dir-sep-string "hyperbole") hyperb:dir)
      (let* ((plist (hypb:straight-package-plist "hyperbole"))
	     (pkg-version (plist-get plist :version))
	     (git-commit (when (string-match " \\([a-f0-9]+\\) " pkg-version)
			   (match-string 1 pkg-version))))
	(list "straight" git-commit)))
     ;; elpa-devel package install -- hyperbole-7.0.0pre0.20220126.1138
     ((string-match "hyperbole-\\([.[:digit:]]+pre[.[:digit:]]+\\)" hypb-dir-name)
      (list "elpa-devel" (match-string 1 hypb-dir-name)))
     ;; melpa/quelpa package install -- hyperbole-20220205.1429
     ((string-match "hyperbole-\\([1-9][0-9][0-9][0-9][0-1][0-9][0-3][0-9]\\.[0-9]+\\)"
		    hypb-dir-name)
      (list "melpa" (match-string 1 hypb-dir-name)))
     ;; git install -- hyperbole d27f4c5197
     ((file-exists-p (expand-file-name ".git" hyperb:dir))
      (ignore-errors
        (let ((default-directory hyperb:dir))
          (list
           "git"
           (substring (shell-command-to-string "git rev-parse HEAD") 0 10)))))
     ;; elpa package install -- /elpa/hyperbole-8.0.0"
     ((and (string-match-p (concat dir-sep-string "elpa" dir-sep-string) hyperb:dir)
	   (string-match "hyperbole-\\([.[:digit:]]+\\)" hypb-dir-name))
      (list "elpa" (match-string 1 hypb-dir-name)))
     ;; tarball archive install -- hyperbole-8.0.0
     ((string-match "hyperbole-\\([.[:digit:]]+\\)" hypb-dir-name)
      (list "archive" (match-string 1 hypb-dir-name)))
     ;; unknown -- hyperbole
     (t (list "unknown" hyperb:dir)))))

;;;###autoload
(defun hypb:locate (search-string &optional filter arg)
  "Find file name match anywhere and put results in the `*Locate*' buffer.
Pass it SEARCH-STRING as argument.  Interactively, prompt for SEARCH-STRING.
With prefix arg ARG, prompt for the exact shell command to run instead.

This program searches for those file names in a database that match
SEARCH-STRING and normally outputs all matching absolute file names,
one per line.  The database normally consists of all files on your
system, or of all files that you have access to.  Consult the
documentation of the program for the details about how it determines
which file names match SEARCH-STRING.  (Those details vary highly with
the version.)

You can specify another program for this command to run by customizing
the variables `locate-command' or `locate-make-command-line'.

The main use of FILTER is to implement `locate-with-filter'.  See
the docstring of that function for its meaning.

After preparing the results buffer, this runs `dired-mode-hook' and
then `locate-post-command-hook'."
  (interactive (list (let ((default (symbol-at-point)))
		       (read-string (format "Locate files anywhere with names that match%s: "
					    (if default
						(format " (default %s)" default)
					      ""))
				    nil nil default))
		     nil
		     current-prefix-arg))
  (locate search-string filter arg))

;;;###autoload
(defun hypb:map-plist (func plist)
  "Apply FUNC of two args, key and value, to key-value pairs in PLIST."
  (unless (hypb:function-p func)
      (error "(hypb:map-plist): Invalid 'func' arg: %s" func))
  (unless (hypb:plist-p plist)
      (error "(hypb:map-plist): Invalid 'plist' arg: %s" plist))
  (cl-loop for (k v) on plist by #'cddr
	   collect (funcall func k v) into result
	   finally return result))

(defun hypb:map-vector (func object)
  "Return list of results of application of FUNC to each element of OBJECT.
OBJECT should be a vector or `byte-code' object."
  (unless (or (vectorp object) (byte-code-function-p object))
    (error "(hypb:map-vector): Second argument must be a vector or byte-code object"))
  (let ((end (length object))
	(i 0)
	(result))
    (while (< i end)
      (setq result (cons (funcall func (aref object i)) result)
	    i (1+ i)))
    (nreverse result)))

(defun hypb:mark-object (object)
  "Mark OBJECT as a Hyperbole object.
If possible to prevent generic functions from changing it.
OBJECT must be a non-empty string or a symbol or this has no effect."
  (cond ((and (stringp object) (not (string-empty-p object)))
	 (put-text-property 0 1 'hyperbole t object))
	((symbolp object)
	 (put object 'hyperbole t))))

;; Derived from "window.el".
(defun hypb:maximize-window-height (&optional window)
  "Maximize WINDOW.
Make WINDOW as large as possible without deleting any windows.
WINDOW must be a valid window and defaults to the selected one.

If the option `window-resize-pixelwise' is non-nil maximize
WINDOW pixelwise."
  (interactive)
  (setq window (window-normalize-window window))
  (window-resize
   window (window-max-delta window nil nil nil nil nil window-resize-pixelwise)
   nil nil window-resize-pixelwise))

(defun hypb:object-p (object)
  "Return t if OBJECT is marked as a Hyperbole object, else nil."
  (cond ((and (stringp object) (not (string-empty-p object)))
	 (get-text-property 0 'hyperbole object))
	((symbolp object)
	 (get object 'hyperbole))))

;;;###autoload
(defun hypb:plist-p (plist)
  "Return t if PLIST is a proper property list, else nil."
  (cl-evenp (% (or (proper-list-p plist) 1) 2)))

(defun hypb:readable-directories (&rest dirs)
  "Flatten rest of DIRS and return or error if any of DIRS are unreadable."
  (setq dirs (flatten-list dirs))
  (let ((unreadable-dirs (delq nil (mapcar (lambda (dir) (unless (file-readable-p dir) dir)) dirs))))
    (when unreadable-dirs
      (error "(hypb:readable-directories): These directories are not readable:\n%s"
	     (string-join unreadable-dirs "\n"))))
  dirs)

;;;###autoload
(defun hypb:require-package (package-name)
  "Prompt user to install, if necessary, and require the Emacs PACKAGE-NAME.
PACKAGE-NAME may be a symbol or a string."
  (when (stringp package-name)
    (setq package-name (intern package-name)))
  (unless (symbolp package-name)
    (error "(hypb:require-package): package-name must be a symbol or string, not '%s'" package-name))
  (unless (package-installed-p package-name)
    (if (y-or-n-p (format "Install package `%s' required by this command?" package-name))
	(package-install package-name)
      (keyboard-quit)))
  (require package-name))

;; Adapted from cl--do-remf in "cl-extra.el" but uses 'equal' for comparisons.
;;;###autoload
(defun hypb:do-remove-from-plist (plist name)
  "Remove from property list PLIST a NAME string."
  (let ((p (cdr plist)))
    ;; Can't use `plist-member' here because it goes to the cons-cell
    ;; of NAME and we need the one before.
    (while (and (cdr p) (not (equal (cadr p) name)))
      (setq p (cddr p)))
    (and (cdr p) (progn (setcdr p (cdddr p)) t))))

;; Adapted from cl-remf in "cl-macs.el" but uses 'equal' for comparisons.
;;;###autoload
(defmacro hypb:remove-from-plist (place name)
  "Remove from property list PLACE a NAME string.
PLACE may be a symbol, or any generalized variable allowed by
`setf'.  The form generated by the macro returns true if NAME was
found and removed, nil otherwise."
  (declare (debug (place form)))
  (gv-letplace (tval setter) place
    (macroexp-let2 macroexp-copyable-p tname name
      `(if (equal ,tname (car ,tval))
           (progn ,(funcall setter `(cddr ,tval))
                  t)
         (hypb:do-remove-from-plist ,tval ,tname)))))

(defun hypb:remove-lines (regexp)
  "Remove lines containing match for REGEXP.
Apply within an active region or to the end of buffer."
    (interactive "sRemove lines with match for regexp: ")
    (flush-lines regexp nil nil t))

(defun hypb:return-process-output (program &optional infile &rest args)
  "Return as a string the output from external PROGRAM with INFILE for input.
Rest of ARGS are passed as arguments to PROGRAM.
Removes any trailing newline at the end of the output."
  (let ((buf (get-buffer-create "*test-output*"))
	(output))
    (with-current-buffer buf
      (setq buffer-read-only nil)
      (erase-buffer)
      (apply 'call-process program infile buf nil args)
      (setq output (buffer-string))
      ;; Remove trailing newline from output.
      (when (> (length output) 0)
        (setq output (substring output 0 -1)))
      (set-buffer-modified-p nil)
      (kill-buffer buf))
    output))

;;;###autoload
(defalias 'hypb:rgrep 'hui-select-rgrep)

(defun hypb:save-lines (regexp)
  "Save only lines containing match for REGEXP.
Apply within an active region or to the end of buffer."
    (interactive "sSave lines with match for regexp: ")
    (keep-lines regexp nil nil t))

(defmacro hypb:save-selected-window-and-input-focus (&rest body)
  "Execute BODY, restore selected windows in frames and frame with input focus.
The value returned is the value of the last form in BODY."
  `(let ((frame (selected-frame)))
     (prog1 (save-selected-window ,@body)
       (select-frame-set-input-focus frame))))

(defun hypb:select-window-frame (window)
  "Select WINDOW and its frame (set input focus there)."
  (if (window-live-p window)
      (progn (select-window window)
	     (select-frame-set-input-focus (window-frame window)))
    (error "(hypb:select-window-frame): Argument must be a live window, not '%s'" window)))

(defun hypb:set-raw-syntax-descriptor (char raw-descriptor &optional syntax-table)
  "Set the syntax of CHAR to RAW-DESCRIPTOR (syntax table value).
Set in the current syntax table or optional SYNTAX-TABLE.  Return
the RAW-DESCRIPTOR.  Use the `syntax-after' function to retrieve
the raw descriptor for a buffer position.

Similar to `modify-syntax-entry' but uses a raw descriptor
previously extracted from a syntax table to set the value rather
than a string.

Syntax tables are char-tables whose values are encoded as raw
descriptors."
  (aset (or syntax-table (syntax-table)) char raw-descriptor))

(defun hypb:split-seq-into-sublists (seq size)
  "Split a sequence SEQ into sublists of length SIZE, preserving item order."
  (let (result)
    (while (> (length seq) 0)
      (push (seq-into (seq-take seq size) 'list) result)
      (setq seq (seq-drop seq size)))
    (nreverse result)))

(defun hypb:straight-package-plist (pkg-string)
  "Return package info for a straight.el built package with name PKG-STRING.
The package info is a property list of package-name,
package-download-source and package-version for PKG-STRING, else
return nil.  This is for the straight.el package manager."
  (when (fboundp 'straight-bug-report-package-info)
    (car (delq nil (mapcar (lambda (pkg-plist)
			     (when (equal (plist-get pkg-plist :package) pkg-string) pkg-plist))
			   (straight-bug-report-package-info))))))

(defun hypb:string-count-matches (regexp str &optional start end)
  "Count occurrences of REGEXP in STR, limited to optional START and END positions.

START is inclusive and indexed from 0; END is exclusive.

This function starts looking for the next match from the end of the
previous match.  Hence, it ignores matches that overlap a previously
found match."
  (let ((str-len (length str))
	(count 0)
	substr)
    (when (and start (or (>= start str-len) (< start 0)))
      (error "(hypb:string-count-matches): 'start' (%d) must be >= 0 and < str length (%d)"
	     start str-len))
    (when (and end (or (> end str-len) (< end 0)))
      (error "(hypb:string-count-matches): 'end' (%d) must be >= 0 and <= str length (%d)"
	     end str-len))
    (setq start (or start 0)
	  end (or end str-len)
	  substr (substring str start end)
	  end (- end start)
	  start 0)
    (while (and (< start str-len)
		(string-match regexp substr start))
      (setq count (1+ count)
	    start (match-end 0)))
    count))

(defun hypb:supercite-p ()
  "Return non-nil iff the Emacs add-on supercite package is in use."
  (let (hook-val)
    (when (memq t (mapcar
		   (lambda (hook-var)
		     (and (boundp hook-var)
			  (progn (setq hook-val (symbol-value hook-var))
			         (cond ((listp hook-val)
				        (when (memq 'sc-cite-original hook-val)
					  t))
				       ((eq hook-val 'sc-cite-original))))))
		   '(mail-citation-hook mail-yank-hooks)))
      t)))

(defun hypb:toggle-isearch-invisible (&optional arg)
  "Toggle interactive invisible searching on or off.
This determines whether to search inside invisible text or not.
Toggles the variable `isearch-invisible' between values
nil and a non-nil value of the option `search-invisible’
\(or `open' if `search-invisible' is nil).

With optional prefix ARG > 0, turn on searching invisible text.
If ARG <= 0, turn search only visible text."
  (interactive "P")
  (if (not (boundp 'isearch-invisible))
      (error "(hypb:toggle-isearch-invisible): Feature not supported by the version of Emacs")
    (setq isearch-invisible (if (if (null arg)
				    (not isearch-invisible)
				  (> (prefix-numeric-value arg) 0))
				(or search-invisible 'open)))
    (message "I-search will %ssearch invisible text"
	     (if isearch-invisible "" "not "))))

;; Next function originally from `org-id-uuid' sans org dependency.
(defun hypb:uuid ()
  "Return string with random (version 4) universally unique id."
  (let ((rnd (md5 (format "%s%s%s%s%s%s%s"
			  (random)
			  (time-convert nil 'list)
			  (user-uid)
			  (emacs-pid)
			  (user-full-name)
			  user-mail-address
			  (recent-keys)))))
    (format "%s-%s-4%s-%s%s-%s"
	    (substring rnd 0 8)
	    (substring rnd 8 12)
	    (substring rnd 13 16)
	    (format "%x"
		    (logior
		     #b10000000
		     (logand
		      #b10111111
		      (string-to-number
		       (substring rnd 16 18) 16))))
	    (substring rnd 18 20)
	    (substring rnd 20 32))))

(defun hypb:user-name ()
  "Return the current user's email or login name (sans any domain name)."
  (if (string-match "@" hyperb:user-email)
      (substring hyperb:user-email 0 (match-beginning 0))
    (user-login-name)))

(defun hypb:window-list (&optional minibuffer-flag)
  "Return a list of Lisp window objects for all Emacs windows in selected frame.
Optional first arg MINIBUFFER-FLAG t means include the minibuffer window
in the list, even if it is not active.  If MINIBUFFER-FLAG is neither t
nor nil it means to not count the minibuffer window even if it is active."
  (window-list nil minibuffer-flag))

;;;###autoload
(defmacro hypb:with-marker (marker &rest body)
  "Set MARKER while executing BODY, then set MARKER to nil.
Return result of last BODY expression."
  (declare (indent 1) (debug t))
  `(prog1 (progn
	    (unless (symbolp ',marker)
	      (error "(with-marker): `marker' must be a symbol, not: '%s'" marker))
	    (unless (boundp ',marker)
	      (setq ,marker nil))
	    (unless (markerp ,marker)
	      (setq ,marker (make-marker)))
	    ,@body)
     (set-marker ,marker nil)))

;;; ************************************************************************
;;; About Hyperbole Setup
;;; ************************************************************************

(defvar hypb:home-page "https://www.gnu.org/software/hyperbole/"
  "The web home page for Hyperbole.")

(defvar hypb:hyperbole-banner-keymap
  (let ((map (make-sparse-keymap)))
    (define-key map [mouse-1]  'hypb:browse-home-page)
    (define-key map [mouse-2]  'hypb:browse-home-page)
    (define-key map "\C-m"     'hypb:browse-home-page)
    map)
  "Keymap used when on the Hyperbole banner glyph.")

;;;###autoload
(defun hypb:display-file-with-logo (file)
  "Display a text FILE in view mode with the Hyperbole banner prepended.
If FILE is not an absolute path, expand it relative to `hyperb:dir'."
  (unless (stringp file)
    (error "(hypb:display-file-with-logo): 'file' must be a string, not '%s'" file))
  (unless (file-name-absolute-p file)
    (setq file (expand-file-name file hyperb:dir)))
  (let ((existing-buf (when (stringp file) (get-file-buffer file)))
	(hsys-org-enable-smart-keys hsys-org-enable-smart-keys))

    ;; Ensure Smart Keys do not defer to Org mode when running tests noninteractively
    (when noninteractive
      (setq hsys-org-enable-smart-keys t))

    (when (and existing-buf noninteractive)
      ;; Likely are running tests when running non-interactively, so
      ;; kill existing buffer, so each test run starts from scratch
      ;; and is consistent.  Trigger an error if buffer has been
      ;; modified.
      (when (buffer-modified-p existing-buf)
	(error "(hypb:display-file-with-logo): Attempt to kill modified buffer: %s" existing-buf))
      (when (kill-buffer existing-buf)
	(setq existing-buf nil)))

    ;; A stub for the `id-browse-file' function is defined in
    ;; "hversion.el" when not running in InfoDock.
    (if (eq (symbol-function #'id-browse-file) #'view-file)
	(find-file file)
      ;; Running under InfoDock
      (id-browse-file file))

    (unless existing-buf
      (let ((buffer-read-only))
	(hypb:insert-hyperbole-banner))
      (goto-char (point-min))
      (set-window-start (selected-window) 1)
      (set-buffer-modified-p nil)
      (org-mode)
      (setq-local org-cycle-global-at-bob t)
      (view-mode)
      ;; Ensure no initial folding of the buffer, possibly from a hook
      (with-suppressed-warnings ((obsolete org-show-all))
        (if (fboundp 'org-fold-show-all)
	    (org-fold-show-all)
	  (org-show-all)))
      ;; On some versions of Emacs like Emacs28, need a slight delay
      ;; for file loading before searches will work properly.
      ;; Otherwise, "test/demo-tests.el" may fail.
      (sit-for 0.10))))

(defun hypb:browse-home-page ()
  "Visit the web home page for Hyperbole."
  (interactive)
  (require 'hsys-www)
  (hact 'www-url hypb:home-page))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun hypb:insert-hyperbole-banner ()
  "Display an optional text FILE with the Hyperbole banner prepended.
Without file, the banner is prepended to the current buffer."
  (let ((hyperbole-banner-path (expand-file-name "hyperbole-banner.png" hyperb:dir)))
    (unless (file-readable-p hyperbole-banner-path)
      (setq hyperbole-banner-path (if (fboundp 'locate-data-file)
				      (locate-data-file "hyperbole-banner.png")
				    (expand-file-name "hyperbole-banner.png"
						      data-directory))))
    (if (or (not (fboundp 'create-image))
	    (not (display-graphic-p))
	    (let ((button (next-button (point-min))))
	      (and button (button-has-type-p button 'hyperbole-banner)))
	    (not hyperbole-banner-path)
	    (not (file-readable-p hyperbole-banner-path)))
	;; Either image support is unavailable, the file cannot be read
	;; or the image has already been inserted, so don't reinsert it.
	nil
      (let ((hyperbole-banner (create-image hyperbole-banner-path))
	     (buffer-read-only)
	     button)
	(goto-char (point-min))
	;; Keep any initial line of variable settings, e.g. for Org
	;; mode as the first line.
	(when (looking-at (regexp-quote "-*- "))
	  (forward-line 1))
	(insert "\n")
	(insert-image hyperbole-banner)
	(insert "\n")
	(setq button (make-button (- (point) 2) (- (point) 1) :type 'hyperbole-banner))
	(button-put button 'help-echo (concat "Click to visit " hypb:home-page))
	(button-put button 'action #'hypb:browse-home-page)
	(button-put button 'face 'default)
	(button-put button 'keymap hypb:hyperbole-banner-keymap)))))

(defun hypb:locate-pathnames ()
  "Return a space-separated string of pathnames in a *Locate* buffer."
  (save-excursion
    (goto-char (point-min))
    (search-forward "\n" nil t 3)
    (replace-regexp-in-string " *\\([^\n]+\\)\n" "\\1 "
			      (buffer-substring-no-properties (point) (point-max)))))

(defun hypb:oct-to-int (oct-num)
  "Return octal integer OCT-NUM converted to a decimal integer."
  (let ((oct-str (int-to-string oct-num))
	(dec-num 0))
    (and (string-match "[^0-7]" oct-str)
	 (error "(hypb:oct-to-int): Bad octal number: %s" oct-str))
    (mapc (lambda (o)
	    (setq dec-num (+ (* dec-num 8)
			     (when (and (>= o ?0) (<= o ?7))
			       (- o ?0)))))
	  oct-str)
    dec-num))

;;; ************************************************************************
;;; Private variables
;;; ************************************************************************

(define-button-type 'hyperbole-banner)

(provide 'hypb)

;;; hypb.el ends here
