;;; helm-elfeed.el --- Helm interface for Elfeed   -*- lexical-binding: t -*-

;; Copyright (C) 2025 Timm Lichte

;; Author: Timm Lichte <timm.lichte@uni-tuebingen.de>
;; URL: https://codeberg.org/timmli/helm-elfeed
;; Package-Version: 20251013.1740
;; Package-Revision: 4d1c853548e3
;; Last modified: 2025-10-13 Mon 19:38:11
;; Package-Requires: ((emacs "29.1") (helm "3.9.6") (elfeed "3.4.2"))
;; Keywords: matching

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; Manage your RSS feeds inside Emacs with Helm and Elfeed.  List and
;; select feeds with Helm's completion mechanisms and perform specific
;; actions:
;; - Show
;; - Update
;; - Edit (requires elfeed-org)

;;; Code:

(require 'helm)
(require 'helm-buffers)
(require 'elfeed)
(require 'org-fold)


;;====================
;;
;; Customization
;;
;;--------------------

(defgroup helm-elfeed nil
  "Helm interface for Elfeed."
  :group 'helm)

(defcustom helm-elfeed-generic-search-queries
  '(("All" . "@6-months-ago")
    ("Marked/starred" . "+star")
    ("Unread" . "@6-months-ago +unread"))
  "List of pairs (NAME . QUERY).
They represent generic search queries used in `helm-elfeed'."
  :type '(cons string sting)
  :group 'helm-elfeed)

(defface helm-elfeed-unread-face
  '((t :inherit font-lock-keyword-face))
  "Font face for the title of an unread entity."
  :group 'helm-elfeed)

(defface helm-elfeed-url-face
  '((t :inherit font-lock-comment-face))
  "Font face for the url part."
  :group 'helm-elfeed)

(defface helm-elfeed-tags-face
  '((t :inherit font-lock-comment-face))
  "Font face for the tags of an entity."
  :group 'helm-elfeed)


;;====================
;;
;; Candidates
;;
;;--------------------

(defun helm-elfeed--trim-or-fill (field-value column-length)
  "Trim or fill a FIELD-VALUE to a specified COLUMN-LENGTH."
  (if (> (length field-value) (- column-length 2))
      (concat (truncate-string-to-width field-value
                                        (- column-length 2))
              "… ")
    (string-pad field-value column-length)))

(defun helm-elfeed--make-candidates ()
  "Make candidates for `helm-elfeed'.  A candidate is a pair (STRING PLIST)."
  (let ((unread-feeds (helm-elfeed--get-unread-feeds))
        (all-feeds (reverse (elfeed-feed-list)))
        (tags-width (let ((max-length 0))
                      (dolist (feed elfeed-feeds max-length)
                        (let* ((tags (cdr feed))
                               (tags-str (format "(%s) "
                                                 (mapconcat #'symbol-name tags ", ")))
                               (tags-length (length tags-str)))
                          (setq max-length (max max-length tags-length))))))
        (generic-searches (cl-loop
                           for generic-search in helm-elfeed-generic-search-queries
                           for search-format = (car generic-search)
                           for search-query = (cdr generic-search)
                           collect `(,search-format
                                     .
                                     ,(list `(:url nil :query ,search-query))))))
    (seq-concatenate
     'list generic-searches
     (cl-loop
      for feed-url in (seq-union unread-feeds all-feeds)
      for feed-tags = (helm-elfeed--get-feed-tags feed-url)
      for feed-object = (elfeed-db-get-feed feed-url)
      for feed-title = (or (elfeed-meta feed-object :title) (elfeed-feed-title feed-object))
      for feed-url-format = (propertize
                             (helm-elfeed--trim-or-fill (format "(%s)" feed-url)
                                                        (max 2
                                                             (- (window-width)
                                                                (length feed-title)
                                                                tags-width)))
                             'face 'helm-elfeed-url-face)
      for feed-tags-format = (propertize
                              (format "(%s)"
                                      (mapconcat #'symbol-name feed-tags ","))
                              'face 'helm-elfeed-tags-face)
      for feed-format = (concat
                         (if (member feed-url unread-feeds)
                             (propertize feed-title
                                         'face 'helm-elfeed-unread-face)
                           feed-title)
                         " " feed-url-format
                         " " feed-tags-format)
      for feed-plist = `(:title ,feed-title
                                :url ,feed-url
                                :query ,(concat
                                         "=" (replace-regexp-in-string "\\([?+]\\)" "\\\\\\1"
                                                                       feed-url)
                                         (when (member feed-url unread-feeds) " +unread")))
      if feed-title
      collect `(,feed-format
                .
                ,(list feed-plist))))))

(defun helm-elfeed--get-unread-feeds ()
  "Return unread feeds in `elfeed-db' as a list of feed URLs."
  (let ((unread-feeds ()))
    (with-elfeed-db-visit (entry feed)
      (let ((feed-url (elfeed-feed-url feed)))
        (when (and (not (member feed-url unread-feeds))
                   (member 'unread (elfeed-entry-tags entry)))
          (push feed-url unread-feeds))))
    (reverse unread-feeds)))

(defun helm-elfeed--get-feed-tags (url)
  "Return list of tag symbols in `elfeed-db' associated with a feed URL."
  (cl-loop
   for feed in elfeed-feeds
   for feed-url = (car feed)
   for feed-tags = (cdr feed)
   if (string= url feed-url)
   return feed-tags))


;;====================
;;
;; Actions
;;
;;--------------------

(defvar helm-elfeed--actions
  (helm-make-actions
   "Show unread feed" #'helm-elfeed-show-feed-action
   "Show complete feed" #'helm-elfeed-show-complete-feed-action
   "Mark feed as read" #'helm-elfeed-mark-feed-as-read-action
   "Update feed" #'helm-elfeed-update-feed-action
   "Update all feeds" #'helm-elfeed-update-action
   "Edit feed" #'helm-elfeed-edit-action)
  "List of pairs (STRING FUNCTIONSYMBOL).
They represent the actions used in `helm-elfeed'.")

(defun helm-elfeed-transformed-actions (actions candidate)
  "Transform ACTIONS for a CANDIDATE of the `helm-elfeed' source."
  (cond
   ;; If candidate is a generic query, do not show the edit action.
   ((not (plist-get (car candidate) :url))
    (helm-make-actions
     "Show unread feed" #'helm-elfeed-show-feed-action
     "Show complete feed" #'helm-elfeed-show-complete-feed-action
     "Mark feed as read" #'helm-elfeed-mark-feed-as-read-action
     "Update all feeds" #'helm-elfeed-update-action))
   ;; Default actions
   (t actions)))

(defun helm-elfeed-show-feed-action (_)
  "Update the Elfeed filter showing only the feeds of marked candidates."
  (let ((search-query (cl-loop
                       for candidate in (helm-marked-candidates)
                       collect (plist-get (car candidate) :query)
                       into collected-strings
                       finally (return (string-join collected-strings " ")))))
    (elfeed-search-set-filter search-query)
    (switch-to-buffer "*elfeed-search*")))

(defun helm-elfeed-show-complete-feed-action (_)
  "Update the Elfeed filter showing only the complete feeds of marked candidates."
  (let ((search-query (cl-loop
                       for candidate in (helm-marked-candidates)
                       collect (plist-get (car candidate) :query)
                       into collected-strings
                       finally (return (replace-regexp-in-string
                                        "+unread"
                                        ""
                                        (string-join collected-strings " "))))))
    (elfeed-search-set-filter search-query)
    (switch-to-buffer "*elfeed-search*")))

(defun helm-elfeed-mark-feed-as-read-action (_)
  "Update the Elfeed search marking the feeds as read."
  (let ((input helm-input)
        (search-query (cl-loop
                       for candidate in (helm-marked-candidates)
                       collect (plist-get (car candidate) :query)
                       into collected-strings
                       finally (return (string-join collected-strings " ")))))
    (elfeed-search-set-filter search-query)
    (switch-to-buffer "*elfeed-search*")
    (call-interactively #'mark-whole-buffer)
    (elfeed-search-untag-all-unread)
    (helm-elfeed input)))

(defun helm-elfeed-update-feed-action (candidate)
  "Update Elfeed database for CANDIDATE feed."
  (let ((feed-url (plist-get (car candidate) :url))
        (search-query (plist-get (car candidate) :query)))
    (elfeed-search-set-filter search-query)
    (switch-to-buffer "*elfeed-search*")
    (if feed-url
        (elfeed-update-feed feed-url)
      (elfeed-update))))

(defun helm-elfeed-update-action (_)
  "Update Elfeed database."
  (elfeed-search-set-filter "@6-months-ago")
  (switch-to-buffer "*elfeed-search*")
  (elfeed-update))

(defun helm-elfeed-edit-action (candidate)
  "Edit feed of CANDIDATE in Elfeed database."
  (let ((feed-url (plist-get (car candidate) :url))
        (feed-files (when (boundp 'rmh-elfeed-org-files)
                      rmh-elfeed-org-files))
        ;; (feed-title (plist-get (car candidate) :title))
        ;; (search-query (plist-get (car candidate) :query))
        )
    (if feed-files
        (cl-loop
         for feed-file in feed-files
         do (progn
              (find-file feed-file)
              ;; Inside Org file
              (goto-char (point-min))
              (org-fold-show-all)
              (when (search-forward feed-url)
                (beginning-of-line)
                (cl-return))))
      (message "elfeed-helm: Could not edit feed due to missing `rmh-elfeed-org-files'."))))


;;====================
;;
;; Main
;;
;;--------------------

;;;###autoload
(defun helm-elfeed (&optional input)
  "Switch between Elfeed feeds with Helm and optionally keep the INPUT."
  (interactive)
  (helm :sources (helm-build-sync-source "Feeds:"
                   :candidates #'helm-elfeed--make-candidates
                   :display-to-real nil ; Transform the selected candidate when passing it to action.
                   :action helm-elfeed--actions
                   :action-transformer (lambda (actions candidate)
                                         (helm-elfeed-transformed-actions actions
                                                                          candidate)))
        :buffer "*helm-elfeed*"
        :truncate-lines helm-buffers-truncate-lines
        :input (or input "")))


(provide 'helm-elfeed)


;; Local Variables:
;; indent-tabs-mode: nil
;; End:

;;; helm-elfeed.el ends here
