;;; ai-code-backends-infra.el --- Infrastructure for AI Code Terminals  -*- lexical-binding: t; -*-

;; Author: Yoav Orot, Kang Tu, AI Agent
;; SPDX-License-Identifier: Apache-2.0

;; Keywords: ai, terminal, vterm, eat

;;; Commentary:
;; This library provides common infrastructure for AI-powered terminal interfaces,
;; including terminal backend abstraction (vterm/eat), window management,
;; and performance optimizations like anti-flicker and reflow glitch prevention.
;; Code was generated by AI Agent, using
;; https://github.com/manzaltu/claude-code-ide.el as reference

;;; Code:

(require 'cl-lib)
(require 'project)

;; Silence native-compiler warnings.
(declare-function vterm "vterm" (&optional buffer-name))
(declare-function vterm-send-string "vterm" (&rest args))
(declare-function vterm-send-escape "vterm" ())
(declare-function vterm-send-return "vterm" ())
(declare-function vterm--window-adjust-process-window-size "vterm" (&rest args))
(declare-function vterm--filter "vterm" (&rest args))
(declare-function eat-term-send-string "eat" (&rest args))
(declare-function eat--adjust-process-window-size "eat" (&rest args))
(declare-function eat-mode "eat" ())
(declare-function eat-exec "eat" (&rest args))
(declare-function ai-code--session-handle-at-input "ai-code-input" ())

;; Declare vterm dynamic variables for let-binding to work with lexical-binding
(defvar vterm-shell)
(defvar vterm-environment)
(defvar vterm-kill-buffer-on-exit)

;;; Customization

(defgroup ai-code-backends-infra nil
  "Infrastructure for AI Code terminals."
  :group 'tools)

(defcustom ai-code-backends-infra-terminal-backend 'vterm
  "Terminal backend to use for sessions.
Can be either `vterm' or `eat'."
  :type '(choice (const :tag "vterm" vterm)
                 (const :tag "eat" eat))
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-window-side 'right
  "Side of the frame where the window should appear."
  :type '(choice (const :tag "Left" left)
                 (const :tag "Right" right)
                 (const :tag "Top" top)
                 (const :tag "Bottom" bottom))
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-window-width 90
  "Width of the side window when opened on left or right."
  :type 'integer
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-window-height 20
  "Height of the side window when opened on top or bottom."
  :type 'integer
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-use-side-window t
  "Whether to display the terminal in a side window."
  :type 'boolean
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-focus-on-open t
  "Whether to focus the terminal window when it opens."
  :type 'boolean
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-vterm-anti-flicker t
  "Enable intelligent flicker reduction for vterm display."
  :type 'boolean
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-vterm-render-delay 0.005
  "Rendering optimization delay for batched terminal updates."
  :type 'number
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-terminal-initialization-delay 0.1
  "Initialization delay for terminal stability."
  :type 'number
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-prevent-reflow-glitch t
  "Workaround for terminal scrolling bug #1422."
  :type 'boolean
  :group 'ai-code-backends-infra)

(defcustom ai-code-backends-infra-eat-preserve-position t
  "Maintain terminal scroll position when switching windows in eat."
  :type 'boolean
  :group 'ai-code-backends-infra)

;;; Variables

(defvar ai-code-backends-infra--processes (make-hash-table :test 'equal)
  "Hash table mapping session keys to their processes.")

(defvar ai-code-backends-infra--last-accessed-buffer nil
  "The most recently accessed AI Code buffer.")

(defvar ai-code-backends-infra--directory-buffer-map (make-hash-table :test 'equal)
  "Hash table mapping (prefix . directory) to last selected session buffer.")

(defvar-local ai-code-backends-infra--idle-timer nil
  "Timer for detecting idle state (response completion).")

(defvar-local ai-code-backends-infra--response-seen nil
  "Non-nil when the current response has been observed.
Observation happens either by the buffer being visible or by a notification
being sent for the response completion.")

(defvar-local ai-code-backends-infra--last-meaningful-output-time nil
  "Float timestamp of the most recent meaningful output.")

(defvar ai-code-cli-args-history nil
  "History list for CLI args prompts.")

(defcustom ai-code-backends-infra-idle-delay 5.0
  "Delay in seconds of inactivity before considering response complete.
After this period of terminal inactivity, a notification may be sent
if the AI session buffer is not currently visible."
  :type 'number
  :group 'ai-code-backends-infra)

;;; Vterm Rendering Optimization

(defvar-local ai-code-backends-infra--vterm-render-queue nil)
(defvar-local ai-code-backends-infra--vterm-render-timer nil)

(defvar ai-code-backends-infra--vterm-advices-installed nil
  "Flag indicating whether vterm filter advices have been installed globally.")

(declare-function ai-code-notifications-response-ready "ai-code-notifications" (&optional backend-name))

(defun ai-code-backends-infra--output-meaningful-p (output)
  "Return non-nil when OUTPUT contains meaningful printable content."
  (let* ((str (or output ""))
         ;; Strip OSC sequences (ESC ] ... BEL or ESC ] ... ESC \).
         (str (replace-regexp-in-string "\x1b\\][^\x07\x1b]*\\(?:\x07\\|\x1b\\\\\\)" "" str))
         ;; Strip ANSI escape sequences.
         (str (replace-regexp-in-string "\x1b\\[[0-9;?]*[ -/]*[@-~]" "" str))
         ;; Strip other control characters.
         (str (replace-regexp-in-string "[\x00-\x1f\x7f]" "" str)))
    (string-match-p "[^ \t\n\r]" str)))

(defun ai-code-backends-infra--buffer-user-visible-p (buffer)
  "Return non-nil when BUFFER is visible in any live window."
  (and (get-buffer-window-list buffer nil t) t))

(defun ai-code-backends-infra--check-response-complete (buffer)
  "Check if AI response is complete in BUFFER and notify if enabled."
  (when (buffer-live-p buffer)
    (with-current-buffer buffer
      (if (ai-code-backends-infra--idle-delay-elapsed-p)
          (let ((visible (ai-code-backends-infra--buffer-user-visible-p buffer)))
            (if visible
                (setq ai-code-backends-infra--response-seen t)
              (when (not ai-code-backends-infra--response-seen)
                (setq ai-code-backends-infra--response-seen t)
                (when (require 'ai-code-notifications nil t)
                  (when (fboundp 'ai-code-notifications-response-ready)
                    (let ((buffer-name (buffer-name buffer)))
                      ;; Extract backend name from buffer name format: *<backend>[<dir>]*
                      ;; Example: "*codex[my-project]*" extracts "codex"
                      ;; Regex breakdown:
                      ;;   \\*       - matches literal asterisk
                      ;;   \\(       - start capture group 1
                      ;;   [^[]+     - one or more chars that are not '['
                      ;;   \\)       - end capture group 1 (this is the backend name)
                      ;;   \\[       - matches literal '['
                      (when (string-match "\\*\\([^[]+\\)\\[" buffer-name)
                        (let ((backend-name (match-string 1 buffer-name)))
                          (ai-code-notifications-response-ready backend-name)))))))))
        (ai-code-backends-infra--schedule-idle-check)))))

(defun ai-code-backends-infra--schedule-idle-check ()
  "Schedule a check for response completion after idle period.
The timer is reset only after meaningful output is observed."
  (when ai-code-backends-infra--idle-timer
    (cancel-timer ai-code-backends-infra--idle-timer))
  (let ((buffer (current-buffer)))
    (setq ai-code-backends-infra--idle-timer
          (run-at-time ai-code-backends-infra-idle-delay nil
                       #'ai-code-backends-infra--check-response-complete
                       buffer))))

(defun ai-code-backends-infra--idle-delay-elapsed-p ()
  "Return non-nil when idle delay has elapsed since last output."
  (let ((last ai-code-backends-infra--last-meaningful-output-time))
    (or (null last)
        (>= (- (float-time) last) ai-code-backends-infra-idle-delay))))

(defun ai-code-backends-infra--note-meaningful-output ()
  "Record meaningful output and schedule idle tracking."
  (setq ai-code-backends-infra--response-seen nil
        ai-code-backends-infra--last-meaningful-output-time (float-time))
  (ai-code-backends-infra--schedule-idle-check))

(defun ai-code-backends-infra--vterm-notification-tracker (orig-fun process input)
  "Track vterm activity for notification purposes, then call ORIG-FUN."
  (when (ai-code-backends-infra--session-buffer-p (process-buffer process))
    (with-current-buffer (process-buffer process)
      (when (ai-code-backends-infra--output-meaningful-p input)
        (ai-code-backends-infra--note-meaningful-output))))
  (funcall orig-fun process input))

(defun ai-code-backends-infra--vterm-smart-renderer (orig-fun process input)
  "Smart rendering filter for optimized vterm display updates.
Activity tracking for notifications is handled separately by
`ai-code-backends-infra--vterm-notification-tracker'."
  (if (or (not ai-code-backends-infra-vterm-anti-flicker)
          (not (ai-code-backends-infra--session-buffer-p (process-buffer process))))
      (funcall orig-fun process input)
    (with-current-buffer (process-buffer process)
      (let* ((complex-redraw-detected
              (string-match-p "\033\\[[0-9]*A.*\033\\[K.*\033\\[[0-9]*A.*\033\\[K" input))
             (clear-count (1- (length (split-string input "\033\\[K"))))
             (escape-count (cl-count ?\033 input))
             (input-length (length input))
             (escape-density (if (> input-length 0) (/ (float escape-count) input-length) 0)))
        (if (or complex-redraw-detected
                (and (> escape-density 0.3) (>= clear-count 2))
                ai-code-backends-infra--vterm-render-queue)
            (progn
              (setq ai-code-backends-infra--vterm-render-queue
                    (concat ai-code-backends-infra--vterm-render-queue input))
              (when ai-code-backends-infra--vterm-render-timer
                (cancel-timer ai-code-backends-infra--vterm-render-timer))
              (setq ai-code-backends-infra--vterm-render-timer
                    (run-at-time ai-code-backends-infra-vterm-render-delay nil
                                 (lambda (buf)
                                   (when (buffer-live-p buf)
                                     (with-current-buffer buf
                                       (when ai-code-backends-infra--vterm-render-queue
                                         (let ((inhibit-redisplay t)
                                               (data ai-code-backends-infra--vterm-render-queue))
                                           (setq ai-code-backends-infra--vterm-render-queue nil
                                                 ai-code-backends-infra--vterm-render-timer nil)
                                           (funcall orig-fun (get-buffer-process buf) data))))))
                                 (current-buffer))))
          (funcall orig-fun process input))))))

(defun ai-code-backends-infra--configure-vterm-buffer ()
  "Configure vterm for enhanced performance."
  (setq-local vterm-scroll-to-bottom-on-output nil)
  (when (boundp 'vterm--redraw-immididately)
    (setq-local vterm--redraw-immididately nil))
  (when (fboundp 'ai-code--session-handle-at-input)
    (local-set-key (kbd "@") #'ai-code--session-handle-at-input))
  (when (fboundp 'ai-code--session-handle-hash-input)
    (local-set-key (kbd "#") #'ai-code--session-handle-hash-input))
  (setq-local cursor-in-non-selected-windows nil)
  (setq-local blink-cursor-mode nil)
  (setq-local cursor-type nil)
  (when-let ((proc (get-buffer-process (current-buffer))))
    (set-process-query-on-exit-flag proc nil)
    (when (fboundp 'process-put)
      (process-put proc 'read-output-max 4096)))
  ;; Install vterm filter advices globally (only once)
  (unless ai-code-backends-infra--vterm-advices-installed
    ;; Always install notification tracker for session buffers
    (advice-add 'vterm--filter :around #'ai-code-backends-infra--vterm-notification-tracker)
    ;; Conditionally install anti-flicker renderer
    (when ai-code-backends-infra-vterm-anti-flicker
      (advice-add 'vterm--filter :around #'ai-code-backends-infra--vterm-smart-renderer))
    (setq ai-code-backends-infra--vterm-advices-installed t)))

;;; Terminal Backend Abstraction

(defun ai-code-backends-infra--terminal-ensure-backend ()
  "Ensure the selected terminal backend is available."
  (cond
   ((eq ai-code-backends-infra-terminal-backend 'vterm)
    (unless (featurep 'vterm) (require 'vterm nil t))
    (unless (featurep 'vterm)
      (user-error "The package vterm is not installed")))
   ((eq ai-code-backends-infra-terminal-backend 'eat)
    (unless (featurep 'eat) (require 'eat nil t))
    (unless (featurep 'eat)
      (user-error "The package eat is not installed")))
   (t (user-error "Invalid terminal backend: %s" ai-code-backends-infra-terminal-backend))))

(defun ai-code-backends-infra--terminal-send-string (string)
  "Send STRING to the terminal in the current buffer."
  (cond
   ((eq ai-code-backends-infra-terminal-backend 'vterm)
    (vterm-send-string string))
   ((eq ai-code-backends-infra-terminal-backend 'eat)
    (when (bound-and-true-p eat-terminal)
      (eat-term-send-string eat-terminal string)))
   (t (error "Unknown terminal backend: %s" ai-code-backends-infra-terminal-backend))))

(defun ai-code-backends-infra--terminal-send-escape ()
  "Send escape key to the terminal in the current buffer."
  (cond
   ((eq ai-code-backends-infra-terminal-backend 'vterm) (vterm-send-escape))
   ((eq ai-code-backends-infra-terminal-backend 'eat)
    (when (bound-and-true-p eat-terminal)
      (eat-term-send-string eat-terminal "\e")))
   (t (error "Unknown terminal backend: %s" ai-code-backends-infra-terminal-backend))))

(defun ai-code-backends-infra--terminal-send-return ()
  "Send return key to the terminal in the current buffer."
  (cond
   ((eq ai-code-backends-infra-terminal-backend 'vterm) (vterm-send-return))
   ((eq ai-code-backends-infra-terminal-backend 'eat)
    (when (bound-and-true-p eat-terminal)
      (eat-term-send-string eat-terminal "\r")))
   (t (error "Unknown terminal backend: %s" ai-code-backends-infra-terminal-backend))))

(defun ai-code-backends-infra--terminal-send-backspace ()
  "Send backspace key to the terminal in the current buffer."
  (cond
   ((eq ai-code-backends-infra-terminal-backend 'vterm)
    (vterm-send-string "\177"))
   ((eq ai-code-backends-infra-terminal-backend 'eat)
    (when (bound-and-true-p eat-terminal)
      (eat-term-send-string eat-terminal "\177")))
   (t (error "Unknown terminal backend: %s" ai-code-backends-infra-terminal-backend))))

;;; Reflow and Window Management

(defun ai-code-backends-infra--terminal-resize-handler ()
  "Retrieve the terminal's resize handling function based on backend."
  (pcase ai-code-backends-infra-terminal-backend
    ('vterm #'vterm--window-adjust-process-window-size)
    ('eat #'eat--adjust-process-window-size)
    (_ (error "Unsupported terminal backend"))))

(defun ai-code-backends-infra--session-buffer-p (buffer)
  "Check if BUFFER belongs to an AI session."
  (when-let ((name (if (stringp buffer) buffer (buffer-name buffer))))
    (string-match-p "\\`\\*.*\\[.*\\].*\\*\\'" name)))

(defun ai-code-backends-infra--terminal-reflow-filter (original-fn &rest args)
  "Filter terminal reflows to prevent height-only resize triggers."
  (let* ((base-result (apply original-fn args))
         (dimensions-stable t))
    (dolist (win (window-list))
      (when-let* ((buf (window-buffer win))
                  ((ai-code-backends-infra--session-buffer-p buf)))
        (let* ((new-width (window-width win))
               (cached-width (window-parameter win 'ai-code-backends-infra-cached-width)))
          (unless (eql new-width cached-width)
            (setq dimensions-stable nil)
            (set-window-parameter win 'ai-code-backends-infra-cached-width new-width)))))
    (if (and ai-code-backends-infra-prevent-reflow-glitch dimensions-stable)
        nil
      base-result)))

(defun ai-code-backends-infra--display-buffer-in-side-window (buffer)
  "Display BUFFER in a side window."
  (let ((window
         (if ai-code-backends-infra-use-side-window
             (let* ((side ai-code-backends-infra-window-side)
                    (display-buffer-alist
                     `((,(regexp-quote (buffer-name buffer))
                        (display-buffer-in-side-window)
                        (side . ,side)
                        (slot . 0)
                        ,@(when (memq side '(left right))
                            `((window-width . ,ai-code-backends-infra-window-width)))
                        ,@(when (memq side '(top bottom))
                            `((window-height . ,ai-code-backends-infra-window-height)))
                        (window-parameters . ((no-delete-other-windows . t)))))))
               (display-buffer buffer))
           (display-buffer buffer))))
    (setq ai-code-backends-infra--last-accessed-buffer buffer)
    (when (and window ai-code-backends-infra-focus-on-open)
      (select-window window))
    window))

;;; Session Helpers

(defun ai-code-backends-infra--session-working-directory ()
  "Return the working directory, preferring the current project root."
  (if-let ((project (project-current)))
      (expand-file-name (project-root project))
    (expand-file-name default-directory)))

(defun ai-code-backends-infra--session-buffer-name (prefix directory &optional instance-name)
  "Return a session buffer name for PREFIX in DIRECTORY.
When INSTANCE-NAME is non-nil and not \"default\", include it in the name."
  (let* ((base (file-name-nondirectory (directory-file-name directory)))
         (instance (and instance-name
                        (not (string= instance-name ""))
                        (not (string= instance-name "default"))
                        instance-name)))
    (format "*%s[%s%s]*"
            prefix
            base
            (if instance (format ":%s" instance) ""))))

(defun ai-code-backends-infra--normalize-instance-name (instance-name)
  "Return a normalized INSTANCE-NAME, defaulting to \"default\"."
  (if (and instance-name (not (string= instance-name "")))
      instance-name
    "default"))

(defun ai-code-backends-infra--session-key (directory instance-name)
  "Return a session key for DIRECTORY and INSTANCE-NAME."
  (cons directory (ai-code-backends-infra--normalize-instance-name instance-name)))

(defun ai-code-backends-infra--session-map-key (prefix directory)
  "Return a map key for PREFIX and DIRECTORY."
  (cons prefix (expand-file-name directory)))

(defun ai-code-backends-infra--parse-session-buffer-name (buffer-name prefix)
  "Parse BUFFER-NAME for PREFIX.
Return a cons of (base-name . instance-name) or nil."
  (when (string-match
         (format "\\`\\*%s\\[\\([^]:]+\\)\\(?::\\([^]]+\\)\\)?\\]\\*\\'"
                 (regexp-quote prefix))
         buffer-name)
    (cons (match-string 1 buffer-name)
          (match-string 2 buffer-name))))

(defun ai-code-backends-infra--session-instance-name (buffer-name prefix)
  "Return instance name for BUFFER-NAME with PREFIX."
  (when-let ((parsed (ai-code-backends-infra--parse-session-buffer-name buffer-name prefix)))
    (ai-code-backends-infra--normalize-instance-name (cdr parsed))))

(defun ai-code-backends-infra--find-session-buffers (prefix directory)
  "Return session buffers for PREFIX in DIRECTORY."
  (let ((base (file-name-nondirectory (directory-file-name directory))))
    (cl-remove-if-not
     (lambda (buf)
       (when-let ((parsed (ai-code-backends-infra--parse-session-buffer-name
                           (buffer-name buf)
                           prefix)))
         (string= (car parsed) base)))
     (buffer-list))))

(defun ai-code-backends-infra--remember-session-buffer (prefix directory buffer)
  "Remember BUFFER as the last selected session for PREFIX and DIRECTORY."
  (when (and prefix directory buffer)
    (puthash (ai-code-backends-infra--session-map-key prefix directory)
             buffer
             ai-code-backends-infra--directory-buffer-map)))

(defun ai-code-backends-infra--forget-session-buffer (prefix directory buffer)
  "Forget BUFFER if it's the remembered session for PREFIX and DIRECTORY."
  (when (and prefix directory buffer)
    (let* ((key (ai-code-backends-infra--session-map-key prefix directory))
           (existing (gethash key ai-code-backends-infra--directory-buffer-map)))
      (when (eq existing buffer)
        (remhash key ai-code-backends-infra--directory-buffer-map)))))

(defun ai-code-backends-infra--select-session-buffer (prefix directory &optional force-prompt)
  "Select a session buffer for PREFIX in DIRECTORY.
Returns the selected buffer or nil if none exist."
  (let ((buffers (ai-code-backends-infra--find-session-buffers prefix directory)))
    (cond
     ((null buffers) nil)
     ((= (length buffers) 1)
      (ai-code-backends-infra--remember-session-buffer prefix directory (car buffers))
      (car buffers))
     (t
      (let* ((remembered (gethash (ai-code-backends-infra--session-map-key prefix directory)
                                  ai-code-backends-infra--directory-buffer-map))
             (choices (mapcar (lambda (buf)
                                (cons (ai-code-backends-infra--session-instance-name
                                       (buffer-name buf)
                                       prefix)
                                      buf))
                              buffers)))
        (if (and (not force-prompt) remembered (memq remembered buffers))
            remembered
          (let ((selection (completing-read
                            (format "Select %s session: " prefix)
                            (mapcar #'car choices)
                            nil t)))
            (let ((buffer (cdr (assoc selection choices))))
              (ai-code-backends-infra--remember-session-buffer prefix directory buffer)
              buffer))))))))

(defun ai-code-backends-infra--prompt-for-instance-name (existing-instance-names &optional force-prompt)
  "Prompt for a new instance name.
EXISTING-INSTANCE-NAMES is a list of existing instance names.
If FORCE-PROMPT is nil and there are no existing instances, return \"default\"."
  (if (or existing-instance-names force-prompt)
      (let ((proposed-name ""))
        (while (or (string= proposed-name "")
                   (member proposed-name existing-instance-names))
          (setq proposed-name
                (read-string (if existing-instance-names
                                 (format "Instance name (existing: %s): "
                                         (mapconcat #'identity existing-instance-names ", "))
                               "Instance name: ")
                             nil nil (and (> (length proposed-name) 0) proposed-name)))
          (cond
           ((string= proposed-name "")
            (message "Instance name cannot be empty. Please enter a name.")
            (sit-for 1))
           ((member proposed-name existing-instance-names)
            (message "Instance name '%s' already exists. Please choose a different name." proposed-name)
            (sit-for 1))))
        proposed-name)
    "default"))

(defun ai-code-backends-infra--resolve-start-command (program switches arg &optional prompt-label)
  "Build command string for PROGRAM and SWITCHES.
When ARG is non-nil, prompt for CLI args using SWITCHES as default input.
PROMPT-LABEL is used in the minibuffer prompt."
  (let* ((default-args (mapconcat #'identity switches " "))
         (prompt (format "%s args: " (or prompt-label "CLI")))
         (prompt-args (when arg
                        (read-string prompt default-args 'ai-code-cli-args-history)))
         (resolved-args (if arg
                            (split-string-shell-command prompt-args)
                          switches))
         (command (mapconcat #'identity
                             (cons program resolved-args)
                             " ")))
    (list :command command :args resolved-args)))

(defun ai-code-backends-infra--cleanup-session (directory buffer-name process-table
                                                          &optional instance-name prefix)
  "Clean up a session for DIRECTORY using BUFFER-NAME and PROCESS-TABLE."
  (let* ((resolved-instance (or instance-name
                                (and prefix
                                     (ai-code-backends-infra--session-instance-name
                                      buffer-name
                                      prefix))
                                "default"))
         (key (ai-code-backends-infra--session-key directory resolved-instance)))
    (remhash key process-table))
  (when-let ((buffer (get-buffer buffer-name)))
    (ai-code-backends-infra--forget-session-buffer prefix directory buffer)
    (when (buffer-live-p buffer)
      (kill-buffer buffer))))

(defun ai-code-backends-infra--toggle-or-create-session (working-dir buffer-name process-table command
                                                                     &optional escape-fn cleanup-fn
                                                                     instance-name prefix force-prompt)
  "Toggle or create a terminal session.
WORKING-DIR is the directory for the session.
BUFFER-NAME is the terminal buffer name.
PROCESS-TABLE maps session keys to processes.
COMMAND is the shell command to run.
ESCAPE-FN is bound to `C-<escape>' inside the session buffer when non-nil.
CLEANUP-FN is called with no arguments when the process exits.
INSTANCE-NAME overrides instance selection when non-nil.
PREFIX enables instance selection when BUFFER-NAME is nil.
When FORCE-PROMPT is non-nil, always prompt for a new instance name."
  (ai-code-backends-infra--cleanup-dead-processes process-table)
  (let* ((existing-buffers (and prefix
                                (ai-code-backends-infra--find-session-buffers
                                 prefix
                                 working-dir)))
         (existing-instance-names (mapcar (lambda (buf)
                                            (ai-code-backends-infra--session-instance-name
                                             (buffer-name buf)
                                             prefix))
                                          existing-buffers))
         (resolved-instance (cond
                             (instance-name (ai-code-backends-infra--normalize-instance-name instance-name))
                             (prefix
                              (ai-code-backends-infra--prompt-for-instance-name
                               existing-instance-names
                               force-prompt))
                             (t "default")))
         (resolved-buffer-name (or buffer-name
                                   (and prefix
                                        (ai-code-backends-infra--session-buffer-name
                                         prefix
                                         working-dir
                                         resolved-instance))))
         (session-key (ai-code-backends-infra--session-key working-dir resolved-instance))
         (existing-process (gethash session-key process-table))
         (buffer (get-buffer resolved-buffer-name)))
    (if (and existing-process (process-live-p existing-process) buffer)
        (if (get-buffer-window buffer)
            (delete-window (get-buffer-window buffer))
          (progn
            (ai-code-backends-infra--remember-session-buffer prefix working-dir buffer)
            (ai-code-backends-infra--display-buffer-in-side-window buffer)))
      (let* ((buffer-and-process
              (ai-code-backends-infra--create-terminal-session
               resolved-buffer-name working-dir command nil))
             (new-buffer (car buffer-and-process))
             (process (cdr buffer-and-process)))
        (puthash session-key process process-table)
        ;; Wait for initialization before checking process status
        (sleep-for ai-code-backends-infra-terminal-initialization-delay)
        ;; Check if process is still alive after initialization delay
        (if (and process (process-live-p process))
            (progn
              ;; Process started successfully, set up sentinel for cleanup on exit
              (set-process-sentinel
               process
               (lambda (_proc _event)
                 (ai-code-backends-infra--cleanup-session
                  working-dir
                  resolved-buffer-name
                  process-table
                  resolved-instance
                  prefix)
                 (when cleanup-fn
                   (funcall cleanup-fn))))
              (when escape-fn
                (with-current-buffer new-buffer
                  (local-set-key (kbd "C-<escape>") escape-fn)))
              (with-current-buffer new-buffer
                (add-hook 'kill-buffer-hook
                          (lambda ()
                            (ai-code-backends-infra--forget-session-buffer
                             prefix
                             working-dir
                             (current-buffer)))
                          nil t))
              (ai-code-backends-infra--remember-session-buffer prefix working-dir new-buffer)
              (ai-code-backends-infra--display-buffer-in-side-window new-buffer))
          ;; Process exited during initialization - show buffer with error to user
          ;; Clean up the session from process table (but keep buffer visible)
          (remhash session-key process-table)
          ;; Display the buffer so user can see the error output
          (if (buffer-live-p new-buffer)
              (progn
                (pop-to-buffer new-buffer)
                (message "CLI failed to start - see buffer for error details"))
            (message "CLI failed to start - process exited immediately")))))))

(defun ai-code-backends-infra--switch-to-session-buffer (buffer-name missing-message
                                                                    &optional prefix working-dir force-prompt)
  "Switch to BUFFER-NAME or signal MISSING-MESSAGE.
When PREFIX and WORKING-DIR are provided, select from multiple sessions."
  (let ((buffer (or (and buffer-name (get-buffer buffer-name))
                    (and prefix working-dir
                         (ai-code-backends-infra--select-session-buffer
                          prefix
                          working-dir
                          force-prompt)))))
    (if buffer
        (progn
          (ai-code-backends-infra--remember-session-buffer prefix working-dir buffer)
          (if-let ((window (get-buffer-window buffer)))
            (select-window window)
            (ai-code-backends-infra--display-buffer-in-side-window buffer)))
      (user-error "%s" missing-message))))

(defun ai-code-backends-infra--send-line-to-session (buffer-name missing-message line
                                                                &optional prefix working-dir force-prompt)
  "Send LINE to BUFFER-NAME or signal MISSING-MESSAGE.
When PREFIX and WORKING-DIR are provided, select from multiple sessions."
  (let ((buffer (or (and buffer-name (get-buffer buffer-name))
                    (and prefix working-dir
                         (ai-code-backends-infra--select-session-buffer
                          prefix
                          working-dir
                          force-prompt)))))
    (if buffer
        (with-current-buffer buffer
          (ai-code-backends-infra--remember-session-buffer prefix working-dir buffer)
          (ai-code-backends-infra--terminal-send-string line)
          (sit-for 0.5) ;; 0.1 might be too low for some cli backends such as github copilot cli
          (ai-code-backends-infra--terminal-send-return))
      (user-error "%s" missing-message))))

;;; Generic Session Creation

(defun ai-code-backends-infra--create-terminal-session (buffer-name working-dir command env-vars)
  "Generic function to create a terminal session.
BUFFER-NAME is the name for the buffer.
WORKING-DIR is the directory.
COMMAND is the shell command to run.
ENV-VARS is a list of environment variables."
  (ai-code-backends-infra--terminal-ensure-backend)
  (let ((default-directory working-dir))
    (cond
     ((eq ai-code-backends-infra-terminal-backend 'vterm)
      (let* ((vterm-shell command)
             (vterm-kill-buffer-on-exit nil)  ; Keep buffer alive to show errors
             (vterm-environment (append env-vars (bound-and-true-p vterm-environment))))
        (let ((buffer (save-window-excursion (vterm buffer-name))))
          (with-current-buffer buffer
            (ai-code-backends-infra--configure-vterm-buffer))
          (cons buffer (get-buffer-process buffer)))))

     ((eq ai-code-backends-infra-terminal-backend 'eat)
      (let* ((buffer (get-buffer-create buffer-name))
             (parts (split-string-shell-command command))
             (program (car parts))
             (args (cdr parts)))
        (with-current-buffer buffer
          (unless (eq major-mode 'eat-mode) (eat-mode))
          (when (fboundp 'ai-code--session-handle-at-input)
            (local-set-key (kbd "@") #'ai-code--session-handle-at-input))
          (when (fboundp 'ai-code--session-handle-hash-input)
            (local-set-key (kbd "#") #'ai-code--session-handle-hash-input))
          (setq-local process-environment (append env-vars process-environment))
          (eat-exec buffer buffer-name program nil args)
          ;; Add process filter to track activity for notifications
          (when-let ((proc (get-buffer-process buffer)))
            (let ((orig-filter (process-filter proc)))
              (set-process-filter
               proc
               (lambda (process output)
                 ;; Call original filter first
                 (when orig-filter
                   (funcall orig-filter process output))
                 ;; Then track activity for notifications
                 (with-current-buffer (process-buffer process)
                   (when (ai-code-backends-infra--output-meaningful-p output)
                     (ai-code-backends-infra--note-meaningful-output)))))))
          (cons buffer (get-buffer-process buffer)))))
     (t (error "Unknown backend")))))

(defun ai-code-backends-infra--cleanup-dead-processes (table)
  "Clean up dead processes from TABLE."
  (maphash (lambda (key proc)
             (unless (process-live-p proc)
               (remhash key table)))
           table))

;;; Evil Mode Integration

(defvar ai-code-backends-infra--evil-original-spc-command nil
  "Original command for SPC in `evil-normal-state-map'.")

(declare-function ai-code-send-command "ai-code" (&optional arg))

(defun ai-code-backends-infra--evil-spc-command ()
  "In AI session buffers, run `ai-code-send-command'.
Otherwise, run the original command for SPC."
  (interactive)
  (if (ai-code-backends-infra--session-buffer-p (current-buffer))
      (call-interactively #'ai-code-send-command)
    (when ai-code-backends-infra--evil-original-spc-command
      (call-interactively ai-code-backends-infra--evil-original-spc-command))))

;;;###autoload
(defun ai-code-backends-infra-evil-setup ()
  "Setup AI Code integration with Evil mode.
This function configures SPC key binding in Evil normal state for
AI session buffers.  Call this function after Evil is loaded,
typically in your Emacs configuration with:
  (with-eval-after-load \\='evil (ai-code-backends-infra-evil-setup))"
  (interactive)
  (when (and (featurep 'evil) (boundp 'evil-normal-state-map))
    (unless ai-code-backends-infra--evil-original-spc-command
      (setq ai-code-backends-infra--evil-original-spc-command
            (lookup-key evil-normal-state-map (kbd "SPC"))))
    (define-key evil-normal-state-map (kbd "SPC")
                #'ai-code-backends-infra--evil-spc-command)))

(provide 'ai-code-backends-infra)
;;; ai-code-backends-infra.el ends here
