#include "uart.h"
#include <dos.h>

int cdecl uart_set_bps_rate(unsigned baseaddr, unsigned divisor)
{
	// returns -1 if no UART found
	// else 0

	unsigned char ov;

	// save old LCR value
	ov=inp(baseaddr+3);
	// make DL accessible
	outp(baseaddr+3,0x80|ov);
	// write DL
	outpw(baseaddr,divisor);
	// read DL
	if (inpw(baseaddr)!=divisor) return UE_ERROR;
	// hide DL
	outp(baseaddr+3,ov);
	return 0;
}

int cdecl uart_set_line_params(unsigned baseaddr, unsigned lineparams)
{
	// returns -1 if no UART found
	// else 0

	// This function resets break condition!

	outp(baseaddr+3,lineparams);
	if (inp(baseaddr+3)!=lineparams) return UE_ERROR;
	return 0;
}

void cdecl uart_set_break(unsigned baseaddr)
{
	// forces UART into break condition
	outp(baseaddr+3,inp(baseaddr+3)|0x40);
}

void cdecl uart_clear_break(unsigned baseaddr)
{
	// clears break condition
	outp(baseaddr+3,inp(baseaddr+3)&0xbf);
}

void cdecl uart_set_dtr(unsigned baseaddr)
{
	// sets the DTR output
	outp(baseaddr+4,inp(baseaddr+4)|0x01);
}

void cdecl uart_clear_dtr(unsigned baseaddr)
{
	// clears the DTR output
	outp(baseaddr+4,inp(baseaddr+4)&0xfe);
}

void cdecl uart_set_rts(unsigned baseaddr)
{
	// sets the RTS output
	outp(baseaddr+4,inp(baseaddr+4)|0x02);
}

void cdecl uart_clear_rts(unsigned baseaddr)
{
	// clears the RTS output
	outp(baseaddr+4,inp(baseaddr+4)&0xfc);
}

void cdecl uart_set_local_loopback(unsigned baseaddr)
{
	// sets the UART to local loopback mode
	outp(baseaddr+4,inp(baseaddr+4)|0x10);
}

void cdecl uart_clear_local_loopback(unsigned baseaddr)
{
	// sets the UART to normal mode again
	outp(baseaddr+4,inp(baseaddr+4)&0xef);
}

int cdecl uart_enable_fifo(unsigned baseaddr, unsigned triggerlevel)
{
	// enables and clears the 16550A built-in FIFOs
	// doesn't do any harm if called with other chips
	// returns: 0 if OK
	//          -10 if 16550 with defective FIFO
	//          -1 if no FIFO available

	unsigned char f;

	f=inp(baseaddr+2)&0xC0;
	if (f==0) return UE_ERROR;
	if (f==0x80) {
		outp(baseaddr+2,0);
		return UE_BAD_FIFO;
		}
	outp(baseaddr+2,triggerlevel&0xC0|0x07);
	return 0;
}

void cdecl uart_disable_fifo(unsigned baseaddr)
{
	// disables the 16550A FIFOs
	// no trouble with other chips
	outp(baseaddr+2,0);
}

int cdecl uart_get_line_status(unsigned baseaddr)
{
	// returns line status (see header file for constants)
	return inp(baseaddr+5);
}

int cdecl uart_get_modem_status(unsigned baseaddr)
{
	// returns modem status (see header file for constants)
	return inp(baseaddr+6);
}

int cdecl uart_read_scratch_register(unsigned baseaddr)
{
	// reads baseaddr+7; provided because with some boards this byte has
	// special meaning (RX/TX enable, int level select, etc.)
	return inp(baseaddr+7);
}

void cdecl uart_write_scratch_register(unsigned baseaddr, int value)
{
	// writes baseaddr+7; provided because with some boards this byte has
	// special meaning (RX/TX enable, int level select, etc.)
	outp(baseaddr+7,value);
}

