/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2024 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

/////////////////////// Stdlib includes

/////////////////////// Qt includes
#include <QDebug>
#include <QString>
#include <QStringList>
#include <QDomDocument>


/////////////////////// libXpertMassCore includes
#include "MsXpS/libXpertMassCore/Isotope.hpp"
#include "MsXpS/libXpertMassCore/PolChemDef.hpp"


/////////////////////// Local includes
#include "tests-config.h"


#pragma once


namespace MsXpS
{
namespace libXpertMassCore
{


class DECLSPEC TestUtils
{
  public:
  TestUtils();
  TestUtils(const QString &pol_chem_def_name, int version);
  virtual ~TestUtils();

  QString m_testsInputDataDir  = QString(TESTS_INPUT_DIR);
  QString m_testsOutputDataDir = QString(TESTS_OUTPUT_DIR);

  QString m_testsInputPolChemDefsDir =
    QString("%1/polymer-chemistry-definitions").arg(m_testsInputDataDir);
  QString m_testsOutputPolChemDefsDir =
    QString("%1/polymer-chemistry-definitions").arg(m_testsOutputDataDir);

  QString m_testsInputPolSeqsDir =
    QString("%1/polymer-sequences").arg(m_testsInputDataDir);
  QString m_testsOutputPolSeqsDir =
    QString("%1/polymer-sequences").arg(m_testsOutputDataDir);

  QString m_polChemDefDirName;
  QString m_polChemDefFileBaseName;
  QString m_polChemDefName;

  PolChemDefSpec m_polChemDefSpec;
  PolChemDefSPtr msp_polChemDef = nullptr;

  QString xml_format_indent_string = " ";

  Isotope m_isotopeC12 =
    Isotope("carbon",
            "C",
            12.0,
            0.989211941850466902614869013632414862513542175292968750000000);
  IsotopeQSPtr msp_isotopeC12 = QSharedPointer<Isotope>::create(m_isotopeC12);

  Isotope m_isotopeC13 =
    Isotope("carbon",
            "C",
            13.0033548352,
            0.010788058149533083507343178553128382191061973571777343750000);
  IsotopeQSPtr msp_isotopeC13 = QSharedPointer<Isotope>::create(m_isotopeC13);

  Isotope m_isotopeN14 =
    Isotope("nitrogen",
            "N",
            14.0030740042,
            0.996358014567941707717579902237048372626304626464843750000000);
  IsotopeQSPtr msp_isotopeN14 = QSharedPointer<Isotope>::create(m_isotopeN14);

  Isotope m_isotopeN15 =
    Isotope("nitrogen",
            "N",
            15.0001088994,
            0.003641985432058271465738386041266494430601596832275390625000);
  IsotopeQSPtr msp_isotopeN15 = QSharedPointer<Isotope>::create(m_isotopeN15);

  std::size_t m_isotopeCount = 0;

  QString m_acetylationFormulaString = "-H2O+CH3COOH";
  QString m_tryptophanFormulaString  = "C11H10N2O1";

  // 157 residues
  QString m_telokinAsMonomerText1Letter =
    "MAMISGMSGRKASGSSPTSPINADKVENEDAFLEEVAEEKPHVKPYFTKTILDMEVVEGSAARFDCKIEGYPDP"
    "EVMWYKDDQPVKESRHFQIDYDEEGNCSLTISEVCGDDDAKYTCKAVNSLGEATCTAELLVETMGKEGEGEGEG"
    "EEDEEEEEE";

  QString m_telokinNtermPeptideAsMonomerText1Letter =
    "MAMISGMSGRKASGSSPTSPINADKV";

  int m_telokinSequenceMonomerCount = 157;

  // 157 residues, 7 spaces
  QString m_telokinAsMonomerText1LetterWithSpaces =
    "MAMISGMSGRKASGSSP TSPINADK "
    "VENEDAFLEEVAEEKPHVKPYFTKTILDMEVVEGSAARFDCKIEGYPDP"
    "EVMWYKDDQPVKESRHFQIDYDEEGNCSL TISEVCGDD DAKYTCKAVNSLG "
    "EATCTA\nELLVETMGKEGEGEGEG"
    "EEDE\tEEEEE";

  // 157 residues
  QString m_telokinAsMonomerText3Letters =
    "MetAlaMetIleSerGlyMetSerGlyArgLysAlaSerGlySerSerProThrSerProIleAsnAlaAspLy"
    "sValGluAsnGluAspAlaPheLeuGluGluValAlaGluGluLysProHisValLysProTyrPheThrLysT"
    "hrIleLeuAspMetGluValValGluGlySerAlaAlaArgPheAspCysLysIleGluGlyTyrProAspPro"
    "GluValMetTrpTyrLysAspAspGlnProValLysGluSerArgHisPheGlnIleAspTyrAspGluGluGl"
    "yAsnCysSerLeuThrIleSerGluValCysGlyAspAspAspAlaLysTyrThrCysLysAlaValAsnSerL"
    "euGlyGluAlaThrCysThrAlaGluLeuLeuValGluThrMetGlyLysGluGlyGluGlyGluGlyGluGly"
    "GluGluAspGluGluGluGluGluGlu";

  QString m_glucoseFormulaString    = QString("C6H12O6");
  QString m_saccharoseFormulaString = QString("C12H22O11");
  QString m_formulaTitle            = "MA dipeptide";

  QString m_actionFormulaStringMetAlaDipeptidyl =
    QString("+C5H11N1O2S1-H2O+C3H7N1O2-H2O");
  QString m_actionFormulaStringMetAlaDipeptidylSpaces =
    QString("+C5 H11 N1\nO2S1-H2O+C3 H7 N1 O2 - H2O");

  //  See the constructor
  QString m_actionFormulaStringMetAlaDipeptidylTitled;

  QString m_naturalIsotopicDataFileName = "light-c-n.dat";
  QString m_manualUserIsotopicDataFileName =
    "one-C14-radioactive-on-glucose-manual-config.dat";

  static void initializeXpertmassLibrary();
  PolChemDefSPtr initializePolChemDef(const QString &pol_chem_def_name,
                                      int version);

  QDomDocument craftFormulaDomDocument(const QStringList &dom_strings);
  QDomDocument craftIonizeruleDomDocument(const QStringList &dom_strings);
  QDomDocument craftMdfDomDocument(const QStringList &dom_strings);
  QDomDocument craftMnmDomDocument(const QStringList &dom_strings);
  QDomDocument craftMonomerDomDocument(const QStringList &dom_strings);
  QDomDocument craftClkDomDocument(const QStringList &dom_strings);
  QDomDocument craftClrDomDocument(const QStringList &dom_strings);
  QDomDocument craftClsDomDocument(const QStringList &dom_strings);
  QDomDocument craftClaDomDocument(const QStringList &dom_strings);
  QDomDocument craftFgrDomDocument(const QStringList &dom_strings);
  QDomDocument craftFgsDomDocument(const QStringList &dom_strings);
};


} // namespace libXpertMassCore
} // namespace MsXpS
