/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  Codec,
  TranslatedString,
  buildCodecForObject,
  codecForBoolean,
} from "@gnu-taler/taler-util";
import {
  buildStorageKey,
  useLocalStorage,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";

interface Preferences {
  showChallangeSetup: boolean;
  showDebugInfo: boolean;
}

export const codecForPreferences = (): Codec<Preferences> =>
  buildCodecForObject<Preferences>()
    .property("showChallangeSetup", codecForBoolean())
    .property("showDebugInfo", codecForBoolean())
    .build("Preferences");

const defaultPreferences: Preferences = {
  showChallangeSetup: false,
  showDebugInfo: false,
};

const PREFERENCES_KEY = buildStorageKey(
  "challenger-preferences",
  codecForPreferences(),
);
/**
 * User preferences.
 *
 * @returns tuple of [state, update()]
 */
export function usePreferences(): [
  Readonly<Preferences>,
  <T extends keyof Preferences>(key: T, value: Preferences[T]) => void,
] {
  const { value, update } = useLocalStorage(
    PREFERENCES_KEY,
    defaultPreferences,
  );

  function updateField<T extends keyof Preferences>(k: T, v: Preferences[T]) {
    const newValue = { ...value, [k]: v };
    update(newValue);
  }
  return [value, updateField];
}

export function getAllBooleanPreferences(): Array<keyof Preferences> {
  return [
    "showChallangeSetup",
    "showDebugInfo",
  ];
}

export function getLabelForPreferences(
  k: keyof Preferences,
  i18n: ReturnType<typeof useTranslationContext>["i18n"],
): TranslatedString {
  switch (k) {
    case "showChallangeSetup":
      return i18n.str`Show challenger setup screen`;
    case "showDebugInfo":
      return i18n.str`Show debug info`;
  }
}
