/*
 * Copyright (c) 2020 iXsystems, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

/*
 * Available Solaris debug functions.  All of the ASSERT() macros will be
 * compiled out when NDEBUG is defined, this is the default behavior for
 * the SPL.  To enable assertions use the --enable-debug with configure.
 * The VERIFY() functions are never compiled out and cannot be disabled.
 *
 * PANIC()	- Panic the node and print message.
 * ASSERT()	- Assert X is true, if not panic.
 * ASSERT3B()	- Assert boolean X OP Y is true, if not panic.
 * ASSERT3S()	- Assert signed X OP Y is true, if not panic.
 * ASSERT3U()	- Assert unsigned X OP Y is true, if not panic.
 * ASSERT3P()	- Assert pointer X OP Y is true, if not panic.
 * ASSERT0()	- Assert value is zero, if not panic.
 * VERIFY()	- Verify X is true, if not panic.
 * VERIFY3B()	- Verify boolean X OP Y is true, if not panic.
 * VERIFY3S()	- Verify signed X OP Y is true, if not panic.
 * VERIFY3U()	- Verify unsigned X OP Y is true, if not panic.
 * VERIFY3P()	- Verify pointer X OP Y is true, if not panic.
 * VERIFY0()	- Verify value is zero, if not panic.
 */

#ifndef _SPL_DEBUG_H
#define	_SPL_DEBUG_H


/*
 * Common DEBUG functionality.
 */
int spl_panic(const char *file, const char *func, int line,
    const char *fmt, ...);
void spl_dumpstack(void);

#ifndef expect
#define	expect(expr, value) (__builtin_expect((expr), (value)))
#endif
#define	likely(expr)   expect((expr) != 0, 1)
#define	unlikely(expr) expect((expr) != 0, 0)

/* BEGIN CSTYLED */
#define	PANIC(fmt, a...)						\
	spl_panic(__FILE__, __FUNCTION__, __LINE__, fmt, ## a)

#define	VERIFY(cond)							\
	(void) (unlikely(!(cond)) &&					\
	    spl_panic(__FILE__, __FUNCTION__, __LINE__,			\
	    "%s", "VERIFY(" #cond ") failed\n"))

#define	VERIFY3B(LEFT, OP, RIGHT)	do {				\
		boolean_t _verify3_left = (boolean_t)(LEFT);		\
		boolean_t _verify3_right = (boolean_t)(RIGHT);		\
		if (!(_verify3_left OP _verify3_right))			\
		    spl_panic(__FILE__, __FUNCTION__, __LINE__,		\
		    "VERIFY3(" #LEFT " "  #OP " "  #RIGHT ") "		\
		    "failed (%d " #OP " %d)\n",				\
		    (boolean_t) (_verify3_left),			\
		    (boolean_t) (_verify3_right));			\
	} while (0)

#define	VERIFY3S(LEFT, OP, RIGHT)	do {				\
		int64_t _verify3_left = (int64_t)(LEFT);		\
		int64_t _verify3_right = (int64_t)(RIGHT);		\
		if (!(_verify3_left OP _verify3_right))			\
		    spl_panic(__FILE__, __FUNCTION__, __LINE__,		\
		    "VERIFY3(" #LEFT " "  #OP " "  #RIGHT ") "		\
		    "failed (%lld " #OP " %lld)\n",			\
		    (long long) (_verify3_left),			\
		    (long long) (_verify3_right));			\
	} while (0)

#define	VERIFY3U(LEFT, OP, RIGHT)	do {				\
		uint64_t _verify3_left = (uint64_t)(LEFT);		\
		uint64_t _verify3_right = (uint64_t)(RIGHT);		\
		if (!(_verify3_left OP _verify3_right))			\
		    spl_panic(__FILE__, __FUNCTION__, __LINE__,		\
		    "VERIFY3(" #LEFT " "  #OP " "  #RIGHT ") "		\
		    "failed (%llu " #OP " %llu)\n",			\
		    (unsigned long long) (_verify3_left),		\
		    (unsigned long long) (_verify3_right));		\
	} while (0)

#define	VERIFY3P(LEFT, OP, RIGHT)	do {				\
		uintptr_t _verify3_left = (uintptr_t)(LEFT);		\
		uintptr_t _verify3_right = (uintptr_t)(RIGHT);		\
		if (!(_verify3_left OP _verify3_right))			\
		    spl_panic(__FILE__, __FUNCTION__, __LINE__,		\
		    "VERIFY3(" #LEFT " "  #OP " "  #RIGHT ") "		\
		    "failed (%px " #OP " %px)\n",			\
		    (void *) (_verify3_left),				\
		    (void *) (_verify3_right));				\
	} while (0)

#define	VERIFY0(RIGHT)	do {				\
		int64_t _verify3_left = (int64_t)(0);		\
		int64_t _verify3_right = (int64_t)(RIGHT);		\
		if (!(_verify3_left == _verify3_right))			\
		    spl_panic(__FILE__, __FUNCTION__, __LINE__,		\
		    "VERIFY3(0 == " #RIGHT ") "				\
		    "failed (0 == %lld)\n",				\
		    (long long) (_verify3_right));			\
	} while (0)
#define	CTASSERT_GLOBAL(x)		CTASSERT(x)

/*
 * Debugging disabled (--disable-debug)
 */
#ifdef NDEBUG

#define	ASSERT(x)		((void)0)
#define	ASSERT3B(x,y,z)		((void)0)
#define	ASSERT3S(x,y,z)		((void)0)
#define	ASSERT3U(x,y,z)		((void)0)
#define	ASSERT3P(x,y,z)		((void)0)
#define	ASSERT0(x)		((void)0)
#define	IMPLY(A, B)		((void)0)
#define	EQUIV(A, B)		((void)0)

/*
 * Debugging enabled (--enable-debug)
 */
#else

#define	ASSERT3B	VERIFY3B
#define	ASSERT3S	VERIFY3S
#define	ASSERT3U	VERIFY3U
#define	ASSERT3P	VERIFY3P
#define	ASSERT0		VERIFY0
#define	ASSERT		VERIFY
#define	IMPLY(A, B) \
	((void)(((!(A)) || (B)) || \
	    spl_panic(__FILE__, __FUNCTION__, __LINE__, \
	    "(" #A ") implies (" #B ")")))
#define	EQUIV(A, B) \
	((void)((!!(A) == !!(B)) || \
	    spl_panic(__FILE__, __FUNCTION__, __LINE__, \
	    "(" #A ") is equivalent to (" #B ")")))
/* END CSTYLED */

#endif /* NDEBUG */

#endif /* SPL_DEBUG_H */
