/*
 * Decompiled with CFR 0.152.
 */
package com.influxdb.client;

import com.influxdb.Arguments;
import com.influxdb.LogLevel;
import com.influxdb.client.write.PointSettings;
import com.influxdb.exceptions.InfluxException;
import java.io.IOException;
import java.io.InputStream;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import okhttp3.HttpUrl;
import okhttp3.OkHttpClient;

public final class InfluxDBClientOptions {
    private static final Pattern TAGS_PROPERTY = Pattern.compile("(influx2\\.tags\\.)(.+)");
    private static final Pattern DURATION_PATTERN = Pattern.compile("^(\\d+)([a-zA-Z]{0,2})$");
    private final String url;
    private final OkHttpClient.Builder okHttpClient;
    private final LogLevel logLevel;
    private AuthScheme authScheme;
    private char[] token;
    private String username;
    private char[] password;
    private String org;
    private String bucket;
    private final PointSettings pointSettings;

    private InfluxDBClientOptions(@Nonnull Builder builder) {
        Arguments.checkNotNull(builder, "InfluxDBClientOptions.Builder");
        this.url = builder.url;
        this.okHttpClient = builder.okHttpClient;
        this.logLevel = builder.logLevel;
        this.authScheme = builder.authScheme;
        this.token = builder.token;
        this.username = builder.username;
        this.password = builder.password;
        this.org = builder.org;
        this.bucket = builder.bucket;
        this.pointSettings = builder.pointSettings;
    }

    @Nonnull
    public String getUrl() {
        return this.url;
    }

    @Nonnull
    public OkHttpClient.Builder getOkHttpClient() {
        return this.okHttpClient;
    }

    @Nonnull
    public LogLevel getLogLevel() {
        return this.logLevel;
    }

    @Nullable
    public AuthScheme getAuthScheme() {
        return this.authScheme;
    }

    @Nullable
    public char[] getToken() {
        return this.token;
    }

    @Nullable
    public String getUsername() {
        return this.username;
    }

    @Nullable
    public char[] getPassword() {
        return this.password;
    }

    @Nullable
    public String getOrg() {
        return this.org;
    }

    @Nullable
    public String getBucket() {
        return this.bucket;
    }

    @Nonnull
    public PointSettings getPointSettings() {
        return this.pointSettings;
    }

    @Nonnull
    public static Builder builder() {
        return new Builder();
    }

    @NotThreadSafe
    public static class Builder {
        private String url;
        private OkHttpClient.Builder okHttpClient;
        private LogLevel logLevel;
        private AuthScheme authScheme;
        private char[] token;
        private String username;
        private char[] password;
        private String org;
        private String bucket;
        private PointSettings pointSettings = new PointSettings();

        @Nonnull
        public Builder url(@Nonnull String url) {
            Arguments.checkNonEmpty(url, "url");
            this.url = url;
            return this;
        }

        @Nonnull
        public Builder okHttpClient(@Nonnull OkHttpClient.Builder okHttpClient) {
            Arguments.checkNotNull(okHttpClient, "OkHttpClient.Builder");
            this.okHttpClient = okHttpClient;
            return this;
        }

        @Nonnull
        public Builder logLevel(@Nonnull LogLevel logLevel) {
            Arguments.checkNotNull((Object)logLevel, "logLevel");
            this.logLevel = logLevel;
            return this;
        }

        @Nonnull
        public Builder authenticate(@Nonnull String username, @Nonnull char[] password) {
            Arguments.checkNonEmpty(username, "username");
            Arguments.checkNotNull(password, "password");
            this.authScheme = AuthScheme.SESSION;
            this.username = username;
            this.password = password;
            return this;
        }

        @Nonnull
        public Builder authenticateToken(char[] token) {
            Arguments.checkNotNull(token, "token");
            this.authScheme = AuthScheme.TOKEN;
            this.token = token;
            return this;
        }

        @Nonnull
        public Builder org(@Nullable String org) {
            this.org = org;
            return this;
        }

        @Nonnull
        public Builder bucket(@Nullable String bucket) {
            this.bucket = bucket;
            return this;
        }

        @Nonnull
        public Builder addDefaultTag(@Nonnull String key, @Nullable String expression) {
            Arguments.checkNotNull(key, "tagName");
            this.pointSettings.addDefaultTag(key, expression);
            return this;
        }

        @Nonnull
        public Builder connectionString(@Nonnull String connectionString) {
            Arguments.checkNonEmpty(connectionString, "url");
            HttpUrl parse = HttpUrl.parse(connectionString);
            if (parse == null) {
                throw new InfluxException("Unable to parse connection string " + connectionString);
            }
            String url = parse.scheme() + "://" + parse.host() + ":" + parse.port() + parse.encodedPath();
            String org = parse.queryParameter("org");
            String bucket = parse.queryParameter("bucket");
            String token = parse.queryParameter("token");
            String logLevel = parse.queryParameter("logLevel");
            String readTimeout = parse.queryParameter("readTimeout");
            String writeTimeout = parse.queryParameter("writeTimeout");
            String connectTimeout = parse.queryParameter("connectTimeout");
            return this.configure(url, org, bucket, token, logLevel, readTimeout, writeTimeout, connectTimeout);
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Nonnull
        public Builder loadProperties() {
            try (InputStream inputStream = this.getClass().getResourceAsStream("/influx2.properties");){
                Properties properties = new Properties();
                properties.load(inputStream);
                String url = properties.getProperty("influx2.url");
                String org = properties.getProperty("influx2.org");
                String bucket = properties.getProperty("influx2.bucket");
                String token = properties.getProperty("influx2.token");
                String logLevel = properties.getProperty("influx2.logLevel");
                String readTimeout = properties.getProperty("influx2.readTimeout");
                String writeTimeout = properties.getProperty("influx2.writeTimeout");
                String connectTimeout = properties.getProperty("influx2.connectTimeout");
                properties.stringPropertyNames().forEach(key -> {
                    Matcher matcher = TAGS_PROPERTY.matcher((CharSequence)key);
                    if (matcher.matches()) {
                        String tagKey = matcher.group(2);
                        this.addDefaultTag(tagKey, properties.getProperty((String)key).trim());
                    }
                });
                Builder builder = this.configure(url, org, bucket, token, logLevel, readTimeout, writeTimeout, connectTimeout);
                return builder;
            }
            catch (IOException e) {
                throw new IllegalStateException(e);
            }
        }

        @Nonnull
        public InfluxDBClientOptions build() {
            if (this.url == null) {
                throw new IllegalStateException("The url to connect to InfluxDB has to be defined.");
            }
            if (this.okHttpClient == null) {
                this.okHttpClient = new OkHttpClient.Builder();
            }
            if (this.logLevel == null) {
                this.logLevel = LogLevel.NONE;
            }
            return new InfluxDBClientOptions(this);
        }

        @Nonnull
        private Builder configure(@Nonnull String url, @Nullable String org, @Nullable String bucket, @Nullable String token, @Nullable String logLevel, @Nullable String readTimeout, @Nullable String writeTimeout, @Nullable String connectTimeout) {
            this.url(url);
            this.org(org);
            this.bucket(bucket);
            if (token != null) {
                this.authenticateToken(token.toCharArray());
            }
            if (logLevel != null) {
                this.logLevel(Enum.valueOf(LogLevel.class, logLevel));
            }
            this.okHttpClient = new OkHttpClient.Builder();
            if (readTimeout != null) {
                this.okHttpClient.readTimeout(this.toDuration(readTimeout));
            }
            if (writeTimeout != null) {
                this.okHttpClient.writeTimeout(this.toDuration(writeTimeout));
            }
            if (connectTimeout != null) {
                this.okHttpClient.connectTimeout(this.toDuration(connectTimeout));
            }
            return this;
        }

        @Nonnull
        private Duration toDuration(@Nonnull String value) {
            ChronoUnit chronoUnit;
            Matcher matcher = DURATION_PATTERN.matcher(value);
            if (!matcher.matches()) {
                throw new InfluxException("'" + value + "' is not a valid duration");
            }
            String amount = matcher.group(1);
            String unit = matcher.group(2);
            switch (unit != null && !unit.isEmpty() ? unit.toLowerCase() : "ms") {
                case "ms": {
                    chronoUnit = ChronoUnit.MILLIS;
                    break;
                }
                case "s": {
                    chronoUnit = ChronoUnit.SECONDS;
                    break;
                }
                case "m": {
                    chronoUnit = ChronoUnit.MINUTES;
                    break;
                }
                default: {
                    throw new InfluxException("unknown unit for '" + value + "'");
                }
            }
            return Duration.of(Long.valueOf(amount), chronoUnit);
        }
    }

    public static enum AuthScheme {
        SESSION,
        TOKEN;

    }
}

