/*
 * Decompiled with CFR 0.152.
 */
package org.apache.zeppelin.notebook;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Tags;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.commons.lang3.StringUtils;
import org.apache.zeppelin.conf.ZeppelinConfiguration;
import org.apache.zeppelin.notebook.Note;
import org.apache.zeppelin.notebook.NoteInfo;
import org.apache.zeppelin.notebook.Notebook;
import org.apache.zeppelin.notebook.exception.NotePathAlreadyExistsException;
import org.apache.zeppelin.notebook.repo.NotebookRepo;
import org.apache.zeppelin.user.AuthenticationInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class NoteManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(NoteManager.class);
    public static final String TRASH_FOLDER = "~Trash";
    private Folder root;
    private Folder trash;
    private NotebookRepo notebookRepo;
    private NoteCache noteCache;
    private Map<String, String> notesInfo;
    private final ZeppelinConfiguration zConf;

    @Inject
    public NoteManager(NotebookRepo notebookRepo, ZeppelinConfiguration zConf) throws IOException {
        this.zConf = zConf;
        this.notebookRepo = notebookRepo;
        this.noteCache = new NoteCache(zConf.getNoteCacheThreshold());
        this.root = new Folder("/", notebookRepo, this.noteCache, zConf);
        this.trash = this.root.getOrCreateFolder(TRASH_FOLDER);
        this.init();
    }

    private void init() throws IOException {
        this.notesInfo = this.notebookRepo.list(AuthenticationInfo.ANONYMOUS).values().stream().collect(Collectors.toConcurrentMap(NoteInfo::getId, NoteInfo::getPath));
        this.notesInfo.entrySet().stream().forEach(entry -> {
            try {
                this.addOrUpdateNoteNode(new NoteInfo((String)entry.getKey(), (String)entry.getValue()));
            }
            catch (IOException e) {
                LOGGER.warn(e.getMessage());
            }
        });
    }

    public Map<String, String> getNotesInfo() {
        return this.notesInfo;
    }

    public void reloadNotes() throws IOException {
        this.root = new Folder("/", this.notebookRepo, this.noteCache, this.zConf);
        this.trash = this.root.getOrCreateFolder(TRASH_FOLDER);
        this.init();
    }

    public int getCacheSize() {
        return this.noteCache.getSize();
    }

    private void addOrUpdateNoteNode(NoteInfo noteInfo, boolean checkDuplicates) throws IOException {
        String notePath = noteInfo.getPath();
        if (checkDuplicates && !this.isNotePathAvailable(notePath)) {
            throw new NotePathAlreadyExistsException("Note '" + notePath + "' existed");
        }
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i])) continue;
            curFolder = curFolder.getOrCreateFolder(tokens[i]);
        }
        curFolder.addNote(tokens[tokens.length - 1], noteInfo);
        this.notesInfo.put(noteInfo.getId(), noteInfo.getPath());
    }

    private void addOrUpdateNoteNode(NoteInfo noteInfo) throws IOException {
        this.addOrUpdateNoteNode(noteInfo, false);
    }

    public boolean containsNote(String notePath) {
        try {
            this.getNoteNode(notePath);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public boolean containsFolder(String folderPath) {
        try {
            this.getFolder(folderPath);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void saveNote(Note note, AuthenticationInfo subject) throws IOException {
        if (note.isRemoved()) {
            LOGGER.warn("Try to save note: {} when it is removed", (Object)note.getId());
        } else {
            this.addOrUpdateNoteNode(new NoteInfo(note));
            this.noteCache.putNote(note);
            NoteManager noteManager = this;
            synchronized (noteManager) {
                this.notebookRepo.save(note, subject);
            }
        }
    }

    public void addNote(Note note, AuthenticationInfo subject) throws IOException {
        this.addOrUpdateNoteNode(new NoteInfo(note), true);
        this.noteCache.putNote(note);
    }

    public void saveNote(Note note) throws IOException {
        this.saveNote(note, AuthenticationInfo.ANONYMOUS);
    }

    public void removeNote(String noteId, AuthenticationInfo subject) throws IOException {
        String notePath = this.notesInfo.remove(noteId);
        Folder folder = this.getOrCreateFolder(this.getFolderName(notePath));
        folder.removeNote(this.getNoteName(notePath));
        this.noteCache.removeNote(noteId);
        this.notebookRepo.remove(noteId, notePath, subject);
    }

    public void moveNote(String noteId, String newNotePath, AuthenticationInfo subject) throws IOException {
        String newNoteName;
        String oldNoteName;
        if (noteId == null) {
            throw new IOException("No metadata found for this note: " + noteId);
        }
        if (!this.isNotePathAvailable(newNotePath)) {
            throw new NotePathAlreadyExistsException("Note '" + newNotePath + "' existed");
        }
        String notePath = this.notesInfo.get(noteId);
        NoteNode noteNode = this.getNoteNode(notePath);
        noteNode.getParent().removeNote(this.getNoteName(notePath));
        noteNode.setNotePath(newNotePath);
        String newParent = this.getFolderName(newNotePath);
        Folder newFolder = this.getOrCreateFolder(newParent);
        newFolder.addNoteNode(noteNode);
        this.notesInfo.put(noteId, newNotePath);
        this.notebookRepo.move(noteId, notePath, newNotePath, subject);
        if (!StringUtils.equals((CharSequence)notePath, (CharSequence)newNotePath)) {
            this.processNote(noteId, note -> {
                note.setPath(newNotePath);
                return null;
            });
        }
        if (!StringUtils.equals((CharSequence)(oldNoteName = this.getNoteName(notePath)), (CharSequence)(newNoteName = this.getNoteName(newNotePath)))) {
            this.processNote(noteId, note -> {
                this.notebookRepo.save(note, subject);
                return null;
            });
        }
    }

    public void moveFolder(String folderPath, String newFolderPath, AuthenticationInfo subject) throws IOException {
        this.notebookRepo.move(folderPath, newFolderPath, subject);
        Folder folder = this.getFolder(folderPath);
        folder.getParent().removeFolder(folder.getName(), subject);
        Folder newFolder = this.getOrCreateFolder(newFolderPath);
        newFolder.getParent().addFolder(newFolder.getName(), folder);
        for (NoteInfo noteInfo : folder.getNoteInfoRecursively()) {
            this.notesInfo.put(noteInfo.getId(), noteInfo.getPath());
        }
    }

    public List<NoteInfo> removeFolder(String folderPath, AuthenticationInfo subject) throws IOException {
        this.notebookRepo.remove(folderPath, subject);
        Folder folder = this.getFolder(folderPath);
        List<NoteInfo> noteInfos = folder.getParent().removeFolder(folder.getName(), subject);
        for (NoteInfo noteInfo : noteInfos) {
            this.notesInfo.remove(noteInfo.getId());
        }
        return noteInfos;
    }

    public <T> T processNote(String noteId, boolean reload, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
        if (this.notesInfo == null || noteId == null || !this.notesInfo.containsKey(noteId)) {
            return noteProcessor.process(null);
        }
        String notePath = this.notesInfo.get(noteId);
        NoteNode noteNode = this.getNoteNode(notePath);
        return noteNode.loadAndProcessNote(reload, noteProcessor);
    }

    public <T> T processNote(String noteId, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
        return this.processNote(noteId, false, noteProcessor);
    }

    public Folder getOrCreateFolder(String folderName) {
        String[] tokens = folderName.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i])) continue;
            curFolder = curFolder.getOrCreateFolder(tokens[i]);
        }
        return curFolder;
    }

    private NoteNode getNoteNode(String notePath) throws IOException {
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            throw new IOException("Can not find note: " + notePath);
        }
        NoteNode noteNode = curFolder.getNote(tokens[tokens.length - 1]);
        if (noteNode == null) {
            throw new IOException("Can not find note: " + notePath);
        }
        return noteNode;
    }

    private Folder getFolder(String folderPath) throws IOException {
        String[] tokens = folderPath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            throw new IOException("Can not find folder: " + folderPath);
        }
        return curFolder;
    }

    public Folder getTrashFolder() {
        return this.trash;
    }

    private String getFolderName(String notePath) {
        int pos = notePath.lastIndexOf(47);
        return notePath.substring(0, pos);
    }

    private String getNoteName(String notePath) {
        int pos = notePath.lastIndexOf(47);
        return notePath.substring(pos + 1);
    }

    private boolean isNotePathAvailable(String notePath) {
        String[] tokens = notePath.split("/");
        Folder curFolder = this.root;
        for (int i = 0; i < tokens.length - 1; ++i) {
            if (StringUtils.isBlank((CharSequence)tokens[i]) || (curFolder = curFolder.getFolder(tokens[i])) != null) continue;
            return true;
        }
        return !curFolder.containsNote(tokens[tokens.length - 1]);
    }

    public String getNoteIdByPath(String notePath) throws IOException {
        NoteNode noteNode = this.getNoteNode(notePath);
        return noteNode.getNoteId();
    }

    private static class NoteCache {
        private static final Logger LOGGER = LoggerFactory.getLogger(NoteCache.class);
        private final int threshold;
        private final Map<String, Note> lruCache;
        private final Counter cacheHit;
        private final Counter cacheMiss;

        public NoteCache(int threshold) {
            this.threshold = (Integer)Metrics.gauge((String)"zeppelin_note_cache_threshold", (Iterable)Tags.empty(), (Number)threshold);
            this.lruCache = Metrics.gaugeMapSize((String)"zeppelin_note_cache", (Iterable)Tags.empty(), Collections.synchronizedMap(new LRUCache()));
            this.cacheHit = Metrics.counter((String)"zeppelin_note_cache_hit", (Iterable)Tags.empty());
            this.cacheMiss = Metrics.counter((String)"zeppelin_note_cache_miss", (Iterable)Tags.empty());
        }

        public int getSize() {
            return this.lruCache.size();
        }

        public Note getNote(String noteId) {
            Note note = this.lruCache.get(noteId);
            if (note != null) {
                this.cacheHit.increment();
            } else {
                this.cacheMiss.increment();
            }
            return note;
        }

        public void putNote(Note note) {
            this.lruCache.put(note.getId(), note);
        }

        public Note removeNote(String noteId) {
            return this.lruCache.remove(noteId);
        }

        private class LRUCache
        extends LinkedHashMap<String, Note> {
            private static final long serialVersionUID = 1L;

            public LRUCache() {
                super(NoteCache.this.threshold, 0.5f, true);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            protected boolean removeEldestEntry(Map.Entry<String, Note> eldest) {
                if (this.size() <= NoteCache.this.threshold) {
                    return false;
                }
                Note eldestNote = eldest.getValue();
                ReentrantReadWriteLock.WriteLock lock = eldestNote.getLock().writeLock();
                if (lock.tryLock()) {
                    try {
                        boolean bl = true;
                        return bl;
                    }
                    finally {
                        lock.unlock();
                    }
                }
                LOGGER.info("Can not evict note {}, because the write lock can not be acquired. {} notes currently loaded.", (Object)eldestNote.getId(), (Object)this.size());
                this.cleanupCache();
                return false;
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            private void cleanupCache() {
                Iterator iterator = this.entrySet().iterator();
                int count = 0;
                while (this.size() - 1 >= NoteCache.this.threshold && iterator.hasNext()) {
                    Map.Entry noteEntry = iterator.next();
                    Note note = (Note)noteEntry.getValue();
                    ReentrantReadWriteLock.WriteLock lock = note.getLock().writeLock();
                    if (!lock.tryLock()) continue;
                    try {
                        iterator.remove();
                        LOGGER.debug("Remove note {} from LRU Cache", (Object)note.getId());
                        ++count;
                    }
                    finally {
                        lock.unlock();
                    }
                }
                LOGGER.info("The cache cleanup removes {} entries", (Object)count);
            }
        }
    }

    public static class NoteNode {
        private Folder parent;
        private NoteInfo noteInfo;
        private NotebookRepo notebookRepo;
        private NoteCache noteCache;
        private ZeppelinConfiguration zConf;

        public NoteNode(NoteInfo noteInfo, Folder parent, NotebookRepo notebookRepo, NoteCache noteCache, ZeppelinConfiguration zConf) {
            this.noteInfo = noteInfo;
            this.parent = parent;
            this.notebookRepo = notebookRepo;
            this.noteCache = noteCache;
            this.zConf = zConf;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public <T> T loadAndProcessNote(boolean reload, Notebook.NoteProcessor<T> noteProcessor) throws IOException {
            Note note;
            NoteNode noteNode = this;
            synchronized (noteNode) {
                note = this.noteCache.getNote(this.noteInfo.getId());
                if (note == null || reload) {
                    note = this.notebookRepo.get(this.noteInfo.getId(), this.noteInfo.getPath(), AuthenticationInfo.ANONYMOUS);
                    if (this.parent.toString().equals("/")) {
                        note.setPath("/" + note.getName());
                    } else {
                        note.setPath(this.parent.toString() + "/" + note.getName());
                    }
                    note.setCronSupported(this.zConf);
                    this.noteCache.putNote(note);
                }
            }
            try {
                note.getLock().readLock().lock();
                noteNode = noteProcessor.process(note);
                return (T)noteNode;
            }
            finally {
                note.getLock().readLock().unlock();
            }
        }

        public String getNoteId() {
            return this.noteInfo.getId();
        }

        public String getNoteName() {
            return this.noteInfo.getNoteName();
        }

        public String getNotePath() {
            if (this.parent.getPath().equals("/")) {
                return this.parent.getPath() + this.noteInfo.getNoteName();
            }
            return this.parent.getPath() + "/" + this.noteInfo.getNoteName();
        }

        public NoteInfo getNoteInfo() {
            return this.noteInfo;
        }

        public Folder getParent() {
            return this.parent;
        }

        public String toString() {
            return this.getNotePath();
        }

        public void setParent(Folder parent) {
            this.parent = parent;
        }

        public void setNotePath(String notePath) {
            this.noteInfo.setPath(notePath);
        }

        public void updateNotePath() {
            this.noteInfo.setPath(this.getNotePath());
        }
    }

    public static class Folder {
        private String name;
        private Folder parent;
        private NotebookRepo notebookRepo;
        private NoteCache noteCache;
        private final ZeppelinConfiguration zConf;
        private Map<String, NoteNode> notes = new ConcurrentHashMap<String, NoteNode>();
        private Map<String, Folder> subFolders = new ConcurrentHashMap<String, Folder>();

        public Folder(String name, NotebookRepo notebookRepo, NoteCache noteCache, ZeppelinConfiguration zConf) {
            this.name = name;
            this.zConf = zConf;
            this.notebookRepo = notebookRepo;
            this.noteCache = noteCache;
        }

        public Folder(String name, Folder parent, NotebookRepo notebookRepo, NoteCache noteCache, ZeppelinConfiguration zConf) {
            this(name, notebookRepo, noteCache, zConf);
            this.parent = parent;
        }

        public synchronized Folder getOrCreateFolder(String folderName) {
            if (StringUtils.isBlank((CharSequence)folderName)) {
                return this;
            }
            if (!this.subFolders.containsKey(folderName)) {
                this.subFolders.put(folderName, new Folder(folderName, this, this.notebookRepo, this.noteCache, this.zConf));
            }
            return this.subFolders.get(folderName);
        }

        public Folder getParent() {
            return this.parent;
        }

        public void setParent(Folder parent) {
            this.parent = parent;
        }

        public String getName() {
            return this.name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public Folder getFolder(String folderName) {
            return this.subFolders.get(folderName);
        }

        public Map<String, Folder> getFolders() {
            return this.subFolders;
        }

        public NoteNode getNote(String noteName) {
            return this.notes.get(noteName);
        }

        public void addNote(String noteName, NoteInfo noteInfo) {
            this.notes.put(noteName, new NoteNode(noteInfo, this, this.notebookRepo, this.noteCache, this.zConf));
        }

        public void addFolder(String folderName, Folder folder) throws IOException {
            this.subFolders.put(folderName, folder);
            folder.setParent(this);
            folder.setName(folderName);
            for (NoteNode noteNode : folder.getNoteNodeRecursively()) {
                noteNode.updateNotePath();
            }
        }

        public boolean containsNote(String noteName) {
            return this.notes.containsKey(noteName);
        }

        public void addNoteNode(NoteNode noteNode) {
            this.notes.put(noteNode.getNoteName(), noteNode);
            noteNode.setParent(this);
        }

        public void removeNote(String noteName) {
            this.notes.remove(noteName);
        }

        public List<NoteInfo> removeFolder(String folderName, AuthenticationInfo subject) throws IOException {
            Folder folder = this.subFolders.remove(folderName);
            return folder.getNoteInfoRecursively();
        }

        public List<NoteInfo> getNoteInfoRecursively() {
            ArrayList<NoteInfo> notesInfo = new ArrayList<NoteInfo>();
            for (NoteNode noteNode : this.notes.values()) {
                notesInfo.add(noteNode.getNoteInfo());
            }
            for (Folder folder : this.subFolders.values()) {
                notesInfo.addAll(folder.getNoteInfoRecursively());
            }
            return notesInfo;
        }

        public List<NoteNode> getNoteNodeRecursively() {
            ArrayList<NoteNode> noteNodeRecursively = new ArrayList<NoteNode>();
            noteNodeRecursively.addAll(this.notes.values());
            for (Folder folder : this.subFolders.values()) {
                noteNodeRecursively.addAll(folder.getNoteNodeRecursively());
            }
            return noteNodeRecursively;
        }

        public Map<String, NoteNode> getNotes() {
            return this.notes;
        }

        public String getPath() {
            if (this.name.equals("/")) {
                return this.name;
            }
            if (this.parent.name.equals("/")) {
                return "/" + this.name;
            }
            return this.parent.toString() + "/" + this.name;
        }

        public String toString() {
            return this.getPath();
        }
    }
}

