/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.memory;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.nio.channels.ClosedByInterruptException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.annotation.concurrent.GuardedBy;
import org.apache.spark.internal.LogKey;
import org.apache.spark.internal.LogKeys;
import org.apache.spark.internal.MDC;
import org.apache.spark.internal.SparkLogger;
import org.apache.spark.internal.SparkLoggerFactory;
import org.apache.spark.memory.MemoryConsumer;
import org.apache.spark.memory.MemoryManager;
import org.apache.spark.memory.MemoryMode;
import org.apache.spark.memory.SparkOutOfMemoryError;
import org.apache.spark.memory.TooLargePageException;
import org.apache.spark.unsafe.memory.MemoryBlock;
import org.apache.spark.util.Utils;
import org.sparkproject.guava.annotations.VisibleForTesting;

public class TaskMemoryManager {
    private static final SparkLogger logger = SparkLoggerFactory.getLogger(TaskMemoryManager.class);
    private static final int PAGE_NUMBER_BITS = 13;
    @VisibleForTesting
    static final int OFFSET_BITS = 51;
    private static final int PAGE_TABLE_SIZE = 8192;
    public static final long MAXIMUM_PAGE_SIZE_BYTES = 0x3FFFFFFF8L;
    private static final long MASK_LONG_LOWER_51_BITS = 0x7FFFFFFFFFFFFL;
    private final MemoryBlock[] pageTable = new MemoryBlock[8192];
    private final BitSet allocatedPages = new BitSet(8192);
    private final MemoryManager memoryManager;
    private final long taskAttemptId;
    final MemoryMode tungstenMemoryMode;
    @GuardedBy(value="this")
    private final HashSet<MemoryConsumer> consumers;
    private volatile long acquiredButNotUsed = 0L;
    private long currentOffHeapMemory = 0L;
    private final Object offHeapMemoryLock = new Object();
    private long currentOnHeapMemory = 0L;
    private final Object onHeapMemoryLock = new Object();
    private volatile long peakOffHeapMemory = 0L;
    private volatile long peakOnHeapMemory = 0L;

    public TaskMemoryManager(MemoryManager memoryManager, long taskAttemptId) {
        this.tungstenMemoryMode = memoryManager.tungstenMemoryMode();
        this.memoryManager = memoryManager;
        this.taskAttemptId = taskAttemptId;
        this.consumers = new HashSet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long acquireExecutionMemory(long required, MemoryConsumer requestingConsumer) {
        assert (required >= 0L);
        assert (requestingConsumer != null);
        MemoryMode mode = requestingConsumer.getMode();
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            Object object;
            long got = this.memoryManager.acquireExecutionMemory(required, this.taskAttemptId, mode);
            if (got < required) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Task {} need to spill {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(required - got), requestingConsumer});
                }
                TreeMap<Long, List> sortedConsumers = new TreeMap<Long, List>();
                for (MemoryConsumer c : this.consumers) {
                    if (c.getUsed() <= 0L || c.getMode() != mode) continue;
                    long key = c == requestingConsumer ? 0L : c.getUsed();
                    List list = sortedConsumers.computeIfAbsent(key, k -> new ArrayList(1));
                    list.add(c);
                }
                while (got < required && !sortedConsumers.isEmpty()) {
                    Map.Entry currentEntry = sortedConsumers.ceilingEntry(required - got);
                    if (currentEntry == null) {
                        currentEntry = sortedConsumers.lastEntry();
                    }
                    List cList = (List)currentEntry.getValue();
                    got += this.trySpillAndAcquire(requestingConsumer, required - got, cList, cList.size() - 1);
                    if (!cList.isEmpty()) continue;
                    sortedConsumers.remove(currentEntry.getKey());
                }
            }
            this.consumers.add(requestingConsumer);
            if (logger.isDebugEnabled()) {
                logger.debug("Task {} acquired {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(got), requestingConsumer});
            }
            if (mode == MemoryMode.OFF_HEAP) {
                object = this.offHeapMemoryLock;
                synchronized (object) {
                    this.currentOffHeapMemory += got;
                    this.peakOffHeapMemory = Math.max(this.peakOffHeapMemory, this.currentOffHeapMemory);
                }
            }
            object = this.onHeapMemoryLock;
            synchronized (object) {
                this.currentOnHeapMemory += got;
                this.peakOnHeapMemory = Math.max(this.peakOnHeapMemory, this.currentOnHeapMemory);
            }
            return got;
        }
    }

    private long trySpillAndAcquire(MemoryConsumer requestingConsumer, long requested, List<MemoryConsumer> cList, int idx) {
        MemoryMode mode = requestingConsumer.getMode();
        final MemoryConsumer consumerToSpill = cList.get(idx);
        if (logger.isDebugEnabled()) {
            logger.debug("Task {} try to spill {} from {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(requested), consumerToSpill, requestingConsumer});
        }
        try {
            long released = consumerToSpill.spill(requested, requestingConsumer);
            if (released > 0L) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Task {} spilled {} of requested {} from {} for {}", new Object[]{this.taskAttemptId, Utils.bytesToString(released), Utils.bytesToString(requested), consumerToSpill, requestingConsumer});
                }
                return this.memoryManager.acquireExecutionMemory(requested, this.taskAttemptId, mode);
            }
            cList.remove(idx);
            return 0L;
        }
        catch (InterruptedIOException | ClosedByInterruptException e) {
            logger.error("Error while calling spill() on {}", (Throwable)e, new MDC[]{MDC.of((LogKey)LogKeys.MEMORY_CONSUMER, (Object)consumerToSpill)});
            throw new RuntimeException(e.getMessage());
        }
        catch (IOException e) {
            logger.error("Error while calling spill() on {}", (Throwable)e, new MDC[]{MDC.of((LogKey)LogKeys.MEMORY_CONSUMER, (Object)consumerToSpill)});
            throw new SparkOutOfMemoryError("SPILL_OUT_OF_MEMORY", (Map<String, String>)new HashMap<String, String>(){
                {
                    this.put("consumerToSpill", consumerToSpill.toString());
                    this.put("message", e.getMessage());
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseExecutionMemory(long size, MemoryConsumer consumer) {
        if (logger.isDebugEnabled()) {
            logger.debug("Task {} release {} from {}", new Object[]{this.taskAttemptId, Utils.bytesToString(size), consumer});
        }
        this.memoryManager.releaseExecutionMemory(size, this.taskAttemptId, consumer.getMode());
        if (consumer.getMode() == MemoryMode.OFF_HEAP) {
            Object object = this.offHeapMemoryLock;
            synchronized (object) {
                this.currentOffHeapMemory -= size;
            }
        }
        Object object = this.onHeapMemoryLock;
        synchronized (object) {
            this.currentOnHeapMemory -= size;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void showMemoryUsage() {
        logger.info("Memory used in task {}", new MDC[]{MDC.of((LogKey)LogKeys.TASK_ATTEMPT_ID, (Object)this.taskAttemptId)});
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            long memoryAccountedForByConsumers = 0L;
            for (MemoryConsumer c : this.consumers) {
                long totalMemUsage = c.getUsed();
                memoryAccountedForByConsumers += totalMemUsage;
                if (totalMemUsage <= 0L) continue;
                logger.info("Acquired by {}: {}", new MDC[]{MDC.of((LogKey)LogKeys.MEMORY_CONSUMER, (Object)c), MDC.of((LogKey)LogKeys.MEMORY_SIZE, (Object)Utils.bytesToString(totalMemUsage))});
            }
            long memoryNotAccountedFor = this.memoryManager.getExecutionMemoryUsageForTask(this.taskAttemptId) - memoryAccountedForByConsumers;
            logger.info("{} bytes of memory were used by task {} but are not associated with specific consumers", new MDC[]{MDC.of((LogKey)LogKeys.MEMORY_SIZE, (Object)memoryNotAccountedFor), MDC.of((LogKey)LogKeys.TASK_ATTEMPT_ID, (Object)this.taskAttemptId)});
            logger.info("{} bytes of memory are used for execution and {} bytes of memory are used for storage", new MDC[]{MDC.of((LogKey)LogKeys.EXECUTION_MEMORY_SIZE, (Object)this.memoryManager.executionMemoryUsed()), MDC.of((LogKey)LogKeys.STORAGE_MEMORY_SIZE, (Object)this.memoryManager.storageMemoryUsed())});
        }
    }

    public long pageSizeBytes() {
        return this.memoryManager.pageSizeBytes();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MemoryBlock allocatePage(long size, MemoryConsumer consumer) {
        int pageNumber;
        assert (consumer != null);
        assert (consumer.getMode() == this.tungstenMemoryMode);
        if (size > 0x3FFFFFFF8L) {
            throw new TooLargePageException(size);
        }
        long acquired = this.acquireExecutionMemory(size, consumer);
        if (acquired <= 0L) {
            return null;
        }
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            pageNumber = this.allocatedPages.nextClearBit(0);
            if (pageNumber >= 8192) {
                this.releaseExecutionMemory(acquired, consumer);
                throw new IllegalStateException("Have already allocated a maximum of 8192 pages");
            }
            this.allocatedPages.set(pageNumber);
        }
        MemoryBlock page = null;
        try {
            page = this.memoryManager.tungstenMemoryAllocator().allocate(acquired);
        }
        catch (OutOfMemoryError e) {
            logger.warn("Failed to allocate a page ({} bytes), try again.", new MDC[]{MDC.of((LogKey)LogKeys.PAGE_SIZE, (Object)acquired)});
            TaskMemoryManager taskMemoryManager2 = this;
            synchronized (taskMemoryManager2) {
                this.acquiredButNotUsed += acquired;
                this.allocatedPages.clear(pageNumber);
            }
            return this.allocatePage(size, consumer);
        }
        page.pageNumber = pageNumber;
        this.pageTable[pageNumber] = page;
        if (logger.isTraceEnabled()) {
            logger.trace("Allocate page number {} ({} bytes)", (Object)pageNumber, (Object)acquired);
        }
        return page;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void freePage(MemoryBlock page, MemoryConsumer consumer) {
        assert (page.pageNumber != -1) : "Called freePage() on memory that wasn't allocated with allocatePage()";
        assert (page.pageNumber != -3) : "Called freePage() on a memory block that has already been freed";
        assert (page.pageNumber != -2) : "Called freePage() on a memory block that has already been freed";
        assert (this.allocatedPages.get(page.pageNumber));
        this.pageTable[page.pageNumber] = null;
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            this.allocatedPages.clear(page.pageNumber);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Freed page number {} ({} bytes)", (Object)page.pageNumber, (Object)page.size());
        }
        long pageSize = page.size();
        page.pageNumber = -2;
        this.memoryManager.tungstenMemoryAllocator().free(page);
        this.releaseExecutionMemory(pageSize, consumer);
    }

    public long encodePageNumberAndOffset(MemoryBlock page, long offsetInPage) {
        if (this.tungstenMemoryMode == MemoryMode.OFF_HEAP) {
            offsetInPage -= page.getBaseOffset();
        }
        return TaskMemoryManager.encodePageNumberAndOffset(page.pageNumber, offsetInPage);
    }

    @VisibleForTesting
    public static long encodePageNumberAndOffset(int pageNumber, long offsetInPage) {
        assert (pageNumber >= 0) : "encodePageNumberAndOffset called with invalid page";
        return (long)pageNumber << 51 | offsetInPage & 0x7FFFFFFFFFFFFL;
    }

    @VisibleForTesting
    public static int decodePageNumber(long pagePlusOffsetAddress) {
        return (int)(pagePlusOffsetAddress >>> 51);
    }

    private static long decodeOffset(long pagePlusOffsetAddress) {
        return pagePlusOffsetAddress & 0x7FFFFFFFFFFFFL;
    }

    public Object getPage(long pagePlusOffsetAddress) {
        if (this.tungstenMemoryMode == MemoryMode.ON_HEAP) {
            int pageNumber = TaskMemoryManager.decodePageNumber(pagePlusOffsetAddress);
            assert (pageNumber >= 0 && pageNumber < 8192);
            MemoryBlock page = this.pageTable[pageNumber];
            assert (page != null);
            assert (page.getBaseObject() != null);
            return page.getBaseObject();
        }
        return null;
    }

    public long getOffsetInPage(long pagePlusOffsetAddress) {
        long offsetInPage = TaskMemoryManager.decodeOffset(pagePlusOffsetAddress);
        if (this.tungstenMemoryMode == MemoryMode.ON_HEAP) {
            return offsetInPage;
        }
        int pageNumber = TaskMemoryManager.decodePageNumber(pagePlusOffsetAddress);
        assert (pageNumber >= 0 && pageNumber < 8192);
        MemoryBlock page = this.pageTable[pageNumber];
        assert (page != null);
        return page.getBaseOffset() + offsetInPage;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long cleanUpAllAllocatedMemory() {
        TaskMemoryManager taskMemoryManager = this;
        synchronized (taskMemoryManager) {
            for (MemoryConsumer c : this.consumers) {
                if (c == null || c.getUsed() <= 0L || !logger.isDebugEnabled()) continue;
                logger.debug("unreleased {} memory from {}", (Object)Utils.bytesToString(c.getUsed()), (Object)c);
            }
            this.consumers.clear();
            for (MemoryBlock page : this.pageTable) {
                if (page == null) continue;
                if (logger.isDebugEnabled()) {
                    logger.debug("unreleased page: {} in task {}", (Object)page, (Object)this.taskAttemptId);
                }
                page.pageNumber = -2;
                this.memoryManager.tungstenMemoryAllocator().free(page);
            }
            Arrays.fill(this.pageTable, null);
        }
        this.memoryManager.releaseExecutionMemory(this.acquiredButNotUsed, this.taskAttemptId, this.tungstenMemoryMode);
        return this.memoryManager.releaseAllExecutionMemoryForTask(this.taskAttemptId);
    }

    public long getMemoryConsumptionForThisTask() {
        return this.memoryManager.getExecutionMemoryUsageForTask(this.taskAttemptId);
    }

    public MemoryMode getTungstenMemoryMode() {
        return this.tungstenMemoryMode;
    }

    public long getPeakOnHeapExecutionMemory() {
        return this.peakOnHeapMemory;
    }

    public long getPeakOffHeapExecutionMemory() {
        return this.peakOffHeapMemory;
    }
}

