/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.confignode.procedure;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Deque;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.iotdb.commons.concurrent.ThreadName;
import org.apache.iotdb.commons.utils.TestOnly;
import org.apache.iotdb.confignode.procedure.CompletedProcedureContainer;
import org.apache.iotdb.confignode.procedure.CompletedProcedureRecycler;
import org.apache.iotdb.confignode.procedure.InternalProcedure;
import org.apache.iotdb.confignode.procedure.Procedure;
import org.apache.iotdb.confignode.procedure.RootProcedureStack;
import org.apache.iotdb.confignode.procedure.StoppableThread;
import org.apache.iotdb.confignode.procedure.TimeoutExecutorThread;
import org.apache.iotdb.confignode.procedure.env.ConfigNodeProcedureEnv;
import org.apache.iotdb.confignode.procedure.exception.ProcedureException;
import org.apache.iotdb.confignode.procedure.scheduler.ProcedureScheduler;
import org.apache.iotdb.confignode.procedure.scheduler.SimpleProcedureScheduler;
import org.apache.iotdb.confignode.procedure.state.ProcedureLockState;
import org.apache.iotdb.confignode.procedure.state.ProcedureState;
import org.apache.iotdb.confignode.procedure.store.IProcedureStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProcedureExecutor<Env> {
    private static final Logger LOG = LoggerFactory.getLogger(ProcedureExecutor.class);
    private final ConcurrentHashMap<Long, CompletedProcedureContainer<Env>> completed = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, RootProcedureStack<Env>> rollbackStack = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, Procedure<Env>> procedures = new ConcurrentHashMap();
    private ThreadGroup threadGroup;
    private CopyOnWriteArrayList<WorkerThread> workerThreads;
    private TimeoutExecutorThread<Env> timeoutExecutor;
    private TimeoutExecutorThread<Env> workerMonitorExecutor;
    private int corePoolSize;
    private int maxPoolSize;
    private final ProcedureScheduler scheduler;
    private final AtomicLong workId = new AtomicLong(0L);
    private final AtomicInteger activeExecutorCount = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final Env environment;
    private final IProcedureStore<Env> store;

    public ProcedureExecutor(Env environment, IProcedureStore<Env> store, ProcedureScheduler scheduler) {
        this.environment = environment;
        this.scheduler = scheduler;
        this.store = store;
    }

    @TestOnly
    public ProcedureExecutor(Env environment, IProcedureStore<Env> store) {
        this(environment, store, new SimpleProcedureScheduler());
    }

    public void init(int numThreads) {
        this.corePoolSize = numThreads;
        this.maxPoolSize = 10 * numThreads;
        this.threadGroup = new ThreadGroup(ThreadName.CONFIG_NODE_PROCEDURE_WORKER.getName());
        this.timeoutExecutor = new TimeoutExecutorThread(this, this.threadGroup, ThreadName.CONFIG_NODE_TIMEOUT_EXECUTOR.getName());
        this.workerMonitorExecutor = new TimeoutExecutorThread(this, this.threadGroup, ThreadName.CONFIG_NODE_WORKER_THREAD_MONITOR.getName());
        this.workId.set(0L);
        this.workerThreads = new CopyOnWriteArrayList();
        for (int i = 0; i < this.corePoolSize; ++i) {
            this.workerThreads.add(new WorkerThread(this.threadGroup));
        }
        this.workerMonitorExecutor.add(new WorkerMonitor());
        this.scheduler.start();
        this.recover();
    }

    private void recover() {
        List<Procedure<Env>> procedureList = this.getProcedureListFromDifferentVersion();
        for (Procedure<Env> proc : procedureList) {
            if (proc.isFinished()) {
                this.completed.putIfAbsent(proc.getProcId(), new CompletedProcedureContainer<Env>(proc));
                continue;
            }
            if (!proc.hasParent()) {
                this.rollbackStack.put(proc.getProcId(), new RootProcedureStack());
            }
            this.procedures.putIfAbsent(proc.getProcId(), proc);
        }
        ArrayList runnableList = new ArrayList();
        ArrayList<Procedure<Env>> failedList = new ArrayList<Procedure<Env>>();
        ArrayList<Procedure<Env>> waitingList = new ArrayList<Procedure<Env>>();
        ArrayList<Procedure<Env>> waitingTimeoutList = new ArrayList<Procedure<Env>>();
        for (Procedure<Env> proc : procedureList) {
            RootProcedureStack<Env> rootStack;
            Procedure<Env> parent;
            if (proc.isFinished() && !proc.hasParent()) continue;
            long rootProcedureId = this.getRootProcedureId(proc);
            if (proc.hasParent() && (parent = this.procedures.get(proc.getParentProcId())) != null && !proc.isFinished()) {
                parent.incChildrenLatch();
            }
            if ((rootStack = this.rollbackStack.get(rootProcedureId)) != null) {
                rootStack.loadStack(proc);
            }
            proc.setRootProcedureId(rootProcedureId);
            switch (proc.getState()) {
                case RUNNABLE: {
                    runnableList.add(proc);
                    break;
                }
                case FAILED: {
                    failedList.add(proc);
                    break;
                }
                case WAITING: {
                    waitingList.add(proc);
                    break;
                }
                case WAITING_TIMEOUT: {
                    waitingTimeoutList.add(proc);
                    break;
                }
                case ROLLEDBACK: 
                case INITIALIZING: {
                    LOG.error("Unexpected state:{} for {}", (Object)proc.getState(), proc);
                    throw new UnsupportedOperationException("Unexpected state");
                }
            }
        }
        waitingList.forEach(procedure -> {
            if (!procedure.hasChildren()) {
                procedure.setState(ProcedureState.RUNNABLE);
                runnableList.add((Procedure)procedure);
            }
        });
        this.restoreLocks();
        waitingTimeoutList.forEach(this.timeoutExecutor::add);
        failedList.forEach(this.scheduler::addBack);
        runnableList.forEach(this.scheduler::addBack);
        this.scheduler.signalAll();
    }

    private List<Procedure<Env>> getProcedureListFromDifferentVersion() {
        if (this.store.isOldVersionProcedureStore()) {
            LOG.info("Old procedure directory detected, upgrade beginning...");
            return this.store.load();
        }
        return this.store.getProcedures();
    }

    Long getRootProcedureId(Procedure<Env> proc) {
        while (proc.hasParent()) {
            if ((proc = this.procedures.get(proc.getParentProcId())) != null) continue;
            return -1L;
        }
        return proc.getProcId();
    }

    private void releaseLock(Procedure<Env> procedure, boolean force) {
        if (force || !procedure.holdLock(this.environment) || procedure.isFinished()) {
            procedure.doReleaseLock(this.environment, this.store);
        }
    }

    private void restoreLock(Procedure procedure, Set<Long> restored) {
        procedure.restoreLock(this.environment);
        restored.add(procedure.getProcId());
    }

    private void restoreLocks(Deque<Procedure<Env>> stack, Set<Long> restored) {
        while (!stack.isEmpty()) {
            this.restoreLock(stack.pop(), restored);
        }
    }

    private void restoreLocks() {
        HashSet restored = new HashSet();
        ArrayDeque stack = new ArrayDeque();
        this.procedures.values().forEach(procedure -> {
            while (procedure != null) {
                if (restored.contains(procedure.getProcId())) {
                    this.restoreLocks(stack, restored);
                    return;
                }
                if (!procedure.hasParent()) {
                    this.restoreLock((Procedure)procedure, restored);
                    this.restoreLocks(stack, restored);
                    return;
                }
                stack.push(procedure);
                procedure = this.procedures.get(procedure.getParentProcId());
            }
        });
    }

    public void startWorkers() {
        if (!this.running.compareAndSet(false, true)) {
            LOG.warn("Already running");
            return;
        }
        this.timeoutExecutor.start();
        this.workerMonitorExecutor.start();
        for (WorkerThread workerThread : this.workerThreads) {
            workerThread.start();
        }
        LOG.info("{} procedure workers are started.", (Object)this.workerThreads.size());
    }

    public void startCompletedCleaner(long cleanTimeInterval, long cleanEvictTTL) {
        this.addInternalProcedure(new CompletedProcedureRecycler<Env>(this.store, this.completed, cleanTimeInterval, cleanEvictTTL));
    }

    public void addInternalProcedure(InternalProcedure interalProcedure) {
        if (interalProcedure == null) {
            return;
        }
        interalProcedure.setState(ProcedureState.WAITING_TIMEOUT);
        this.timeoutExecutor.add(interalProcedure);
    }

    public boolean removeInternalProcedure(InternalProcedure internalProcedure) {
        if (internalProcedure == null) {
            return true;
        }
        internalProcedure.setState(ProcedureState.SUCCESS);
        return this.timeoutExecutor.remove(internalProcedure);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void executeProcedure(Procedure<Env> proc) {
        block27: {
            if (proc.isFinished()) {
                LOG.debug("{} is already finished.", proc);
                return;
            }
            Long rootProcId = this.getRootProcedureId(proc);
            if (rootProcId == null) {
                LOG.warn("Rollback because parent is done/rolledback, proc is {}", proc);
                this.executeRollback(proc);
                return;
            }
            RootProcedureStack<Env> rootProcStack = this.rollbackStack.get(rootProcId);
            if (rootProcStack == null) {
                LOG.warn("Rollback stack is null for {}", (Object)proc.getProcId());
                return;
            }
            ProcedureLockState lockState = null;
            try {
                do {
                    if (!rootProcStack.acquire()) {
                        if (rootProcStack.setRollback()) {
                            lockState = this.executeRootStackRollback(rootProcId, rootProcStack);
                            switch (lockState) {
                                case LOCK_ACQUIRED: {
                                    break block27;
                                }
                                case LOCK_EVENT_WAIT: {
                                    LOG.info("LOCK_EVENT_WAIT rollback {}", proc);
                                    rootProcStack.unsetRollback();
                                    break block27;
                                }
                                case LOCK_YIELD_WAIT: {
                                    rootProcStack.unsetRollback();
                                    this.scheduler.yield(proc);
                                    break block27;
                                }
                                default: {
                                    throw new UnsupportedOperationException();
                                }
                            }
                        }
                        if (proc.wasExecuted()) break;
                        switch (this.executeRollback(proc)) {
                            case LOCK_ACQUIRED: {
                                break block27;
                            }
                            case LOCK_EVENT_WAIT: {
                                LOG.info("LOCK_EVENT_WAIT can't rollback child running for {}", proc);
                                break block27;
                            }
                            case LOCK_YIELD_WAIT: {
                                this.scheduler.yield(proc);
                                break block27;
                            }
                            default: {
                                throw new UnsupportedOperationException();
                            }
                        }
                    }
                    lockState = this.acquireLock(proc);
                    switch (lockState) {
                        case LOCK_ACQUIRED: {
                            this.executeProcedure(rootProcStack, proc);
                            break;
                        }
                        case LOCK_EVENT_WAIT: 
                        case LOCK_YIELD_WAIT: {
                            LOG.info("{} lockstate is {}", proc, (Object)lockState);
                            break;
                        }
                        default: {
                            throw new UnsupportedOperationException();
                        }
                    }
                    rootProcStack.release();
                    if (!proc.isSuccess()) continue;
                    proc.updateMetricsOnFinish(this.getEnvironment(), proc.elapsedTime(), true);
                    LOG.debug("{} finished in {}ms successfully.", proc, (Object)proc.elapsedTime());
                    if (proc.getProcId() == rootProcId.longValue()) {
                        this.rootProcedureCleanup(proc);
                    } else {
                        this.executeCompletionCleanup(proc);
                    }
                    return;
                } while (rootProcStack.isFailed());
            }
            finally {
                if (Objects.equals((Object)lockState, (Object)ProcedureLockState.LOCK_EVENT_WAIT)) {
                    LOG.info("procedureId {} wait for lock.", (Object)proc.getProcId());
                    ((ConfigNodeProcedureEnv)this.environment).getNodeLock().waitProcedure(proc);
                }
            }
        }
    }

    private void executeProcedure(RootProcedureStack rootProcStack, Procedure<Env> proc) {
        boolean reExecute;
        if (proc.getState() != ProcedureState.RUNNABLE) {
            LOG.error("The executing procedure should in RUNNABLE state, but it's not. Procedure is {}", proc);
            return;
        }
        Object[] subprocs = null;
        do {
            reExecute = false;
            try {
                subprocs = proc.doExecute(this.environment);
                if (subprocs != null && subprocs.length == 0) {
                    subprocs = null;
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupt during execution, suspend or retry it later.", (Throwable)e);
                this.yieldProcedure(proc);
            }
            catch (Throwable e) {
                LOG.error("CODE-BUG:{}", proc, (Object)e);
                proc.setFailure(new ProcedureException(e.getMessage(), e));
            }
            if (!proc.isFailed()) {
                if (subprocs != null) {
                    if (subprocs.length == 1 && subprocs[0] == proc) {
                        subprocs = null;
                        reExecute = true;
                    } else {
                        subprocs = this.initializeChildren(rootProcStack, proc, (Procedure<Env>[])subprocs);
                        LOG.info("Initialized sub procs:{}", (Object)Arrays.toString(subprocs));
                    }
                } else if (proc.getState() == ProcedureState.WAITING_TIMEOUT) {
                    LOG.info("Added into timeoutExecutor {}", proc);
                } else {
                    proc.setState(ProcedureState.SUCCESS);
                }
            }
            rootProcStack.addRollbackStep(proc);
            this.updateStoreOnExecution(rootProcStack, proc, (Procedure<Env>[])subprocs);
            if (!this.store.isRunning()) {
                return;
            }
            if (!proc.isRunnable() || !proc.isYieldAfterExecution(this.environment)) continue;
            this.yieldProcedure(proc);
            return;
        } while (reExecute);
        if (subprocs != null && !proc.isFailed()) {
            this.submitChildrenProcedures((Procedure<Env>[])subprocs);
        }
        this.releaseLock(proc, false);
        if (proc.isFinished() && proc.hasParent()) {
            this.countDownChildren(rootProcStack, proc);
        }
    }

    private void countDownChildren(RootProcedureStack rootProcStack, Procedure<Env> proc) {
        Procedure<Env> parent = this.procedures.get(proc.getParentProcId());
        if (parent == null && rootProcStack.isRollingback()) {
            return;
        }
        if (parent != null && parent.tryRunnable()) {
            this.store.update(parent);
            this.scheduler.addFront(parent);
            LOG.info("Finished subprocedure pid={}, resume processing ppid={}", (Object)proc.getProcId(), (Object)parent.getProcId());
        }
    }

    private void submitChildrenProcedures(Procedure<Env>[] subprocs) {
        for (Procedure<Env> subproc : subprocs) {
            subproc.updateMetricsOnSubmit(this.getEnvironment());
            this.procedures.put(subproc.getProcId(), subproc);
            this.scheduler.addFront(subproc);
            LOG.info("Sub-Procedure pid={} has been submitted", (Object)subproc.getProcId());
        }
    }

    private void updateStoreOnExecution(RootProcedureStack rootProcStack, Procedure<Env> proc, Procedure<Env>[] subprocs) {
        if (subprocs != null && !proc.isFailed()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stored {}, children {}", proc, (Object)Arrays.toString(subprocs));
            }
            this.store.update(subprocs);
        } else {
            LOG.debug("Store update {}", proc);
            if (proc.isFinished() && !proc.hasParent()) {
                long[] childProcIds = rootProcStack.getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(childProcIds);
                    for (long childProcId : childProcIds) {
                        this.procedures.remove(childProcId);
                    }
                } else {
                    this.store.update(proc);
                }
            } else {
                this.store.update(proc);
            }
        }
    }

    private Procedure<Env>[] initializeChildren(RootProcedureStack rootProcStack, Procedure<Env> proc, Procedure<Env>[] subprocs) {
        long rootProcedureId = this.getRootProcedureId(proc);
        for (int i = 0; i < subprocs.length; ++i) {
            Procedure<Env> subproc = subprocs[i];
            if (subproc == null) {
                String errMsg = "subproc[" + i + "] is null, aborting procedure";
                proc.setFailure(new ProcedureException(errMsg, new IllegalArgumentException(errMsg)));
                return null;
            }
            subproc.setParentProcId(proc.getProcId());
            subproc.setRootProcId(rootProcedureId);
            subproc.setProcId(this.store.getNextProcId());
            subproc.setProcRunnable();
            rootProcStack.addSubProcedure(subproc);
        }
        if (!proc.isFailed()) {
            proc.setChildrenLatch(subprocs.length);
            switch (proc.getState()) {
                case RUNNABLE: {
                    proc.setState(ProcedureState.WAITING);
                    break;
                }
                case WAITING_TIMEOUT: {
                    this.timeoutExecutor.add(proc);
                    break;
                }
            }
        }
        return subprocs;
    }

    private void yieldProcedure(Procedure<Env> proc) {
        this.releaseLock(proc, false);
        this.scheduler.yield(proc);
    }

    private ProcedureLockState executeRootStackRollback(Long rootProcId, RootProcedureStack procedureStack) {
        Procedure<Env> rootProcedure = this.procedures.get(rootProcId);
        ProcedureException exception = rootProcedure.getException();
        if (exception == null) {
            exception = procedureStack.getException();
            rootProcedure.setFailure(exception);
            this.store.update(rootProcedure);
        }
        List subprocStack = procedureStack.getSubproceduresStack();
        int stackTail = subprocStack.size();
        while (stackTail-- > 0) {
            Procedure<Env> procedure = subprocStack.get(stackTail);
            if (procedure.isSuccess()) {
                subprocStack.remove(stackTail);
                this.cleanupAfterRollback(procedure);
                continue;
            }
            ProcedureLockState lockState = this.acquireLock(procedure);
            if (lockState != ProcedureLockState.LOCK_ACQUIRED) {
                return lockState;
            }
            lockState = this.executeRollback(procedure);
            this.releaseLock(procedure, false);
            boolean abortRollback = lockState != ProcedureLockState.LOCK_ACQUIRED;
            if (abortRollback |= !this.isRunning() || !this.store.isRunning()) {
                return lockState;
            }
            if (!procedure.isFinished() && procedure.isYieldAfterExecution(this.environment)) {
                return ProcedureLockState.LOCK_YIELD_WAIT;
            }
            if (procedure == rootProcedure) continue;
            this.executeCompletionCleanup(procedure);
        }
        LOG.info("Rolled back {}, time duration is {}", rootProcedure, (Object)rootProcedure.elapsedTime());
        this.rootProcedureCleanup(rootProcedure);
        return ProcedureLockState.LOCK_ACQUIRED;
    }

    private ProcedureLockState acquireLock(Procedure<Env> proc) {
        if (proc.hasLock()) {
            return ProcedureLockState.LOCK_ACQUIRED;
        }
        return proc.doAcquireLock(this.environment, this.store);
    }

    private ProcedureLockState executeRollback(Procedure<Env> procedure) {
        try {
            procedure.doRollback(this.environment);
        }
        catch (IOException e) {
            LOG.error("Roll back failed for {}", procedure, (Object)e);
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted exception occured for {}", procedure, (Object)e);
        }
        catch (Throwable t) {
            LOG.error("CODE-BUG: runtime exception for {}", procedure, (Object)t);
        }
        this.cleanupAfterRollback(procedure);
        return ProcedureLockState.LOCK_ACQUIRED;
    }

    private void cleanupAfterRollback(Procedure<Env> procedure) {
        if (procedure.removeStackIndex()) {
            if (!procedure.isSuccess()) {
                procedure.setState(ProcedureState.ROLLEDBACK);
            }
            procedure.updateMetricsOnFinish(this.getEnvironment(), procedure.elapsedTime(), false);
            if (procedure.hasParent()) {
                this.store.delete(procedure.getProcId());
                this.procedures.remove(procedure.getProcId());
            } else {
                long[] childProcIds = this.rollbackStack.get(procedure.getProcId()).getSubprocedureIds();
                if (childProcIds != null) {
                    this.store.delete(childProcIds);
                } else {
                    this.store.update(procedure);
                }
            }
        } else {
            this.store.update(procedure);
        }
    }

    private void executeCompletionCleanup(Procedure<Env> proc) {
        if (proc.hasLock()) {
            this.releaseLock(proc, true);
        }
    }

    private void rootProcedureCleanup(Procedure<Env> proc) {
        this.executeCompletionCleanup(proc);
        CompletedProcedureContainer<Env> retainer = new CompletedProcedureContainer<Env>(proc);
        this.completed.put(proc.getProcId(), retainer);
        this.rollbackStack.remove(proc.getProcId());
        this.procedures.remove(proc.getProcId());
    }

    private long pushProcedure(Procedure<Env> procedure) {
        long currentProcId = procedure.getProcId();
        procedure.updateMetricsOnSubmit(this.getEnvironment());
        RootProcedureStack stack = new RootProcedureStack();
        this.rollbackStack.put(currentProcId, stack);
        this.procedures.put(currentProcId, procedure);
        this.scheduler.addBack(procedure);
        return procedure.getProcId();
    }

    public int getWorkerThreadCount() {
        return this.workerThreads.size();
    }

    public long getActiveWorkerThreadCount() {
        return this.workerThreads.stream().filter(worker -> ((WorkerThread)worker).activeProcedure.get() != null).count();
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public void stop() {
        if (!this.running.getAndSet(false)) {
            return;
        }
        LOG.info("Stopping");
        this.scheduler.stop();
        this.timeoutExecutor.sendStopSignal();
    }

    public void join() {
        this.timeoutExecutor.awaitTermination();
        this.workerMonitorExecutor.awaitTermination();
        for (WorkerThread workerThread : this.workerThreads) {
            workerThread.awaitTermination();
        }
        try {
            this.threadGroup.destroy();
        }
        catch (IllegalThreadStateException e) {
            LOG.warn("ProcedureExecutor threadGroup {} contains running threads which are used by non-procedure module.", (Object)this.threadGroup);
            this.threadGroup.list();
        }
    }

    public boolean isStarted(long procId) {
        Procedure<Env> procedure = this.procedures.get(procId);
        if (procedure == null) {
            return this.completed.get(procId) != null;
        }
        return procedure.wasExecuted();
    }

    public boolean isFinished(long procId) {
        return !this.procedures.containsKey(procId);
    }

    public ConcurrentHashMap<Long, Procedure<Env>> getProcedures() {
        return this.procedures;
    }

    public long submitProcedure(Procedure<Env> procedure) {
        Preconditions.checkArgument((procedure.getState() == ProcedureState.INITIALIZING ? 1 : 0) != 0);
        Preconditions.checkArgument((!procedure.hasParent() ? 1 : 0) != 0, (String)"Unexpected parent", procedure);
        procedure.setProcId(this.store.getNextProcId());
        procedure.setProcRunnable();
        this.store.update(procedure);
        LOG.debug("{} is stored.", procedure);
        return this.pushProcedure(procedure);
    }

    public ProcedureScheduler getScheduler() {
        return this.scheduler;
    }

    public Env getEnvironment() {
        return this.environment;
    }

    public IProcedureStore getStore() {
        return this.store;
    }

    public RootProcedureStack<Env> getRollbackStack(long rootProcId) {
        return this.rollbackStack.get(rootProcId);
    }

    private class WorkerThread
    extends StoppableThread {
        private final AtomicLong startTime;
        private final AtomicReference<Procedure<Env>> activeProcedure;
        protected long keepAliveTime;

        public WorkerThread(ThreadGroup threadGroup) {
            this(threadGroup, "ProcedureCoreWorker-");
        }

        public WorkerThread(ThreadGroup threadGroup, String prefix) {
            super(threadGroup, prefix + ProcedureExecutor.this.workId.incrementAndGet());
            this.startTime = new AtomicLong(Long.MAX_VALUE);
            this.activeProcedure = new AtomicReference();
            this.keepAliveTime = -1L;
            this.setDaemon(true);
        }

        @Override
        public void sendStopSignal() {
            ProcedureExecutor.this.scheduler.signalAll();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            long lastUpdated = System.currentTimeMillis();
            try {
                while (ProcedureExecutor.this.isRunning() && this.keepAlive(lastUpdated)) {
                    Procedure procedure = ProcedureExecutor.this.scheduler.poll(this.keepAliveTime, TimeUnit.MILLISECONDS);
                    if (procedure == null) {
                        Thread.sleep(1000L);
                        continue;
                    }
                    this.activeProcedure.set(procedure);
                    ProcedureExecutor.this.activeExecutorCount.incrementAndGet();
                    this.startTime.set(System.currentTimeMillis());
                    ProcedureExecutor.this.executeProcedure(procedure);
                    ProcedureExecutor.this.activeExecutorCount.decrementAndGet();
                    LOG.trace("Halt pid={}, activeCount={}", (Object)procedure.getProcId(), (Object)ProcedureExecutor.this.activeExecutorCount.get());
                    this.activeProcedure.set(null);
                    lastUpdated = System.currentTimeMillis();
                    this.startTime.set(lastUpdated);
                }
            }
            catch (Exception e) {
                if (this.activeProcedure.get() != null) {
                    LOG.warn("Exception happened when worker {} execute procedure {}", new Object[]{this.getName(), this.activeProcedure.get(), e});
                }
            }
            finally {
                LOG.info("Procedure worker {} terminated.", (Object)this.getName());
            }
            ProcedureExecutor.this.workerThreads.remove(this);
        }

        protected boolean keepAlive(long lastUpdated) {
            return true;
        }

        @Override
        public String toString() {
            Procedure p = this.activeProcedure.get();
            return this.getName() + "(pid=" + (p == null ? Long.valueOf(-1L) : p.getProcId() + ")");
        }

        public long getCurrentRunTime() {
            return System.currentTimeMillis() - this.startTime.get();
        }
    }

    private final class WorkerMonitor
    extends InternalProcedure<Env> {
        private static final int DEFAULT_WORKER_MONITOR_INTERVAL = 30000;
        private static final int DEFAULT_WORKER_STUCK_THRESHOLD = 60000;
        private static final float DEFAULT_WORKER_ADD_STUCK_PERCENTAGE = 0.5f;

        public WorkerMonitor() {
            super(30000L);
            this.updateTimestamp();
        }

        private int calculateRunningAndStuckWorkers() {
            int runningCount = 0;
            int stuckCount = 0;
            for (WorkerThread worker : ProcedureExecutor.this.workerThreads) {
                if (worker.activeProcedure.get() == null) continue;
                ++runningCount;
                if (worker.getCurrentRunTime() < 60000L) {
                    ++stuckCount;
                    LOG.warn("Worker stuck {}({}), run time {} ms", new Object[]{worker, ((Procedure)worker.activeProcedure.get()).getProcType(), worker.getCurrentRunTime()});
                }
                LOG.info("Procedure workers: {} is running, {} is running and stuck", (Object)runningCount, (Object)stuckCount);
            }
            return stuckCount;
        }

        private void checkThreadCount(int stuckCount) {
            if (stuckCount < 1 || !ProcedureExecutor.this.scheduler.hasRunnables()) {
                return;
            }
            float stuckPerc = (float)stuckCount / (float)ProcedureExecutor.this.workerThreads.size();
            if (stuckPerc >= 0.5f && ProcedureExecutor.this.workerThreads.size() < ProcedureExecutor.this.maxPoolSize) {
                TemporaryWorkerThread worker = new TemporaryWorkerThread(ProcedureExecutor.this.threadGroup);
                ProcedureExecutor.this.workerThreads.add(worker);
                worker.start();
                LOG.debug("Added new worker thread {}", (Object)worker);
            }
        }

        @Override
        protected void periodicExecute(Env env) {
            int stuckCount = this.calculateRunningAndStuckWorkers();
            this.checkThreadCount(stuckCount);
            this.updateTimestamp();
        }
    }

    private final class TemporaryWorkerThread
    extends WorkerThread {
        public TemporaryWorkerThread(ThreadGroup group) {
            super(group, "ProcedureTemporaryWorker-");
            this.keepAliveTime = TimeUnit.SECONDS.toMillis(10L);
        }

        @Override
        protected boolean keepAlive(long lastUpdate) {
            return System.currentTimeMillis() - lastUpdate < this.keepAliveTime;
        }
    }
}

