/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.loader;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hugegraph.driver.HugeClient;
import org.apache.hugegraph.loader.constant.Constants;
import org.apache.hugegraph.loader.constant.ElemType;
import org.apache.hugegraph.loader.exception.InitException;
import org.apache.hugegraph.loader.exception.LoadException;
import org.apache.hugegraph.loader.exception.ReadException;
import org.apache.hugegraph.loader.executor.GroovyExecutor;
import org.apache.hugegraph.loader.executor.LoadContext;
import org.apache.hugegraph.loader.executor.LoadOptions;
import org.apache.hugegraph.loader.mapping.ElementMapping;
import org.apache.hugegraph.loader.mapping.InputStruct;
import org.apache.hugegraph.loader.mapping.LoadMapping;
import org.apache.hugegraph.loader.metrics.LoadMetrics;
import org.apache.hugegraph.loader.metrics.LoadSummary;
import org.apache.hugegraph.loader.reader.InputReader;
import org.apache.hugegraph.loader.reader.line.Line;
import org.apache.hugegraph.loader.task.ParseTaskBuilder;
import org.apache.hugegraph.loader.task.TaskManager;
import org.apache.hugegraph.loader.util.HugeClientHolder;
import org.apache.hugegraph.loader.util.LoadUtil;
import org.apache.hugegraph.loader.util.Printer;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public final class HugeGraphLoader {
    public static final Logger LOG = Log.logger(HugeGraphLoader.class);
    private final LoadContext context;
    private final LoadMapping mapping;
    private final TaskManager manager;

    public static void main(String[] args) {
        HugeGraphLoader loader;
        try {
            loader = new HugeGraphLoader(args);
        }
        catch (Throwable e) {
            Printer.printError("Failed to start loading", LoadUtil.targetRuntimeException(e));
            throw e;
        }
        loader.load();
    }

    public HugeGraphLoader(String[] args) {
        this(LoadOptions.parseOptions(args));
    }

    public HugeGraphLoader(LoadOptions options) {
        this(options, LoadMapping.of(options.file));
    }

    public HugeGraphLoader(LoadOptions options, LoadMapping mapping) {
        this.context = new LoadContext(options);
        this.mapping = mapping;
        this.manager = new TaskManager(this.context);
        this.addShutdownHook();
    }

    private void addShutdownHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            LOG.info("Shutdown hook was triggered");
            this.stopThenShutdown();
        }));
    }

    public LoadContext context() {
        return this.context;
    }

    public boolean load() {
        try {
            this.context.setLoadingMode();
            this.clearAllDataIfNeeded();
            this.createSchema();
            this.loadInputs();
            Printer.printSummary(this.context);
        }
        catch (Throwable t) {
            RuntimeException e = LoadUtil.targetRuntimeException(t);
            Printer.printError("Failed to load", e);
            if (this.context.options().testMode) {
                throw e;
            }
        }
        finally {
            this.stopThenShutdown();
        }
        return this.context.noError();
    }

    private void clearAllDataIfNeeded() {
        LoadOptions options = this.context.options();
        if (!options.clearAllData) {
            return;
        }
        int requestTimeout = options.timeout;
        options.timeout = options.clearTimeout;
        HugeClient client = HugeClientHolder.create(options);
        String message = "I'm sure to delete all data";
        LOG.info("Prepare to clear the data of graph '{}'", (Object)options.graph);
        client.graphs().clearGraph(options.graph, message);
        LOG.info("The graph '{}' has been cleared successfully", (Object)options.graph);
        options.timeout = requestTimeout;
        client.close();
    }

    private void createSchema() {
        LoadOptions options = this.context.options();
        if (!StringUtils.isEmpty((CharSequence)options.schema)) {
            String script;
            File file = FileUtils.getFile((String[])new String[]{options.schema});
            HugeClient client = this.context.client();
            GroovyExecutor groovyExecutor = new GroovyExecutor();
            groovyExecutor.bind("schema", client.schema());
            try {
                script = FileUtils.readFileToString((File)file, (Charset)Constants.CHARSET);
            }
            catch (IOException e) {
                throw new LoadException("Failed to read schema file '%s'", (Throwable)e, options.schema);
            }
            groovyExecutor.execute(script, client);
        }
        this.context.updateSchemaCache();
    }

    private void loadInputs() {
        Printer.printRealtimeProgress(this.context);
        LoadOptions options = this.context.options();
        LoadSummary summary = this.context.summary();
        summary.initMetrics(this.mapping);
        summary.startTotalTimer();
        try {
            if (!options.failureMode) {
                this.loadInputs(this.mapping.structs());
            } else {
                this.loadInputs(this.mapping.structsForFailure(options));
            }
            this.manager.waitFinished();
        }
        finally {
            summary.calculateTotalTime(ElemType.VERTEX);
            summary.calculateTotalTime(ElemType.EDGE);
            summary.stopTotalTimer();
        }
        Printer.printFinalProgress(this.context);
    }

    private void loadInputs(List<InputStruct> structs) {
        if (this.context.options().checkVertex) {
            LOG.info("Forced to load vertices before edges since set option check-vertex=true");
            SplitInputStructs split = this.splitStructs(structs);
            this.loadStructs(split.vertexInputStructs);
            this.manager.waitFinished("vertex insert tasks");
            this.loadStructs(split.edgeInputStructs);
        } else {
            this.loadStructs(structs);
        }
    }

    private void loadStructs(List<InputStruct> structs) {
        for (InputStruct struct : structs) {
            if (this.context.stopped()) break;
            if (struct.skip()) continue;
            try {
                InputReader reader = InputReader.create(struct.input());
                Throwable throwable = null;
                try {
                    reader.init(this.context, struct);
                    this.loadStruct(struct, reader);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (reader == null) continue;
                    if (throwable != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    reader.close();
                }
            }
            catch (InitException e) {
                throw new LoadException("Failed to init input reader", e);
            }
        }
    }

    private void loadStruct(InputStruct struct, InputReader reader) {
        LOG.info("Start loading '{}'", (Object)struct);
        LoadMetrics metrics = this.context.summary().metrics(struct);
        metrics.startInFlight();
        ParseTaskBuilder taskBuilder = new ParseTaskBuilder(this.context, struct);
        int batchSize = this.context.options().batchSize;
        ArrayList<Line> lines = new ArrayList<Line>(batchSize);
        boolean finished = false;
        while (!finished && !this.context.stopped()) {
            try {
                if (reader.hasNext()) {
                    lines.add((Line)reader.next());
                    metrics.increaseReadSuccess();
                } else {
                    finished = true;
                }
            }
            catch (ReadException e) {
                metrics.increaseReadFailure();
                this.handleReadFailure(struct, e);
            }
            boolean reachedMaxReadLines = this.reachedMaxReadLines();
            if (reachedMaxReadLines) {
                finished = true;
            }
            if (lines.size() < batchSize && !finished) continue;
            List<ParseTaskBuilder.ParseTask> tasks = taskBuilder.build(lines);
            for (ParseTaskBuilder.ParseTask task : tasks) {
                this.executeParseTask(struct, task.mapping(), task);
            }
            reader.confirmOffset();
            this.context.newProgress().markLoaded(struct, finished);
            this.handleParseFailure();
            if (reachedMaxReadLines) {
                LOG.warn("Read lines exceed limit, stopped loading tasks");
                this.context.stopLoading();
            }
            lines = new ArrayList(batchSize);
        }
        metrics.stopInFlight();
        LOG.info("Finish loading '{}'", (Object)struct);
    }

    private void executeParseTask(InputStruct struct, ElementMapping mapping, ParseTaskBuilder.ParseTask task) {
        long start = System.currentTimeMillis();
        Object batches = task.get();
        long end = System.currentTimeMillis();
        this.context.summary().addTimeRange(mapping.type(), start, end);
        if (this.context.options().dryRun || CollectionUtils.isEmpty((Collection)batches)) {
            return;
        }
        Iterator iterator = batches.iterator();
        while (iterator.hasNext()) {
            List batch = (List)iterator.next();
            this.manager.submitBatch(struct, mapping, batch);
        }
    }

    private void handleReadFailure(InputStruct struct, ReadException e) {
        LOG.error("Read {} error", (Object)struct, (Object)e);
        this.context.occurredError();
        LoadOptions options = this.context.options();
        if (options.testMode) {
            throw e;
        }
        this.context.failureLogger(struct).write(e);
        long failures = this.context.summary().totalReadFailures();
        if (options.maxReadErrors != -1 && failures >= (long)options.maxReadErrors) {
            Printer.printError("More than %s read error, stop reading and waiting all parse/insert tasks stopped", options.maxReadErrors);
            this.context.stopLoading();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleParseFailure() {
        LoadOptions options = this.context.options();
        long failures = this.context.summary().totalParseFailures();
        if (options.maxParseErrors != -1 && failures >= (long)options.maxParseErrors) {
            if (this.context.stopped()) {
                return;
            }
            LoadContext loadContext = this.context;
            synchronized (loadContext) {
                if (!this.context.stopped()) {
                    Printer.printError("More than %s parse error, stop parsing and waiting all insert tasks stopped", options.maxParseErrors);
                    this.context.stopLoading();
                }
            }
        }
    }

    private SplitInputStructs splitStructs(List<InputStruct> structs) {
        InputStruct result;
        SplitInputStructs split = new SplitInputStructs();
        for (InputStruct struct : structs) {
            result = struct.extractVertexStruct();
            if (result == InputStruct.EMPTY) continue;
            split.vertexInputStructs.add(result);
        }
        for (InputStruct struct : structs) {
            result = struct.extractEdgeStruct();
            if (result == InputStruct.EMPTY) continue;
            split.edgeInputStructs.add(result);
        }
        return split;
    }

    private boolean reachedMaxReadLines() {
        long maxReadLines = this.context.options().maxReadLines;
        if (maxReadLines == -1L) {
            return false;
        }
        return this.context.summary().totalReadLines() >= maxReadLines;
    }

    private synchronized void stopThenShutdown() {
        if (this.context.closed()) {
            return;
        }
        LOG.info("Stop loading then shutdown HugeGraphLoader");
        try {
            this.context.stopLoading();
            if (this.manager != null) {
                this.manager.waitFinished();
                this.manager.shutdown();
            }
        }
        finally {
            try {
                this.context.unsetLoadingMode();
            }
            finally {
                this.context.close();
            }
        }
    }

    private static class SplitInputStructs {
        private final List<InputStruct> vertexInputStructs = new ArrayList<InputStruct>();
        private final List<InputStruct> edgeInputStructs = new ArrayList<InputStruct>();
    }
}

