/*
 * Decompiled with CFR 0.152.
 */
package mil.nga.sf.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import mil.nga.sf.CircularString;
import mil.nga.sf.CompoundCurve;
import mil.nga.sf.Curve;
import mil.nga.sf.CurvePolygon;
import mil.nga.sf.Geometry;
import mil.nga.sf.GeometryCollection;
import mil.nga.sf.GeometryEnvelope;
import mil.nga.sf.GeometryType;
import mil.nga.sf.Line;
import mil.nga.sf.LineString;
import mil.nga.sf.MultiLineString;
import mil.nga.sf.MultiPoint;
import mil.nga.sf.MultiPolygon;
import mil.nga.sf.Point;
import mil.nga.sf.Polygon;
import mil.nga.sf.PolyhedralSurface;
import mil.nga.sf.TIN;
import mil.nga.sf.Triangle;
import mil.nga.sf.util.SFException;
import mil.nga.sf.util.centroid.CentroidCurve;
import mil.nga.sf.util.centroid.CentroidPoint;
import mil.nga.sf.util.centroid.CentroidSurface;
import mil.nga.sf.util.centroid.DegreesCentroid;

public class GeometryUtils {
    private static final Logger logger = Logger.getLogger(GeometryUtils.class.getName());

    public static int getDimension(Geometry geometry) {
        int dimension = -1;
        GeometryType geometryType = geometry.getGeometryType();
        switch (geometryType) {
            case POINT: 
            case MULTIPOINT: {
                dimension = 0;
                break;
            }
            case LINESTRING: 
            case MULTILINESTRING: 
            case CIRCULARSTRING: 
            case COMPOUNDCURVE: {
                dimension = 1;
                break;
            }
            case POLYGON: 
            case CURVEPOLYGON: 
            case MULTIPOLYGON: 
            case POLYHEDRALSURFACE: 
            case TIN: 
            case TRIANGLE: {
                dimension = 2;
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                List geometries = geomCollection.getGeometries();
                for (Geometry subGeometry : geometries) {
                    dimension = Math.max(dimension, GeometryUtils.getDimension(subGeometry));
                }
                break;
            }
            default: {
                throw new SFException("Unsupported Geometry Type: " + geometryType);
            }
        }
        return dimension;
    }

    public static double distance(Point point1, Point point2) {
        double diffX = point1.getX() - point2.getX();
        double diffY = point1.getY() - point2.getY();
        return Math.sqrt(diffX * diffX + diffY * diffY);
    }

    public static double distance(Line line) {
        return GeometryUtils.distance(line.startPoint(), line.endPoint());
    }

    public static double distanceHaversine(Point point1, Point point2) {
        double lat1 = point1.getY();
        double lon1 = point1.getX();
        double lat2 = point2.getY();
        double lon2 = point2.getX();
        double diffLat = GeometryUtils.degreesToRadians(lat2 - lat1);
        double diffLon = GeometryUtils.degreesToRadians(lon2 - lon1);
        double a = Math.sin(diffLat / 2.0) * Math.sin(diffLat / 2.0) + Math.cos(GeometryUtils.degreesToRadians(lat1)) * Math.cos(GeometryUtils.degreesToRadians(lat2)) * Math.sin(diffLon / 2.0) * Math.sin(diffLon / 2.0);
        double c = 2.0 * Math.atan2(Math.sqrt(a), Math.sqrt(1.0 - a));
        return 6378137.0 * c;
    }

    public static double distanceHaversine(Line line) {
        return GeometryUtils.distanceHaversine(line.startPoint(), line.endPoint());
    }

    public static double bearing(Point point1, Point point2) {
        double y1 = GeometryUtils.degreesToRadians(point1.getY());
        double y2 = GeometryUtils.degreesToRadians(point2.getY());
        double xDiff = GeometryUtils.degreesToRadians(point2.getX() - point1.getX());
        double y = Math.sin(xDiff) * Math.cos(y2);
        double x = Math.cos(y1) * Math.sin(y2) - Math.sin(y1) * Math.cos(y2) * Math.cos(xDiff);
        return (GeometryUtils.radiansToDegrees(Math.atan2(y, x)) + 360.0) % 360.0;
    }

    public static double bearing(Line line) {
        return GeometryUtils.bearing(line.startPoint(), line.endPoint());
    }

    public static boolean isNorthBearing(double bearing) {
        return (bearing %= 360.0) < 90.0 || bearing > 270.0;
    }

    public static boolean isEastBearing(double bearing) {
        return (bearing %= 360.0) > 0.0 && bearing < 180.0;
    }

    public static boolean isSouthBearing(double bearing) {
        return (bearing %= 360.0) > 90.0 && bearing < 270.0;
    }

    public static boolean isWestBearing(double bearing) {
        return bearing % 360.0 > 180.0;
    }

    public static Point geodesicMidpoint(Point point1, Point point2) {
        Point point1Radians = GeometryUtils.degreesToRadians(point1);
        Point point2Radians = GeometryUtils.degreesToRadians(point2);
        Point midpointRadians = GeometryUtils.geodesicMidpointRadians(point1Radians, point2Radians);
        return GeometryUtils.radiansToDegrees(midpointRadians);
    }

    public static Point geodesicMidpointRadians(Point point1, Point point2) {
        double xDiff = point2.getX() - point1.getX();
        double y1 = point1.getY();
        double y2 = point2.getY();
        double x1 = point1.getX();
        double bx = Math.cos(y2) * Math.cos(xDiff);
        double by = Math.cos(y2) * Math.sin(xDiff);
        double y = Math.atan2(Math.sin(y1) + Math.sin(y2), Math.sqrt((Math.cos(y1) + bx) * (Math.cos(y1) + bx) + by * by));
        double x = x1 + Math.atan2(by, Math.cos(y1) + bx);
        Point midpoint = new Point(x, y);
        return midpoint;
    }

    public static double degreesToRadians(double degrees) {
        return degrees * (Math.PI / 180);
    }

    public static double radiansToDegrees(double radians) {
        return radians * 57.29577951308232;
    }

    public static Point degreesToRadians(Point point) {
        double x = GeometryUtils.degreesToRadians(point.getX());
        double y = GeometryUtils.degreesToRadians(point.getY());
        return new Point(x, y);
    }

    public static Point radiansToDegrees(Point point) {
        double x = GeometryUtils.radiansToDegrees(point.getX());
        double y = GeometryUtils.radiansToDegrees(point.getY());
        return new Point(x, y);
    }

    public static Point getCentroid(Geometry geometry) {
        Point centroid = null;
        int dimension = GeometryUtils.getDimension(geometry);
        switch (dimension) {
            case 0: {
                CentroidPoint point = new CentroidPoint(geometry);
                centroid = point.getCentroid();
                break;
            }
            case 1: {
                CentroidCurve curve = new CentroidCurve(geometry);
                centroid = curve.getCentroid();
                break;
            }
            case 2: {
                CentroidSurface surface = new CentroidSurface(geometry);
                centroid = surface.getCentroid();
            }
        }
        return centroid;
    }

    public static Point getDegreesCentroid(Geometry geometry) {
        return DegreesCentroid.getCentroid(geometry);
    }

    public static void minimizeWGS84(Geometry geometry) {
        GeometryUtils.minimize(geometry, 180.0);
    }

    public static void minimizeWebMercator(Geometry geometry) {
        GeometryUtils.minimize(geometry, 2.0037508342789244E7);
    }

    public static void minimizeGeometry(Geometry geometry, double maxX) {
        GeometryUtils.minimize(geometry, maxX);
    }

    public static void minimize(Geometry geometry, double maxX) {
        GeometryType geometryType = geometry.getGeometryType();
        switch (geometryType) {
            case LINESTRING: {
                GeometryUtils.minimize((LineString)geometry, maxX);
                break;
            }
            case POLYGON: {
                GeometryUtils.minimize((Polygon)geometry, maxX);
                break;
            }
            case MULTILINESTRING: {
                GeometryUtils.minimize((MultiLineString)geometry, maxX);
                break;
            }
            case MULTIPOLYGON: {
                GeometryUtils.minimize((MultiPolygon)geometry, maxX);
                break;
            }
            case CIRCULARSTRING: {
                GeometryUtils.minimize((CircularString)geometry, maxX);
                break;
            }
            case COMPOUNDCURVE: {
                GeometryUtils.minimize((CompoundCurve)geometry, maxX);
                break;
            }
            case CURVEPOLYGON: {
                CurvePolygon curvePolygon = (CurvePolygon)geometry;
                GeometryUtils.minimize(curvePolygon, maxX);
                break;
            }
            case POLYHEDRALSURFACE: {
                GeometryUtils.minimize((PolyhedralSurface)geometry, maxX);
                break;
            }
            case TIN: {
                GeometryUtils.minimize((TIN)geometry, maxX);
                break;
            }
            case TRIANGLE: {
                GeometryUtils.minimize((Triangle)geometry, maxX);
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                for (Geometry subGeometry : geomCollection.getGeometries()) {
                    GeometryUtils.minimize(subGeometry, maxX);
                }
                break;
            }
        }
    }

    private static void minimize(LineString lineString, double maxX) {
        List<Point> points = lineString.getPoints();
        if (points.size() > 1) {
            Point point = points.get(0);
            for (int i = 1; i < points.size(); ++i) {
                Point nextPoint = points.get(i);
                if (point.getX() < nextPoint.getX()) {
                    if (!(nextPoint.getX() - point.getX() > point.getX() - nextPoint.getX() + maxX * 2.0)) continue;
                    nextPoint.setX(nextPoint.getX() - maxX * 2.0);
                    continue;
                }
                if (!(point.getX() > nextPoint.getX()) || !(point.getX() - nextPoint.getX() > nextPoint.getX() - point.getX() + maxX * 2.0)) continue;
                nextPoint.setX(nextPoint.getX() + maxX * 2.0);
            }
        }
    }

    private static void minimize(MultiLineString multiLineString, double maxX) {
        List<LineString> lineStrings = multiLineString.getLineStrings();
        for (LineString lineString : lineStrings) {
            GeometryUtils.minimize(lineString, maxX);
        }
    }

    private static void minimize(Polygon polygon, double maxX) {
        for (LineString ring : polygon.getRings()) {
            GeometryUtils.minimize(ring, maxX);
        }
    }

    private static void minimize(MultiPolygon multiPolygon, double maxX) {
        List<Polygon> polygons = multiPolygon.getPolygons();
        for (Polygon polygon : polygons) {
            GeometryUtils.minimize(polygon, maxX);
        }
    }

    private static void minimize(CompoundCurve compoundCurve, double maxX) {
        for (LineString lineString : compoundCurve.getLineStrings()) {
            GeometryUtils.minimize(lineString, maxX);
        }
    }

    private static void minimize(CurvePolygon<Curve> curvePolygon, double maxX) {
        for (Curve ring : curvePolygon.getRings()) {
            GeometryUtils.minimize(ring, maxX);
        }
    }

    private static void minimize(PolyhedralSurface polyhedralSurface, double maxX) {
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            GeometryUtils.minimize(polygon, maxX);
        }
    }

    public static void normalizeWGS84(Geometry geometry) {
        GeometryUtils.normalize(geometry, 180.0);
    }

    public static void normalizeWebMercator(Geometry geometry) {
        GeometryUtils.normalize(geometry, 2.0037508342789244E7);
    }

    public static void normalizeGeometry(Geometry geometry, double maxX) {
        GeometryUtils.normalize(geometry, maxX);
    }

    public static void normalize(Geometry geometry, double maxX) {
        GeometryType geometryType = geometry.getGeometryType();
        switch (geometryType) {
            case POINT: {
                GeometryUtils.normalize((Point)geometry, maxX);
                break;
            }
            case LINESTRING: {
                GeometryUtils.normalize((LineString)geometry, maxX);
                break;
            }
            case POLYGON: {
                GeometryUtils.normalize((Polygon)geometry, maxX);
                break;
            }
            case MULTIPOINT: {
                GeometryUtils.normalize((MultiPoint)geometry, maxX);
                break;
            }
            case MULTILINESTRING: {
                GeometryUtils.normalize((MultiLineString)geometry, maxX);
                break;
            }
            case MULTIPOLYGON: {
                GeometryUtils.normalize((MultiPolygon)geometry, maxX);
                break;
            }
            case CIRCULARSTRING: {
                GeometryUtils.normalize((CircularString)geometry, maxX);
                break;
            }
            case COMPOUNDCURVE: {
                GeometryUtils.normalize((CompoundCurve)geometry, maxX);
                break;
            }
            case CURVEPOLYGON: {
                CurvePolygon curvePolygon = (CurvePolygon)geometry;
                GeometryUtils.normalize(curvePolygon, maxX);
                break;
            }
            case POLYHEDRALSURFACE: {
                GeometryUtils.normalize((PolyhedralSurface)geometry, maxX);
                break;
            }
            case TIN: {
                GeometryUtils.normalize((TIN)geometry, maxX);
                break;
            }
            case TRIANGLE: {
                GeometryUtils.normalize((Triangle)geometry, maxX);
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                for (Geometry subGeometry : geomCollection.getGeometries()) {
                    GeometryUtils.normalize(subGeometry, maxX);
                }
                break;
            }
        }
    }

    private static void normalize(Point point, double maxX) {
        point.setX(GeometryUtils.normalize(point.getX(), maxX));
    }

    private static double normalize(double x, double maxX) {
        if (x < -maxX) {
            x += maxX * 2.0;
        } else if (x > maxX) {
            x -= maxX * 2.0;
        }
        return x;
    }

    private static void normalize(MultiPoint multiPoint, double maxX) {
        List<Point> points = multiPoint.getPoints();
        for (Point point : points) {
            GeometryUtils.normalize(point, maxX);
        }
    }

    private static void normalize(LineString lineString, double maxX) {
        for (Point point : lineString.getPoints()) {
            GeometryUtils.normalize(point, maxX);
        }
    }

    private static void normalize(MultiLineString multiLineString, double maxX) {
        List<LineString> lineStrings = multiLineString.getLineStrings();
        for (LineString lineString : lineStrings) {
            GeometryUtils.normalize(lineString, maxX);
        }
    }

    private static void normalize(Polygon polygon, double maxX) {
        for (LineString ring : polygon.getRings()) {
            GeometryUtils.normalize(ring, maxX);
        }
    }

    private static void normalize(MultiPolygon multiPolygon, double maxX) {
        List<Polygon> polygons = multiPolygon.getPolygons();
        for (Polygon polygon : polygons) {
            GeometryUtils.normalize(polygon, maxX);
        }
    }

    private static void normalize(CompoundCurve compoundCurve, double maxX) {
        for (LineString lineString : compoundCurve.getLineStrings()) {
            GeometryUtils.normalize(lineString, maxX);
        }
    }

    private static void normalize(CurvePolygon<Curve> curvePolygon, double maxX) {
        for (Curve ring : curvePolygon.getRings()) {
            GeometryUtils.normalize(ring, maxX);
        }
    }

    private static void normalize(PolyhedralSurface polyhedralSurface, double maxX) {
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            GeometryUtils.normalize(polygon, maxX);
        }
    }

    public static List<Point> simplifyPoints(List<Point> points, double tolerance) {
        return GeometryUtils.simplifyPoints(points, tolerance, 0, points.size() - 1);
    }

    private static List<Point> simplifyPoints(List<Point> points, double tolerance, int startIndex, int endIndex) {
        List<Point> result = null;
        double dmax = 0.0;
        int index = 0;
        Point startPoint = points.get(startIndex);
        Point endPoint = points.get(endIndex);
        for (int i = startIndex + 1; i < endIndex; ++i) {
            Point point = points.get(i);
            double d = GeometryUtils.perpendicularDistance(point, startPoint, endPoint);
            if (!(d > dmax)) continue;
            index = i;
            dmax = d;
        }
        if (dmax > tolerance) {
            List<Point> recResults1 = GeometryUtils.simplifyPoints(points, tolerance, startIndex, index);
            List<Point> recResults2 = GeometryUtils.simplifyPoints(points, tolerance, index, endIndex);
            result = recResults1.subList(0, recResults1.size() - 1);
            result.addAll(recResults2);
        } else {
            result = new ArrayList<Point>();
            result.add(startPoint);
            result.add(endPoint);
        }
        return result;
    }

    public static List<Point> geodesicPath(LineString lineString, double maxDistance) {
        return GeometryUtils.geodesicPath(lineString.getPoints(), maxDistance);
    }

    public static List<Point> geodesicPath(List<Point> points, double maxDistance) {
        ArrayList<Point> path = new ArrayList<Point>();
        if (!points.isEmpty()) {
            path.add(points.get(0));
            for (int i = 0; i < points.size() - 1; ++i) {
                List<Point> subPath = GeometryUtils.geodesicPath(points.get(i), points.get(i + 1), maxDistance);
                path.addAll(subPath.subList(1, subPath.size()));
            }
        }
        return path;
    }

    public static List<Point> geodesicPath(Point point1, Point point2, double maxDistance) {
        ArrayList<Point> path = new ArrayList<Point>();
        path.add(point1);
        GeometryUtils.geodesicPath(point1, point2, maxDistance, path);
        path.add(point2);
        return path;
    }

    private static void geodesicPath(Point point1, Point point2, double maxDistance, List<Point> path) {
        double distance = GeometryUtils.distanceHaversine(point1, point2);
        if (distance > maxDistance) {
            Point midpoint = GeometryUtils.geodesicMidpoint(point1, point2);
            GeometryUtils.geodesicPath(point1, midpoint, maxDistance, path);
            path.add(midpoint);
            GeometryUtils.geodesicPath(midpoint, point2, maxDistance, path);
        }
    }

    public static GeometryEnvelope geodesicEnvelope(GeometryEnvelope envelope) {
        Point right;
        Point left;
        Point midpoint;
        double y;
        GeometryEnvelope geodesic = envelope.copy();
        if (envelope.getMinY() < 0.0 && (y = (midpoint = GeometryUtils.geodesicMidpoint(left = envelope.getBottomLeft(), right = envelope.getBottomRight())).getY()) < geodesic.getMinY()) {
            geodesic.setMinY(y);
        }
        if (envelope.getMaxY() > 0.0 && (y = (midpoint = GeometryUtils.geodesicMidpoint(left = envelope.getTopLeft(), right = envelope.getTopRight())).getY()) > geodesic.getMaxY()) {
            geodesic.setMaxY(y);
        }
        return geodesic;
    }

    public static double perpendicularDistance(Point point, Point lineStart, Point lineEnd) {
        double y2;
        double x2;
        double x = point.getX();
        double y = point.getY();
        double startX = lineStart.getX();
        double startY = lineStart.getY();
        double endX = lineEnd.getX();
        double endY = lineEnd.getY();
        double vX = endX - startX;
        double vY = endY - startY;
        double wX = x - startX;
        double wY = y - startY;
        double c1 = wX * vX + wY * vY;
        double c2 = vX * vX + vY * vY;
        if (c1 <= 0.0) {
            x2 = startX;
            y2 = startY;
        } else if (c2 <= c1) {
            x2 = endX;
            y2 = endY;
        } else {
            double b = c1 / c2;
            x2 = startX + b * vX;
            y2 = startY + b * vY;
        }
        double distance = Math.sqrt(Math.pow(x2 - x, 2.0) + Math.pow(y2 - y, 2.0));
        return distance;
    }

    public static boolean pointInPolygon(Point point, Polygon polygon) {
        return GeometryUtils.pointInPolygon(point, polygon, 1.0E-15);
    }

    public static boolean pointInPolygon(Point point, Polygon polygon, double epsilon) {
        boolean contains = false;
        List rings = polygon.getRings();
        if (!rings.isEmpty() && (contains = GeometryUtils.pointInPolygon(point, (LineString)rings.get(0), epsilon))) {
            for (int i = 1; i < rings.size(); ++i) {
                if (!GeometryUtils.pointInPolygon(point, (LineString)rings.get(i), epsilon)) continue;
                contains = false;
                break;
            }
        }
        return contains;
    }

    public static boolean pointInPolygon(Point point, LineString ring) {
        return GeometryUtils.pointInPolygon(point, ring, 1.0E-15);
    }

    public static boolean pointInPolygon(Point point, LineString ring, double epsilon) {
        return GeometryUtils.pointInPolygon(point, ring.getPoints(), epsilon);
    }

    public static boolean pointInPolygon(Point point, List<Point> points) {
        return GeometryUtils.pointInPolygon(point, points, 1.0E-15);
    }

    public static boolean pointInPolygon(Point point, List<Point> points, double epsilon) {
        boolean contains = false;
        int i = 0;
        int j = points.size() - 1;
        if (GeometryUtils.closedPolygon(points)) {
            j = i++;
        }
        while (i < points.size()) {
            Point point1 = points.get(i);
            Point point2 = points.get(j);
            if (Math.abs(point1.getX() - point.getX()) <= epsilon && Math.abs(point1.getY() - point.getY()) <= epsilon) {
                contains = true;
                break;
            }
            if (point1.getY() > point.getY() != point2.getY() > point.getY() && point.getX() < (point2.getX() - point1.getX()) * (point.getY() - point1.getY()) / (point2.getY() - point1.getY()) + point1.getX()) {
                contains = !contains;
            }
            j = i++;
        }
        if (!contains) {
            contains = GeometryUtils.pointOnPolygonEdge(point, points);
        }
        return contains;
    }

    public static boolean pointOnPolygonEdge(Point point, Polygon polygon) {
        return GeometryUtils.pointOnPolygonEdge(point, polygon, 1.0E-15);
    }

    public static boolean pointOnPolygonEdge(Point point, Polygon polygon, double epsilon) {
        return polygon.numRings() > 0 && GeometryUtils.pointOnPolygonEdge(point, (LineString)polygon.getRings().get(0), epsilon);
    }

    public static boolean pointOnPolygonEdge(Point point, LineString ring) {
        return GeometryUtils.pointOnPolygonEdge(point, ring, 1.0E-15);
    }

    public static boolean pointOnPolygonEdge(Point point, LineString ring, double epsilon) {
        return GeometryUtils.pointOnPolygonEdge(point, ring.getPoints(), epsilon);
    }

    public static boolean pointOnPolygonEdge(Point point, List<Point> points) {
        return GeometryUtils.pointOnPolygonEdge(point, points, 1.0E-15);
    }

    public static boolean pointOnPolygonEdge(Point point, List<Point> points, double epsilon) {
        return GeometryUtils.pointOnPath(point, points, epsilon, !GeometryUtils.closedPolygon(points));
    }

    public static boolean closedPolygon(Polygon polygon) {
        return polygon.numRings() > 0 && GeometryUtils.closedPolygon((LineString)polygon.getRings().get(0));
    }

    public static boolean closedPolygon(LineString ring) {
        return GeometryUtils.closedPolygon(ring.getPoints());
    }

    public static boolean closedPolygon(List<Point> points) {
        boolean closed = false;
        if (!points.isEmpty()) {
            Point first = points.get(0);
            Point last = points.get(points.size() - 1);
            closed = first.getX() == last.getX() && first.getY() == last.getY();
        }
        return closed;
    }

    public static boolean pointOnLine(Point point, LineString line) {
        return GeometryUtils.pointOnLine(point, line, 1.0E-15);
    }

    public static boolean pointOnLine(Point point, LineString line, double epsilon) {
        return GeometryUtils.pointOnLine(point, line.getPoints(), epsilon);
    }

    public static boolean pointOnLine(Point point, List<Point> points) {
        return GeometryUtils.pointOnLine(point, points, 1.0E-15);
    }

    public static boolean pointOnLine(Point point, List<Point> points, double epsilon) {
        return GeometryUtils.pointOnPath(point, points, epsilon, false);
    }

    public static boolean pointOnPath(Point point, Point point1, Point point2) {
        return GeometryUtils.pointOnPath(point, point1, point2, 1.0E-15);
    }

    public static boolean pointOnPath(Point point, Point point1, Point point2, double epsilon) {
        double length21;
        double lengthP1;
        double yP1;
        boolean contains = false;
        double x21 = point2.getX() - point1.getX();
        double y21 = point2.getY() - point1.getY();
        double xP1 = point.getX() - point1.getX();
        double dp = xP1 * x21 + (yP1 = point.getY() - point1.getY()) * y21;
        if (dp >= 0.0 && (lengthP1 = xP1 * xP1 + yP1 * yP1) <= (length21 = x21 * x21 + y21 * y21)) {
            contains = Math.abs(dp * dp - lengthP1 * length21) <= epsilon;
        }
        return contains;
    }

    private static boolean pointOnPath(Point point, List<Point> points, double epsilon, boolean circular) {
        boolean onPath = false;
        int i = 0;
        int j = points.size() - 1;
        if (!circular) {
            j = i++;
        }
        while (i < points.size()) {
            Point point2;
            Point point1 = points.get(i);
            if (GeometryUtils.pointOnPath(point, point1, point2 = points.get(j), epsilon)) {
                onPath = true;
                break;
            }
            j = i++;
        }
        return onPath;
    }

    public static Point intersection(Line line1, Line line2) {
        return GeometryUtils.intersection(line1.startPoint(), line1.endPoint(), line2.startPoint(), line2.endPoint());
    }

    public static Point intersection(Point line1Point1, Point line1Point2, Point line2Point1, Point line2Point2) {
        Point intersection = null;
        double a1 = line1Point2.getY() - line1Point1.getY();
        double b1 = line1Point1.getX() - line1Point2.getX();
        double c1 = a1 * line1Point1.getX() + b1 * line1Point1.getY();
        double a2 = line2Point2.getY() - line2Point1.getY();
        double b2 = line2Point1.getX() - line2Point2.getX();
        double c2 = a2 * line2Point1.getX() + b2 * line2Point1.getY();
        double determinant = a1 * b2 - a2 * b1;
        if (determinant != 0.0) {
            double x = (b2 * c1 - b1 * c2) / determinant;
            double y = (a1 * c2 - a2 * c1) / determinant;
            intersection = new Point(x, y);
        }
        return intersection;
    }

    public static Geometry degreesToMeters(Geometry geometry) {
        Geometry meters = null;
        switch (geometry.getGeometryType()) {
            case POINT: {
                meters = GeometryUtils.degreesToMeters((Point)geometry);
                break;
            }
            case LINESTRING: {
                meters = GeometryUtils.degreesToMeters((LineString)geometry);
                break;
            }
            case POLYGON: {
                meters = GeometryUtils.degreesToMeters((Polygon)geometry);
                break;
            }
            case MULTIPOINT: {
                meters = GeometryUtils.degreesToMeters((MultiPoint)geometry);
                break;
            }
            case MULTILINESTRING: {
                meters = GeometryUtils.degreesToMeters((MultiLineString)geometry);
                break;
            }
            case MULTIPOLYGON: {
                meters = GeometryUtils.degreesToMeters((MultiPolygon)geometry);
                break;
            }
            case CIRCULARSTRING: {
                meters = GeometryUtils.degreesToMeters((CircularString)geometry);
                break;
            }
            case COMPOUNDCURVE: {
                meters = GeometryUtils.degreesToMeters((CompoundCurve)geometry);
                break;
            }
            case CURVEPOLYGON: {
                CurvePolygon curvePolygon = (CurvePolygon)geometry;
                meters = GeometryUtils.degreesToMeters(curvePolygon);
                break;
            }
            case POLYHEDRALSURFACE: {
                meters = GeometryUtils.degreesToMeters((PolyhedralSurface)geometry);
                break;
            }
            case TIN: {
                meters = GeometryUtils.degreesToMeters((TIN)geometry);
                break;
            }
            case TRIANGLE: {
                meters = GeometryUtils.degreesToMeters((Triangle)geometry);
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection<Geometry> metersCollection = new GeometryCollection<Geometry>();
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                for (Geometry subGeometry : geomCollection.getGeometries()) {
                    metersCollection.addGeometry(GeometryUtils.degreesToMeters(subGeometry));
                }
                meters = metersCollection;
                break;
            }
        }
        return meters;
    }

    public static Point degreesToMeters(Point point) {
        Point value = GeometryUtils.degreesToMeters(point.getX(), point.getY());
        value.setZ(point.getZ());
        value.setM(point.getM());
        return value;
    }

    public static Point degreesToMeters(double x, double y) {
        x = GeometryUtils.normalize(x, 180.0);
        y = Math.min(y, 90.0);
        y = Math.max(y, -89.99999999999999);
        double xValue = x * 2.0037508342789244E7 / 180.0;
        double yValue = Math.log(Math.tan((90.0 + y) * Math.PI / 360.0)) / (Math.PI / 180);
        yValue = yValue * 2.0037508342789244E7 / 180.0;
        return new Point(xValue, yValue);
    }

    public static MultiPoint degreesToMeters(MultiPoint multiPoint) {
        MultiPoint meters = new MultiPoint(multiPoint.hasZ(), multiPoint.hasM());
        for (Point point : multiPoint.getPoints()) {
            meters.addPoint(GeometryUtils.degreesToMeters(point));
        }
        return meters;
    }

    public static LineString degreesToMeters(LineString lineString) {
        LineString meters = new LineString(lineString.hasZ(), lineString.hasM());
        for (Point point : lineString.getPoints()) {
            meters.addPoint(GeometryUtils.degreesToMeters(point));
        }
        return meters;
    }

    public static Line degreesToMeters(Line line) {
        Line meters = new Line(line.hasZ(), line.hasM());
        for (Point point : line.getPoints()) {
            meters.addPoint(GeometryUtils.degreesToMeters(point));
        }
        return meters;
    }

    public static MultiLineString degreesToMeters(MultiLineString multiLineString) {
        MultiLineString meters = new MultiLineString(multiLineString.hasZ(), multiLineString.hasM());
        for (LineString lineString : multiLineString.getLineStrings()) {
            meters.addLineString(GeometryUtils.degreesToMeters(lineString));
        }
        return meters;
    }

    public static Polygon degreesToMeters(Polygon polygon) {
        Polygon meters = new Polygon(polygon.hasZ(), polygon.hasM());
        for (LineString ring : polygon.getRings()) {
            meters.addRing(GeometryUtils.degreesToMeters(ring));
        }
        return meters;
    }

    public static MultiPolygon degreesToMeters(MultiPolygon multiPolygon) {
        MultiPolygon meters = new MultiPolygon(multiPolygon.hasZ(), multiPolygon.hasM());
        for (Polygon polygon : multiPolygon.getPolygons()) {
            meters.addPolygon(GeometryUtils.degreesToMeters(polygon));
        }
        return meters;
    }

    public static CircularString degreesToMeters(CircularString circularString) {
        CircularString meters = new CircularString(circularString.hasZ(), circularString.hasM());
        for (Point point : circularString.getPoints()) {
            meters.addPoint(GeometryUtils.degreesToMeters(point));
        }
        return meters;
    }

    public static CompoundCurve degreesToMeters(CompoundCurve compoundCurve) {
        CompoundCurve meters = new CompoundCurve(compoundCurve.hasZ(), compoundCurve.hasM());
        for (LineString lineString : compoundCurve.getLineStrings()) {
            meters.addLineString(GeometryUtils.degreesToMeters(lineString));
        }
        return meters;
    }

    public static CurvePolygon<Curve> degreesToMeters(CurvePolygon<Curve> curvePolygon) {
        CurvePolygon<Curve> meters = new CurvePolygon<Curve>(curvePolygon.hasZ(), curvePolygon.hasM());
        for (Curve ring : curvePolygon.getRings()) {
            meters.addRing((Curve)GeometryUtils.degreesToMeters(ring));
        }
        return meters;
    }

    public static PolyhedralSurface degreesToMeters(PolyhedralSurface polyhedralSurface) {
        PolyhedralSurface meters = new PolyhedralSurface(polyhedralSurface.hasZ(), polyhedralSurface.hasM());
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            meters.addPolygon(GeometryUtils.degreesToMeters(polygon));
        }
        return meters;
    }

    public static TIN degreesToMeters(TIN tin) {
        TIN meters = new TIN(tin.hasZ(), tin.hasM());
        for (Polygon polygon : tin.getPolygons()) {
            meters.addPolygon(GeometryUtils.degreesToMeters(polygon));
        }
        return meters;
    }

    public static Triangle degreesToMeters(Triangle triangle) {
        Triangle meters = new Triangle(triangle.hasZ(), triangle.hasM());
        for (LineString ring : triangle.getRings()) {
            meters.addRing(GeometryUtils.degreesToMeters(ring));
        }
        return meters;
    }

    public static Geometry metersToDegrees(Geometry geometry) {
        Geometry degrees = null;
        switch (geometry.getGeometryType()) {
            case POINT: {
                degrees = GeometryUtils.metersToDegrees((Point)geometry);
                break;
            }
            case LINESTRING: {
                degrees = GeometryUtils.metersToDegrees((LineString)geometry);
                break;
            }
            case POLYGON: {
                degrees = GeometryUtils.metersToDegrees((Polygon)geometry);
                break;
            }
            case MULTIPOINT: {
                degrees = GeometryUtils.metersToDegrees((MultiPoint)geometry);
                break;
            }
            case MULTILINESTRING: {
                degrees = GeometryUtils.metersToDegrees((MultiLineString)geometry);
                break;
            }
            case MULTIPOLYGON: {
                degrees = GeometryUtils.metersToDegrees((MultiPolygon)geometry);
                break;
            }
            case CIRCULARSTRING: {
                degrees = GeometryUtils.metersToDegrees((CircularString)geometry);
                break;
            }
            case COMPOUNDCURVE: {
                degrees = GeometryUtils.metersToDegrees((CompoundCurve)geometry);
                break;
            }
            case CURVEPOLYGON: {
                CurvePolygon curvePolygon = (CurvePolygon)geometry;
                degrees = GeometryUtils.metersToDegrees(curvePolygon);
                break;
            }
            case POLYHEDRALSURFACE: {
                degrees = GeometryUtils.metersToDegrees((PolyhedralSurface)geometry);
                break;
            }
            case TIN: {
                degrees = GeometryUtils.metersToDegrees((TIN)geometry);
                break;
            }
            case TRIANGLE: {
                degrees = GeometryUtils.metersToDegrees((Triangle)geometry);
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection<Geometry> degreesCollection = new GeometryCollection<Geometry>();
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                for (Geometry subGeometry : geomCollection.getGeometries()) {
                    degreesCollection.addGeometry(GeometryUtils.metersToDegrees(subGeometry));
                }
                degrees = degreesCollection;
                break;
            }
        }
        return degrees;
    }

    public static Point metersToDegrees(Point point) {
        Point value = GeometryUtils.metersToDegrees(point.getX(), point.getY());
        value.setZ(point.getZ());
        value.setM(point.getM());
        return value;
    }

    public static Point metersToDegrees(double x, double y) {
        double xValue = x * 180.0 / 2.0037508342789244E7;
        double yValue = y * 180.0 / 2.0037508342789244E7;
        yValue = Math.atan(Math.exp(yValue * (Math.PI / 180))) / Math.PI * 360.0 - 90.0;
        return new Point(xValue, yValue);
    }

    public static MultiPoint metersToDegrees(MultiPoint multiPoint) {
        MultiPoint degrees = new MultiPoint(multiPoint.hasZ(), multiPoint.hasM());
        for (Point point : multiPoint.getPoints()) {
            degrees.addPoint(GeometryUtils.metersToDegrees(point));
        }
        return degrees;
    }

    public static LineString metersToDegrees(LineString lineString) {
        LineString degrees = new LineString(lineString.hasZ(), lineString.hasM());
        for (Point point : lineString.getPoints()) {
            degrees.addPoint(GeometryUtils.metersToDegrees(point));
        }
        return degrees;
    }

    public static Line metersToDegrees(Line line) {
        Line degrees = new Line(line.hasZ(), line.hasM());
        for (Point point : line.getPoints()) {
            degrees.addPoint(GeometryUtils.metersToDegrees(point));
        }
        return degrees;
    }

    public static MultiLineString metersToDegrees(MultiLineString multiLineString) {
        MultiLineString degrees = new MultiLineString(multiLineString.hasZ(), multiLineString.hasM());
        for (LineString lineString : multiLineString.getLineStrings()) {
            degrees.addLineString(GeometryUtils.metersToDegrees(lineString));
        }
        return degrees;
    }

    public static Polygon metersToDegrees(Polygon polygon) {
        Polygon degrees = new Polygon(polygon.hasZ(), polygon.hasM());
        for (LineString ring : polygon.getRings()) {
            degrees.addRing(GeometryUtils.metersToDegrees(ring));
        }
        return degrees;
    }

    public static MultiPolygon metersToDegrees(MultiPolygon multiPolygon) {
        MultiPolygon degrees = new MultiPolygon(multiPolygon.hasZ(), multiPolygon.hasM());
        for (Polygon polygon : multiPolygon.getPolygons()) {
            degrees.addPolygon(GeometryUtils.metersToDegrees(polygon));
        }
        return degrees;
    }

    public static CircularString metersToDegrees(CircularString circularString) {
        CircularString degrees = new CircularString(circularString.hasZ(), circularString.hasM());
        for (Point point : circularString.getPoints()) {
            degrees.addPoint(GeometryUtils.metersToDegrees(point));
        }
        return degrees;
    }

    public static CompoundCurve metersToDegrees(CompoundCurve compoundCurve) {
        CompoundCurve degrees = new CompoundCurve(compoundCurve.hasZ(), compoundCurve.hasM());
        for (LineString lineString : compoundCurve.getLineStrings()) {
            degrees.addLineString(GeometryUtils.metersToDegrees(lineString));
        }
        return degrees;
    }

    public static CurvePolygon<Curve> metersToDegrees(CurvePolygon<Curve> curvePolygon) {
        CurvePolygon<Curve> degrees = new CurvePolygon<Curve>(curvePolygon.hasZ(), curvePolygon.hasM());
        for (Curve ring : curvePolygon.getRings()) {
            degrees.addRing((Curve)GeometryUtils.metersToDegrees(ring));
        }
        return degrees;
    }

    public static PolyhedralSurface metersToDegrees(PolyhedralSurface polyhedralSurface) {
        PolyhedralSurface degrees = new PolyhedralSurface(polyhedralSurface.hasZ(), polyhedralSurface.hasM());
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            degrees.addPolygon(GeometryUtils.metersToDegrees(polygon));
        }
        return degrees;
    }

    public static TIN metersToDegrees(TIN tin) {
        TIN degrees = new TIN(tin.hasZ(), tin.hasM());
        for (Polygon polygon : tin.getPolygons()) {
            degrees.addPolygon(GeometryUtils.metersToDegrees(polygon));
        }
        return degrees;
    }

    public static Triangle metersToDegrees(Triangle triangle) {
        Triangle degrees = new Triangle(triangle.hasZ(), triangle.hasM());
        for (LineString ring : triangle.getRings()) {
            degrees.addRing(GeometryUtils.metersToDegrees(ring));
        }
        return degrees;
    }

    public static GeometryEnvelope wgs84Envelope() {
        return new GeometryEnvelope(-180.0, -90.0, 180.0, 90.0);
    }

    public static GeometryEnvelope wgs84TransformableEnvelope() {
        return new GeometryEnvelope(-180.0, -89.99999999999999, 180.0, 90.0);
    }

    public static GeometryEnvelope webMercatorEnvelope() {
        return new GeometryEnvelope(-2.0037508342789244E7, -2.0037508342789244E7, 2.0037508342789244E7, 2.0037508342789244E7);
    }

    public static GeometryEnvelope wgs84EnvelopeWithWebMercator() {
        return new GeometryEnvelope(-180.0, -85.05112877980659, 180.0, 85.0511287798066);
    }

    public static Geometry cropWebMercator(Geometry geometry) {
        return GeometryUtils.crop(geometry, GeometryUtils.webMercatorEnvelope());
    }

    public static Geometry crop(Geometry geometry, GeometryEnvelope envelope) {
        Geometry crop = null;
        if (GeometryUtils.contains(envelope, geometry.getEnvelope())) {
            crop = geometry;
        } else {
            switch (geometry.getGeometryType()) {
                case POINT: {
                    crop = GeometryUtils.crop((Point)((Object)geometry), envelope);
                    break;
                }
                case LINESTRING: {
                    crop = GeometryUtils.crop((LineString)((Object)geometry), envelope);
                    break;
                }
                case POLYGON: {
                    crop = GeometryUtils.crop((Polygon)geometry, envelope);
                    break;
                }
                case MULTIPOINT: {
                    crop = GeometryUtils.crop((MultiPoint)((Object)geometry), envelope);
                    break;
                }
                case MULTILINESTRING: {
                    crop = GeometryUtils.crop((MultiLineString)((Object)geometry), envelope);
                    break;
                }
                case MULTIPOLYGON: {
                    crop = GeometryUtils.crop((MultiPolygon)((Object)geometry), envelope);
                    break;
                }
                case CIRCULARSTRING: {
                    crop = GeometryUtils.crop((CircularString)((Object)geometry), envelope);
                    break;
                }
                case COMPOUNDCURVE: {
                    crop = GeometryUtils.crop((CompoundCurve)((Object)geometry), envelope);
                    break;
                }
                case CURVEPOLYGON: {
                    CurvePolygon curvePolygon = geometry;
                    crop = GeometryUtils.crop(curvePolygon, envelope);
                    break;
                }
                case POLYHEDRALSURFACE: {
                    crop = GeometryUtils.crop((PolyhedralSurface)((Object)geometry), envelope);
                    break;
                }
                case TIN: {
                    crop = GeometryUtils.crop((TIN)((Object)geometry), envelope);
                    break;
                }
                case TRIANGLE: {
                    crop = GeometryUtils.crop((Triangle)geometry, envelope);
                    break;
                }
                case GEOMETRYCOLLECTION: 
                case MULTICURVE: 
                case MULTISURFACE: {
                    GeometryCollection<Geometry> cropCollection = new GeometryCollection<Geometry>();
                    GeometryCollection geomCollection = (GeometryCollection)((Object)geometry);
                    for (Geometry subGeometry : geomCollection.getGeometries()) {
                        cropCollection.addGeometry(GeometryUtils.crop(subGeometry, envelope));
                    }
                    crop = cropCollection;
                    break;
                }
            }
        }
        return crop;
    }

    public static Point crop(Point point, GeometryEnvelope envelope) {
        Point crop = null;
        if (GeometryUtils.contains(envelope, point)) {
            crop = new Point(point);
        }
        return crop;
    }

    public static List<Point> crop(List<Point> points, GeometryEnvelope envelope) {
        ArrayList<Point> crop = new ArrayList<Point>();
        Line left = envelope.getLeft();
        Line bottom = envelope.getBottom();
        Line right = envelope.getRight();
        Line top = envelope.getTop();
        Point previousPoint = null;
        boolean previousContains = false;
        for (Point point : points) {
            boolean contains = GeometryUtils.contains(envelope, point);
            if (!(previousPoint == null || contains && previousContains)) {
                Line line = new Line(previousPoint, point);
                double bearing = GeometryUtils.bearing(GeometryUtils.metersToDegrees(line));
                boolean westBearing = GeometryUtils.isWestBearing(bearing);
                boolean eastBearing = GeometryUtils.isEastBearing(bearing);
                boolean southBearing = GeometryUtils.isSouthBearing(bearing);
                boolean northBearing = GeometryUtils.isNorthBearing(bearing);
                Line vertLine = null;
                if (point.getX() > envelope.getMaxX()) {
                    if (eastBearing) {
                        vertLine = right;
                    }
                } else if (point.getX() < envelope.getMinX()) {
                    if (westBearing) {
                        vertLine = left;
                    }
                } else if (eastBearing) {
                    vertLine = left;
                } else if (westBearing) {
                    vertLine = right;
                }
                Line horizLine = null;
                if (point.getY() > envelope.getMaxY()) {
                    if (northBearing) {
                        horizLine = top;
                    }
                } else if (point.getY() < envelope.getMinY()) {
                    if (southBearing) {
                        horizLine = bottom;
                    }
                } else if (northBearing) {
                    horizLine = bottom;
                } else if (southBearing) {
                    horizLine = top;
                }
                Point vertIntersection = null;
                if (vertLine != null && (vertIntersection = GeometryUtils.intersection(line, vertLine)) != null && !GeometryUtils.contains(envelope, vertIntersection)) {
                    vertIntersection = null;
                }
                Point horizIntersection = null;
                if (horizLine != null && (horizIntersection = GeometryUtils.intersection(line, horizLine)) != null && !GeometryUtils.contains(envelope, horizIntersection)) {
                    horizIntersection = null;
                }
                Point intersection1 = null;
                Point intersection2 = null;
                if (vertIntersection != null && horizIntersection != null) {
                    double horizDistance;
                    double vertDistance = GeometryUtils.distance(previousPoint, vertIntersection);
                    if (vertDistance <= (horizDistance = GeometryUtils.distance(previousPoint, horizIntersection))) {
                        intersection1 = vertIntersection;
                        intersection2 = horizIntersection;
                    } else {
                        intersection1 = horizIntersection;
                        intersection2 = vertIntersection;
                    }
                } else {
                    intersection1 = vertIntersection != null ? vertIntersection : horizIntersection;
                }
                if (intersection1 != null && !GeometryUtils.isEqual(intersection1, point) && !GeometryUtils.isEqual(intersection1, previousPoint)) {
                    crop.add(intersection1);
                    if (!(contains || previousContains || intersection2 == null || GeometryUtils.isEqual(intersection2, intersection1))) {
                        crop.add(intersection2);
                    }
                }
            }
            if (contains) {
                crop.add(point);
            }
            previousPoint = point;
            previousContains = contains;
        }
        if (crop.isEmpty()) {
            crop = null;
        } else if (crop.size() > 1) {
            if (points.get(0).equals(points.get(points.size() - 1)) && !((Point)crop.get(0)).equals(crop.get(crop.size() - 1))) {
                crop.add(new Point((Point)crop.get(0)));
            }
            if (crop.size() > 2) {
                ArrayList<Point> simplified = new ArrayList<Point>();
                simplified.add((Point)crop.get(0));
                for (int i = 1; i < crop.size() - 1; ++i) {
                    Point next;
                    Point previous = (Point)simplified.get(simplified.size() - 1);
                    Point point = (Point)crop.get(i);
                    if (GeometryUtils.pointOnPath(point, previous, next = (Point)crop.get(i + 1))) continue;
                    simplified.add(point);
                }
                simplified.add((Point)crop.get(crop.size() - 1));
                crop = simplified;
            }
        }
        return crop;
    }

    public static MultiPoint crop(MultiPoint multiPoint, GeometryEnvelope envelope) {
        MultiPoint crop = null;
        ArrayList<Point> cropPoints = new ArrayList<Point>();
        for (Point point : multiPoint.getPoints()) {
            Point cropPoint = GeometryUtils.crop(point, envelope);
            if (cropPoint == null) continue;
            cropPoints.add(cropPoint);
        }
        if (!cropPoints.isEmpty()) {
            crop = new MultiPoint(multiPoint.hasZ(), multiPoint.hasM());
            crop.setPoints(cropPoints);
        }
        return crop;
    }

    public static LineString crop(LineString lineString, GeometryEnvelope envelope) {
        LineString crop = null;
        List<Point> cropPoints = GeometryUtils.crop(lineString.getPoints(), envelope);
        if (cropPoints != null) {
            crop = new LineString(lineString.hasZ(), lineString.hasM());
            crop.setPoints(cropPoints);
        }
        return crop;
    }

    public static Line crop(Line line, GeometryEnvelope envelope) {
        Line crop = null;
        List<Point> cropPoints = GeometryUtils.crop(line.getPoints(), envelope);
        if (cropPoints != null) {
            crop = new Line(line.hasZ(), line.hasM());
            crop.setPoints(cropPoints);
        }
        return crop;
    }

    public static MultiLineString crop(MultiLineString multiLineString, GeometryEnvelope envelope) {
        MultiLineString crop = null;
        ArrayList<LineString> cropLineStrings = new ArrayList<LineString>();
        for (LineString lineString : multiLineString.getLineStrings()) {
            LineString cropLineString = GeometryUtils.crop(lineString, envelope);
            if (cropLineString == null) continue;
            cropLineStrings.add(cropLineString);
        }
        if (!cropLineStrings.isEmpty()) {
            crop = new MultiLineString(multiLineString.hasZ(), multiLineString.hasM());
            crop.setLineStrings(cropLineStrings);
        }
        return crop;
    }

    public static Polygon crop(Polygon polygon, GeometryEnvelope envelope) {
        Polygon crop = null;
        ArrayList<LineString> cropRings = new ArrayList<LineString>();
        for (LineString ring : polygon.getRings()) {
            List<Point> cropPoints;
            List<Point> points = ring.getPoints();
            if (!ring.isClosed()) {
                points.add(new Point(points.get(0)));
            }
            if ((cropPoints = GeometryUtils.crop(points, envelope)) == null) continue;
            LineString cropRing = new LineString(ring.hasZ(), ring.hasM());
            cropRing.setPoints(cropPoints);
            cropRings.add(cropRing);
        }
        if (!cropRings.isEmpty()) {
            crop = new Polygon(polygon.hasZ(), polygon.hasM());
            crop.setRings(cropRings);
        }
        return crop;
    }

    public static MultiPolygon crop(MultiPolygon multiPolygon, GeometryEnvelope envelope) {
        MultiPolygon crop = null;
        ArrayList<Polygon> cropPolygons = new ArrayList<Polygon>();
        for (Polygon polygon : multiPolygon.getPolygons()) {
            Polygon cropPolygon = GeometryUtils.crop(polygon, envelope);
            if (cropPolygon == null) continue;
            cropPolygons.add(cropPolygon);
        }
        if (!cropPolygons.isEmpty()) {
            crop = new MultiPolygon(multiPolygon.hasZ(), multiPolygon.hasM());
            crop.setPolygons(cropPolygons);
        }
        return crop;
    }

    public static CircularString crop(CircularString circularString, GeometryEnvelope envelope) {
        CircularString crop = null;
        List<Point> cropPoints = GeometryUtils.crop(circularString.getPoints(), envelope);
        if (cropPoints != null) {
            crop = new CircularString(circularString.hasZ(), circularString.hasM());
            crop.setPoints(cropPoints);
        }
        return crop;
    }

    public static CompoundCurve crop(CompoundCurve compoundCurve, GeometryEnvelope envelope) {
        CompoundCurve crop = null;
        ArrayList<LineString> cropLineStrings = new ArrayList<LineString>();
        for (LineString lineString : compoundCurve.getLineStrings()) {
            LineString cropLineString = GeometryUtils.crop(lineString, envelope);
            if (cropLineString == null) continue;
            cropLineStrings.add(cropLineString);
        }
        if (!cropLineStrings.isEmpty()) {
            crop = new CompoundCurve(compoundCurve.hasZ(), compoundCurve.hasM());
            crop.setLineStrings(cropLineStrings);
        }
        return crop;
    }

    public static CurvePolygon<Curve> crop(CurvePolygon<Curve> curvePolygon, GeometryEnvelope envelope) {
        CurvePolygon<Curve> crop = null;
        ArrayList<Curve> cropRings = new ArrayList<Curve>();
        for (Curve ring : curvePolygon.getRings()) {
            Geometry cropRing = GeometryUtils.crop(ring, envelope);
            if (cropRing == null) continue;
            cropRings.add((Curve)cropRing);
        }
        if (!cropRings.isEmpty()) {
            crop = new CurvePolygon<Curve>(curvePolygon.hasZ(), curvePolygon.hasM());
            crop.setRings(cropRings);
        }
        return crop;
    }

    public static PolyhedralSurface crop(PolyhedralSurface polyhedralSurface, GeometryEnvelope envelope) {
        PolyhedralSurface crop = null;
        ArrayList<Polygon> cropPolygons = new ArrayList<Polygon>();
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            Polygon cropPolygon = GeometryUtils.crop(polygon, envelope);
            if (cropPolygon == null) continue;
            cropPolygons.add(cropPolygon);
        }
        if (!cropPolygons.isEmpty()) {
            crop = new PolyhedralSurface(polyhedralSurface.hasZ(), polyhedralSurface.hasM());
            crop.setPolygons(cropPolygons);
        }
        return crop;
    }

    public static TIN crop(TIN tin, GeometryEnvelope envelope) {
        TIN crop = null;
        ArrayList<Polygon> cropPolygons = new ArrayList<Polygon>();
        for (Polygon polygon : tin.getPolygons()) {
            Polygon cropPolygon = GeometryUtils.crop(polygon, envelope);
            if (cropPolygon == null) continue;
            cropPolygons.add(cropPolygon);
        }
        if (!cropPolygons.isEmpty()) {
            crop = new TIN(tin.hasZ(), tin.hasM());
            crop.setPolygons(cropPolygons);
        }
        return crop;
    }

    public static Triangle crop(Triangle triangle, GeometryEnvelope envelope) {
        Triangle crop = null;
        ArrayList<LineString> cropRings = new ArrayList<LineString>();
        for (LineString ring : triangle.getRings()) {
            List<Point> cropPoints;
            List<Point> points = ring.getPoints();
            if (!ring.isClosed()) {
                points.add(new Point(points.get(0)));
            }
            if ((cropPoints = GeometryUtils.crop(points, envelope)) == null) continue;
            LineString cropRing = new LineString(ring.hasZ(), ring.hasM());
            cropRing.setPoints(cropPoints);
            cropRings.add(cropRing);
        }
        if (!cropRings.isEmpty()) {
            crop = new Triangle(triangle.hasZ(), triangle.hasM());
            crop.setRings(cropRings);
        }
        return crop;
    }

    public static boolean isEqual(Point point1, Point point2) {
        return GeometryUtils.isEqual(point1, point2, 1.0E-8);
    }

    public static boolean isEqual(Point point1, Point point2, double epsilon) {
        boolean equal;
        boolean bl = equal = Math.abs(point1.getX() - point2.getX()) <= epsilon && Math.abs(point1.getY() - point2.getY()) <= epsilon && point1.hasZ() == point2.hasZ() && point1.hasM() == point2.hasM();
        if (equal) {
            if (point1.hasZ()) {
                boolean bl2 = equal = Math.abs(point1.getZ() - point2.getZ()) <= epsilon;
            }
            if (equal && point1.hasM()) {
                equal = Math.abs(point1.getM() - point2.getM()) <= epsilon;
            }
        }
        return equal;
    }

    public static boolean contains(GeometryEnvelope envelope, Point point) {
        return envelope.contains(point, 1.0E-8);
    }

    public static boolean contains(GeometryEnvelope envelope1, GeometryEnvelope envelope2) {
        return envelope1.contains(envelope2, 1.0E-8);
    }

    public static void boundWGS84(Geometry geometry) {
        GeometryUtils.bound(geometry, GeometryUtils.wgs84Envelope());
    }

    public static void boundWGS84Transformable(Geometry geometry) {
        GeometryUtils.bound(geometry, GeometryUtils.wgs84TransformableEnvelope());
    }

    public static void boundWebMercator(Geometry geometry) {
        GeometryUtils.bound(geometry, GeometryUtils.webMercatorEnvelope());
    }

    public static void boundWGS84WithWebMercator(Geometry geometry) {
        GeometryUtils.bound(geometry, GeometryUtils.wgs84EnvelopeWithWebMercator());
    }

    public static void bound(Geometry geometry, GeometryEnvelope envelope) {
        GeometryType geometryType = geometry.getGeometryType();
        switch (geometryType) {
            case POINT: {
                GeometryUtils.bound((Point)geometry, envelope);
                break;
            }
            case LINESTRING: {
                GeometryUtils.bound((LineString)geometry, envelope);
                break;
            }
            case POLYGON: {
                GeometryUtils.bound((Polygon)geometry, envelope);
                break;
            }
            case MULTIPOINT: {
                GeometryUtils.bound((MultiPoint)geometry, envelope);
                break;
            }
            case MULTILINESTRING: {
                GeometryUtils.bound((MultiLineString)geometry, envelope);
                break;
            }
            case MULTIPOLYGON: {
                GeometryUtils.bound((MultiPolygon)geometry, envelope);
                break;
            }
            case CIRCULARSTRING: {
                GeometryUtils.bound((CircularString)geometry, envelope);
                break;
            }
            case COMPOUNDCURVE: {
                GeometryUtils.bound((CompoundCurve)geometry, envelope);
                break;
            }
            case CURVEPOLYGON: {
                CurvePolygon curvePolygon = (CurvePolygon)geometry;
                GeometryUtils.bound(curvePolygon, envelope);
                break;
            }
            case POLYHEDRALSURFACE: {
                GeometryUtils.bound((PolyhedralSurface)geometry, envelope);
                break;
            }
            case TIN: {
                GeometryUtils.bound((TIN)geometry, envelope);
                break;
            }
            case TRIANGLE: {
                GeometryUtils.bound((Triangle)geometry, envelope);
                break;
            }
            case GEOMETRYCOLLECTION: 
            case MULTICURVE: 
            case MULTISURFACE: {
                GeometryCollection geomCollection = (GeometryCollection)geometry;
                for (Geometry subGeometry : geomCollection.getGeometries()) {
                    GeometryUtils.bound(subGeometry, envelope);
                }
                break;
            }
        }
    }

    private static void bound(Point point, GeometryEnvelope envelope) {
        double x = point.getX();
        double y = point.getY();
        if (x < envelope.getMinX()) {
            point.setX(envelope.getMinX());
        } else if (x > envelope.getMaxX()) {
            point.setX(envelope.getMaxX());
        }
        if (y < envelope.getMinY()) {
            point.setY(envelope.getMinY());
        } else if (y > envelope.getMaxY()) {
            point.setY(envelope.getMaxY());
        }
    }

    private static void bound(MultiPoint multiPoint, GeometryEnvelope envelope) {
        for (Point point : multiPoint.getPoints()) {
            GeometryUtils.bound(point, envelope);
        }
    }

    private static void bound(LineString lineString, GeometryEnvelope envelope) {
        for (Point point : lineString.getPoints()) {
            GeometryUtils.bound(point, envelope);
        }
    }

    private static void bound(MultiLineString multiLineString, GeometryEnvelope envelope) {
        for (LineString lineString : multiLineString.getLineStrings()) {
            GeometryUtils.bound(lineString, envelope);
        }
    }

    private static void bound(Polygon polygon, GeometryEnvelope envelope) {
        for (LineString ring : polygon.getRings()) {
            GeometryUtils.bound(ring, envelope);
        }
    }

    private static void bound(MultiPolygon multiPolygon, GeometryEnvelope envelope) {
        for (Polygon polygon : multiPolygon.getPolygons()) {
            GeometryUtils.bound(polygon, envelope);
        }
    }

    private static void bound(CompoundCurve compoundCurve, GeometryEnvelope envelope) {
        for (LineString lineString : compoundCurve.getLineStrings()) {
            GeometryUtils.bound(lineString, envelope);
        }
    }

    private static void bound(CurvePolygon<Curve> curvePolygon, GeometryEnvelope envelope) {
        for (Curve ring : curvePolygon.getRings()) {
            GeometryUtils.bound(ring, envelope);
        }
    }

    private static void bound(PolyhedralSurface polyhedralSurface, GeometryEnvelope envelope) {
        for (Polygon polygon : polyhedralSurface.getPolygons()) {
            GeometryUtils.bound(polygon, envelope);
        }
    }

    public static <T extends Geometry> boolean hasZ(List<T> geometries) {
        boolean hasZ = false;
        for (Geometry geometry : geometries) {
            if (!geometry.hasZ()) continue;
            hasZ = true;
            break;
        }
        return hasZ;
    }

    public static <T extends Geometry> boolean hasM(List<T> geometries) {
        boolean hasM = false;
        for (Geometry geometry : geometries) {
            if (!geometry.hasM()) continue;
            hasM = true;
            break;
        }
        return hasM;
    }

    public static List<GeometryType> parentHierarchy(GeometryType geometryType) {
        ArrayList<GeometryType> hierarchy = new ArrayList<GeometryType>();
        GeometryType parentType = GeometryUtils.parentType(geometryType);
        while (parentType != null) {
            hierarchy.add(parentType);
            parentType = GeometryUtils.parentType(parentType);
        }
        return hierarchy;
    }

    public static GeometryType parentType(GeometryType geometryType) {
        GeometryType parentType = null;
        switch (geometryType) {
            case GEOMETRY: {
                break;
            }
            case POINT: {
                parentType = GeometryType.GEOMETRY;
                break;
            }
            case LINESTRING: {
                parentType = GeometryType.CURVE;
                break;
            }
            case POLYGON: {
                parentType = GeometryType.CURVEPOLYGON;
                break;
            }
            case MULTIPOINT: {
                parentType = GeometryType.GEOMETRYCOLLECTION;
                break;
            }
            case MULTILINESTRING: {
                parentType = GeometryType.MULTICURVE;
                break;
            }
            case MULTIPOLYGON: {
                parentType = GeometryType.MULTISURFACE;
                break;
            }
            case GEOMETRYCOLLECTION: {
                parentType = GeometryType.GEOMETRY;
                break;
            }
            case CIRCULARSTRING: {
                parentType = GeometryType.LINESTRING;
                break;
            }
            case COMPOUNDCURVE: {
                parentType = GeometryType.CURVE;
                break;
            }
            case CURVEPOLYGON: {
                parentType = GeometryType.SURFACE;
                break;
            }
            case MULTICURVE: {
                parentType = GeometryType.GEOMETRYCOLLECTION;
                break;
            }
            case MULTISURFACE: {
                parentType = GeometryType.GEOMETRYCOLLECTION;
                break;
            }
            case CURVE: {
                parentType = GeometryType.GEOMETRY;
                break;
            }
            case SURFACE: {
                parentType = GeometryType.GEOMETRY;
                break;
            }
            case POLYHEDRALSURFACE: {
                parentType = GeometryType.SURFACE;
                break;
            }
            case TIN: {
                parentType = GeometryType.POLYHEDRALSURFACE;
                break;
            }
            case TRIANGLE: {
                parentType = GeometryType.POLYGON;
                break;
            }
            default: {
                throw new SFException("Geometry Type not supported: " + geometryType);
            }
        }
        return parentType;
    }

    public static Map<GeometryType, Map<GeometryType, ?>> childHierarchy(GeometryType geometryType) {
        HashMap hierarchy = null;
        List<GeometryType> childTypes = GeometryUtils.childTypes(geometryType);
        if (!childTypes.isEmpty()) {
            hierarchy = new HashMap();
            for (GeometryType childType : childTypes) {
                hierarchy.put(childType, GeometryUtils.childHierarchy(childType));
            }
        }
        return hierarchy;
    }

    public static List<GeometryType> childTypes(GeometryType geometryType) {
        ArrayList<GeometryType> childTypes = new ArrayList<GeometryType>();
        switch (geometryType) {
            case GEOMETRY: {
                childTypes.add(GeometryType.POINT);
                childTypes.add(GeometryType.GEOMETRYCOLLECTION);
                childTypes.add(GeometryType.CURVE);
                childTypes.add(GeometryType.SURFACE);
                break;
            }
            case POINT: {
                break;
            }
            case LINESTRING: {
                childTypes.add(GeometryType.CIRCULARSTRING);
                break;
            }
            case POLYGON: {
                childTypes.add(GeometryType.TRIANGLE);
                break;
            }
            case MULTIPOINT: {
                break;
            }
            case MULTILINESTRING: {
                break;
            }
            case MULTIPOLYGON: {
                break;
            }
            case GEOMETRYCOLLECTION: {
                childTypes.add(GeometryType.MULTIPOINT);
                childTypes.add(GeometryType.MULTICURVE);
                childTypes.add(GeometryType.MULTISURFACE);
                break;
            }
            case CIRCULARSTRING: {
                break;
            }
            case COMPOUNDCURVE: {
                break;
            }
            case CURVEPOLYGON: {
                childTypes.add(GeometryType.POLYGON);
                break;
            }
            case MULTICURVE: {
                childTypes.add(GeometryType.MULTILINESTRING);
                break;
            }
            case MULTISURFACE: {
                childTypes.add(GeometryType.MULTIPOLYGON);
                break;
            }
            case CURVE: {
                childTypes.add(GeometryType.LINESTRING);
                childTypes.add(GeometryType.COMPOUNDCURVE);
                break;
            }
            case SURFACE: {
                childTypes.add(GeometryType.CURVEPOLYGON);
                childTypes.add(GeometryType.POLYHEDRALSURFACE);
                break;
            }
            case POLYHEDRALSURFACE: {
                childTypes.add(GeometryType.TIN);
                break;
            }
            case TIN: {
                break;
            }
            case TRIANGLE: {
                break;
            }
            default: {
                throw new SFException("Geometry Type not supported: " + geometryType);
            }
        }
        return childTypes;
    }

    public static byte[] serialize(Geometry geometry) {
        byte[] bytes = null;
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        ObjectOutputStream out = null;
        try {
            out = new ObjectOutputStream(bos);
            out.writeObject(geometry);
            out.flush();
            bytes = bos.toByteArray();
        }
        catch (IOException e) {
            throw new SFException("Failed to serialize geometry into bytes", e);
        }
        finally {
            try {
                bos.close();
            }
            catch (IOException e) {
                logger.log(Level.WARNING, "Failed to close stream", e);
            }
            if (out != null) {
                try {
                    out.close();
                }
                catch (IOException e) {
                    logger.log(Level.WARNING, "Failed to close stream", e);
                }
            }
        }
        return bytes;
    }

    public static Geometry deserialize(byte[] bytes) {
        Geometry geometry = null;
        ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
        ObjectInputStream in = null;
        try {
            in = new ObjectInputStream(bis);
            geometry = (Geometry)in.readObject();
        }
        catch (IOException | ClassNotFoundException e) {
            throw new SFException("Failed to deserialize geometry into bytes", e);
        }
        finally {
            try {
                bis.close();
            }
            catch (IOException e) {
                logger.log(Level.WARNING, "Failed to close stream", e);
            }
            if (in != null) {
                try {
                    in.close();
                }
                catch (IOException e) {
                    logger.log(Level.WARNING, "Failed to close stream", e);
                }
            }
        }
        return geometry;
    }
}

