/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.transport;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import io.netty.channel.Channel;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.channel.ChannelPromise;
import io.netty.channel.unix.Errors;
import java.io.IOException;
import java.net.SocketAddress;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLException;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.exceptions.OverloadedException;
import org.apache.cassandra.exceptions.OversizedCQLMessageException;
import org.apache.cassandra.metrics.ClientMetrics;
import org.apache.cassandra.net.FrameEncoder;
import org.apache.cassandra.transport.CQLMessageHandler;
import org.apache.cassandra.transport.Envelope;
import org.apache.cassandra.transport.ProtocolException;
import org.apache.cassandra.transport.ProtocolVersion;
import org.apache.cassandra.transport.messages.ErrorMessage;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExceptionHandlers {
    private static final Logger logger = LoggerFactory.getLogger(ExceptionHandlers.class);

    public static ChannelInboundHandlerAdapter postV5Handler(FrameEncoder.PayloadAllocator allocator, ProtocolVersion version) {
        return new PostV5ExceptionHandler(allocator, version);
    }

    static void logClientNetworkingExceptions(Throwable cause, SocketAddress clientAddress) {
        if (Throwables.anyCauseMatches(cause, t -> t instanceof ProtocolException)) {
            if (Throwables.anyCauseMatches(cause, t -> t instanceof ProtocolException && !((ProtocolException)t).isSilent())) {
                ClientMetrics.instance.markProtocolException();
                NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 1L, TimeUnit.MINUTES, "Protocol exception with client networking: " + cause.getMessage(), new Object[0]);
            }
        } else if (Throwables.anyCauseMatches(cause, t -> t instanceof OverloadedException)) {
            NoSpamLogger.log(logger, NoSpamLogger.Level.INFO, 1L, TimeUnit.MINUTES, cause.getMessage(), new Object[0]);
        } else if (Throwables.anyCauseMatches(cause, t -> t instanceof OversizedCQLMessageException)) {
            NoSpamLogger.log(logger, NoSpamLogger.Level.INFO, 1L, TimeUnit.MINUTES, cause.getMessage(), new Object[0]);
        } else if (Throwables.anyCauseMatches(cause, t -> t instanceof Errors.NativeIoException)) {
            ClientMetrics.instance.markUnknownException();
            logger.trace("Native exception in client networking", cause);
        } else if (Throwables.anyCauseMatches(cause, t -> t instanceof SSLException)) {
            NoSpamLogger.log(logger, NoSpamLogger.Level.WARN, 1L, TimeUnit.MINUTES, "SSLException in client networking with peer {} {}", clientAddress, cause.getMessage());
        } else {
            ClientMetrics.instance.markUnknownException();
            logger.warn("Unknown exception in client networking with peer {} {}", (Object)clientAddress, (Object)cause.getMessage());
        }
    }

    static Predicate<Throwable> getUnexpectedExceptionHandler(Channel channel, boolean alwaysLogAtError) {
        SocketAddress address = channel.remoteAddress();
        if (DatabaseDescriptor.getClientErrorReportingExclusions().contains(address)) {
            return cause -> {
                logger.debug("Excluding client exception for {}; address contained in client_error_reporting_exclusions", (Object)address, cause);
                return true;
            };
        }
        return new UnexpectedChannelExceptionHandler(channel, alwaysLogAtError);
    }

    static final class UnexpectedChannelExceptionHandler
    implements Predicate<Throwable> {
        private static final Set<String> ioExceptionsAtDebugLevel = ImmutableSet.builder().add((Object)"Connection reset by peer").add((Object)"Broken pipe").add((Object)"Connection timed out").build();
        private final Channel channel;
        private final boolean alwaysLogAtError;

        UnexpectedChannelExceptionHandler(Channel channel, boolean alwaysLogAtError) {
            this.channel = channel;
            this.alwaysLogAtError = alwaysLogAtError;
        }

        public boolean apply(Throwable exception) {
            String message;
            try {
                message = "Unexpected exception during request; channel = " + this.channel;
            }
            catch (Exception ignore) {
                message = "Unexpected exception during request; channel = <unprintable>";
            }
            if (!this.alwaysLogAtError && (exception instanceof IOException || exception.getCause() instanceof IOException)) {
                String errorMessage = exception.getMessage();
                boolean logAtTrace = false;
                for (String ioException : ioExceptionsAtDebugLevel) {
                    if (!errorMessage.contains(ioException)) continue;
                    logAtTrace = true;
                    break;
                }
                if (logAtTrace) {
                    logger.trace(message, exception);
                } else {
                    logger.info(message, exception);
                }
            } else {
                logger.error(message, exception);
            }
            return true;
        }
    }

    private static final class PostV5ExceptionHandler
    extends ChannelInboundHandlerAdapter {
        private final FrameEncoder.PayloadAllocator allocator;
        private final ProtocolVersion version;

        public PostV5ExceptionHandler(FrameEncoder.PayloadAllocator allocator, ProtocolVersion version) {
            this.allocator = allocator;
            this.version = version;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
            if (ctx.channel().isOpen()) {
                Predicate<Throwable> handler = ExceptionHandlers.getUnexpectedExceptionHandler(ctx.channel(), false);
                ErrorMessage errorMessage = ErrorMessage.fromException(cause, handler);
                Envelope response = errorMessage.encode(this.version);
                FrameEncoder.Payload payload = this.allocator.allocate(true, CQLMessageHandler.envelopeSize(response.header));
                try {
                    response.encodeInto(payload.buffer);
                    response.release();
                    payload.finish();
                    ChannelPromise promise = ctx.newPromise();
                    if (PostV5ExceptionHandler.isFatal(cause)) {
                        promise.addListener(future -> ctx.close());
                    }
                    ctx.writeAndFlush((Object)payload, promise);
                }
                finally {
                    payload.release();
                    JVMStabilityInspector.inspectThrowable(cause);
                }
            }
            if (DatabaseDescriptor.getClientErrorReportingExclusions().contains(ctx.channel().remoteAddress())) {
                logger.debug("Excluding client exception for {}; address contained in client_error_reporting_exclusions", (Object)ctx.channel().remoteAddress(), (Object)cause);
                return;
            }
            ExceptionHandlers.logClientNetworkingExceptions(cause, ctx.channel().remoteAddress());
        }

        private static boolean isFatal(Throwable cause) {
            return Throwables.anyCauseMatches(cause, t -> t instanceof ProtocolException && ((ProtocolException)t).isFatal());
        }
    }
}

