/*
 *   convert.c
 *
 *   (C) Richard Drummond 2001-2007
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */

#include "defs.h"
#include "cmap.h"
#include "convert.h"
#include "surface.h"

#ifdef SUPPORT_VGA16FB
#include "vga.h"
#endif

/*
 * Return a function for converting framebuffer pixels to RGB24 format,
 * based on the type, color model and depth of the framebuffer.
 *
 * Returns 0 for unsupported framebuffer types
 */
convert_fn_ptr get_convert_fn( __u32 type, __u32 color_model )
{
  convert_fn_ptr fn = 0;

  switch( color_model )
  {
    case FB_VISUAL_PSEUDOCOLOR:
    case FB_VISUAL_STATIC_PSEUDOCOLOR:
     if ( type == PIXEL_TYPE_CLUT8 )
	fn = convert_clut8;
#ifdef SUPPORT_VGA16FB
     else if( type == PIXEL_TYPE_VGA4 )
        fn = convert_vga4;
#endif
     break;
    case FB_VISUAL_DIRECTCOLOR:
      switch( type )
      {
       case PIXEL_TYPE_x1r5g5b5: fn = convert_directcolor_x1r5g5b5; break;
       case PIXEL_TYPE_r5g6b5:   fn = convert_directcolor_r5g6b5;   break;
       case PIXEL_TYPE_r8g8b8:   fn = convert_directcolor_r8g8b8;   break;
       case PIXEL_TYPE_x8r8g8b8: fn = convert_directcolor_x8r8g8b8; break;
      }
      break;
    case FB_VISUAL_TRUECOLOR:
      switch( type )
      {
        case PIXEL_TYPE_x1r5g5b5: fn = convert_truecolor_x1r5g5b5; break;
        case PIXEL_TYPE_r5g6b5:   fn = convert_truecolor_r5g6b5;   break;
        case PIXEL_TYPE_r8g8b8:   fn = convert_truecolor_r8g8b8;   break;
        case PIXEL_TYPE_x8r8g8b8: fn = convert_truecolor_x8r8g8b8; break;
      }
      break;
  } // switch( color_model )

  return fn;
}


/*
 * Convert a row of CLUT8 (Pseudocolor) pixels to RGB24
 */
void convert_clut8( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  for ( ; count; count--, src++ )
  {
    *dest++ = ColorMap_getRed(   cmap, *src );
    *dest++ = ColorMap_getGreen( cmap, *src );
    *dest++ = ColorMap_getBlue(  cmap, *src );
  }
}


#ifdef SUPPORT_VGA16FB

/*
 * Convert a row of 4-plane VGA pixels to RGB24
 */
void convert_vga4( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  unsigned int pixel;
  unsigned int start = 0; // Pixel at which to start converting - currently unused.

  /* Force src pointer to be volatile - so that the compiler won't optimize
     out reads to the 'same' location when switching between planes */
  volatile __u8 *v_src = (volatile __u8 *) src;

  VGA_setReadMode0();

  for ( pixel=0; pixel < start+count; v_src++ )
  {
    __u8 p0, p1, p2, p3;
    int bit;

    /* Read a byte (8 pixels) from each bit plane */
    VGA_setReadPlane( 0 ); p0 = *v_src;
    VGA_setReadPlane( 1 ); p1 = *v_src;
    VGA_setReadPlane( 2 ); p2 = *v_src;
    VGA_setReadPlane( 3 ); p3 = *v_src;

    /* For each pixel in that group of eight . . . */
    for( bit = 7; bit >= 0; bit--, pixel++ )
    {
      __u8 pen  = 0;
      __u8 mask = 1<<bit;

      /* that is, providing it's after our start pixel . . . */
      if( pixel > start )
      {
        /* Covert the pixel to a pen number */
        if( p0&mask ) pen|=1;
        if( p1&mask ) pen|=2;
        if( p2&mask ) pen|=4;
        if( p3&mask ) pen|=8;

        /* Look up the pen's RGB value and output that */
        *dest++ = ColorMap_getRed(   cmap, pen );
        *dest++ = ColorMap_getGreen( cmap, pen );
        *dest++ = ColorMap_getBlue(  cmap, pen );
      }
    } /* end for(bit) */
  } /* end for(pixel) */
}

#endif


void convert_directcolor_x1r5g5b5( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u16 *src_w = (__u16 *)src;

  for ( ; count; count--, src_w++ )
  {
    *dest++ = ColorMap_getRed(   cmap, ( *src_w >> 10 ) & 0x1F );
    *dest++ = ColorMap_getGreen( cmap, ( *src_w >> 5 )  & 0x1F );
    *dest++ = ColorMap_getBlue(  cmap, ( *src_w )       & 0x1F );
  }
  return;
}


void convert_directcolor_r5g6b5( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u16 *src_w = (__u16 *) src;

  for ( ; count; count--, src_w++ )
  {
    *dest++ = ColorMap_getRed(   cmap, ( *src_w >> 11 ) & 0x1F );
    *dest++ = ColorMap_getGreen( cmap, ( *src_w >> 6 )  & 0x3F );
    *dest++ = ColorMap_getBlue(  cmap, ( *src_w )       & 0x1F );
  }
  return;
}


void convert_directcolor_r8g8b8( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  for ( ; count; count-- )
  {
    int green, blue;

    blue    = ColorMap_getBlue(  cmap, *src++ );
    green   = ColorMap_getGreen( cmap, *src++ );
    *dest++ = ColorMap_getRed(   cmap, *src++ );
    *dest++ = green;
    *dest++ = blue;
  }
  return;
}


void convert_directcolor_x8r8g8b8( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u32 *src_l = (__u32 *) src;

  for ( ; count; count--, src_l++ )
  {
    *dest++ = ColorMap_getRed(   cmap, (*src_l >> 16) & 0xFF );
    *dest++ = ColorMap_getGreen( cmap, (*src_l >> 8)  & 0xFF );
    *dest++ = ColorMap_getBlue(  cmap,  *src_l        & 0xFF );
  }
  return;
}


void convert_truecolor_x1r5g5b5( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u16 * src_w = (__u16 *)src;
  const ColorMap __attribute__((unused)) *cmap_unused = cmap;

  for ( ; count; count--, src_w++ )
  {
    *dest++ = ( *src_w >> 7 ) & 0xF8;
    *dest++ = ( *src_w >> 2 ) & 0xF8;
    *dest++ =   *src_w << 3;
  }
  return;
}


void convert_truecolor_r5g6b5( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u16 *src_w = (__u16 *)src;
  const ColorMap __attribute__((unused)) *cmap_unused = cmap;

  for ( ; count; count--, src_w++ )
  {
    *dest++ = ( *src_w >> 8 ) & 0xF8;
    *dest++ = ( *src_w >> 3 ) & 0xFC;
    *dest++ =   *src_w << 3;
  }
  return;
}


void convert_truecolor_r8g8b8( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  const ColorMap __attribute__((unused)) *cmap_unused = cmap;

  for ( ; count; count-- )
  {
#ifdef WORDS_BIGENDIAN
    *dest++ = *src++;
    *dest++ = *src++;
    *dest++ = *src++;
#else
    int green, blue;

    /* Swap order of pixel components from BGR to RGB */
    blue    = *src++;
    green   = *src++;
    *dest++ = *src++;
    *dest++ = green;
    *dest++ = blue;
#endif
  }
  return;
}


void convert_truecolor_x8r8g8b8( const __u8 *src, __u8 *dest, __u32 count, const ColorMap *cmap )
{
  __u32 *src_l = (__u32 *)src;
  const ColorMap __attribute__((unused)) *cmap_unused = cmap;

  for ( ; count; count--, src_l++ )
  {
    *dest++ = *src_l >> 16;
    *dest++ = *src_l >> 8;
    *dest++ = *src_l;
  }
  return;
}
