/*
 * Python object definition of the folder type libpff item
 *
 * Copyright (c) 2008-2013, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <types.h>

#if defined( HAVE_STDLIB_H ) || defined( HAVE_WINAPI )
#include <stdlib.h>
#endif

#include "pypff.h"
#include "pypff_folder.h"
#include "pypff_item.h"
#include "pypff_libcerror.h"
#include "pypff_libcstring.h"
#include "pypff_libpff.h"
#include "pypff_message.h"
#include "pypff_python.h"
#include "pypff_unused.h"

PyMethodDef pypff_folder_object_methods[] = {

	/* Functions to access the folder values */

	/* Functions to access the sub folders */

	{ "get_number_of_sub_folders",
	  (PyCFunction) pypff_folder_get_number_of_sub_folders,
	  METH_NOARGS,
	  "get_number_of_sub_folders() -> Integer\n"
	  "\n"
	  "Retrieves the number of sub folders." },

	{ "get_sub_folder",
	  (PyCFunction) pypff_folder_get_sub_folder,
	  METH_VARARGS | METH_KEYWORDS,
	  "get_sub_folder(sub_folder_index) -> Object or None\n"
	  "\n"
	  "Retrieves a specific sub folder." },

	/* Functions to access the sub messages */

	{ "get_number_of_sub_messages",
	  (PyCFunction) pypff_folder_get_number_of_sub_messages,
	  METH_NOARGS,
	  "get_number_of_sub_messages() -> Integer\n"
	  "\n"
	  "Retrieves the number of sub messages." },

	{ "get_sub_message",
	  (PyCFunction) pypff_folder_get_sub_message,
	  METH_VARARGS | METH_KEYWORDS,
	  "get_sub_message(sub_message_index) -> Object or None\n"
	  "\n"
	  "Retrieves a specific sub message." },

	/* Sentinel */
	{ NULL, NULL, 0, NULL }
};

PyGetSetDef pypff_folder_object_get_set_definitions[] = {

	{ "number_of_sub_folders",
	  (getter) pypff_folder_get_number_of_sub_folders,
	  (setter) 0,
	  "The number of sub folders.",
	  NULL },

	{ "number_of_sub_messages",
	  (getter) pypff_folder_get_number_of_sub_messages,
	  (setter) 0,
	  "The number of sub messages.",
	  NULL },

	/* Sentinel */
	{ NULL, NULL, NULL, NULL, NULL }
};

PyTypeObject pypff_folder_type_object = {
	PyObject_HEAD_INIT( NULL )

	/* ob_size */
	0,
	/* tp_name */
	"pypff.folder",
	/* tp_basicsize */
	sizeof( pypff_item_t ),
	/* tp_itemsize */
	0,
	/* tp_dealloc */
	0,
	/* tp_print */
	0,
	/* tp_getattr */
	0,
	/* tp_setattr */
	0,
	/* tp_compare */
	0,
	/* tp_repr */
	0,
	/* tp_as_number */
	0,
	/* tp_as_sequence */
	0,
	/* tp_as_mapping */
	0,
	/* tp_hash */
	0,
	/* tp_call */
	0,
	/* tp_str */
	0,
	/* tp_getattro */
	0,
	/* tp_setattro */
	0,
	/* tp_as_buffer */
	0,
        /* tp_flags */
	Py_TPFLAGS_DEFAULT,
	/* tp_doc */
	"pypff folder object (wraps folder type libpff_item_t)",
	/* tp_traverse */
	0,
	/* tp_clear */
	0,
	/* tp_richcompare */
	0,
	/* tp_weaklistoffset */
	0,
	/* tp_iter */
	0,
	/* tp_iternext */
	0,
	/* tp_methods */
	pypff_folder_object_methods,
	/* tp_members */
	0,
	/* tp_getset */
	pypff_folder_object_get_set_definitions,
	/* tp_base */
	&pypff_item_type_object,
	/* tp_dict */
	0,
	/* tp_descr_get */
	0,
	/* tp_descr_set */
	0,
	/* tp_dictoffset */
	0,
	/* tp_init */
	0,
	/* tp_alloc */
	0,
	/* tp_new */
	0,
	/* tp_free */
	0,
	/* tp_is_gc */
	0,
	/* tp_bases */
	NULL,
	/* tp_mro */
	NULL,
	/* tp_cache */
	NULL,
	/* tp_subclasses */
	NULL,
	/* tp_weaklist */
	NULL,
	/* tp_del */
	0
};

/* Retrieves the number of sub folders
 * Returns a Python object if successful or NULL on error
 */
PyObject *pypff_folder_get_number_of_sub_folders(
           pypff_item_t *pypff_item,
           PyObject *arguments PYPFF_ATTRIBUTE_UNUSED )
{
	char error_string[ PYPFF_ERROR_STRING_SIZE ];

	libcerror_error_t *error  = NULL;
	static char *function     = "pypff_folder_get_number_of_sub_folders";
	int number_of_sub_folders = 0;
	int result                = 0;

	PYPFF_UNREFERENCED_PARAMETER( arguments )

	if( pypff_item == NULL )
	{
		PyErr_Format(
		 PyExc_TypeError,
		 "%s: invalid item.",
		 function );

		return( NULL );
	}
	Py_BEGIN_ALLOW_THREADS

	result = libpff_folder_get_number_of_sub_folders(
	          pypff_item->item,
	          &number_of_sub_folders,
	          &error );

	Py_END_ALLOW_THREADS

	if( result != 1 )
	{
		if( libcerror_error_backtrace_sprint(
		     error,
		     error_string,
		     PYPFF_ERROR_STRING_SIZE ) == -1 )
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve number of sub folders.",
			 function );
		}
		else
		{
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve number of sub folders.\n%s",
			 function,
			 error_string );
		}
		libcerror_error_free(
		 &error );

		return( NULL );
	}
	return( PyInt_FromLong(
	         (long) number_of_sub_folders ) );
}

/* Retrieves a specific sub folder
 * Returns a Python object if successful or NULL on error
 */
PyObject *pypff_folder_get_sub_folder(
           pypff_item_t *pypff_item,
           PyObject *arguments,
           PyObject *keywords )
{
	char error_string[ PYPFF_ERROR_STRING_SIZE ];

	libcerror_error_t *error    = NULL;
	libpff_item_t *sub_folder   = NULL;
	PyObject *folder_object     = NULL;
	static char *keyword_list[] = { "sub_folder_index", NULL };
	static char *function       = "pypff_folder_get_sub_folder";
	int result                  = 0;
	int sub_folder_index        = 0;

	if( pypff_item == NULL )
	{
		PyErr_Format(
		 PyExc_TypeError,
		 "%s: invalid item.",
		 function );

		return( NULL );
	}
	if( PyArg_ParseTupleAndKeywords(
	     arguments,
	     keywords,
	     "i",
	     keyword_list,
	     &sub_folder_index ) == 0 )
        {
		goto on_error;
        }
	Py_BEGIN_ALLOW_THREADS

	result = libpff_folder_get_sub_folder(
	          pypff_item->item,
	          sub_folder_index,
	          &sub_folder,
	          &error );

	Py_END_ALLOW_THREADS

	if( result != 1 )
	{
		if( libcerror_error_backtrace_sprint(
		     error,
		     error_string,
		     PYPFF_ERROR_STRING_SIZE ) == -1 )
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve sub folder: %d.",
			 function,
			 sub_folder_index );
		}
		else
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve sub folder: %d.\n%s",
			 function,
			 sub_folder_index,
			 error_string );
		}
		libcerror_error_free(
		 &error );

		goto on_error;
	}
	folder_object = pypff_item_new(
	                 &pypff_folder_type_object,
	                 sub_folder,
	                 pypff_item->file_object );

	if( folder_object == NULL )
	{
		PyErr_Format(
		 PyExc_MemoryError,
		 "%s: unable to create folder object.",
		 function );

		goto on_error;
	}
	return( folder_object );

on_error:
	if( sub_folder != NULL )
	{
		libpff_item_free(
		 &sub_folder,
		 NULL );
	}
	return( NULL );
}

/* Retrieves the number of sub messages
 * Returns a Python object if successful or NULL on error
 */
PyObject *pypff_folder_get_number_of_sub_messages(
           pypff_item_t *pypff_item,
           PyObject *arguments PYPFF_ATTRIBUTE_UNUSED )
{
	char error_string[ PYPFF_ERROR_STRING_SIZE ];

	libcerror_error_t *error   = NULL;
	static char *function      = "pypff_folder_get_number_of_sub_messages";
	int number_of_sub_messages = 0;
	int result                 = 0;

	PYPFF_UNREFERENCED_PARAMETER( arguments )

	if( pypff_item == NULL )
	{
		PyErr_Format(
		 PyExc_TypeError,
		 "%s: invalid item.",
		 function );

		return( NULL );
	}
	Py_BEGIN_ALLOW_THREADS

	result = libpff_folder_get_number_of_sub_messages(
	          pypff_item->item,
	          &number_of_sub_messages,
	          &error );

	Py_END_ALLOW_THREADS

	if( result != 1 )
	{
		if( libcerror_error_backtrace_sprint(
		     error,
		     error_string,
		     PYPFF_ERROR_STRING_SIZE ) == -1 )
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve number of sub messages.",
			 function );
		}
		else
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve number of sub messages.\n%s",
			 function,
			 error_string );
		}
		libcerror_error_free(
		 &error );

		return( NULL );
	}
	return( PyInt_FromLong(
	         (long) number_of_sub_messages ) );
}

/* Retrieves a specific sub message
 * Returns a Python object if successful or NULL on error
 */
PyObject *pypff_folder_get_sub_message(
           pypff_item_t *pypff_item,
           PyObject *arguments,
           PyObject *keywords )
{
	char error_string[ PYPFF_ERROR_STRING_SIZE ];

	libcerror_error_t *error    = NULL;
	libpff_item_t *sub_message  = NULL;
	PyObject *message_object    = NULL;
	static char *keyword_list[] = { "sub_message_index", NULL };
	static char *function       = "pypff_folder_get_sub_message";
	int result                  = 0;
	int sub_message_index       = 0;

	if( pypff_item == NULL )
	{
		PyErr_Format(
		 PyExc_TypeError,
		 "%s: invalid item.",
		 function );

		return( NULL );
	}
	if( PyArg_ParseTupleAndKeywords(
	     arguments,
	     keywords,
	     "i",
	     keyword_list,
	     &sub_message_index ) == 0 )
        {
		goto on_error;
        }
	Py_BEGIN_ALLOW_THREADS

	result = libpff_folder_get_sub_message(
	          pypff_item->item,
	          sub_message_index,
	          &sub_message,
	          &error );

	Py_END_ALLOW_THREADS

	if( result != 1 )
	{
		if( libcerror_error_backtrace_sprint(
		     error,
		     error_string,
		     PYPFF_ERROR_STRING_SIZE ) == -1 )
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve sub message: %d.",
			 function,
			 sub_message_index );
		}
		else
                {
			PyErr_Format(
			 PyExc_IOError,
			 "%s: unable to retrieve sub message: %d.\n%s",
			 function,
			 sub_message_index,
			 error_string );
		}
		libcerror_error_free(
		 &error );

		goto on_error;
	}
	message_object = pypff_item_new(
	                 &pypff_message_type_object,
	                 sub_message,
	                 pypff_item->file_object );

	if( message_object == NULL )
	{
		PyErr_Format(
		 PyExc_MemoryError,
		 "%s: unable to create message object.",
		 function );

		goto on_error;
	}
	return( message_object );

on_error:
	if( sub_message != NULL )
	{
		libpff_item_free(
		 &sub_message,
		 NULL );
	}
	return( NULL );
}


