/*
 * Record entry functions
 *
 * Copyright (c) 2008-2013, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <common.h>
#include <byte_stream.h>
#include <memory.h>
#include <types.h>

#include "libpff_descriptor_data_stream.h"
#include "libpff_libcerror.h"
#include "libpff_libuna.h"
#include "libpff_mapi.h"
#include "libpff_record_entry.h"

/* Function to determine if there are zero bytes in a string
 * Trailing zero bytes not included
 * Returns 1 if the buffer contains zero bytes, 0 if not or -1 on error
 */
int libpff_record_entry_string_contains_zero_bytes(
     uint8_t *buffer,
     size_t buffer_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_string_contains_zero_bytes";
	size_t buffer_iterator  = 0;
	uint8_t zero_byte_found = 0;

	if( buffer == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid buffer.",
		 function );

		return( -1 );
	}
	if( buffer_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid buffer size value exceeds maximum.",
		 function );

		return( -1 );
	}
	for( buffer_iterator = 0;
	     buffer_iterator < buffer_size;
	     buffer_iterator++ )
	{
		if( zero_byte_found == 0 )
		{
			if( buffer[ buffer_iterator ] == 0 )
			{
				zero_byte_found = 1;
			}
		}
		else
		{
			if( buffer[ buffer_iterator ] != 0 )
			{
				return( 1 );
			}
		}
	}
	return( 0 );
}

/* Creates a record entry
 * Make sure the value record_entry is referencing, is set to NULL
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_initialize(
     libpff_record_entry_t **record_entry,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_initialize";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( *record_entry != NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_ALREADY_SET,
		 "%s: invalid record entry value already set.",
		 function );

		return( -1 );
	}
	*record_entry = memory_allocate_structure(
	                 libpff_record_entry_t );

	if( *record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
		 "%s: unable to create record entry.",
		 function );

		goto on_error;
	}
	if( memory_set(
	     *record_entry,
	     0,
	     sizeof( libpff_record_entry_t ) ) == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_SET_FAILED,
		 "%s: unable to clear record entry.",
		 function );

		goto on_error;
	}
	return( 1 );

on_error:
	if( *record_entry != NULL )
	{
		memory_free(
		 *record_entry );

		*record_entry = NULL;
	}
	return( -1 );
}

/* Frees a record entry
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_free(
     libpff_record_entry_t **record_entry,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_free";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( *record_entry != NULL )
	{
		if( ( *record_entry )->value_data != NULL )
		{
			memory_free(
			 ( *record_entry )->value_data );
		}
		memory_free(
		 *record_entry );

		*record_entry = NULL;
	}
	return( 1 );
}

/* Clones the record entry
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_clone(
     libpff_record_entry_t **destination_record_entry,
     libpff_record_entry_t *source_record_entry,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_clone";

	if( destination_record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( *destination_record_entry != NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_ALREADY_SET,
		 "%s: invalid destination record entry value already set.",
		 function );

		return( -1 );
	}
	if( source_record_entry == NULL )
	{
		*destination_record_entry = NULL;

		return( 1 );
	}
	if( libpff_record_entry_initialize(
	     destination_record_entry,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_INITIALIZE_FAILED,
		 "%s: unable to create destination record entry.",
		 function );

		goto on_error;
	}
	if( *destination_record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: missing destination record entry.",
		 function );

		goto on_error;
	}
	if( memory_copy(
	     &( ( *destination_record_entry )->identifier ),
	     &( source_record_entry->identifier ),
	     sizeof( libpff_record_entry_identifier_t ) ) == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_COPY_FAILED,
		 "%s: unable to copy identifier.",
		 function );

		goto on_error;
	}
	if( source_record_entry->value_data != NULL )
	{
		( *destination_record_entry )->value_data = (uint8_t *) memory_allocate(
		                                                         source_record_entry->value_data_size );

		if( ( *destination_record_entry )->value_data == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
			 "%s: unable to create value data.",
			 function );

			goto on_error;
		}
		if( memory_copy(
		     ( *destination_record_entry )->value_data,
		     source_record_entry->value_data,
		     source_record_entry->value_data_size ) == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_COPY_FAILED,
			 "%s: unable to copy value data.",
			 function );

			goto on_error;
		}
		( *destination_record_entry )->value_data_size = source_record_entry->value_data_size;
	}
	( *destination_record_entry )->name_to_id_map_entry = source_record_entry->name_to_id_map_entry;
	( *destination_record_entry )->flags                = source_record_entry->flags;

	return( 1 );

on_error:
	if( *destination_record_entry != NULL )
	{
		libpff_record_entry_free(
		 destination_record_entry,
		 NULL );
	}
	return( -1 );
}

/* Retrieves the value data size
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_get_value_data_size(
     libpff_record_entry_t *record_entry,
     size_t *value_data_size,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_get_value_data_size";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( value_data_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value data size.",
		 function );

		return( -1 );
	}
	*value_data_size = record_entry->value_data_size;

	return( 1 );
}

/* Sets the value data in the record entry
 * The function makes a local copy
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_set_value_data(
     libpff_record_entry_t *record_entry,
     const uint8_t *value_data,
     size_t value_data_size,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_set_value_data";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data != NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_ALREADY_SET,
		 "%s: invalid record entry - value data already set.",
		 function );

		return( -1 );
	}
	if( value_data_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid value data size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( value_data_size > 0 )
	{
		if( value_data == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
			 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
			 "%s: invalid value data.",
			 function );

			return( -1 );
		}
		record_entry->value_data = (uint8_t *) memory_allocate(
		                                        value_data_size );

		if( record_entry->value_data == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
			 "%s: unable to create value data.",
			 function );

			goto on_error;
		}
		if( memory_copy(
		     record_entry->value_data,
		     value_data,
		     value_data_size ) == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_COPY_FAILED,
			 "%s: unable to copy value data.",
			 function );

			goto on_error;
		}
		record_entry->value_data_size = value_data_size;
	}
	return( 1 );

on_error:
	if( record_entry->value_data != NULL )
	{
		memory_free(
		 record_entry->value_data );

		record_entry->value_data = NULL;
	}
	record_entry->value_data_size = 0;

	return( -1 );
}

/* Sets the value data from the list in the record entry
 * The function makes a local copy
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_set_value_data_from_list(
     libpff_record_entry_t *record_entry,
     libbfio_handle_t *file_io_handle,
     libfdata_list_t *value_data_list,
     libfcache_cache_t *value_data_cache,
     libcerror_error_t **error )
{
	libfdata_stream_t *value_data_stream = NULL;
	static char *function                = "libpff_record_entry_set_value_data_from_list";

	if( libpff_descriptor_data_stream_initialize(
	     &value_data_stream,
	     value_data_list,
	     value_data_cache,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_INITIALIZE_FAILED,
		 "%s: unable to create descriptor data stream.",
		 function );

		goto on_error;
	}
	if( libpff_record_entry_set_value_data_from_stream(
	     record_entry,
	     file_io_handle,
	     value_data_stream,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set value data from stream.",
		 function );

		goto on_error;
	}
	if( libfdata_stream_free(
	     &value_data_stream,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_FINALIZE_FAILED,
		 "%s: unable to free descriptor data stream.",
		 function );

		goto on_error;
	}
	return( 1 );

on_error:
	if( value_data_stream != NULL )
	{
		libfdata_stream_free(
		 &value_data_stream,
		 NULL );
	}
	return( -1 );
}

/* Sets the value data from the stream in the record entry
 * The function makes a local copy
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_set_value_data_from_stream(
     libpff_record_entry_t *record_entry,
     libbfio_handle_t *file_io_handle,
     libfdata_stream_t *value_data_stream,
     libcerror_error_t **error )
{
	static char *function    = "libpff_record_entry_set_value_data_from_stream";
	size64_t value_data_size = 0;
	ssize_t read_count       = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data != NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_ALREADY_SET,
		 "%s: invalid record entry - value data already set.",
		 function );

		return( -1 );
	}
	if( libfdata_stream_get_size(
	     value_data_stream,
	     &value_data_size,
	     error ) != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
		 "%s: unable to retrieve value data stream size.",
		 function );

		return( -1 );
	}
	if( value_data_size > (size64_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid value data size value out of bounds.",
		 function );

		return( -1 );
	}
	if( value_data_size > 0 )
	{
		if( libfdata_stream_seek_offset(
		     value_data_stream,
		     0,
		     SEEK_SET,
		     error ) == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_IO,
			 LIBCERROR_IO_ERROR_SEEK_FAILED,
			 "%s: unable to seek offset: 0 in value data stream.",
			 function );

			goto on_error;
		}
		record_entry->value_data = (uint8_t *) memory_allocate(
		                                        (size_t) value_data_size );

		if( record_entry->value_data == NULL )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_MEMORY,
			 LIBCERROR_MEMORY_ERROR_INSUFFICIENT,
			 "%s: unable to create value data.",
			 function );

			goto on_error;
		}
		record_entry->value_data_size = (size_t) value_data_size;

		read_count = libfdata_stream_read_buffer(
			      value_data_stream,
			      (intptr_t *) file_io_handle,
			      record_entry->value_data,
			      record_entry->value_data_size,
			      0,
			      error );

		if( read_count != (ssize_t) record_entry->value_data_size )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_IO,
			 LIBCERROR_IO_ERROR_READ_FAILED,
			 "%s: unable to read buffer from value data stream.",
			 function );

			goto on_error;
		}
	}
	return( 1 );

on_error:
	if( record_entry->value_data != NULL )
	{
		memory_free(
		 record_entry->value_data );

		record_entry->value_data = NULL;
	}
	record_entry->value_data_size = 0;

	return( -1 );
}

/* Copies the value data
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_value_data(
     libpff_record_entry_t *record_entry,
     uint8_t *value_data,
     size_t value_data_size,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_value_data";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value data.",
		 function );

		return( -1 );
	}
	if( value_data_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid value data size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( value_data_size < record_entry->value_data_size )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_TOO_SMALL,
		 "%s: invalid value data value too small.",
		 function );

		return( -1 );
	}
	if( memory_copy(
	     value_data,
	     record_entry->value_data,
	     record_entry->value_data_size ) == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_COPY_FAILED,
		 "%s: unable to copy value data.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Reads value data from the current offset into a buffer
 * Returns the number of bytes read or -1 on error
 */
ssize_t libpff_record_entry_read_buffer(
         libpff_record_entry_t *record_entry,
         uint8_t *buffer,
         size_t buffer_size,
         libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_read_buffer";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data_offset < 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid record entry - value data offset value out of bounds.",
		 function );

		return( -1 );
	}
	if( buffer == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid buffer.",
		 function );

		return( -1 );
	}
	if( buffer_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid buffer size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_offset >= (off64_t) record_entry->value_data_size ) )
	{
		return( 0 );
	}
	if( ( buffer_size > record_entry->value_data_size )
	 || ( ( (size_t) record_entry->value_data_offset + buffer_size ) > record_entry->value_data_size ) )
	{
		buffer_size = (size_t) ( record_entry->value_data_size - record_entry->value_data_offset );
	}
	if( memory_copy(
	     buffer,
	     &( record_entry->value_data[ record_entry->value_data_offset ] ),
	     buffer_size ) == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_MEMORY,
		 LIBCERROR_MEMORY_ERROR_COPY_FAILED,
		 "%s: unable to copy value data to buffer.",
		 function );

		return( -1 );
	}
	record_entry->value_data_offset += buffer_size;

	return( (ssize_t) buffer_size );
}

/* Seeks a certain offset of the value data
 * Returns the offset if seek is successful or -1 on error
 */
off64_t libpff_record_entry_seek_offset(
         libpff_record_entry_t *record_entry,
         off64_t offset,
         int whence,
         libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_seek_offset";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data_offset < 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: invalid record entry - value data offset value out of bounds.",
		 function );

		return( -1 );
	}
	if( ( whence != SEEK_CUR )
	 && ( whence != SEEK_END )
	 && ( whence != SEEK_SET ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported whence.",
		 function );

		return( -1 );
	}
	if( whence == SEEK_CUR )
	{
		offset += record_entry->value_data_offset;
	}
	else if( whence == SEEK_END )
	{
		offset += (off64_t) record_entry->value_data_size;
	}
#if defined( HAVE_DEBUG_OUTPUT )
	if( libcnotify_verbose != 0 )
	{
		libcnotify_printf(
		 "%s: seeking offset: %" PRIi64 ".\n",
		 function,
		 offset );
	}
#endif
	if( offset < 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_OUT_OF_BOUNDS,
		 "%s: offset value out of bounds.",
		 function );

		return( -1 );
	}
	record_entry->value_data_offset = offset;

	return( offset );
}

/* Copies the value data to a boolean value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_boolean(
     libpff_record_entry_t *record_entry,
     uint8_t *value_boolean,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_to_boolean";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_boolean == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value boolean.",
		 function );

		return( -1 );
	}
	/* The value data size of a boolean value is 1
	 */
	if( record_entry->value_data_size != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	*value_boolean = record_entry->value_data[ 0 ];

	return( 1 );
}

/* Copies the value data to a 16-bit value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_16bit(
     libpff_record_entry_t *record_entry,
     uint16_t *value_16bit,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_to_16bit";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_16bit == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value 16-bit.",
		 function );

		return( -1 );
	}
	/* The value data size of a 16-bit value is 2
	 */
	if( record_entry->value_data_size != 2 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	byte_stream_copy_to_uint16_little_endian(
	 record_entry->value_data,
	 *value_16bit );

	return( 1 );
}

/* Copies the value data to a 32-bit value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_32bit(
     libpff_record_entry_t *record_entry,
     uint32_t *value_32bit,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_to_32bit";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_32bit == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value 32-bit.",
		 function );

		return( -1 );
	}
	/* The value data size of a 32-bit value is 4
	 */
	if( record_entry->value_data_size != 4 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	byte_stream_copy_to_uint32_little_endian(
	 record_entry->value_data,
	 *value_32bit );

	return( 1 );
}

/* Copies the value data to a 64-bit value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_64bit(
     libpff_record_entry_t *record_entry,
     uint64_t *value_64bit,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_to_64bit";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_64bit == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value 64-bit.",
		 function );

		return( -1 );
	}
	/* The value data size of a 64-bit value is 8
	 */
	if( record_entry->value_data_size != 8 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	byte_stream_copy_to_uint64_little_endian(
	 record_entry->value_data,
	 *value_64bit );

	return( 1 );
}

/* Copies the value data to a size value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_size(
     libpff_record_entry_t *record_entry,
     size_t *value_size,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_to_size";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value size.",
		 function );

		return( -1 );
	}
	/* The value data size of a size value is 4 or 8
	 */
	if( ( record_entry->value_data_size != 4 )
	 && ( record_entry->value_data_size != 8 ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data_size == 4 )
	{
		byte_stream_copy_to_uint32_little_endian(
		 record_entry->value_data,
		 *value_size );
	}
	else if( record_entry->value_data_size == 8 )
	{
		byte_stream_copy_to_uint64_little_endian(
		 record_entry->value_data,
		 *value_size );
	}
	return( 1 );
}

/* Copies the value data to a size value
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_floating_point(
     libpff_record_entry_t *record_entry,
     double *value_floating_point,
     libcerror_error_t **error )
{
	byte_stream_float64_t value_double;
	byte_stream_float32_t value_float;

	static char *function = "libpff_record_entry_copy_to_floating_point";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( value_floating_point == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid value floating point.",
		 function );

		return( -1 );
	}
	/* The value data size of a floating point value is 4 or 8
	 */
	if( ( record_entry->value_data_size != 4 )
	 && ( record_entry->value_data_size != 8 ) )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data_size == 4 )
	{
		byte_stream_copy_to_uint32_little_endian(
		 record_entry->value_data,
		 value_float.integer );

		*value_floating_point = (double) value_float.floating_point;
	}
	else if( record_entry->value_data_size == 8 )
	{
		byte_stream_copy_to_uint64_little_endian(
		 record_entry->value_data,
		 value_double.integer );

		*value_floating_point = value_double.floating_point;
	}
	return( 1 );
}

/* Determines the UTF-8 string size from the value data
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_get_utf8_string_size(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     size_t *utf8_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_get_utf8_string_size";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf8_string_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-8 string size.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		*utf8_string_size = 0;

		return( 1 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf8_string_size_from_utf16_stream(
		          record_entry->value_data,
		          record_entry->value_data_size,
		          LIBUNA_ENDIAN_LITTLE,
		          utf8_string_size,
		          error );
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf8_string_size_from_utf7_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  utf8_string_size,
			  error );
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf8_string_size_from_utf8_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  utf8_string_size,
			  error );
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf8_string_size_from_byte_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  (int) ascii_codepage,
			  utf8_string_size,
			  error );
	}
	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
		 "%s: unable to determine UTF-8 string size.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Converts the value data into an UTF-8 string
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_utf8_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     uint8_t *utf8_string,
     size_t utf8_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_copy_to_utf8_string";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf8_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-8 string.",
		 function );

		return( -1 );
	}
	if( utf8_string_size == 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_ZERO_OR_LESS,
		 "%s: invalid UTF-8 string size value zero or less.",
		 function );

		return( -1 );
	}
	if( utf8_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-8 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		utf8_string[ 0 ] = 0;

		return( 1 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf8_string_copy_from_utf16_stream(
		          utf8_string,
		          utf8_string_size,
		          record_entry->value_data,
		          record_entry->value_data_size,
		          LIBUNA_ENDIAN_LITTLE,
		          error );
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf8_string_copy_from_utf7_stream(
		          utf8_string,
		          utf8_string_size,
		          record_entry->value_data,
		          record_entry->value_data_size,
		          error );
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf8_string_copy_from_utf8_stream(
		          utf8_string,
		          utf8_string_size,
		          record_entry->value_data,
		          record_entry->value_data_size,
		          error );
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf8_string_copy_from_byte_stream(
		          utf8_string,
		          utf8_string_size,
		          record_entry->value_data,
		          record_entry->value_data_size,
		          (int) ascii_codepage,
		          error );
	}
	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set UTF-8 string.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Compares the value data with an UTF-8 string
 * Returns 1 if the strings are equal, 0 if not or -1 on error
 */
int libpff_record_entry_compare_with_utf8_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     const uint8_t *utf8_string,
     size_t utf8_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_compare_with_utf8_string";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf8_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-8 string.",
		 function );

		return( -1 );
	}
	if( utf8_string_size == 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_ZERO_OR_LESS,
		 "%s: invalid UTF-8 string size value zero or less.",
		 function );

		return( -1 );
	}
	if( utf8_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-8 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		return( 0 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf8_string_compare_with_utf16_stream(
			  utf8_string,
			  utf8_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  LIBUNA_ENDIAN_LITTLE,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-8 string with UTF-16 stream.",
			 function );

			return( -1 );
		}
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf8_string_compare_with_utf7_stream(
			  utf8_string,
			  utf8_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-8 string with UTF-7 stream.",
			 function );

			return( -1 );
		}
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf8_string_compare_with_utf8_stream(
			  utf8_string,
			  utf8_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-8 string with UTF-8 stream.",
			 function );

			return( -1 );
		}
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf8_string_compare_with_byte_stream(
			  utf8_string,
			  utf8_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  (int) ascii_codepage,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-8 string with byte stream.",
			 function );

			return( -1 );
		}
	}
	return( result );
}

/* Determines the UTF-16 string size from the value data
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_get_utf16_string_size(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     size_t *utf16_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_get_utf16_string_size";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf16_string_size == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-16 string size.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		*utf16_string_size = 0;

		return( 1 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf16_string_size_from_utf16_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
		          LIBUNA_ENDIAN_LITTLE,
		          utf16_string_size,
		          error );
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf16_string_size_from_utf7_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
		          utf16_string_size,
		          error );
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf16_string_size_from_utf8(
			  record_entry->value_data,
			  record_entry->value_data_size,
		          utf16_string_size,
		          error );
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf16_string_size_from_byte_stream(
			  record_entry->value_data,
			  record_entry->value_data_size,
		          (int) ascii_codepage,
		          utf16_string_size,
		          error );
	}
	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
		 "%s: unable to determine UTF-16 string size.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Converts the value data into an UTF-16 string
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_to_utf16_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     uint16_t *utf16_string,
     size_t utf16_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_copy_to_utf16_string";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf16_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-16 string.",
		 function );

		return( -1 );
	}
	if( utf16_string_size == 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_ZERO_OR_LESS,
		 "%s: invalid UTF-16 string size value zero or less.",
		 function );

		return( -1 );
	}
	if( utf16_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-16 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		utf16_string[ 0 ] = 0;

		return( 1 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf16_string_copy_from_utf16_stream(
		          utf16_string,
		          utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
		          LIBUNA_ENDIAN_LITTLE,
		          error );
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf16_string_copy_from_utf7_stream(
		          utf16_string,
		          utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
		          error );
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf16_string_copy_from_utf8(
		          utf16_string,
		          utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
		          error );
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf16_string_copy_from_byte_stream(
		          utf16_string,
		          utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
		          (int) ascii_codepage,
		          error );
	}
	if( result != 1 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_SET_FAILED,
		 "%s: unable to set UTF-16 string.",
		 function );

		return( -1 );
	}
	return( 1 );
}

/* Compares the value data with an UTF-16 string
 * Returns 1 if the strings are equal, 0 if not or -1 on error
 */
int libpff_record_entry_compare_with_utf16_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     const uint16_t *utf16_string,
     size_t utf16_string_size,
     libcerror_error_t **error )
{
	static char *function   = "libpff_record_entry_compare_with_utf16_string";
	uint8_t is_ascii_string = 0;
	int result              = 0;

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( utf16_string == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid UTF-16 string.",
		 function );

		return( -1 );
	}
	if( utf16_string_size == 0 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_ZERO_OR_LESS,
		 "%s: invalid UTF-16 string size value zero or less.",
		 function );

		return( -1 );
	}
	if( utf16_string_size > (size_t) SSIZE_MAX )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_VALUE_EXCEEDS_MAXIMUM,
		 "%s: invalid UTF-16 string size value exceeds maximum.",
		 function );

		return( -1 );
	}
	if( ( record_entry->value_data == NULL )
	 || ( record_entry->value_data_size == 0 ) )
	{
		return( 0 );
	}
	if( record_entry->identifier.value_type == LIBPFF_VALUE_TYPE_STRING_ASCII )
	{
		is_ascii_string = 1;
	}
	/* Codepage 1200 represents Unicode
	 * If the codepage is 1200 find out if the string is encoded in UTF-8 or UTF-16 little-endian
	 */
	if( ( is_ascii_string != 0 )
	 && ( ascii_codepage == 1200 ) )
	{
		result = libpff_record_entry_string_contains_zero_bytes(
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GET_FAILED,
			 "%s: unable to determine if value data contains zero bytes.",
			 function );

			return( -1 );
		}
		else if( result != 0 )
		{
			is_ascii_string = 0;
		}
	}
	/* String is in UTF-16 little-endian
	 */
	if( is_ascii_string == 0 )
	{
		result = libuna_utf16_string_compare_with_utf16_stream(
		          utf16_string,
		          utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
		          LIBUNA_ENDIAN_LITTLE,
		          error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-16 string with UTF-16 stream.",
			 function );

			return( -1 );
		}
	}
	/* Codepage 65000 represents UTF-7
	 */
	else if( ascii_codepage == 65000 )
	{
		result = libuna_utf16_string_compare_with_utf7_stream(
			  utf16_string,
			  utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-16 string with UTF-7 stream.",
			 function );

			return( -1 );
		}
	}
	/* Codepage 1200 or 65001 represents UTF-8
	 */
	else if( ( ascii_codepage == 1200 )
	      || ( ascii_codepage == 65001 ) )
	{
		result = libuna_utf16_string_compare_with_utf8_stream(
			  utf16_string,
			  utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-16 string with UTF-8 stream.",
			 function );

			return( -1 );
		}
	}
	else
	{
		/* TODO currently libuna uses the same numeric values for the codepages as PFF
		 * add a mapping function if this implementation changes
		 */
		result = libuna_utf16_string_compare_with_byte_stream(
			  utf16_string,
			  utf16_string_size,
			  record_entry->value_data,
			  record_entry->value_data_size,
			  (int) ascii_codepage,
			  error );

		if( result == -1 )
		{
			libcerror_error_set(
			 error,
			 LIBCERROR_ERROR_DOMAIN_RUNTIME,
			 LIBCERROR_RUNTIME_ERROR_GENERIC,
			 "%s: unable to compare UTF-16 string with byte stream.",
			 function );

			return( -1 );
		}
	}
	return( result );
}

/* Copies the value data to an object identifier
 * Returns 1 if successful or -1 on error
 */
int libpff_record_entry_copy_object_identifier(
     libpff_record_entry_t *record_entry,
     uint32_t *object_identifier,
     libcerror_error_t **error )
{
	static char *function = "libpff_record_entry_copy_object_identifier";

	if( record_entry == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid record entry.",
		 function );

		return( -1 );
	}
	if( record_entry->identifier.value_type != LIBPFF_VALUE_TYPE_OBJECT )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value type.",
		 function );

		return( -1 );
	}
	if( record_entry->value_data == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_VALUE_MISSING,
		 "%s: invalid record entry - missing value data.",
		 function );

		return( -1 );
	}
	if( object_identifier == NULL )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_ARGUMENTS,
		 LIBCERROR_ARGUMENT_ERROR_INVALID_VALUE,
		 "%s: invalid object identifier.",
		 function );

		return( -1 );
	}
	/* The value data size of an object is 8
	 * where the first 4 bytes are the identifier
	 */
	if( record_entry->value_data_size != 8 )
	{
		libcerror_error_set(
		 error,
		 LIBCERROR_ERROR_DOMAIN_RUNTIME,
		 LIBCERROR_RUNTIME_ERROR_UNSUPPORTED_VALUE,
		 "%s: unsupported value data size.",
		 function );

		return( -1 );
	}
	byte_stream_copy_to_uint32_little_endian(
	 record_entry->value_data,
	 *object_identifier );

	return( 1 );
}

