/*
 * Record entry functions
 *
 * Copyright (c) 2008-2013, Joachim Metz <joachim.metz@gmail.com>
 *
 * Refer to AUTHORS for acknowledgements.
 *
 * This software is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#if !defined( _LIBPFF_RECORD_ENTRY_H )
#define _LIBPFF_RECORD_ENTRY_H

#include <common.h>
#include <types.h>

#include "libpff_libbfio.h"
#include "libpff_libcerror.h"
#include "libpff_libfcache.h"
#include "libpff_libfdata.h"
#include "libpff_name_to_id_map.h"
#include "libpff_types.h"

#if defined( __cplusplus )
extern "C" {
#endif

typedef struct libpff_record_entry_identifier libpff_record_entry_identifier_t;

struct libpff_record_entry_identifier
{
	/* The entry identifier format
	 */
	uint8_t format;

	/* The entry identifier
	 */
	union
	{
		/* The MAPI property based entry identifier
		 */
		struct
		{
			/* The entry type
			 */
			uint32_t entry_type;

			/* The value type
			 */
			uint32_t value_type;
		};
		/* The GUID based entry identifier
		 */
		uint8_t guid[ 16 ];

		/* The PRQ_ID_SECURE4 based entry identifier
		 */
		uint64_t secure4;
	};
};

typedef struct libpff_record_entry libpff_record_entry_t;

struct libpff_record_entry
{
	/* The record entry identifier
	 */
	libpff_record_entry_identifier_t identifier;

	/* The value data
	 */
	uint8_t *value_data;

	/* The value data size
	 */
	size_t value_data_size;

	/* The value data offset
	 */
	off64_t value_data_offset;

	/* The name to id map entry
	 */
	libpff_internal_name_to_id_map_entry_t *name_to_id_map_entry;

	/* The flags
	 */
	uint8_t flags;
};

int libpff_record_entry_string_contains_zero_bytes(
     uint8_t *buffer,
     size_t buffer_size,
     libcerror_error_t **error );

int libpff_record_entry_initialize(
     libpff_record_entry_t **record_entry,
     libcerror_error_t **error );

int libpff_record_entry_free(
     libpff_record_entry_t **record_entry,
     libcerror_error_t **error );

int libpff_record_entry_clone(
     libpff_record_entry_t **destination_record_entry,
     libpff_record_entry_t *source_record_entry,
     libcerror_error_t **error );

int libpff_record_entry_get_value_data_size(
     libpff_record_entry_t *record_entry,
     size_t *value_data_size,
     libcerror_error_t **error );

int libpff_record_entry_set_value_data(
     libpff_record_entry_t *record_entry,
     const uint8_t *value_data,
     size_t value_data_size,
     libcerror_error_t **error );

int libpff_record_entry_set_value_data_from_list(
     libpff_record_entry_t *record_entry,
     libbfio_handle_t *file_io_handle,
     libfdata_list_t *value_data_list,
     libfcache_cache_t *value_data_cache,
     libcerror_error_t **error );

int libpff_record_entry_set_value_data_from_stream(
     libpff_record_entry_t *record_entry,
     libbfio_handle_t *file_io_handle,
     libfdata_stream_t *value_data_stream,
     libcerror_error_t **error );

int libpff_record_entry_copy_value_data(
     libpff_record_entry_t *record_entry,
     uint8_t *value_data,
     size_t value_data_size,
     libcerror_error_t **error );

ssize_t libpff_record_entry_read_buffer(
         libpff_record_entry_t *record_entry,
         uint8_t *buffer,
         size_t buffer_size,
         libcerror_error_t **error );

off64_t libpff_record_entry_seek_offset(
         libpff_record_entry_t *record_entry,
         off64_t offset,
         int whence,
         libcerror_error_t **error );

int libpff_record_entry_copy_to_boolean(
     libpff_record_entry_t *record_entry,
     uint8_t *value_boolean,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_16bit(
     libpff_record_entry_t *record_entry,
     uint16_t *value_16bit,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_32bit(
     libpff_record_entry_t *record_entry,
     uint32_t *value_32bit,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_64bit(
     libpff_record_entry_t *record_entry,
     uint64_t *value_64bit,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_size(
     libpff_record_entry_t *record_entry,
     size_t *value_size,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_floating_point(
     libpff_record_entry_t *record_entry,
     double *value_floating_point,
     libcerror_error_t **error );

int libpff_record_entry_get_utf8_string_size(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     size_t *utf8_string_size,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_utf8_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     uint8_t *utf8_string,
     size_t utf8_string_size,
     libcerror_error_t **error );

int libpff_record_entry_compare_with_utf8_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     const uint8_t *utf8_string,
     size_t utf8_string_size,
     libcerror_error_t **error );

int libpff_record_entry_get_utf16_string_size(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     size_t *utf16_string_size,
     libcerror_error_t **error );

int libpff_record_entry_copy_to_utf16_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     uint16_t *utf16_string,
     size_t utf16_string_size,
     libcerror_error_t **error );

int libpff_record_entry_compare_with_utf16_string(
     libpff_record_entry_t *record_entry,
     uint32_t ascii_codepage,
     const uint16_t *utf16_string,
     size_t utf16_string_size,
     libcerror_error_t **error );

int libpff_record_entry_copy_object_identifier(
     libpff_record_entry_t *record_entry,
     uint32_t *object_identifier,
     libcerror_error_t **error );

#if defined( __cplusplus )
}
#endif

#endif

