/* Copyright (C) 2005 to 2014 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <glib.h>

#include <c++-gtk-utils/window.h>
#include <c++-gtk-utils/application.h>
#include <c++-gtk-utils/thread.h>

#ifdef CGU_USE_GTK

namespace Cgu {
class WinBase::CB {
public:
  static void delete_event(void* data) {
    static_cast<Cgu::WinBase*>(data)->on_delete_event();
  }
};
} // namespace Cgu

// the GObject callback function with both C linkage
// specification and internal linkage
extern "C" {
  static gboolean cgu_winbase_delete_event(GtkWidget*,
					   GdkEvent*,
					   void* data) {
    // functions with C linkage specification cannot propagate C++
    // exceptions, so execute within a try block and deal with any
    // exceptions internally

    // provide a CancelBlock here to make this function NPTL friendly,
    // as we have a catch-all without rethrowing
    Cgu::Thread::CancelBlock b;
    try {
      Cgu::WinBase::CB::delete_event(data);
    }
    catch (...) {
      g_critical("Exception thrown in cgu_winbase_delete_event()\n");
    }
    return true; // returning true prevents destroy sig being emitted we call
                 // gtk_object_destroy() ourselves in the WinBase destructor
  }
} // extern "C"

namespace Cgu {

WinBase::WinBase(const char* caption, GdkPixbuf* icon_p, bool modal,
		 GtkWindow* parent_p_, GtkWindow* window_p):
			     in_exec_loop(false), is_modal(modal),
			     close_guard(false), parent_p(parent_p_) {

#if GTK_CHECK_VERSION(2,99,0)
  app_p = 0;
#endif

  if (window_p) g_window_p = window_p;
  else {
    g_window_p = (GtkWindow*)gtk_window_new(GTK_WINDOW_TOPLEVEL);
  }
  
  if (caption) gtk_window_set_title(g_window_p, caption);

  if (is_modal) {
    gtk_window_set_modal(g_window_p, true);
    if (parent_p) {
      gtk_window_set_transient_for(g_window_p, parent_p);
      gtk_widget_set_sensitive(GTK_WIDGET(parent_p), false);
    }
  }

  g_signal_connect(G_OBJECT(g_window_p), "delete_event",
		   G_CALLBACK(cgu_winbase_delete_event), this);

  if (icon_p) gtk_window_set_icon(g_window_p, icon_p);
}

WinBase::~WinBase() {
#if GTK_CHECK_VERSION(2,99,0)
  if (app_p) app_p->remove(this);
#endif

  gtk_widget_destroy(GTK_WIDGET(g_window_p));
  // call gtk_main_quit() if we have previously called WinBase::exec() on this
  // object and we have not reached this destructor via a call to WinBase::close()
  // (perhaps a parent window has called WinBase::close() or gtk_main_quit() via
  // one of its callbacks so hoping to destroy itself by going out of scope but
  // has caused this WinBase object to unblock instead - this further call to
  // gtk_main_quit() will balance up the recursive calls to gtk_main() to ensure
  // that both the parent and this object are unblocked and so destroyed)
  if (in_exec_loop) gtk_main_quit();
}

int WinBase::get_exec_val() const {
  return 0;
}

void WinBase::close() {
  // protect against a case where a user has connected a GtkDialog
  // object to the response signal and called this method in the
  // response signal handler, but has forgotten to override
  // on_delete_event() to prevent a double call here.  This guard will
  // in practice only have an effect (and be necessary) if exec() has
  // been called - otherwise the first call to close() from the
  // response event will delete this WinBase object and destroy the
  // GtkWindow object, so any subsequent delete event will not be
  // delivered by GTK+
  if (close_guard) return;
  close_guard = true;

  if (is_modal && parent_p) gtk_widget_set_sensitive(GTK_WIDGET(parent_p), true);
  gtk_widget_hide(GTK_WIDGET(g_window_p));
  if (in_exec_loop) {
    in_exec_loop = false;
    gtk_main_quit();
  }
  // if we have not called exec(), then the dialog is self-owning
  // and it is safe and necessary to call 'delete this'
  else delete this;
}

void WinBase::on_delete_event() {
  close();
}

int WinBase::exec() {
#if GTK_CHECK_VERSION(2,99,0)
  if (app_p) {
    g_critical("Cgu::WinBase::exec() called in relation to a WinBase "
	       "object added to a Cgu::Application object\n");
    return -1;
  }
#endif

  in_exec_loop = true;
  gtk_main();
  return get_exec_val();
}

} // namespace Cgu

#endif // CGU_USE_GTK
