#! /bin/bash

# /usr/local/sbin/mktwpol.sh
# https://sourceforge.net/projects/mktwpol

[[ "${EUID}" != "0" ]]		&& echo " Got root?"		&& exit
[[ ! "${BASH}" =~ "bash" ]]	&& echo " ${0##*/} needs bash"	&& exit

VERSION=1.0.1		# 15 DEC 2018

# Copyright (C) 2013-2018 Chuck Seyboldt <c.cboldt@gmail.com>
# This work is licensed under a Creative Commons Attribution-ShareAlike 3.0 Unported License
# http://creativecommons.org/licenses/by-sa/3.0/

# A Gentoo-oriented Tripwire Policy Text Generator
# Creates plain-text tripwire policy from lists of package names
#
# - Optional mktwpol.cfg (or other) script configuration file can be used to:
#	- set default command-line switches
#	- automate passphrase response to tripwire prompts
#	- substitute, augment, or modify package lists and file lists
#
# See mktwpol-generic.sh for script that is adaptable to other package managers
# https://sourceforge.net/projects/mktwpol

###########  Default Package Lists and RuleName Definitions

#  The default mktwpol.sh RULENAME[] and related variables are
#  provided in a separate file.
#  The default name and location is /etc/tripwire/mktwpol*.rules

###########  Subroutines for Generating Policy Text Output
#################################################################

# "select_policy" routine runs each filename through a gauntlet.
# $Filetype assignment is based on which attribute matched LAST.

# Bin, Lib and Kernel have the same tripwire policy template
# Config and RootFile have the same tripwire policy template
# separated for future differentiation

select_policy ()
{
						   Filetype=Config	# Default
  [[ $targetfile =~ ^/etc/ ]]			&& Filetype=Config
  [[ $targetfile =~ /log/ ]]			&& Filetype=Log
  [ -x $targetfile ]				&& Filetype=Bin
  [[ $targetfile =~ /lib/ ]]			&& Filetype=Lib
  [ -b $targetfile ]				&& Filetype=Block
  [ -c $targetfile ]				&& Filetype=Char
  [ -p $targetfile ]				&& Filetype=Pipe
  [ -S $targetfile ]				&& Filetype=Socket
  [[ $targetfile =~ ^/lib/modules/ ]]		&& Filetype=Kernel
  [ -d $targetfile ]				&& Filetype=Dir
  [ -h $targetfile ]				&& Filetype=SoftLink
  [[ $targetfile =~ ^/dev/tty ]]		&& Filetype=Tty
  [[ $targetfile =~ ^/root/ ]]			&& Filetype=RootFile
  [ $targetfile == "/root" ]			&& Filetype=RootDir
  [ ! -d $targetfile ] && \
  [ -u $targetfile -o -g $targetfile ]		&& Filetype=SUID
  [[ "${EXCEPT[$i]}" =~ $targetfile ]]		&& Filetype=Special

  printf "  %-32s\t" $targetfile		# Output at least one TAB

# The below command line can be used to study a few things.
#   - count the total number of package-managed filenames to be considered
#   - count the number of considered files of any $Filetype
#   - time each step in the $Filetype gauntlet
#
#	time for i in $(grep -v ^dir /var/db/pkg/*/*/CONTENTS | \
#	cut -d" "  -f2 | grep -e bin/ -e /etc/ -e /usr/libexec/ \
#	-e /var/log/ -e [.]sh$ | sort -u); do \
#	[ -S $targetfile ] && echo $i; done  | wc
#
# For grins, compare with
#
#      find /var/db/pkg -name CONTENTS -exec cat '{}' \; | sort -u | wc


case $Filetype in
  Bin	   )	echo "-> \$(${BIN_SEC[$i]:-ReadOnly})${SEC_MOD[$i]} ;"	;;
  Config   )	echo "-> \$(${ETC_SEC[$i]:-Dynamic})${SEC_MOD[$i]} ;"	;;
  SoftLink )	echo "-> \$(SoftLink)${SFT_MOD[$i]} ;  # Softlink"	;;
  Lib	   )	echo "-> \$(${BIN_SEC[$i]:-ReadOnly})${SEC_MOD[$i]} ;"	;;
  Dir	   )	echo "-> \$(${ETC_SEC[$i]:-Dynamic})${SEC_MOD[$i]}${RECURSE[$i]} ;  # Directory"	;;
  Tty      )	echo "-> \$(Dynamic)-ipug ;"				;;
  Block	   )	echo "-> \$(Device) ;  # Block device"			;;
  Char	   )	echo "-> \$(Device) ;  # Character device"		;;
  Log	   )	echo "-> \$(${LOG_SEC[$i]:-Growing})${SEC_MOD[$i]} ;"	;;
  Kernel   )	echo "-> \$(${BIN_SEC[$i]:-ReadOnly})${SEC_MOD[$i]} ;  # Kernel"	;;
  Special  )	echo "-> ${SEC_EX[$i]} ;  # Exception"			;;
  SUID	   )	echo "-> \$(IgnoreNone)-aSH ;  # SUID or SGID"		;;
  RootFile )	echo "-> \$(${ETC_SEC[$i]:-Dynamic})${SEC_MOD[$i]} ;  # Rootfile"	;;
  Pipe	   )	echo "-> \$(Device) ;  # Pipe"				;;
  Socket   )	echo "-> \$(Device) ;  # Socket"			;;
  RootDir  )	echo "-> \$(IgnoreNone)-amcSH ;  # Catch changes to /root"	;;
esac
}

print_header ()
{
if [[ "${RULES_VERSION,,}" =~ default ]]; then
  echo "
  ############       !!!!  WARNING  !!!!      ###########
  #  Default policies may be weak on your system ...    #
  #   - the script may have overlooked critical files   #
  #   - tripwire inspection policies may be too lax     #
  #######################################################

  #  To view scope of tripwire inspection policies:
  #  \`twprint --print-dbfile -d ${DBFILE} | less\`
"
fi
echo "  # =================================================="
echo "  # Tripwire Policy File  http://bugs.gentoo.org/34662"
echo "  #     https://sourceforge.net/projects/mktwpol"
echo "  # =================================================="
echo "  # Generated by : ${0##*/} v. $VERSION"
echo "  # Generated on : `date '+%B %e, %Y at %R'`"
echo "  # Rules File   : ${RULES_VERSION}"
echo
echo '  # ============================================================================'
echo '  #'
echo '  # Tripwire, Inc. permission statements apply to some fully hardcoded'
echo "  # document, not to generated content produced by the script, ${0##*/}"
echo '  #'
echo '  # That said, here are the Tripwire, Inc. permission statements ...'
echo '  #'
echo '  # Permission is granted to make and distribute verbatim copies of this'
echo '  # document provided the copyright notice and this permission notice are'
echo '  # preserved on all copies.'
echo '  #'
echo '  # Permission is granted to copy and distribute modified versions of'
echo '  # this document under the conditions for verbatim copying, provided'
echo '  # that the entire resulting derived work is distributed under the'
echo '  # terms of a permission notice identical to this one.'
echo '  #'
echo '  # Tripwire is a registered trademark of Tripwire, Inc.'
echo '  # (in the United States and other countries)'
echo '  # All rights reserved.'
echo '  #'
echo '  # ============================================================================'
echo
echo "@@section GLOBAL"
echo "HOSTNAME=\"`uname -n`\" ;"
echo
echo '  # Standard Tripwire File Property Mask Aliases (from `man twpolicy`)'
echo '  # --------------------------------------------'
echo '  #	ReadOnly	+pinugtsdbmCM-rlacSH'
echo '  #	Dynamic		+pinugtd-srlbamcCMSH'
echo '  #	Growing		+pinugtdl-srbamcCMSH'
echo '  #	Device		+pugsdr-intlbamcCMSH'
echo '  #	IgnoreAll	-pinugtsdrlbamcCMSH'
echo '  #	IgnoreNone	+pinugtsdrbamcCMSH-l'
echo '  #'
echo '  # Non-standard File Property Mask Aliases'
echo '  # --------------------------------------------'
echo '  	Invariant	= +pugt ;	 # Permissions, UID, GID, and filetype'
echo '  	SoftLink	= +pinugtsdbmc ; # Skip checking hash values'
echo
echo '@@section FS'
echo
echo '# ================== [ Begin Hardcoded Tripwire Rules ] ======================'
echo
echo '('
echo '  rulename = "Tripwire CFG and Data",'
echo '  severity = 100'
echo ')'
echo '{'
echo "  # Tripwire file locations were transposed from ${TW_CFG}"
echo '  # Tripwire creates backup files by renaming "tw.cfg" and "tw.pol",'
echo '  # then creating new files.  The new files have new inode numbers.'
echo
echo "  ${TW_CFG}			-> \$(ReadOnly) -i ;"
echo "  ${POLFILE}			-> \$(ReadOnly) -i ;"
echo "  ${SITEKEYFILE}		-> \$(ReadOnly) ;"
echo "  ${LOCALKEYFILE}	-> \$(ReadOnly) ;"
echo "  ${DBDIR}			-> \$(Dynamic) ;"
echo
echo "  # Database backup file inode may change"
echo '  # on invocation of `tripwire --update`.'
echo
echo "  ${DBFILE}.bak	-> \$(Dynamic) -i ;"
echo
echo '  # Do not scan individual `tripwire --check` reports'
echo '  # Size of directory changes under Reiser filesystem.'
echo
echo "  ${REPORTDIR}		-> \$(ReadOnly) -mbs (recurse = 0) ;"
if [[ "${TXT_POLFILE}" != "STDOUT" ]]; then
  echo
  echo '  # Ignore deletion of plain-text policy file.'
  echo
  echo "  !${TXT_POLFILE} ;"
fi
echo '}'
echo
echo '# ================== [ Begin Generated Tripwire Rules ] ======================'
}

print_footer ()
{
echo
echo '# ============================================================================'
echo '#'
echo '# Hardcoded and generated output is based on:'
echo '#'
echo '#	- tripwire.pol.gentoo : Darren Kirby : September 5, 2006'
echo '#	  http://bugs.gentoo.org/34662'
echo '#	- Generic Policy file : V1.2.0rh : August 9, 2001'
echo '#	- FreeBSD: ports/security/tripwire/files/twpol.txt : v 1.3 : 2005/08/09'
echo '#	  http://lists.freebsd.org/pipermail/freebsd-security/2005-October/003221.html'
echo '#	- Examples found in tripwire-2.4.2-src.tar.bz2 source code distribution'
echo '#'
echo '# FreeBSD is a registered trademark of the FreeBSD Project Inc.'
echo '#'
echo '#################      END of tripwire Policy Text File      #################'
echo
}

###########  Cycle through RULENAME variable arrays

# "print_generated_rules" routine cycles each group of array variables through "print_a_rule"

print_generated_rules ()
{
for (( i = 0 ; i < ${#RULENAME[@]} ; i++ ))
do
  printf "\r Processing rule $i of $[(10#${#RULENAME[@]}-1)] tripwire policy rules" >&2
  print_a_rule
done
echo >&2
}

# -------------------------------
# "print_a_rule" routine runs once for each RULENAME[]
#   - make a title header for the tripwire rule, including optional "emailto" field
#   - print stop points (STOPLIST) aka ignorefiles, if any
#   - forward proposed package names, one-by-one, to process_packagename
#   - forward filelist array(s), if any, to process_filelist

print_a_rule ()
{
echo
echo "# ========================================================================="
echo "#  RuleName[$i]: ${RULENAME[$i]}"
echo "# -------------------------------------------------------------------------"
[ -n "${PACKAGES[$i]}" ] && printf " Packages : ${PACKAGES[$i]}\n" | eval ${FOLD} | sed s/^/"# "/
[ -n "${PACKAGES[$i]}" -a -n "${FILELIST[$i]}" ] && printf "#\n"
[ -n "${FILELIST[$i]}" ] && printf " FileNames: ${FILELIST[$i]}\n" | eval ${FOLD} | sed s/^/"# "/
echo "# ========================================================================="
echo \(
echo "  rulename = \"${RULENAME[$i]}\","
printf "  severity = ${SEVERITY[$i]:-100}"
[ -n "${EMAILTO[$i]}" ] && printf ",\n  emailto = ${EMAILTO[$i]}\n" || echo
echo \)

echo \{
[ -n "${STOPLIST[$i]}" ] && \
  printf "\n# ${RULENAME[$i]}: Ignore changes to these files/directories\n"
  case_stoplist=+\($(sed "s/ /\|/g" <<< "${STOPLIST[$i]}")\)
for targetfile in ${STOPLIST[$i]}
  do [ -e "$targetfile" ] && echo "  !$targetfile ;"
done

if [ "${SKIP_PACKAGES}" == "Yes" ]; then
   [ ! -z "${PACKAGES[$i]}" ] && \
   printf "\n# !! NOTICE !!\n# Skipping ${RULENAME[$i]} Packages !!"
else
  for package in ${PACKAGES[$i]}
    do process_packagename
  done
fi

[ -n "${FILELIST[$i]}" ] && process_filelist

# Code to process pseudo-two-dimensional arrays.
# FLST, CMTS, SCMD, SFMD, and RCRS hold numerically-specific variable names,
# Those variable names are in the style of an array name, e.g., FILELIST_2[26]
# The specific variable names are then indirectly expanded to their contents

for j in {2..100}; do
  FLST=FILELIST_$j[$i]
  CMTS=COMMENTS_$j[$i]
  SCMD=SEC_MOD_$j[$i]
  SFMD=SFT_MOD_$j[$i]
  RCRS=RECURSE_$j[$i]
  FILELIST[$i]="${!FLST}"
  COMMENTS[$i]="${!CMTS}"
  SEC_MOD[$i]="${!SCMD}"
  SFT_MOD[$i]="${!SFMD}"
  RECURSE[$i]="${!RCRS}"
  [ -z "${FILELIST[$i]}" ] && break || process_filelist
done
echo \}
}

# -------------------------------
# "process_packagename" routine is applied to every listed package name
#   - to qualify for being listed in tmp_array[], and eventually in policy text file:
#     - $targetfile contains a string from the ${INCL_PATHS} list
#     - $targetfile is not a directory or zero-size file
#   - calls filename and rule printing subroutines for each targetfile in $tmp_array[]
# Note: adding "lib/" results in MUCH longer list, and is not necessary because
#       all files in "/usr/lib" and "/usr/local/lib" directories can be watched
#       under a FILELIST[] rule, albeit not associated with a particular package
# Note: adding the test for executables also results in a MUCH longer list

# These values are set in config_mktwpol() below
# Copied here merely for reference
# INCL_PATHS=${INCL_PATHS:="*bin/* /etc/* /usr/libexec/* /var/log/* *.sh"}
# case_incl_paths=+\($(sed "s/ /\|/g" <<< "${INCL_PATHS}")\)
# grep_incl_paths=$(sed -e "s/ / -e /g" -e "s/*//g" -e "s/.sh/[.]sh$/" <<< "${INCL_PATHS}")

process_packagename ()
{
[[ "${package}" =~ "/" ]] || package="*/${package}"

# Trick to view only the first of multiple files (if exist) in a glob wildcard
# $CONTENTS always contains one element (the glob, or the name of an existing file)
# ${CONTENTS[@]} may contain the names of multiple existing files

CONTENTS=($vdb_path/${package}-[0-9]*/CONTENTS)
[ -f $CONTENTS ] || return

# The INCL_EXEC loop iterates for all non-directory lines in the CONTENTS files,
# so as to view executable files that are outside of $INCL_PATHS.
# Repetition of the `case $case_incl_paths` function for each file name is much
# less efficient than piping a list of file names to `grep -e $grep_incl_paths`

# Note: the second test for directory isn't redundant.  A few softlinks
#	(always "executable") point to directories, and "grep -v ^dir" misses those.

  if [ "${INCL_EXEC}" == "Yes" ]; then
  tmp_array=(`
  for fname in $(grep -v ^dir $vdb_path/${package}-[0-9]*/CONTENTS | cut -d" " -f2 | sort -u)
  do
    case $fname in
	/etc/hosts | $case_stoplist )	: ;;
	$case_incl_paths )		[ -s ${fname} ] && echo $fname ;;
	* )				[ ! -d ${fname} -a -x ${fname} ] && echo $fname ;;
    esac
  done`)

# The default ${tmp_array[]} building subroutine is below.
# It aims to minimize iterations through the tmp_array-building loop.
# Loop only for fnames that satisfy INCL_PATHS pattern matching
#
# The order of commands in the pipeline matters.
# `cut` must precede `grep -e grep_incl_paths` to match "-e [.]sh$"

  else
  tmp_array=(`
  for fname in $(grep -v ^dir $vdb_path/${package}-[0-9]*/CONTENTS | \
		cut -d" " -f2 | grep -e ${grep_incl_paths} | sort -u)
  do
    case $fname in
	/etc/hosts | $case_stoplist )	: ;;
	* ) 				[ -s ${fname} ] && echo $fname ;;
    esac
  done`)
  fi	# End processing CONTENTS file or files associated with $package

  if [ -n "${tmp_array}" ]; then		# Empty array results in NUL output
    printf "\n# ${RULENAME[$i]}: $package\n\n"
    for targetfile in ${tmp_array[@]}
      do  select_policy
    done
  fi
}

# -------------------------------
# "process_filelist" routine is used only for FILELIST[] arrays
#   - outputs COMMENTS[], if any, for FILELIST[] array
#   - if file exists, calls for printing filename and tripwire policy
#   - blocks creating a policy for:
#	- any file listed in a STOPLIST[] array
#	- any file named "lost+found"
#	- any directory in the /proc/* wildcard
#	- any non-zero size file in the /proc/* wildcard

process_filelist ()
{
printf "\n# ${RULENAME[$i]}: ${COMMENTS[$i]}\n\n"
for targetfile in ${FILELIST[$i]}; do
  case ${targetfile} in
    */lost+found* | $case_stoplist )		:	;;
    /proc/* )	[ ! -d "$targetfile" -a ! -s "$targetfile" ] && select_policy	;;
    * )		[ -e "$targetfile" ] && select_policy	;;
  esac
done
}

###########  Main Routine for Generating Policy Text Output
#################################################################

print_policy_text ()
{
[ "$TXT_POLFILE" == "" ] && TXT_POLFILE=STDOUT || \
echo " Plain-text Policy File : $TXT_POLFILE"		>&2
echo " Using Rules File       : $RULES_FILE_NAME"	>&2
print_header
print_generated_rules
print_footer
}

###########  Subroutines for the User Interface
#################################################################

###########  Subroutines for Informing the User

# Some messages sent to STDERR and only STDERR (>&2)
#  - to avoid appearing in redirected STDOUT output (the text policy)
#  - to avoid disappearing from view when STDOUT has been redirected

recite_ver ()
{
echo "
 This is ${0##*/} version $VERSION
 A Gentoo-oriented Tripwire Policy Text Generator
 https://sourceforge.net/projects/mktwpol
"
}

recite_help ()
{
recite_ver
echo " Usage: ${0##*/}	[-c tw_cfg_dir]  [-R rules_file] [-C config_file] \\
			[-u[-r]] [-s] [-x] [-h|-V] [debug [#]]

	-c Use tripwire configuration files found in \"tw_cfg_dir\"
	   Default tw_cfg_dir = /etc/tripwire
	-R Read RULENAME[], PACKAGES[], FILELIST[] from rules_file
	   Default rules_file = most recent mktwpol*.rules
	   in \"tw_cfg_dir\", /etc/mktwpol, /root, or /var/lib/mktwpol
	-C Change defaults (or modify RULE[] definitions) from config_file
	   Default (optional) config_file = most recent mktwpol*.cfg
	   in \"tw_cfg_dir\", /etc/mktwpol, /root, or /var/lib/mktwpol
	-u Create tripwire policy and database after producing policy text file
	-r Remove policy text file after tripwire has processed it
	-s Skip processing of PACKAGES[] arrays
	-x List executable files outside of catchall rule (not recommended)
	-h Show version and help information
	-V Show version information

 \`${0##*/}\` without \"-u\" command line parameter:
	- sends policy text to STDOUT, suitable for redirection with \">\"

 \`${0##*/} -u\`  produces no policy on STDOUT. -- ! WON'T REDIRECT ! --
	- sends policy text to a date-named file in \"tw_cfg_dir\"
	- calls \`twadmin\`  to create tw.pol from that file
	- calls \`tripwire\` to create the system database

 \`${0##*/} debug\`
	- limits output to one selected rule.  Default is RULENAME[0]
"
exit
}

###########  Subroutines for Configuring the Script

# -------------------------------
# "config_mktwpol" checks for and reads two mktwpol configuration files

config_mktwpol ()
{
TW_CFG=${TW_CFG:=/etc/tripwire/tw.cfg}

# If mktwpol.sh is called with TW_CFG set to an existing directory,
# or if the "basename" part of TW_CFG has no dot, as though being a
# config filename, then treat as though TW_CFG is a directory name,
# and use the "tw.cfg" and "twcfg.txt" tripwire naming conventions
# for the tripwire configuration files.

if [ -d ${TW_CFG} -o "${TW_CFG}" == "${TW_CFG%.*}" ]; then
  TW_CFG_DIR=${TW_CFG}
  TW_CFG=${TW_CFG_DIR}/tw.cfg
fi

# If TW_CFG has no slash "/" in it, treat it as a config file name
# and use /etc/tripwire default installation directory

if [ "${TW_CFG}" == "${TW_CFG%/*}" ]; then
  TW_CFG=/etc/tripwire/${TW_CFG}
fi

TW_CFG_DIR=${TW_CFG%/*}
TW_TXT_CFG=$TW_CFG_DIR/twcfg.txt

# TW_CFG, TW_CFG_DIR and TW_TXT_CFG are frozen at this point
# ----------------------------------------------------------
# This is protection from attempted mischief later
# $TW_CFG is a full path/filename

readonly TW_CFG TW_CFG_DIR TW_TXT_CFG

# If "-C" CONFIG_FILE is null, assign it the newest config file or a default
CONFIG_FILE=${CONFIG_FILE:="`ls -t ${TW_CFG_DIR}/mktwpol*.cfg /etc/{mktwpol,tripwire}/mktwpol*.cfg /root/mktwpol.cfg 2> /dev/null | head -1`"}
CONFIG_FILE=${CONFIG_FILE:=$TW_CFG_DIR/mktwpol.cfg}

# If CONFIG_FILE has no slash "/" in it, treat it as a config file name
# to be found in the already defined TW_CFG_DIR, whatever that is.

[ "${CONFIG_FILE}" == "${CONFIG_FILE%/*}" ] && \
     CONFIG_FILE=$TW_CFG_DIR/$CONFIG_FILE

# Warn on "non-existence" of non-default script configuration file
if [ "$CONFIG_FILE" != "$TW_CFG_DIR/mktwpol.cfg" -a ! -f "$CONFIG_FILE" ]; then
  echo "
 ${0##*/} configuration file, $CONFIG_FILE, does not exist.
 Continuing with default configuration.
"
fi

# If "-R" RULES_FILE is null, Check CONFIG_FILE for an assignment
RULES_FILE=${RULES_FILE:="`grep ^RULES_FILE $CONFIG_FILE 2> /dev/null | cut -d= -f2`"}
# If both "-R" and CONFIG_FILE assignments are null, assign newest, or default
RULES_FILE=${RULES_FILE:="`ls -t ${TW_CFG_DIR}/mktwpol*.rules /etc/{mktwpol,tripwire}/mktwpol*.rules /root/mktwpol*.rules /var/lib/mktwpol/mktwpol*.rules 2> /dev/null | head -1`"}
RULES_FILE=${RULES_FILE:=$TW_CFG_DIR/mktwpol.rules}

# If RULES_FILE has no slash "/" in it, treat it as a rules file name
# to be found in the already defined TW_CFG_DIR, whatever that is.

[ "${RULES_FILE}" == "${RULES_FILE%/*}" ] && \
     RULES_FILE=$TW_CFG_DIR/$RULES_FILE

# Save RULES_FILE_NAME for reporting.  While "-R" command assignment will have priority,
# the script will read RULES_FILE variable from CONFIG_FILE, after reading the "-R" assigned
# RULES_FILE.  The script would then report the RULES_FILE named in CONFIG_FILE even though
# the "-R" rules file has already been read into memory.  Use of `readonly RULES_FILE` works
# but throws an error message if a RULES_FILE definition appears in CONFIG_FILE.

RULES_FILE_NAME=$RULES_FILE

# Warn on "non-existence" of non-default script rules file
if [ "$RULES_FILE" != "$TW_CFG_DIR/mktwpol.rules" -a ! -f "$RULES_FILE" ]; then
  echo "
 ${0##*/} rules file, $RULES_FILE, does not exist.
 Continuing anyway.
"
fi

for config_file in $RULES_FILE $CONFIG_FILE; do

# Config file security demands

  if [ -r ${config_file} ]; then
    if [ "`stat -c %A ${config_file}`" != "-rw-------" ]; then
      echo " Exiting because of file permissions."		>&2
      echo " Run \`chmod 600 ${config_file}\` and try again."	>&2
      exit 8
    elif [ "`stat -c %U ${config_file}`" != "root" ]; then
      echo " Exiting because file owner is not root."		>&2
      echo " Run \`chown root ${config_file}\` and try again."	>&2
      exit 8
    fi

# Before reading each config_file, seek command to discard
# the previously set rules.

    grep -q ^UNSET_DEFAULT_RULES=Y  $config_file 2> /dev/null
    if [ "$?" == "0" ]; then
      unset	RULENAME SEVERITY EMAILTO \
		BIN_SEC  ETC_SEC  LOG_SEC \
		STOPLIST \
		PACKAGES \
		FILELIST FILELIST_2 FILELIST_3 FILELIST_4 FILELIST_5 \
		COMMENTS COMMENTS_2 COMMENTS_3 COMMENTS_4 COMMENTS_5 \
		SEC_MOD  SEC_MOD_2  SEC_MOD_3  SEC_MOD_4  SEC_MOD_5 \
		SFT_MOD  SFT_MOD_2  SFT_MOD_3  SFT_MOD_4  SFT_MOD_5 \
		RECURSE  RECURSE_2  RECURSE_3  RECURSE_4  RECURSE_5 \
		EXCEPT   SEC_EX
    fi
  source ${config_file}
  fi
done

if [ "${#RULENAME[@]}" == "0" ]; then
  [ "$TW_CFG_DIR" != "/etc/tripwire" ] && \
     NO_TW_RULES="$TW_CFG_DIR, /etc/tripwire" || \
     NO_TW_RULES="/etc/tripwire"
  echo "
 No RULENAME variables have been defined
 ${0##*/} absolutely needs to have at least one defined RULENAME variable.
 These are defined in a file named "mktwpol*.rules".  I looked in
 $NO_TW_RULES, /etc/mktwpol, /root, and /var/lib/mktwpol.
 If you want to have the "mktwpol*.rules" file elsewhere, point to it
 with the \"-R\" switch.

 Exiting.
" >&2
  exit 3
fi

# Default settings unless changed by script configuration file or rules file

RULES_VERSION=${RULES_VERSION:="Undefined External"}

INCL_PATHS=${INCL_PATHS:="*bin/* /etc/* /usr/libexec/* /var/log/* *.sh"}
case_incl_paths=+\($(sed "s/ /\|/g" <<< "${INCL_PATHS}")\)
grep_incl_paths=$(sed -e "s/ / -e /g" -e "s/*//g" -e "s/.sh/[.]sh$/" <<< "${INCL_PATHS}")

FOLD="${FOLD:=fmt -u}"

# NONSTD_TW_CFG is used to disambiguate some tripwire commands, if TW_CFG
# is something other than tripwire's default /etc/tripwire/tw.cfg

[ "$TW_CFG" != "/etc/tripwire/tw.cfg" ] && NONSTD_TW_CFG=" -c ${TW_CFG}"
}

test_for_dupe_rules ()
{
DUPE_PACKAGES="`echo ${PACKAGES[@]} | tr [:space:] '\n' | sort | uniq -d`"
DUPE_FILELIST="`echo ${FILELIST[@]} ${FILELIST_2[@]} ${FILELIST_3[@]} ${FILELIST_4[@]} ${FILELIST_5[@]} \
		| tr [:space:] '\n' | sort | uniq -d`"

if [ -n "${DUPE_PACKAGES}" -o -n "${DUPE_FILELIST}" ]; then
  echo " Uh Oh!  Duplicates Found!"		>&2
  echo						>&2
  [ -n "${DUPE_PACKAGES}" ] && echo " In more than one PACKAGES[]:" ${DUPE_PACKAGES}	>&2
  [ -n "${DUPE_FILELIST}" ] && echo " In more than one FILELIST[]:" ${DUPE_FILELIST}	>&2
  echo						>&2
  echo " To resolve the duplication:"		>&2
  echo " Edit $RULES_FILE or $CONFIG_FILE"	>&2
  echo " Exiting.  Goodbye."			>&2
  exit 4
fi
}

# -------------------------------
# "get_twcfg_variables" reads TRIPWIRE config file (not the script config file)
# Assuming existence of twadmin ... if tw.cfg exists, it was made by twadmin
# Filename "twcfg.txt" is hardcoded here

get_twcfg_variables ()
{
if [ -r ${TW_CFG} ]; then
  tmp_array=(`twadmin --print-cfgfile -c ${TW_CFG}`)
elif [ -r ${TW_CFG_DIR}/twcfg.txt ]; then
  tmp_array=(`cat ${TW_CFG_DIR}/twcfg.txt`)
else
  echo " ${0##*/} depends on finding a tripwire configuration file
	${TW_CFG}         is made and directly used by tripwire
	${TW_CFG_DIR}/twcfg.txt      is only used for set-up purposes
 Run \`twsetup.sh\`
 Exiting.  Goodbye.
"
  exit 3
fi

# `z+=2` incrementing shaves a few thousandths of a second
# but risks skipping over the required variable names

for (( z = 0 ; z < ${#tmp_array[@]} ; z++ ))
do
  case ${tmp_array[@]:$z:1} in
    POLFILE | DBFILE | REPORTFILE | SITEKEYFILE | LOCALKEYFILE  )
      assignment=${tmp_array[@]:$((z+1)):1}
      [ "${assignment:0:1}" == "=" ] && \
        export ${tmp_array[@]:$z:1}${assignment}
  ;;
  esac
done
REPORTDIR=${REPORTFILE%/*}	# bash shell equivalent of the `dirname` command
DBDIR=${DBFILE%/*}
DBFILE=${DBFILE/\$\(HOSTNAME\)/`uname -n`}
}

assign_query_package ()
{
vdb_path=$(portageq vdb_path 2> /dev/null)
vdb_path=${vdb_path:=/var/db/pkg/}
if [ -d $vdb_path ]; then
  return
else
  echo "
 I did not find an installed packages database at $vdb_path
 Unable to extract filenames for installed packages.
 See mktwpol-generic.sh for Linux distributions other than Gentoo.

 https://sourceforge.net/p/mktwpol/code/ci/master/tree/

 Generating a generic tripwire policy including the FILELIST[] specifications ...
" >&2
  SKIP_PACKAGES=Yes
fi
}

###########  Subroutines for Operating

# -------------------------------
mode_auto_update ()
{
TXT_POLFILE=${TXT_POLFILE:=${TW_CFG_DIR}/twpol-`date +%y%m%d-%H%M`.txt}
echo "  # $TXT_POLFILE" > $TXT_POLFILE
print_policy_text >> $TXT_POLFILE
chmod 600 $TXT_POLFILE

echo >&2
if [ -n "${SITE_PASSPHRASE}${NULL_PASSPHRASE}" ]; then
  if  [ -n "${NULL_PASSPHRASE}" ]; then
  twadmin --create-polfile $NONSTD_TW_CFG \
          --site-passphrase "${SITE_PASSPHRASE}" $TXT_POLFILE > /dev/null
  else
  twadmin --create-polfile $NONSTD_TW_CFG \
          --site-passphrase "${SITE_PASSPHRASE}" $TXT_POLFILE
  fi
else
  echo "Ready to encrypt $TXT_POLFILE ..." >&2
  twadmin --create-polfile $NONSTD_TW_CFG $TXT_POLFILE
fi
chmod 600 $POLFILE

if [ -n "${LOCAL_PASSPHRASE}${NULL_PASSPHRASE}" ]; then
  if  [ -n "${NULL_PASSPHRASE}" ]; then
  printf " *** Processing Unix File System ***"
  tripwire --init $NONSTD_TW_CFG --local-passphrase "${LOCAL_PASSPHRASE}" > /dev/null
  printf "\n The database was successfully generated.\n"
  else
  tripwire --init $NONSTD_TW_CFG --local-passphrase "${LOCAL_PASSPHRASE}"
  fi
else
  echo >&2
  echo "Ready to make encrypted tripwire database ..."	>&2
  tripwire --init $NONSTD_TW_CFG
fi

if [[ "${AUTO_RM^}" =~ "Y" ]] || [ -n "${NULL_PASSPHRASE}" ]; then
  rm -f $TXT_POLFILE
else
  echo >&2
  printf " Delete $TXT_POLFILE now? [Y/n]: "	>&2
  read -n 1 -t 15 RM_TW_POL
  if [ "${RM_TW_POL^}" != "N" ]; then
    rm -f $TXT_POLFILE
  fi
  echo >&2
fi
}

# -------------------------------
mode_debug ()
{
DEBUGME=y	# Unused variable.  Can be useful for internal debugging.
i=${1:-0}	# User can debug any single rule, default RULENAME[0]
echo
echo "  !! WARNING !!  ${0##*/} is in DEBUG Mode !!"
echo "  !! WARNING !!  Processing --ONLY-- RULENAME[${i}]"
echo
print_a_rule
echo
echo "  !! WARNING !!  ${0##*/} was in DEBUG Mode !!"
echo "  !! WARNING !!  Processed --ONLY-- RULENAME[${i}]"
exit 1
}

###########  Main Routine
#################################################################

# Shell option extglob of extended pattern matching is used to process the
# "case_stoplist" variable in process_filelist() routine and
# "case_incl_paths" variable in process_packagename()
#
# case_stoplist=	+($STOPLIST[]) with pipes in place of spaces
# case_incl_paths=	+($INCL_PATHS) with pipes in place of spaces

shopt -s extglob

# Process command line parameters

while getopts :c:C:R:ursxhV OPTION
do
  case $OPTION in
    c	) TW_CFG=$OPTARG	;;
    C	) CONFIG_FILE=$OPTARG	;;
    R	) RULES_FILE=$OPTARG	;;
    u	) AUTO_UPDATE=Yes	;;
    r	) AUTO_RM=Yes		;;
    s	) SKIP_PACKAGES=Yes	;;
    x	) INCL_EXEC=Yes		;;
    h	) config_mktwpol; recite_help		;;
    V	) recite_ver; exit	;;
    *	) config_mktwpol; recite_help		;;
  esac
done
shift $(($OPTIND - 1))

trap "echo; echo; echo ${0##*/} interrupted >&2 ; echo; exit" SIGHUP SIGINT SIGTERM

hash tripwire 2> /dev/null || \
echo "
 This script has no known function aside from tripwire.
 The header of the policy text file will have fatal flaws.
 Continuing even though tripwire was not found ...
" >&2

config_mktwpol
test_for_dupe_rules
if [ "$1" == "dump" ]; then
  echo ${PACKAGES[@]} | tr [:space:] '\n' | sort
  exit
fi
if [ "$1" == "cruft" ]; then
  PORTDIR=`portageq get_repo_path / gentoo`
  printf " \n %s packages listed in \"%s\" ...\n Absent from Portage tree (%s):\n\n" \
	`wc -w <<< ${PACKAGES[@]}` "${RULES_VERSION}" $PORTDIR
  for i in ${PACKAGES[@]}; do
    pkgdir=$i
    [[ $pkgdir =~ / ]] || pkgdir=*/$pkgdir
    stat $PORTDIR/$pkgdir -c %n > /dev/null 2>&1 || printf " $i\n"
  done
  printf "\n"
  exit
fi
get_twcfg_variables
assign_query_package

# Output Policy Text.  Debug mode has priority.
# FORCE_PRINT=Y overrides AUTO_UPDATE=Y setting in mktwpol.cfg
# FORCE_PRINT=Y used by twsetup.sh to force printing to STDOUT

if [ "$1" == "debug" ]; then
  mode_debug $2
elif [[ "${AUTO_UPDATE^}" =~ "Y" ]] && [[ ! "${FORCE_PRINT^}" =~ "Y" ]]; then
  mode_auto_update
else
  print_policy_text
fi
