/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { getClosureSafeProperty } from '../../util/property';
/**
 * Construct an injectable definition which defines how a token will be constructed by the DI
 * system, and in which injectors (if any) it will be available.
 *
 * This should be assigned to a static `ɵprov` field on a type, which will then be an
 * `InjectableType`.
 *
 * Options:
 * * `providedIn` determines which injectors will include the injectable, by either associating it
 *   with an `@NgModule` or other `InjectorType`, or by specifying that this injectable should be
 *   provided in the `'root'` injector, which will be the application-level injector in most apps.
 * * `factory` gives the zero argument function which will create an instance of the injectable.
 *   The factory can call `inject` to access the `Injector` and request injection of dependencies.
 *
 * @codeGenApi
 * @publicApi This instruction has been emitted by ViewEngine for some time and is deployed to npm.
 */
export function ɵɵdefineInjectable(opts) {
    return {
        token: opts.token,
        providedIn: opts.providedIn || null,
        factory: opts.factory,
        value: undefined,
    };
}
/**
 * @deprecated in v8, delete after v10. This API should be used only by generated code, and that
 * code should now use ɵɵdefineInjectable instead.
 * @publicApi
 */
export const defineInjectable = ɵɵdefineInjectable;
/**
 * Construct an `InjectorDef` which configures an injector.
 *
 * This should be assigned to a static injector def (`ɵinj`) field on a type, which will then be an
 * `InjectorType`.
 *
 * Options:
 *
 * * `providers`: an optional array of providers to add to the injector. Each provider must
 *   either have a factory or point to a type which has a `ɵprov` static property (the
 *   type must be an `InjectableType`).
 * * `imports`: an optional array of imports of other `InjectorType`s or `InjectorTypeWithModule`s
 *   whose providers will also be added to the injector. Locally provided types will override
 *   providers from imports.
 *
 * @codeGenApi
 */
export function ɵɵdefineInjector(options) {
    return { providers: options.providers || [], imports: options.imports || [] };
}
/**
 * Read the injectable def (`ɵprov`) for `type` in a way which is immune to accidentally reading
 * inherited value.
 *
 * @param type A type which may have its own (non-inherited) `ɵprov`.
 */
export function getInjectableDef(type) {
    return getOwnDefinition(type, NG_PROV_DEF) || getOwnDefinition(type, NG_INJECTABLE_DEF);
}
export function isInjectable(type) {
    return getInjectableDef(type) !== null;
}
/**
 * Return definition only if it is defined directly on `type` and is not inherited from a base
 * class of `type`.
 */
function getOwnDefinition(type, field) {
    return type.hasOwnProperty(field) ? type[field] : null;
}
/**
 * Read the injectable def (`ɵprov`) for `type` or read the `ɵprov` from one of its ancestors.
 *
 * @param type A type which may have `ɵprov`, via inheritance.
 *
 * @deprecated Will be removed in a future version of Angular, where an error will occur in the
 *     scenario if we find the `ɵprov` on an ancestor only.
 */
export function getInheritedInjectableDef(type) {
    const def = type && (type[NG_PROV_DEF] || type[NG_INJECTABLE_DEF]);
    if (def) {
        const typeName = getTypeName(type);
        // TODO(FW-1307): Re-add ngDevMode when closure can handle it
        // ngDevMode &&
        console.warn(`DEPRECATED: DI is instantiating a token "${typeName}" that inherits its @Injectable decorator but does not provide one itself.\n` +
            `This will become an error in a future version of Angular. Please add @Injectable() to the "${typeName}" class.`);
        return def;
    }
    else {
        return null;
    }
}
/** Gets the name of a type, accounting for some cross-browser differences. */
function getTypeName(type) {
    // `Function.prototype.name` behaves differently between IE and other browsers. In most browsers
    // it'll always return the name of the function itself, no matter how many other functions it
    // inherits from. On IE the function doesn't have its own `name` property, but it takes it from
    // the lowest level in the prototype chain. E.g. if we have `class Foo extends Parent` most
    // browsers will evaluate `Foo.name` to `Foo` while IE will return `Parent`. We work around
    // the issue by converting the function to a string and parsing its name out that way via a regex.
    if (type.hasOwnProperty('name')) {
        return type.name;
    }
    const match = ('' + type).match(/^function\s*([^\s(]+)/);
    return match === null ? '' : match[1];
}
/**
 * Read the injector def type in a way which is immune to accidentally reading inherited value.
 *
 * @param type type which may have an injector def (`ɵinj`)
 */
export function getInjectorDef(type) {
    return type && (type.hasOwnProperty(NG_INJ_DEF) || type.hasOwnProperty(NG_INJECTOR_DEF)) ?
        type[NG_INJ_DEF] :
        null;
}
export const NG_PROV_DEF = getClosureSafeProperty({ ɵprov: getClosureSafeProperty });
export const NG_INJ_DEF = getClosureSafeProperty({ ɵinj: getClosureSafeProperty });
// We need to keep these around so we can read off old defs if new defs are unavailable
export const NG_INJECTABLE_DEF = getClosureSafeProperty({ ngInjectableDef: getClosureSafeProperty });
export const NG_INJECTOR_DEF = getClosureSafeProperty({ ngInjectorDef: getClosureSafeProperty });
//# sourceMappingURL=data:application/json;base64,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