// Copyright 2023 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_OPENTELEMETRY_OPTIONS_H
#define GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_OPENTELEMETRY_OPTIONS_H

#include "google/cloud/version.h"

namespace google {
namespace cloud {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN

/**
 * Enables tracing with [OpenTelemetry]
 *
 * Setting this option enables the generation of [traces] by the client
 * library. The library uses the global [tracer provider] to generate traces.
 *
 * @par Prerequisites
 * The library must be compiled with OpenTelemetry in order for this option to
 * take effect.
 *
 * @see [The OpenTelemetry tracing quickstart][gc-quickstart] for instructions
 * on how to compile google-cloud-cpp with opentelemetry-cpp.
 *
 * @par Effect
 * Setting this option enables the generation of [traces] by the client
 * library. The library uses the global [tracer provider] to generate traces.
 *
 * To create a client that traces, this option must be provided to the
 * `Make*Connection(...)` factory functions. It does not have an effect when
 * passed to a client's constructor. If you use custom credentials, this option
 * must also be provided to the `Make*Credentials(...)` factory functions in
 * order to trace the authentication components.
 *
 * @par Exporting traces
 * Providing this option only enables the *generation* of traces. It does not
 * enable the *export* of traces.
 *
 * In order to export the traces, the application must set the global tracer
 * provider. The client library recommends using
 * `#google::cloud::otel::ConfigureBasicTracing()` to configure the global
 * tracer provider. This will send the traces generated by the client library to
 * [Cloud Trace], Google Cloud's visualizer for distributed traces.
 *
 * @see [Google Cloud C++ OpenTelemetry reference docs][gc-otel-docs].
 *
 * @par Environment variable
 * This option is controlled by the `GOOGLE_CLOUD_CPP_OPENTELEMETRY_TRACING`
 * environment variable. If the environment variable is set to a non-empty
 * value, tracing with OpenTelemetry is enabled.
 *
 * @ingroup options
 *
 * [Cloud Trace]: https://cloud.google.com/trace
 * [gc-quickstart]:
 * https://github.com/googleapis/google-cloud-cpp/tree/main/google/cloud/opentelemetry/quickstart
 * [gc-otel-docs]:
 * https://cloud.google.com/cpp/docs/reference/opentelemetry/latest
 * [opentelemetry]: https://opentelemetry.io/docs/instrumentation/cpp/
 * [tracer provider]:
 * https://opentelemetry.io/docs/concepts/signals/traces/#tracer-provider
 * [traces]:
 * https://opentelemetry.io/docs/concepts/observability-primer/#distributed-traces
 */
struct OpenTelemetryTracingOption {
  using Type = bool;
};

namespace experimental {
/// @deprecated Use google::cloud::OpenTelemetryTracingOption instead.
using OpenTelemetryTracingOption = ::google::cloud::OpenTelemetryTracingOption;
}  // namespace experimental
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace cloud
}  // namespace google

#endif  // GOOGLE_CLOUD_CPP_GOOGLE_CLOUD_OPENTELEMETRY_OPTIONS_H
